<?php
/**
 * Publish Post Group
 *
 * @package     AutomatorWP\Integrations\Facebook\Actions\Publish_Post_Group
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined('ABSPATH') ) exit;


class AutomatorWP_Facebook_Publish_Post_Group extends AutomatorWP_Integration_Action {
    
    public $integration = 'facebook';
    public $action = 'facebook_publish_post_group';

    /**
     * Store the action result
     *
     * @since 1.0.0
     *
     * @var string $result
     */
    public $result = '';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration' => $this->integration,
            'label' => __( 'Publish a post in a Facebook group', 'automatorwp-pro' ),
            'select_option' => __( 'Publish a <strong>post</strong> to a Facebook group', 'automatorwp-pro' ),
            /* translators: %1$s: Facebook group. */
            'edit_label' => sprintf( __( 'Publish a post in %1$s', 'automatorwp-pro' ), '{group}' ),
            /* translators: %1$s: Facebook group. */
            'log_label' => sprintf( __( 'Publish a post in %1$s', 'automatorwp-pro' ), '{group}' ),
            'options' => array(
                'group' => array(
                    'from' => 'group',
                    'default' => __( 'a Facebook group', 'automatorwp-pro' ),
                    'fields' => array(
                        'group' => automatorwp_utilities_ajax_selector_field( array(
                            'field'             => 'group',
                            'option_default'    => __( 'group', 'automatorwp-pro' ),
                            'option_custom'     => true,
                            'placeholder'       => __( 'Select a group', 'automatorwp-pro' ),
                            'name'              => __( 'Group:', 'automatorwp-pro' ),
                            'action_cb'         => 'automatorwp_facebook_get_groups',
                            'options_cb'        => 'automatorwp_facebook_options_cb_group',
                            'default'           => ''
                        ) ),
                        'group_custom' => automatorwp_utilities_custom_field( array(
                            'option_custom_desc'    => __( 'Group ID', 'automatorwp-pro' ),
                        ) ),
                        'message' => array(
                            'name' => __( 'Post:', 'automatorwp-pro' ),
                            'desc' => __( 'Enter the content for your post on Facebook. Please, be careful, this action might fail if executed in short intervals of time ', 'automatorwp-pro' ),
                            'type' => 'textarea',
                        ),
                    )
                )
            ),
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation){

        // Bail if not selected group
        if( empty( $action_options['group'] ) ) {
            $this->result = __( 'No page selected on the action\'s setup.', 'automatorwp-pro' );
            return;
        }

        // Bail if not message
        if( empty( $action_options['message'] ) ) {
            $this->result = __( 'No content to post on Facebook page.', 'automatorwp-pro' );
            return;
        }

        // Shorthand
        $group_id = $action_options['group'];

        $message = wpautop( $action_options['message'] );
		$tags_to_clean = array( '<p>', '</p>', '<br />', '<br/>', '<br>', '<b>', '</b>' );
		$message = str_replace( $tags_to_clean, '', $message);

        $params = automatorwp_facebook_get_request_parameters();

        // Bail if the authorization has not been setup from settings
        if( $params === false ) {
            return;
        }

        $api_version = automatorwp_facebook_api_version();
        $auth = get_option( 'automatorwp_facebook_auth' );

        $args = array(
            'body' => array(
                'message' => $message,
                'access_token'  => $auth['access_token'],
            )
        );

        // Publish the content on Facebook group
        $response = wp_remote_post('https://graph.facebook.com/' . $api_version . '/' . $group_id . '/feed', $args );

        if ( isset ( $response['response']['code'] ) && absint( $response['response']['code'] ) !== 200 ) {
			$this->result = __( 'The post could not be published in the group. Please, check your group ID', 'automatorwp-pro' );
		} else{
			$this->result = __( 'Post published on Facebook group successfully!', 'automatorwp-pro' );	
		}
        
    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Configuration notice
        add_filter( 'automatorwp_automation_ui_after_item_label', array( $this, 'configuration_notice' ), 10, 2 );

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();

    }

    /**
     * Configuration notice
     *
     * @since 1.0.0
     *
     * @param stdClass  $object     The trigger/action object
     * @param string    $item_type  The object type (trigger|action)
     */
    public function configuration_notice( $object, $item_type ) {

        // Bail if action type don't match this action
        if( $item_type !== 'action' ) {
            return;
        }

        if( $object->type !== $this->action ) {
            return;
        }

        $auth = get_option( 'automatorwp_facebook_auth' );

        // Warn user if the authorization has not been setup from settings
        if( empty ( $auth['access_token'] ) ) : ?>
            <div class="automatorwp-notice-warning" style="margin-top: 10px; margin-bottom: 0;">
                <?php echo sprintf(
                    __( 'You need to configure the <a href="%s" target="_blank">Facebook settings</a> to get this action to work.', 'automatorwp-pro' ),
                    get_admin_url() . 'admin.php?page=automatorwp_settings&tab=opt-tab-facebook'
                ); ?>
                <?php echo sprintf(
                    __( '<a href="%s" target="_blank">Documentation</a>', 'automatorwp-pro' ),
                    'https://automatorwp.com/docs/facebook/'
                ); ?>
            </div>
        <?php endif;

    }

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        $log_meta['result'] = $this->result;

        return $log_meta;

    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;

    }
    
}

new AutomatorWP_Facebook_Publish_Post_Group();