<?php
/**
 * Custom User Fields
 *
 * @package     AutomatorWP\Custom_User_Fields\Custom_Tables\Custom_User_Fields
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

/**
 * Parse query args for fields
 *
 * @since 1.0.0
 *
 * @param string $where
 * @param CT_Query $ct_query
 *
 * @return string
 */
function automatorwp_custom_user_fields_query_where( $where, $ct_query ) {

    global $ct_table;

    if( $ct_table->name !== 'automatorwp_custom_user_fields' ) {
        return $where;
    }

    // Shorthand
    $qv = $ct_query->query_vars;

    // Name
    $where .= automatorwp_custom_table_where( $qv, 'name', 'string' );

    // Meta Key
    $where .= automatorwp_custom_table_where( $qv, 'meta_key', 'string' );

    return $where;
}
add_filter( 'ct_query_where', 'automatorwp_custom_user_fields_query_where', 10, 2 );

/**
 * Define the search fields for custom fields
 *
 * @since 1.0.0
 *
 * @param array $search_fields
 *
 * @return array
 */
function automatorwp_custom_user_fields_search_fields( $search_fields ) {

    $search_fields[] = 'name';
    $search_fields[] = 'meta_key';

    return $search_fields;

}
add_filter( 'ct_query_automatorwp_custom_user_fields_search_fields', 'automatorwp_custom_user_fields_search_fields' );

/**
 * Columns for custom fields list view
 *
 * @since 1.0.0
 *
 * @param array $columns
 *
 * @return array
 */
function automatorwp_custom_user_fields_manage_custom_fields_columns( $columns = array() ) {

    $columns['name']                = __( 'Name', 'automatorwp-pro' );
    $columns['meta_key']            = __( 'Meta Key', 'automatorwp-pro' );
    $columns['shortcode']           = __( 'Shortcode', 'automatorwp-pro' );

    return $columns;
}
add_filter( 'manage_automatorwp_custom_user_fields_columns', 'automatorwp_custom_user_fields_manage_custom_fields_columns' );

/**
 * Sortable columns for custom fields list view
 *
 * @since 1.0.0
 *
 * @param array $sortable_columns
 *
 * @return array
 */
function automatorwp_custom_user_fields_manage_custom_fields_sortable_columns( $sortable_columns ) {

    $sortable_columns['name']               = array( 'name', false );
    $sortable_columns['meta_key']           = array( 'meta_key', false );

    return $sortable_columns;

}
add_filter( 'manage_automatorwp_custom_user_fields_sortable_columns', 'automatorwp_custom_user_fields_manage_custom_fields_sortable_columns' );

/**
 * Columns rendering for custom fields list view
 *
 * @since  1.0.0
 *
 * @param string $column_name
 * @param integer $object_id
 */
function automatorwp_custom_user_fields_manage_custom_fields_custom_column(  $column_name, $object_id ) {

    // Setup vars
    $custom_user_field = ct_get_object( $object_id );

    switch( $column_name ) {
        case 'name':
            ?>
            <strong><a href="<?php echo ct_get_edit_link( 'automatorwp_custom_user_fields', $custom_user_field->id ); ?>"><?php echo $custom_user_field->name; ?></a></strong>
            <?php

            break;
        case 'meta_key':
            ?>
            <?php echo $custom_user_field->meta_key; ?>
            <?php

            break;
        case 'shortcode':
            $shortcode = '[awp_cuf field="' . $custom_user_field->meta_key . '"]';
            ?>
            <input value="<?php echo esc_attr($shortcode); ?>" onclick="this.focus(); this.select();" readonly class="automatorwp-custom-user-fields-shortcode-input"/>
            <?php
            break;
    }
}
add_action( 'manage_automatorwp_custom_user_fields_custom_column', 'automatorwp_custom_user_fields_manage_custom_fields_custom_column', 10, 2 );

/**
 * Default data when creating a new item (similar to WP auto draft) see ct_insert_object()
 *
 * @since  1.0.0
 *
 * @param array $default_data
 *
 * @return array
 */
function automatorwp_custom_user_fields_default_data( $default_data = array() ) {

    $count = automatorwp_custom_user_fields_get_count();
    $count++;

    while( automatorwp_custom_user_fields_get_custom_field_by_meta_key( 'my_custom_field_' . $count ) !== null ) {
        $count++;
    }

    $default_data['name'] = __( 'My Custom Field', 'automatorwp-pro' ) . ' ' . $count;
    $default_data['meta_key'] = 'my_custom_field_' . $count;

    return $default_data;
}
add_filter( 'ct_automatorwp_custom_user_fields_default_data', 'automatorwp_custom_user_fields_default_data' );

/**
 * custom fields meta boxes
 *
 * @since  1.0.0
 */
function automatorwp_custom_user_fields_add_custom_fields_meta_boxes() {

    add_meta_box( 'automatorwp_custom_user_field_data', __( 'Data', 'automatorwp-pro' ), 'automatorwp_custom_user_fields_data_meta_box', 'automatorwp_custom_user_fields', 'normal', 'default' );

}
add_action( 'add_meta_boxes', 'automatorwp_custom_user_fields_add_custom_fields_meta_boxes' );

/**
 * Render the log data meta box
 *
 * @since  1.0.0
 *
 * @param  stdClass $object The current object
 */
function automatorwp_custom_user_fields_data_meta_box( $object = null ) {

    $custom_field_fields = automatorwp_custom_user_fields_get_custom_field_fields( $object );

    // Setup the CMB2 form
    $cmb2 = new CMB2( array(
        'id'        => 'automatorwp_custom_user_fields_form',
        'object_types' => array( 'automatorwp_custom_user_fields' ),
        'classes'   => 'automatorwp-form automatorwp-custom-user-field-form',
        'hookup'    => false,
    ), $object->id );

    // Setup the options fields
    foreach ( $custom_field_fields as $field_id => $field ) {

        $field['id'] = $field_id;

        // Add the field to the form
        $cmb2->add_field( $field );

    }

    // Render the form
    CMB2_Hookup::enqueue_cmb_css();
    CMB2_Hookup::enqueue_cmb_js();
    $cmb2->show_form();

    ?>

    <?php
}

/**
 * Get custom field fields
 *
 * @since 1.0.0
 *
 * @param stdClass $custom_user_field The custom user field object
 *
 * @return array
 */
function automatorwp_custom_user_fields_get_custom_field_fields( $custom_user_field ) {

    // Setup the custom user field field
    $fields = array(
        'name' => array(
            'name' => __( 'Name:', 'automatorwp-pro' ),
            'desc' => __( 'The field name.', 'automatorwp-pro' ),
            'type' => 'text',
        ),
        'meta_key' => array(
            'name' => __( 'Meta Key:', 'automatorwp-pro' ),
            'desc' => __( 'The user meta key.', 'automatorwp-pro' ),
            'type' => 'text',
        ),
        'shortcode' => array(
            'name' => __( 'Shortcode:', 'automatorwp-pro' ),
            'desc' => __( 'Place this shortcode anywhere to display the user custom field value.', 'automatorwp-pro' ),
            'type' => 'text',
            'attributes'    => array(
                'readonly'  => 'readonly',
                'onclick'   => 'this.focus(); this.select();'
            ),
            'default_cb'    => 'automatorwp_custom_user_fields_shortcode_field_default_cb',
            'save_field' => false, // Prevent save
        ),
    );

    /**
     * Filter to set custom fields
     *
     * @since 1.0.0
     *
     * @param array     $fields             The custom user field fields
     * @param stdClass  $custom_user_field        The custom user field object
     *
     * @return array
     */
    return apply_filters( 'automatorwp_custom_user_fields_custom_field_fields', $fields, $custom_user_field );

}

// Shortcode field default cb
function automatorwp_custom_user_fields_shortcode_field_default_cb( $field_args, $field ) {
    $object = ct_get_object( $field->object_id );

    return '[awp_cuf field="' . $object->meta_key . '"]';
}