<?php
/**
 * Functions
 *
 * @package     AutomatorWP\BuddyPress\Functions
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

/**
 * Register custom component
 *
 * @since 1.0.0
 *
 * @param array $component_names
 *
 * @return array
 */
function automatorwp_buddypress_register_notifications_component( $component_names = array() ) {

    // Force $component_names to be an array
    if ( ! is_array( $component_names ) ) {
        $component_names = array();
    }

    // Add 'automatorwp' component
    array_push( $component_names, 'automatorwp' );

    return $component_names;

}
add_filter( 'bp_notifications_get_registered_components', 'automatorwp_buddypress_register_notifications_component' );

/**
 * Options callback for select2 fields assigned to groups
 *
 * @since 1.0.0
 *
 * @param stdClass $field
 *
 * @return array
 */
function automatorwp_buddypress_options_cb_group( $field ) {

    // Setup vars
    $value = $field->escaped_value;
    $none_value = 'any';
    $none_label = __( 'any group', 'automatorwp-pro' );
    $options = automatorwp_options_cb_none_option( $field, $none_value, $none_label );

    if( ! empty( $value ) ) {
        if( ! is_array( $value ) ) {
            $value = array( $value );
        }

        foreach( $value as $group_id ) {

            // Skip option none
            if( $group_id === $none_value ) {
                continue;
            }

            $options[$group_id] = automatorwp_buddypress_get_group_title( $group_id );
        }
    }

    return $options;

}

/**
 * Get the group title
 *
 * @since 1.0.0
 *
 * @param int $group_id
 *
 * @return string|null
 */
function automatorwp_buddypress_get_group_title( $group_id ) {

    // Empty title if no ID provided
    if( absint( $group_id ) === 0 ) {
        return '';
    }

    $group = groups_get_group( $group_id );

    return $group->name;

}

/**
 * Options callback for select fields assigned to group roles
 *
 * @since 1.0.0
 *
 * @param stdClass $field
 *
 * @return array
 */
function automatorwp_buddypress_group_roles_options_cb( $field ) {

    $none_value = 'any';
    $none_label = __( 'any group role', 'automatorwp-pro' );
    $options = automatorwp_options_cb_none_option( $field, $none_value, $none_label );

    // Bail if BuddyPress function does not exist
    if ( ! function_exists( 'bp_groups_get_group_roles' ) ) {
        return $options;
    }

    $roles = bp_groups_get_group_roles( );

    foreach( $roles as $rol => $rol_obj ) {
        $options[$rol] = $rol_obj->name;
    }

    return $options;

}

/**
 * Options callback for select fields assigned to member types
 *
 * @since 1.0.0
 *
 * @param stdClass $field
 *
 * @return array
 */
function automatorwp_buddypress_member_types_options_cb( $field ) {

    $none_value = 'any';
    $none_label = __( 'any profile type', 'automatorwp-pro' );
    $options = automatorwp_options_cb_none_option( $field, $none_value, $none_label );

    $member_types = bp_get_member_types( array(), 'objects' );

    foreach( $member_types as $member_type => $member_type_obj ) {
        $options[$member_type] = $member_type_obj->labels['singular_name'];
    }

    return $options;

}

/**
 * Options callback for select2 fields assigned to profile fields
 *
 * @since 1.0.0
 *
 * @param stdClass $field
 *
 * @return array
 */
function automatorwp_buddypress_options_cb_profile_field( $field ) {

    // Setup vars
    $value = $field->escaped_value;
    $none_value = 'any';
    $none_label = __( 'any field', 'automatorwp-pro' );
    $options = automatorwp_options_cb_none_option( $field, $none_value, $none_label );

    if( ! empty( $value ) ) {
        if( ! is_array( $value ) ) {
            $value = array( $value );
        }

        foreach( $value as $profile_field_id ) {

            // Skip option none
            if( $profile_field_id === $none_value ) {
                continue;
            }

            $options[$profile_field_id] = automatorwp_buddypress_get_profile_field_title( $profile_field_id );
        }
    }

    return $options;

}

/**
 * Get the profile field title
 *
 * @since 1.0.0
 *
 * @param int $profile_field_id
 *
 * @return string
 */
function automatorwp_buddypress_get_profile_field_title( $profile_field_id ) {

    global $wpdb;

    // Empty title if no ID provided
    if( absint( $profile_field_id ) === 0 ) {
        return '';
    }

    $prefix = automatorwp_buddypress_get_table_prefix();

    $profile_field_name = $wpdb->get_var( "SELECT name FROM {$prefix}bp_xprofile_fields WHERE id = {$profile_field_id}");

    return ( $profile_field_name ? $profile_field_name : '' );

}

/**
 * Helper function to get the table prefix
 *
 * @since 1.0.0
 *
 * @return string
 */
function automatorwp_buddypress_get_table_prefix() {

    global $wpdb;

    if( function_exists( 'bp_core_get_table_prefix' ) ) {
        return bp_core_get_table_prefix();
    }

    return $wpdb->prefix;

}

/**
 * Options callback for select2 fields assigned to group types
 *
 * @since 1.3.2
 *
 * @param stdClass $field
 *
 * @return array
 */
function automatorwp_buddypress_options_cb_group_type( $field ) {

    // Setup vars
    $value = $field->escaped_value;
    $none_value = 'any';
    $none_label = __( 'any type', 'automatorwp-pro' );
    $options = automatorwp_options_cb_none_option( $field, $none_value, $none_label );

    if( ! empty( $value ) ) {
        if( ! is_array( $value ) ) {
            $value = array( $value );
        }

        foreach( $value as $group_type ) {
            // Skip option none
            if( $group_type === $none_value ) {
                continue;
            }

            $options[$group_type] = $group_type;
        }
    }

    return $options;

}