<?php
/**
 * User Not In Group Role
 *
 * @package     AutomatorWP\Integrations\BuddyPress\Filters\User_Not_In_Group_Role
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_BuddyPress_User_Not_In_Group_Role_Filter extends AutomatorWP_Integration_Filter {

    public $integration = 'buddypress';
    public $filter = 'buddypress_user_not_in_group_role';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_filter( $this->filter, array(
            'integration'       => $this->integration,
            'label'             => __( 'User is not in group', 'automatorwp-pro' ),
            'select_option'     => __( 'User is not in <strong>group</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Group. %2$s: Condition. %3$s: Role. */
            'edit_label'        => sprintf( __( '%1$s with role %2$s %3$s', 'automatorwp-pro' ), '{group}', '{condition}', '{role}' ),
            /* translators: %1$s: Group. %2$s: Condition. %3$s: Role. */
            'log_label'         => sprintf( __( '%1$s with role %2$s %3$s', 'automatorwp-pro' ), '{group}', '{condition}', '{role}' ),
            'options'           => array(
                'group' => automatorwp_utilities_ajax_selector_option( array(
                    'field'             => 'group',
                    'option_default'    => __( 'group', 'automatorwp-pro' ),
                    'name'              => __( 'Group:', 'automatorwp-pro' ),
                    'option_custom'         => true,
                    'option_custom_desc'    => __( 'Group ID', 'automatorwp-pro' ),
                    'action_cb'         => 'automatorwp_buddypress_get_groups',
                    'options_cb'        => 'automatorwp_buddypress_options_cb_group',
                    'placeholder'       => __( 'Select a group', 'automatorwp-pro' ),
                    'default'           => '',
                ) ),
                'condition' => array(
                    'from' => 'condition',
                    'fields' => array(
                        'condition' => array(
                            'name' => __( 'Condition:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options' => array(
                                'equal'             => __( 'equal to', 'automatorwp-pro' ),
                                'not_equal'         => __( 'not equal to', 'automatorwp-pro' ),
                            ),
                            'default' => 'equal'
                        )
                    )
                ),
                'role' => array(
                    'from' => 'role',
                    'default' => 'any',
                    'fields' => array(
                        'role' => array(
                            'name' => __( 'Group role:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options_cb' => 'automatorwp_buddypress_group_roles_options_cb',
                            'option_none' => true,
                            'default' => 'any'
                        ),
                    ),
                ),
            ),
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_filter    True if user deserves filter, false otherwise
     * @param stdClass  $filter             The filter object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $filter_options     The filter's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_filter( $deserves_filter, $filter, $user_id, $event, $filter_options, $automation ) {

        // Bail if BuddyPress function does not exist
        if ( ! function_exists( 'groups_get_group' ) ) {
            $this->result = __( 'Filter not passed. Groups component is not active.', 'automatorwp-pro' );
            return false;
        }

        // Shorthand
        $group_id = absint( $filter_options['group'] );
        $condition = $filter_options['condition'];
        $role = $filter_options['role'];
        $field = (object) array( 'args' => array() );
        $role_labels = automatorwp_buddypress_group_roles_options_cb( $field );

        // Bail if wrong configured
        if( empty( $role ) ) {
            $this->result = __( 'Filter not passed. Group role option has not been configured.', 'automatorwp-pro' );
            return false;
        }

        $group = groups_get_group( $group_id );

        if( ! $group ) {
            $this->result = sprintf( __( 'Filter not passed. Group with ID #%1$s not found.', 'automatorwp-pro' ), $group_id );
            return false;
        }

        // Deserve if user is not member in the group
        if ( ! BP_Groups_Member::check_is_member( $user_id, $group_id ) ) {
            $this->result = sprintf( __( 'Filter passed. User is not member of group "%1$s".', 'automatorwp-pro' ),
                $group->name
            );
            return true;
        }

        $role_in_group = '';
        
        // Check for organizers
        $is_organizer = BP_Groups_Member::check_is_admin( $user_id, $group_id );
        if ( $is_organizer === 1 ) $role_in_group = 'admin';
        
        // Check for moderators
        $is_moderator = BP_Groups_Member::check_is_mod( $user_id, $group_id );
        if ( $is_moderator === 1 ) $role_in_group = 'mod';

        // Check for members        
        $is_member = BP_Groups_Member::check_is_member( $user_id, $group_id );
        if ( ( $is_organizer === 0 && $is_moderator === 0 && $is_member === 1 ) ) $role_in_group = 'member';

        // Check for banned user
        $is_banned = BP_Groups_Member::check_is_banned( $user_id, $group_id );
        if ( $is_banned === 1 ) $role_in_group = 'banned';
        
        if ( $role !== 'any' ){
            $role_condition = $role_labels[$role];
        } else {
            $role_condition = 'any group role';
        }

        switch( $condition ) {
            case 'equal':
                // Don't deserve if user is not in this group role
                if( $role_in_group === $role || $role === 'any' ) {
                    $this->result = sprintf( __( 'Filter not passed. User role in group %1$s is "%2$s" and does not meets the condition %3$s "%4$s".', 'automatorwp-pro' ),
                        $group->name,
                        $role_labels[$role_in_group],
                        automatorwp_utilities_get_condition_label( $condition ),
                        $role_condition
                    );
                    return false;
                }
                break;
            case 'not_equal':
                // Don't deserve if user is in this group role
                if( $role_in_group !== $role || $role !== 'any' ) {
                    $this->result = sprintf( __( 'Filter not passed. User role in group %1$s is "%2$s" and does not meets the condition %3$s "%4$s".', 'automatorwp-pro' ),
                        $group->name,
                        $role_labels[$role_in_group],
                        automatorwp_utilities_get_condition_label( $condition ),
                        $role_condition
                    );
                    return false;
                }
                break;
        }

        $this->result = sprintf( __( 'Filter passed. User is not in group %1$s with role %2$s "%3$s".', 'automatorwp-pro' ),
            $group->name,
            automatorwp_utilities_get_condition_label( $condition ),
            $role_condition
        );

        return $deserves_filter;

    }

}

new AutomatorWP_BuddyPress_User_Not_In_Group_Role_Filter();