<?php
/**
 * Get Favorite
 *
 * @package     AutomatorWP\Integrations\BuddyBoss\Triggers\Get_Favorite
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_BuddyBoss_Get_Favorite extends AutomatorWP_Integration_Trigger {

    public $integration = 'buddyboss';
    public $trigger = 'buddyboss_get_favorite';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User gets a favorite in a topic', 'automatorwp-pro' ),
            'select_option'     => __( 'User <strong>gets a favorite</strong> in a topic', 'automatorwp-pro' ),
            /* translators: %1$s: Topic title. %2$s: Forum title. %2$s: Number of times. */
            'edit_label'        => sprintf( __( 'User gets a favorite in %1$s of %2$s %3$s time(s)', 'automatorwp-pro' ), '{post}', '{forum}', '{times}' ),
            /* translators: %1$s: Topic title. %2$s: Forum title. */
            'log_label'         => sprintf( __( 'User gets a favorite in %1$s of %2$s', 'automatorwp-pro' ), '{post}', '{forum}' ),
            'action'            => 'bbp_add_user_favorite',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 2,
            'options'           => array(
                'post' => array(
                    'from' => 'post',
                    'fields' => array(
                        'post' => automatorwp_utilities_post_field( array(
                            'name' => __( 'Topic:', 'automatorwp-pro' ),
                            'option_none_label' => __( 'any topic', 'automatorwp-pro' ),
                            'post_type_cb' => 'bbp_get_topic_post_type'
                        ) )
                    )
                ),
                'forum' => array(
                    'from' => 'forum',
                    'fields' => array(
                        'forum' => automatorwp_utilities_post_field( array(
                            'name' => __( 'Forum:', 'automatorwp-pro' ),
                            'option_none_label' => __( 'any forum', 'automatorwp-pro' ),
                            'post_type_cb' => 'bbp_get_forum_post_type'
                        ) )
                    )
                ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_utilities_post_tags(),
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param int $user_id
     * @param int $topic_id
     */
    public function listener( $user_id, $topic_id ) {

        // Bail if not user provided
        if( $user_id === 0 ) {
            return;
        }

        $topic_author = get_post_field( 'post_author', $topic_id );

        // Bail if topic author favorite himself
        if( absint( $user_id ) === absint( $topic_author ) ) {
            return;
        }

        $forum_id = bbp_get_topic_forum_id( $topic_id );

        // Trigger the get a favorite
        automatorwp_trigger_event( array(
            'trigger'       => $this->trigger,
            'user_id'       => $topic_author,
            'post_id'       => $topic_id,
            'forum_id'      => $forum_id,
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Don't deserve if post is not received
        if( ! isset( $event['post_id'] ) && ! isset( $event['forum_id'] ) ) {
            return false;
        }

        // Don't deserve if post doesn't match with the trigger option
        if( ! automatorwp_posts_matches( $event['post_id'], $trigger_options['post'] ) ) {
            return false;
        }

        // Don't deserve if forum doesn't match with the trigger option
        if( ! automatorwp_posts_matches( $event['forum_id'], $trigger_options['forum'] ) ) {
            return false;
        }

        return $deserves_trigger;

    }

}

new AutomatorWP_BuddyBoss_Get_Favorite();