<?php
/**
 * Add Group Activity
 *
 * @package     AutomatorWP\Integrations\BuddyBoss\Actions\Add_Group_Activity
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_BuddyBoss_Add_Group_Activity extends AutomatorWP_Integration_Action {

    public $integration = 'buddyboss';
    public $action = 'buddyboss_add_group_activity';

    /**
     * Store the action result
     *
     * @since 1.0.0
     *
     * @var string $result
     */
    public $result = '';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Add an activity to a group', 'automatorwp-pro' ),
            'select_option'     => __( 'Add an <strong>activity</strong> to a group', 'automatorwp-pro' ),
            /* translators: %1$s: Activity. %2$s: Group. */
            'edit_label'        => sprintf( __( 'Add an %1$s to %2$s', 'automatorwp-pro' ), '{activity}', '{group}' ),
            /* translators: %1$s: Activity. %2$s: Group. */
            'log_label'         => sprintf( __( 'Add an %1$s to %2$s', 'automatorwp-pro' ), '{activity}', '{group}' ),
            'options'           => array(
                'activity' => array(
                    'default' => __( 'activity', 'automatorwp-pro' ),
                    'fields' => array(
                        // Note: "action" key is reserved
                        'activity_action' => array(
                            'name' => __( 'Action (Optional):', 'automatorwp-pro' ),
                            'desc' => __( 'Activity\'s action.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                        'author' => array(
                            'name' => __( 'Author:', 'automatorwp-pro' ),
                            'desc' => __( 'The user ID of the activity\'s author. Leave blank to set the user who completed the automation.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                        'content' => array(
                            'name' => __( 'Content:', 'automatorwp-pro' ),
                            'desc' => __( 'Activity\'s content.', 'automatorwp-pro' ),
                            'type' => 'wysiwyg',
                            'default' => ''
                        ),
                        'link_preview' => array(
                            'name' => __( 'Link Preview (Optional):', 'automatorwp-pro' ),
                            'desc' => __( 'URL link to show as preview with the activity.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                        'link' => array(
                            'name' => __( 'Link (Optional):', 'automatorwp-pro' ),
                            'desc' => __( 'URL link associated to the activity.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                        'hide_sitewide' => array(
                            'name' => __( 'Hide on the sitewide activity stream:', 'automatorwp' ),
                            'desc' => __( 'Check this option to hide this activity from the sitewide activity stream.', 'automatorwp' ),
                            'type' => 'checkbox',
                            'classes' => 'cmb2-switch'
                        ),
                    )
                ),
                'group' => automatorwp_utilities_ajax_selector_option( array(
                    'field'             => 'group',
                    'option_default'    => __( 'group', 'automatorwp-pro' ),
                    'name'              => __( 'Group:', 'automatorwp-pro' ),
                    'option_none'       => false,
                    'option_custom'         => true,
                    'option_custom_desc'    => __( 'Group ID', 'automatorwp-pro' ),
                    'action_cb'         => 'automatorwp_buddyboss_get_groups',
                    'options_cb'        => 'automatorwp_buddyboss_options_cb_group',
                    'placeholder'       => __( 'Select a group', 'automatorwp-pro' ),
                    'default'           => '',
                ) )
            ),
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        $this->result = '';

        if( ! function_exists( 'bp_activity_add' ) ) {
            return;
        }

        // Bail if groups component is not active
        if ( ! bp_is_active( 'groups' ) ) {
            $this->result = __( 'Groups component is not active.', 'automatorwp-pro' );
            return;
        }

        // Shorthand
        $activity_action    = $action_options['activity_action'];
        $author             = $action_options['author'];
        $content            = $action_options['content'];
        $link_preview       = $action_options['link_preview'];
        $link               = $action_options['link'];
        $group              = $action_options['group'];
        $hide               = $action_options['hide_sitewide'];

        // Setup the author
        if( empty( $author ) ) {
            $author = $user_id;
        }

        if( absint( $author ) === 0 ) {
            return;
        }

        // Check the group
        if( absint( $group ) === 0 ) {
            return;
        }

        $activity_id = groups_post_update( array(
			'id'         => false,
			'content'    => $content,
			'user_id'    => $author,
			'group_id'   => $group,
			'privacy'    => 'public',
			'error_type' => 'bool',
		),);

        // Activity link preview
        if( $activity_id && ! empty( $link_preview ) ) {

            $preview_data = automatorwp_buddyboss_get_link_preview( $link_preview );

            if( $preview_data !== false ) {
                bp_activity_update_meta( $activity_id, '_link_embed', '1' );
                bp_activity_update_meta( $activity_id, '_link_preview_data', $preview_data );
            }
        }

        $this->result = __( 'Activity added to group successfully.', 'automatorwp-pro' );
    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();

    }    

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        $log_meta['result'] = $this->result;

        return $log_meta;

    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;

    }    

}

new AutomatorWP_BuddyBoss_Add_Group_Activity();