<?php
/**
 * Update Page Field
 *
 * @package     AutomatorWP\Integrations\Advance_Custom_Fields\Actions\Update_Page_Field
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Advanced_Custom_Fields_Update_Page_Field extends AutomatorWP_Integration_Action {

    public $integration = 'advanced_custom_fields';
    public $action = 'advanced_custom_fields_update_page_field';

    /**
     * The action result
     *
     * @since 1.0.0
     *
     * @var array $result
     */
    public $result = array();

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Update page field with a value', 'automatorwp-pro' ),
            'select_option'     => __( 'Update <strong>page field</strong> with a value', 'automatorwp-pro' ),
            /* translators: %1$s: User. */
            'edit_label'        => sprintf( __( 'Update %1$s %2$s with %3$s', 'automatorwp-pro' ), '{page}', '{field_name}', '{field_value}' ),
            /* translators: %1$s: User. */
            'log_label'         => sprintf( __( 'Update %1$s %2$s with %3$s', 'automatorwp-pro' ), '{page}', '{field_name}', '{field_value}' ),
            'options'           => array(
                'page' => array(
                    'default' => __( 'any page', 'automatorwp' ),
                    'from' => 'page_id',
                    'fields' => array(
                        'page_id' => automatorwp_utilities_post_field( array(
                            'name' => __( 'Page:', 'automatorwp-pro' ),
                            'post_type' => 'page',
                            'placeholder'           => __( 'Select a page', 'automatorwp-pro' ),
                            'option_none_label'     => __( 'any page', 'automatorwp-pro' ),
                            'option_custom'         => true,
                            'option_custom_desc'    => __( 'Page ID', 'automatorwp-pro' ),
                        ) ),
                        'page_id_custom' => automatorwp_utilities_custom_field( array(
                            'option_custom_desc'    => __( 'Page ID', 'automatorwp-pro' ),
                        ) ),
                    )),

                    'field_name' => array(
                        'from' => 'field_name',
                        'default' => __( 'field', 'automatorwp-pro' ),
                        'fields' => array(
                            'field_name' => array(
                                'name' => __( 'Field:', 'automatorwp-pro' ),
                                'type' => 'select',
                                'classes' => 'automatorwp-selector',
                                'options_cb' => 'automatorwp_advanced_custom_fields_options_cb_fields_pages',
                                'default' => ''
                            )
                        )
                    ),

                'field_value' => array(
                    'from' => 'field_value',
                    'default' => __( 'value', 'automatorwp-pro' ),
                    'fields' => array(
                        'field_value' => array(
                            'name' => __( 'Value:', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                    )
                ),
            ),
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        global $wpdb;

        // Shorthand
        $field_name = $action_options['field_name'];
        $field_value = $action_options['field_value'];
        $page_id = absint( $action_options['page_id'] );

        $this->result = array();

        // Bail if no page
        if( empty( $action_options['page_id'] ) ) {
            $this->result[] = sprintf( __( 'Page not found by the ID %1$s.', 'automatorwp-pro' ), $page_id );
            return;
        }

        // Update post meta
        update_post_meta( $page_id, $field_name, $field_value );
        
        $this->result[] = __( 'Page field updated successfully.', 'automatorwp-pro' );

    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();
    }

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        // Store page ID
        $log_meta['page_id'] = $action_options['page_id'];

        // Store page field
        $log_meta['field_name'] = $action_options['field_name'];

        // Store page value field
        $log_meta['field_value'] = $action_options['field_value'];

        // Store result
        $log_meta['result'] = $this->result;

        return $log_meta;
    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['page_title'] = array(
            'name' => __( 'Page Information', 'automatorwp-pro' ),
            'desc' => __( 'Information about the page updated.', 'automatorwp-pro' ),
            'type' => 'title',
        );

        $log_fields['page_id'] = array(
            'name' => __( 'Page ID:', 'automatorwp-pro' ),
            'desc' => __( 'The page\'s ID updated.', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['field_name'] = array(
            'name' => __( 'Page Field:', 'automatorwp-pro' ),
            'desc' => __( 'The updated field.', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['field_value'] = array(
            'name' => __( 'Page Field Value:', 'automatorwp-pro' ),
            'desc' => __( 'The updated field value.', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;
    }

}

new AutomatorWP_Advanced_Custom_Fields_Update_Page_Field();