<?php
/**
 * Remove User Webinar
 *
 * @package     AutomatorWP\Integrations\Zoom\Actions\Remove_User_Webinar
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Zoom_Remove_User_Webinar extends AutomatorWP_Integration_Action {

    public $integration = 'zoom';
    public $action = 'zoom_remove_user_webinar';

    /**
     * Registrant Email
     *
     * @since 1.0.0
     *
     * @var string $registrant_email
     */
    public $registrant_email = '';

    /**
     * Registrant ID
     *
     * @since 1.0.0
     *
     * @var string $registrant_id
     */
    public $registrant_id = '';

    /**
     * Store the action result
     *
     * @since 1.0.0
     *
     * @var string $result
     */
    public $result = '';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Remove user from webinar', 'automatorwp-pro' ),
            'select_option'     => __( 'Remove user from <strong>webinar</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Webinar. */
            'edit_label'        => sprintf( __( 'Remove user from %1$s', 'automatorwp-pro' ), '{webinar}' ),
            /* translators: %1$s: Webinar. */
            'log_label'         => sprintf( __( 'Remove user from %1$s', 'automatorwp-pro' ), '{webinar}' ),
            'options'           => array(
                'webinar' => automatorwp_utilities_ajax_selector_option( array(
                    'field'             => 'webinar',
                    'option_default'    => __( 'webinar', 'automatorwp-pro' ),
                    'placeholder'       => __( 'Select a webinar', 'automatorwp-pro' ),
                    'name'              => __( 'Webinar:', 'automatorwp-pro' ),
                    'action_cb'         => 'automatorwp_zoom_get_webinars',
                    'options_cb'        => 'automatorwp_zoom_options_cb_webinars',
                    'default'           => ''
                ) ),
            ),
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        $prefix = '_automatorwp_zoom_webinars_';

        $this->registrant_email = '';
        $this->registrant_id = '';
        $this->result = '';

        // Shorthand
        $webinar_id = $action_options['webinar'];

        // Bail if empty webinar to assign
        if( empty( $webinar_id ) ) {
            $this->result = __( 'Could not add registrant to webinar, missing webinar', 'automatorwp-pro' );
            return;
        }

        $params = automatorwp_zoom_get_request_parameters( 'webinars' );

        // Bail if the authorization has not been setup from settings
        if( $params === false ) {
            $this->result = __( 'Could not add registrant to webinar, Zoom authentication failed', 'automatorwp-pro' );
            return;
        }

        $user = get_userdata( $user_id );

        $this->registrant_email = $user->user_email;

        $this->registrant_id = get_user_meta( $user_id, $prefix . $webinar_id . '_registrant_id', true );

        if( empty( $this->registrant_id ) ) {
            $this->registrant_id = automatorwp_zoom_get_webinar_registrant_id( $webinar_id,  $this->registrant_email );

            if( $this->registrant_id === false ) {
                $this->result = __( 'Could not add registrant to webinar, registrant was not registered on this webinar', 'automatorwp-pro' );
                return;
            }
        }

        $params['method'] = 'PUT';
        $params['body'] = json_encode( array(
            'action'      => 'cancel',
            'registrants' => array(
                array(
                    'id'    => $this->registrant_id,
                    'email' => $this->registrant_email,
                ),
            ),
        ) );

        // Setup the URL
        $url = 'https://api.zoom.us/v2/webinars/' . $webinar_id . '/registrants/status';

        // Execute the request
        $response = wp_remote_post( $url, $params );

        if ( is_wp_error( $response ) ) {
            $this->result = sprintf( __( 'Could not remove registrant from webinar, error received: %1$s', 'automatorwp-pro' ), $response->get_error_message() );
            return;
        }

        $response_code = wp_remote_retrieve_response_code( $response );
        $body = json_decode( $response['body'], true, 512, JSON_BIGINT_AS_STRING );

        if ( $response_code === 201 || $response_code === 204 ) {
            // Delete user metas with information from this webinar
            delete_user_meta( $user_id, $prefix . $webinar_id . '_id', $body['id'] );
            delete_user_meta( $user_id, $prefix . $webinar_id . '_registrant_id', $body['registrant_id'] );
            delete_user_meta( $user_id, $prefix . $webinar_id . '_join_url', $body['join_url'] );

            $this->result = __( 'Registrant successfully removed', 'automatorwp-pro' );
        } else {
            $this->result = sprintf( __( 'Could not remove registrant from webinar, error received: %1$s', 'automatorwp-pro' ), $body['message'] );
        }

    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Configuration notice
        add_filter( 'automatorwp_automation_ui_after_item_label', array( $this, 'configuration_notice' ), 10, 2 );

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();
    }

    /**
     * Configuration notice
     *
     * @since 1.0.0
     *
     * @param stdClass  $object     The trigger/action object
     * @param string    $item_type  The object type (trigger|action)
     */
    public function configuration_notice( $object, $item_type ) {

        // Bail if action type don't match this action
        if( $item_type !== 'action' || $object->type !== $this->action ) {
            return;
        }

        $params = automatorwp_zoom_get_request_parameters( 'webinars' );

        // Warn user if the authorization has not been setup from settings
        if( $params === false ) : ?>
            <div class="automatorwp-notice-warning" style="margin-top: 10px; margin-bottom: 0;">
                <?php echo sprintf(
                    __( 'You need to configure the <a href="%s" target="_blank">Zoom Webinars settings</a> to get this action to work.', 'automatorwp-pro' ),
                    get_admin_url() . 'admin.php?page=automatorwp_settings&tab=opt-tab-zoom'
                ); ?>
                <?php echo sprintf(
                    __( '<a href="%s" target="_blank">Documentation</a>', 'automatorwp-pro' ),
                    'https://automatorwp.com/docs/zoom/'
                ); ?>
            </div>
        <?php endif;

    }

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        // Store the action's result
        $log_meta['registrant_email'] = $this->registrant_email;
        $log_meta['registrant_id'] = $this->registrant_id;
        $log_meta['result'] = $this->result;

        return $log_meta;
    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['registrant_email'] = array(
            'name' => __( 'Registrant Email:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['registrant_id'] = array(
            'name' => __( 'Registrant ID:', 'automatorwp-pro' ),
            'desc' => __( 'Unique identifier assigned to the registrant.', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;
    }

}

new AutomatorWP_Zoom_Remove_User_Webinar();