<?php
/**
 * Submit Essay
 *
 * @package     AutomatorWP\Integrations\LearnDash\Triggers\Submit_Essay
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_LearnDash_Submit_Essay extends AutomatorWP_Integration_Trigger {

    public $integration = 'learndash';
    public $trigger = 'learndash_submit_essay';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User submits an essay for a quiz', 'automatorwp-pro' ),
            'select_option'     => __( 'User submits an <strong>essay</strong> for a quiz', 'automatorwp-pro' ),
            /* translators: %1$s: Post title. %2$s: Number of times. */
            'edit_label'        => sprintf( __( 'User submits an essay for %1$s %2$s time(s)', 'automatorwp-pro' ), '{post}', '{times}' ),
            /* translators: %1$s: Post title. */
            'log_label'         => sprintf( __( 'User submits an essay for %1$s', 'automatorwp-pro' ), '{post}' ),
            'action'            => 'learndash_new_essay_submitted',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 2,
            'options'           => array(
                'post' => automatorwp_utilities_post_option( array(
                    'name' => __( 'Quiz:', 'automatorwp-pro' ),
                    'option_none_label' => __( 'any quiz', 'automatorwp-pro' ),
                    'post_type' => 'sfwd-quiz'
                ) ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_learndash_get_essay_tags(),
                automatorwp_utilities_post_tags( __( 'Quiz', 'automatorwp-pro' ) ),
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @param int   $essay_id  The new Essay ID created after essay submission.
	 * @param array $essay_args An array of essay arguments.
     * 
     * @since 1.0.0
     */
    public function listener($essay_id, $essay_args) {

        $user_id = get_current_user_id();

        // Bail if no user
        if ( $user_id === 0 ) {
            return;
        }

        $quiz_id = get_post_meta( $essay_id, 'quiz_post_id', true );
        
        automatorwp_trigger_event( array(
            'trigger'   => $this->trigger,
            'user_id'   => $user_id,
            'post_id'   => $quiz_id,
            'essay_id'  => $essay_id,
            'essay_title' => $essay_args['post_title'],
            'essay_content' => $essay_args['post_content'],
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Don't deserve if post is not received
        if( ! isset( $event['post_id'] ) ) {
            return false;
        }

        // Don't deserve if post doesn't match with the trigger option
        if( ! automatorwp_posts_matches( $event['post_id'], $trigger_options['post'] ) ) {
            return false;
        }

        return $deserves_trigger;

    }

    /**
     * Register the required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_trigger_log_meta', array( $this, 'log_meta' ), 10, 6 );

        parent::hooks();
    }

    /**
     * Trigger custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return array
     */
    function log_meta( $log_meta, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Bail if action type don't match this action
        if( $trigger->type !== $this->trigger ) {
            return $log_meta;
        }

        $log_meta['essay_id'] = ( isset( $event['essay_id'] ) ? $event['essay_id'] : '' );
        $log_meta['essay_title'] = ( isset( $event['essay_title'] ) ? $event['essay_title'] : '' );
        $log_meta['essay_content'] = ( isset( $event['essay_content'] ) ? $event['essay_content'] : '' );

        return $log_meta;

    }
}

new AutomatorWP_LearnDash_Submit_Essay();