<?php
/**
 * Complete Quiz Percentage Between
 *
 * @package     AutomatorWP\Integrations\LearnDash\Triggers\Complete_Quiz_Percentage_Between
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_LearnDash_Complete_Quiz_Percentage_Between extends AutomatorWP_Integration_Trigger {

    public $integration = 'learndash';
    public $trigger = 'learndash_complete_quiz_score_between';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User completes a quiz with a grade percentage between percentages', 'automatorwp-pro' ),
            'select_option'     => __( 'User completes a quiz with a grade percentage <strong>between</strong> percentages', 'automatorwp-pro' ),
            /* translators: %1$s: Post title. %2$s: Condition. %3$s: Percent. %4$s: Number of times. */
            'edit_label'        => sprintf( __( 'User completes %1$s with a grade percentage between %2$s and %3$s %4$s time(s)', 'automatorwp-pro' ), '{post}', '{min_percentage}', '{max_percentage}', '{times}' ),
            /* translators: %1$s: Post title. %2$s: Condition. %3$s: Percent. */
            'log_label'         => sprintf( __( 'User completes %1$s with a grade percentage between %2$s and %3$s', 'automatorwp-pro' ), '{post}', '{min_percentage}', '{max_percentage}' ),
            'action'            => ( defined( 'LEARNDASH_VERSION' ) && version_compare( LEARNDASH_VERSION, '3.0.0', '>=' ) ? 'learndash_quiz_submitted' : 'learndash_quiz_completed' ),
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 2,
            'options'           => array(
                'post' => automatorwp_utilities_post_option( array(
                    'name' => __( 'Quiz:', 'automatorwp-pro' ),
                    'option_none_label' => __( 'any quiz', 'automatorwp-pro' ),
                    'post_type' => 'sfwd-quiz'
                ) ),
                'min_percentage' => array(
                    'from' => 'min_percentage',
                    'fields' => array(
                        'min_percentage' => array(
                            'name' => __( 'Minimum percentage:', 'automatorwp-pro' ),
                            'desc' => __( 'The minimum grade percentage required.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'attributes' => array(
                                'type' => 'number',
                                'min' => '0',
                            ),
                            'default' => 0
                        )
                    )
                ),
                'max_percentage' => array(
                    'from' => 'max_percentage',
                    'fields' => array(
                        'max_percentage' => array(
                            'name' => __( 'Maximum percentage:', 'automatorwp-pro' ),
                            'desc' => __( 'The maximum grade percentage required.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'attributes' => array(
                                'type' => 'number',
                                'min' => '0',
                            ),
                            'default' => 0
                        )
                    )
                ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_utilities_post_tags(),
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param array $quiz_data array(
     *      'course' => WP_Post,
     *      'quiz' => WP_Post,
     *      'pass' => integer,
     *      'percentage' => integer,
     * )
     * @param WP_User $current_user
     */
    public function listener( $quiz_data, $current_user ) {

        $user_id = $current_user->ID;
        $quiz_id = automatorwp_learndash_get_post_id( $quiz_data['quiz'] );
        $course_id = automatorwp_learndash_get_post_id( $quiz_data['course'] );

        automatorwp_trigger_event( array(
            'trigger'       => $this->trigger,
            'user_id'       => $user_id,
            'post_id'       => $quiz_id,
            'course_id'     => $course_id,
            'percentage'    => $quiz_data['percentage'],
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Don't deserve if post is not received
        if( ! isset( $event['post_id'] ) && ! isset( $event['percentage'] ) ) {
            return false;
        }

        // Don't deserve if post doesn't match with the trigger option
        if( ! automatorwp_posts_matches( $event['post_id'], $trigger_options['post'] ) ) {
            return false;
        }

        $percentage = (float) $event['percentage'];
        $min_percentage =  (float) $trigger_options['min_percentage'];
        $max_percentage =  (float) $trigger_options['max_percentage'];

        $passed = ( $percentage >= $min_percentage && $percentage <= $max_percentage );

        // Don't deserve if percentage doesn't match with the trigger option
        if( ! $passed ) {
            return false;
        }

        return $deserves_trigger;

    }

}

new AutomatorWP_LearnDash_Complete_Quiz_Percentage_Between();