<?php
/**
 * Email Groups Leaders
 *
 * @package     AutomatorWP\Integrations\LearnDash\Actions\Email_Groups_Leaders
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_LearnDash_Email_Groups_Leaders extends AutomatorWP_Integration_Action {

    public $integration = 'learndash';
    public $action = 'learndash_email_groups_leaders';

    /**
     * Store the action email recipients
     *
     * @since 1.0.0
     *
     * @var array $to
     */
    public $to = array();

    /**
     * Store the action result
     *
     * @since 1.0.0
     *
     * @var bool $result
     */
    public $result = false;

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();
    }

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Send an email to user\'s groups leaders', 'automatorwp-pro' ),
            'select_option'     => __( 'Send <strong>an email</strong> to user\'s groups leaders', 'automatorwp-pro' ),
            /* translators: %1$s: Email. */
            'edit_label'        => sprintf( __( 'Send an %1$s to user\'s groups leaders', 'automatorwp-pro' ), '{email}' ),
            /* translators: %1$s: Email. */
            'log_label'         => sprintf( __( 'Send an %1$s to user\'s groups leaders', 'automatorwp-pro' ), '{email}' ),
            'options'           => array(
                'email' => array(
                    'default' => __( 'email', 'automatorwp-pro' ),
                    'fields' => array(
                        'from' => array(
                            'name' => __( 'From:', 'automatorwp-pro' ),
                            'desc' => __( 'Leave empty to use default WordPress email.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                        'cc' => array(
                            'name' => __( 'CC:', 'automatorwp-pro' ),
                            'desc' => __( 'Email address(es) that will receive a copy of this email. Accepts single or comma-separated list of emails.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                        'bcc' => array(
                            'name' => __( 'BCC:', 'automatorwp-pro' ),
                            'desc' => __( 'Email address(es) that will receive a copy of this email. Accepts single or comma-separated list of emails.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                        'subject' => array(
                            'name' => __( 'Subject:', 'automatorwp-pro' ),
                            'desc' => __( 'Email\'s subject.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                        'content' => array(
                            'name' => __( 'Content:', 'automatorwp-pro' ),
                            'desc' => __( 'Email\'s content.', 'automatorwp-pro' ),
                            'type' => 'wysiwyg',
                            'default' => ''
                        ),
                    )
                )
            ),
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        // Shorthand
        $from       = $action_options['from'];
        $to         = array();
        $cc         = $action_options['cc'];
        $bcc        = $action_options['bcc'];
        $subject    = $action_options['subject'];
        $content    = $action_options['content'];

        // Setup to
        $user_groups_ids = learndash_get_users_group_ids( $user_id );

        if( is_array( $user_groups_ids ) ) {

            // Loop all user groups
            foreach( $user_groups_ids as $group_id ) {

                // Get group leaders
                $group_leaders = learndash_get_groups_administrators( $group_id );

                if( is_array( $group_leaders ) ) {

                    // Loop group leaders to get their email address
                    foreach( $group_leaders as $leader ) {

                        // Prevent to add the same email twice
                        if( ! in_array( $leader->data->user_email , $to ) ) {
                            $to[] = $leader->data->user_email;
                        }
                    }

                }
            }

        }

        $this->to = $to;

        // Setup headers
        $headers = array();

        if( ! empty( $from ) ) {
            $headers[] = 'From: <' . $from . '>';
        }

        if ( ! empty( $cc ) ) {
            $headers[] = 'Cc: ' . $cc;
        }

        if ( ! empty( $bcc ) ) {
            $headers[] = 'Bcc: ' . $bcc;
        }

        $headers[] = 'Content-Type: text/html; charset=' . get_option( 'blog_charset' );

        // Setup the content
        $content = wpautop( $content );

        add_filter( 'wp_mail_content_type', array( $this, 'set_html_content_type' ) );

        // Send email
        $this->result = wp_mail( $to, $subject, $content, $headers );

        remove_filter( 'wp_mail_content_type', array( $this, 'set_html_content_type' ) );

    }

    /**
     * Function to set the mail content type
     *
     * @since 1.0.0
     *
     * @param string $content_type
     *
     * @return string
     */
    public function set_html_content_type( $content_type = 'text/html' ) {
        return 'text/html';
    }

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        $log_meta['from']       = $action_options['from'];
        $log_meta['to']         = $this->to;
        $log_meta['cc']         = $action_options['cc'];
        $log_meta['bcc']        = $action_options['bcc'];
        $log_meta['subject']    = $action_options['subject'];
        $log_meta['content']    = $action_options['content'];
        $log_meta['result']     = ( $this->result ? 'Sent' : 'Not sent' );

        return $log_meta;
    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['email_info'] = array(
            'name' => __( 'Sending Information', 'automatorwp-pro' ),
            'desc' => __( 'Information about email sent.', 'automatorwp-pro' ),
            'type' => 'title',
        );

        $log_fields['from'] = array(
            'name' => __( 'From:', 'automatorwp-pro' ),
            'desc' => __( 'Email sender.', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['to'] = array(
            'name' => __( 'To:', 'automatorwp-pro' ),
            'desc' => __( 'Email recipient.', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['cc'] = array(
            'name' => __( 'CC:', 'automatorwp-pro' ),
            'desc' => __( 'Carbon copy emails.', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['bcc'] = array(
            'name' => __( 'BCC:', 'automatorwp-pro' ),
            'desc' => __( 'Blind carbon copy emails.', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['subject'] = array(
            'name' => __( 'Subject:', 'automatorwp-pro' ),
            'desc' => __( 'Email\'s subject.', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['content'] = array(
            'name' => __( 'Content:', 'automatorwp-pro' ),
            'desc' => __( 'Email\'s content.', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['result'] = array(
            'name' => __( 'Sending result:', 'automatorwp-pro' ),
            'desc' => __( 'If sending result is "Not sent" you need to check if your server\'s wp_mail() function is correctly configured.', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;
    }

}

new AutomatorWP_LearnDash_Email_Groups_Leaders();