<?php
/**
 * Send Webhook
 *
 * @package     AutomatorWP\Integrations\AutomatorWP\Actions\Send_Webhook
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Webhooks_Send_Webhook extends AutomatorWP_Integration_Action {

    /**
     * Store the action result
     *
     * @since 1.0.0
     *
     * @var bool $result
     */
    public $result = false;

    /**
     * Store the request fields
     *
     * @since 1.0.0
     *
     * @var array $request_fields
     */
    public $request_fields = array();

    /**
     * Store the request headers
     *
     * @since 1.0.0
     *
     * @var array $request_headers
     */
    public $request_headers = array();

    /**
     * The request response code
     *
     * @since 1.0.0
     *
     * @var int $response
     */
    public $response_code = 0;

    /**
     * The request response
     *
     * @since 1.0.0
     *
     * @var string $response
     */
    public $response = '';

    /**
     * The integration label
     *
     * @since 1.0.0
     *
     * @var string $label
     */
    public $label = '';

    /**
     * Initialize the action
     *
     * @since 1.0.0
     */
    public function __construct( $integration, $label ) {

        $this->integration = $integration;
        $this->action = $integration . '_send_webhook';
        $this->label = $label;

        parent::__construct();

    }

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            /* translators: %1$s: Integration. */
            'label'             => sprintf( __( 'Send data to %1$s', 'automatorwp-pro' ), $this->label ),
            /* translators: %1$s: Integration. */
            'select_option'     => sprintf( __( 'Send data to %1$s', 'automatorwp-pro' ), "<strong>{$this->label}</strong>" ),
            /* translators: %1$s: Integration. */
            'edit_label'        => sprintf( __( 'Send data to %1$s', 'automatorwp-pro' ), '{webhook}' ),
            /* translators: %1$s: Integration. */
            'log_label'         => sprintf( __( 'Send data to %1$s', 'automatorwp-pro' ), '{webhook}' ),
            'options'           => array(
                'webhook' => array(
                    'default' => $this->label,
                    'fields' => array(
                        'url' => array(
                            'name' => __( 'URL:', 'automatorwp-pro' ),
                            'type' => 'text',
                            'required' => true,
                            'default' => ''
                        ),
                        'method' => array(
                            'name' => __( 'Request Method:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options_cb' => 'automatorwp_webhooks_get_request_methods',
                            'default' => 'POST'
                        ),
                        'format' => array(
                            'name' => __( 'Request Format:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options_cb' => 'automatorwp_webhooks_get_request_formats',
                            'default' => 'FORM'
                        ),
                        'fields' => array(
                            'name' => __( 'Fields:', 'automatorwp-pro' ),
                            'type' => 'group',
                            'classes' => 'automatorwp-fields-table',
                            'options'     => array(
                                'add_button'        => __( 'Add pair', 'automatorwp-pro' ),
                                'remove_button'     => '<span class="dashicons dashicons-no-alt"></span>',
                            ),
                            'fields' => array(
                                'key' => array(
                                    'name' => __( 'Key:', 'automatorwp-pro' ),
                                    'type' => 'text',
                                    'default' => ''
                                ),
                                'value' => array(
                                    'name' => __( 'Value:', 'automatorwp-pro' ),
                                    'type' => 'text',
                                    'default' => ''
                                ),
                            ),
                        ),
                        'headers' => array(
                            'name' => __( 'Headers:', 'automatorwp-pro' ),
                            'type' => 'group',
                            'classes' => 'automatorwp-fields-table',
                            'options'     => array(
                                'add_button'        => __( 'Add pair', 'automatorwp-pro' ),
                                'remove_button'     => '<span class="dashicons dashicons-no-alt"></span>',
                            ),
                            'fields' => array(
                                'key' => array(
                                    'name' => __( 'Key:', 'automatorwp-pro' ),
                                    'type' => 'text',
                                    'default' => ''
                                ),
                                'value' => array(
                                    'name' => __( 'Value:', 'automatorwp-pro' ),
                                    'type' => 'text',
                                    'default' => ''
                                ),
                            ),
                        ),
                    )
                )
            ),
            'tags' => automatorwp_webhooks_get_actions_response_tags()
        ) );

    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Send data button
        add_filter( 'automatorwp_automation_ui_before_option_form', array( $this, 'before_option_form' ), 10, 5 );

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        // Dynamic action tags replacements
        add_filter( 'automatorwp_action_tags_replacements', array( $this, 'dynamic_action_tags_replacements' ), 10, 4 );

        parent::hooks();
    }

    /**
     * Before option form custom content
     *
     * @since 1.0.0
     *
     * @param stdClass  $object     The trigger/action object
     * @param string    $item_type  The object type (trigger|action)
     * @param string    $option     The option key
     * @param array     $args       The option arguments
     */
    public function before_option_form( $object, $item_type, $option, $args ) {

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return;
        }

        ?>
        <div class="automatorwp-webhook-send-test-response"></div>
        <button type="button" class="button automatorwp-webhooks-send-test"><?php _e( 'Send Test', 'automatorwp-pro' ); ?></button>
        <div class="clear" style="margin-bottom: 15px;"></div>
        <?php

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        // Shorthand
        $url        = $action_options['url'];
        $method     = $action_options['method'];
        $format     = $action_options['format'];
        $fields     = $action_options['fields'];
        $headers    = $action_options['headers'];

        // Setup request fields
        $this->request_fields = array();

        if( is_array( $fields ) ) {

            // Turn array( 'key' => '', 'value' => '' ) into array( 'key' => 'value' )
            foreach( $fields as $field ) {

                // Parse tags replacements to both, key and value
                $key = automatorwp_parse_automation_tags( $automation->id, $user_id, $field['key'] );
                $value = automatorwp_parse_automation_tags( $automation->id, $user_id, $field['value'] );

                // Can't sent empty field key
                if( empty( $key ) ) {
                    continue;
                }

                // TODO: Remove in AutomatorWP 3.0, Check for backward compatibility
                if( function_exists( 'automatorwp_parse_function_arg_value' ) ) {
                    // Parse values like "null", "true" or "12" to their correct value types
                    $value = automatorwp_parse_function_arg_value( $value );
                }

                // Check if field already exists
                if( isset( $this->request_fields[$key] ) ) {

                    // Ensure that field is an array
                    if( ! is_array( $this->request_fields[$key] ) ) {
                        $this->request_fields[$key] = array( $this->request_fields[$key] );
                    }

                    $this->request_fields[$key][] = $value;

                } else {
                    $this->request_fields[$key] = $value;
                }


            }

        }

        /**
         * Available filter to override the request fields without key parsed (keys still as key/sub_key)
         *
         * @since 1.0.0
         *
         * @param array                                 $request_fields     The request fields (keys still as key/sub_key)
         * @param stdClass                              $action             The action object
         * @param int                                   $user_id            The user ID
         * @param array                                 $action_options     The action's stored options (with tags already passed)
         * @param stdClass                              $automation         The action's automation object
         * @param AutomatorWP_Webhooks_Send_Webhook  $send_webhook       The send webhook object
         */
        $this->request_fields = apply_filters( 'automatorwp_webhooks_send_webhook_request_fields', $this->request_fields, $action, $user_id, $action_options, $automation, $this );

        // Turn array( 'key/sub_key' => '' ) into array( 'key' => array( 'sub_key' => ''  )  )
        foreach( $this->request_fields as $key => $value ) {

            if( strpos( $key, '/' ) === false ) {
                continue;
            }

            $keys = explode( '/', $key );

            $sub_fields = array();
            $ref = &$sub_fields;
            foreach ( $keys as $i => $v ) {
                if( $i === ( count( $keys ) - 1 ) ) {
                    $ref[$v] = $value;
                } else {
                    $ref[$v] = array();
                }

                $ref = &$ref[$v];
            }

            $this->request_fields = automatorwp_webhooks_array_merge_recursive( $this->request_fields, $sub_fields );

            unset( $this->request_fields[$key] );
        }

        /**
         * Available filter to override the request fields with key parsed (keys are key => array( sub_key ) )
         *
         * @since 1.0.0
         *
         * @param array                                 $request_fields     The request fields (keys are key => array( sub_key ) )
         * @param stdClass                              $action             The action object
         * @param int                                   $user_id            The user ID
         * @param array                                 $action_options     The action's stored options (with tags already passed)
         * @param stdClass                              $automation         The action's automation object
         * @param AutomatorWP_Webhooks_Send_Webhook  $send_webhook       The send webhook object
         */
        $this->request_fields = apply_filters( 'automatorwp_webhooks_send_webhook_request_fields_keys_passed', $this->request_fields, $action, $user_id, $action_options, $automation, $this );


        // Setup request headers
        $this->request_headers = array();

        if( is_array( $headers ) ) {

            // Turn array( 'key' => '', 'value' => '' ) into array( 'key' => 'value' )
            foreach( $headers as $header ) {

                // Parse tags replacements to both, key and value
                $key = automatorwp_parse_automation_tags( $automation->id, $user_id, $header['key'] );
                $value = automatorwp_parse_automation_tags( $automation->id, $user_id, $header['value'] );

                // Can't sent empty header key
                if( empty( $key ) ) {
                    continue;
                }

                // Headers should match the patter "{key}: {value}" like "Content-Type: text/html"
                $this->request_headers[$key] = $value;
            }

        }

        /**
         * Available filter to override the request headers
         *
         * @since 1.0.0
         *
         * @param array                                 $request_headers    The request headers
         * @param stdClass                              $action             The action object
         * @param int                                   $user_id            The user ID
         * @param array                                 $action_options     The action's stored options (with tags already passed)
         * @param stdClass                              $automation         The action's automation object
         * @param AutomatorWP_Webhooks_Send_Webhook  $send_webhook       The send webhook object
         */
        $this->request_headers = apply_filters( 'automatorwp_webhooks_send_webhook_request_headers', $this->request_headers, $action, $user_id, $action_options, $automation, $this );

        // Setup the request fields
        $fields_string = http_build_query( $this->request_fields );

        // Parse the format option
        if( $format === 'JSON' && in_array( $method, array( 'POST', 'PUT' ) ) ) {
            $fields_string = json_encode( $this->request_fields );
            $fields_string = str_replace( "\\\\\\\\n", "\\n", $fields_string );
            $fields_string = str_replace( "\\\\n", "\\n", $fields_string );
			$fields_string = str_replace( '\\\"', '"', $fields_string );
            
            $this->request_headers['Content-Type'] = 'application/json';
        }

        /**
         * Available filter to modify the fields string
         *
         * @since 1.0.0
         *         
         * @param string                                $fields_string      The fields string
         * @param stdClass                              $action             The action object
         * @param int                                   $user_id            The user ID
         * @param array                                 $action_options     The action's stored options (with tags already passed)
         * @param stdClass                              $automation         The action's automation object
         * @param AutomatorWP_Webhooks_Send_Webhook  $send_webhook       The send webhook object
         */
        $fields_string = apply_filters( 'automatorwp_webhooks_send_webhook_fields_string', $fields_string, $action, $user_id, $action_options, $automation, $this );

        $body    = $fields_string; 
        $headers = $this->request_headers; 
       
        $args = [
            'headers' => $headers,
            'body'    => $body,
            'timeout' => 20,
        ];

        // Get method
        if ( $method === 'POST' || $method === '' ) {
            $args['method'] = 'POST';
        } elseif ( $method === 'GET' ) {
            $url .= '?' . $fields_string;
            $args['method'] = 'GET';
        } elseif ( $method === 'PUT' ) {
            $args['method'] = 'PUT';
        } elseif ( $method === 'CUSTOM' ) {
            $args['method'] = $method;
        }

        /**
         * Available hook to extend the cURL handle before execute the request
         *
         * @since 1.0.0
         *
         * @param stdClass                              $action             The action object
         * @param int                                   $user_id            The user ID
         * @param array                                 $action_options     The action's stored options (with tags already passed)
         * @param stdClass                              $automation         The action's automation object
         * @param AutomatorWP_Webhooks_Send_Webhook  $send_webhook       The send webhook object
         */
        do_action( 'automatorwp_webhooks_send_webhook_before_exec', $action, $user_id, $action_options, $automation, $this );

        $response = wp_remote_request( $url, $args );

        // Handle response
        if ( is_wp_error( $response ) ) {
            $this->result = $response->get_error_message();
        } else {
            // Process the server response
            $this->result = wp_remote_retrieve_body( $response );
            $this->response_code = wp_remote_retrieve_response_code( $response );
        }

        $this->response = ( $this->result ? $this->result : '' );  

        /**
         * Available hook to handle the request response
         *
         * @since 1.0.0
         *
         * @param stdClass                              $response           The server response
         * @param int                                   $response_code      The server response code
         * @param stdClass                              $action             The action object
         * @param int                                   $user_id            The user ID
         * @param array                                 $action_options     The action's stored options (with tags already passed)
         * @param stdClass                              $automation         The action's automation object
         * @param AutomatorWP_Webhooks_Send_Webhook  $send_webhook       The send webhook object
         */
        do_action( 'automatorwp_webhooks_send_webhook_response', $this->result, $this->response_code, $action, $user_id, $action_options, $automation, $this );

    }

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        $fields = $this->request_fields;

        if( $action_options['format'] === 'JSON' ) {
            $fields = json_encode( $fields );
        }

        $log_meta['url']            = $action_options['url'];
        $log_meta['method']         = $action_options['method'];
        $log_meta['format']         = $action_options['format'];
        $log_meta['fields']         = $fields;
        $log_meta['headers']        = $this->request_headers;
        $log_meta['response_code']  = $this->response_code;
        $log_meta['response']       = $this->response;
        $log_meta['result']         = ( $this->result ? __( 'Success', 'automatorwp-pro' ) : __( 'No response received', 'automatorwp-pro' ) );

        return $log_meta;

    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['request_info'] = array(
            'name' => __( 'Sending Information', 'automatorwp-pro' ),
            'desc' => __( 'Information about request sent.', 'automatorwp-pro' ),
            'type' => 'title',
        );

        $log_fields['url'] = array(
            'name' => __( 'URL:', 'automatorwp-pro' ),
            'desc' => __( 'URL of the request.', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['method'] = array(
            'name' => __( 'Method:', 'automatorwp-pro' ),
            'desc' => __( 'HTTP method of the request.', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['format'] = array(
            'name' => __( 'Format:', 'automatorwp-pro' ),
            'desc' => __( 'Request format.', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['fields'] = array(
            'name' => __( 'Fields:', 'automatorwp-pro' ),
            'desc' => __( 'Data sent on the request.', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['headers'] = array(
            'name' => __( 'Headers:', 'automatorwp-pro' ),
            'desc' => __( 'Custom headers sent on the request.', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['response_code'] = array(
            'name' => __( 'Response code:', 'automatorwp-pro' ),
            'desc' => __( 'Response code received from server.', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['response'] = array(
            'name' => __( 'Response:', 'automatorwp-pro' ),
            'desc' => __( 'Response received from server.', 'automatorwp-pro' ),
            'type' => 'text',
        );

        $log_fields['result'] = array(
            'name' => __( 'Request result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;

    }

    /**
     * Setup action dynamic tags replacements
     *
     * @since 1.0.0
     *
     * @param array     $replacements   The trigger replacements
     * @param stdClass  $action        The trigger object
     * @param int       $user_id        The user ID
     * @param stdClass  $log            The last trigger log object
     *
     * @return array
     */
    function dynamic_action_tags_replacements( $replacements, $action, $user_id, $log ) {

        // Bail if action type doesn't match this action
        if ( $action->type !== $this->action ) {
            return $replacements;
        }

        // Get fields stored on last log
        $fields = ct_get_object_meta( $log->id, 'response', true );

        $fields_clean = $fields;

        // Clean fields
        $fields_clean = str_replace(["\r\n", "\n", "\r", "\\r\\n"], '', $fields_clean);
        $fields_clean = stripslashes( $fields_clean );
        $fields_decoded = json_decode( $fields_clean, true );

        if ( is_array( $fields_decoded ) ) {
            
            $replacements = array_merge( $replacements, automatorwp_webhooks_action_nested_fields( $fields_decoded, 'response' ) );
                
                // Append other fields values as replacements
                foreach( $fields_decoded as $key => $value ) {
                    
                    if ( $key !== 'response' ) {
                        $replacements[$key] = $value;
                    }
                    
                }
        } else {
            
            $replacements['response'] = trim( $fields_clean, '"' );
        }

        return $replacements;
        
    }

}