<?php
/**
 * Purchase Ticket
 *
 * @package     AutomatorWP\Integrations\The_Events_Calendar\Triggers\Purchase_Ticket
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_The_Events_Calendar_Purchase_Ticket extends AutomatorWP_Integration_Trigger {

    public $integration = 'the_events_calendar';
    public $trigger = 'the_events_calendar_purchase_ticket';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User purchases a ticket for an event', 'automatorwp-pro' ),
            'select_option'     => __( 'User purchases <strong>a ticket</strong> for an event', 'automatorwp-pro' ),
            /* translators: %1$s: Post title. %2$s: Number of times. */
            'edit_label'        => sprintf( __( 'User purchases a ticket for %1$s %2$s time(s)', 'automatorwp-pro' ), '{post}', '{times}' ),
            /* translators: %1$s: Post title. */
            'log_label'         => sprintf( __( 'User purchases a ticket for %1$s', 'automatorwp-pro' ), '{post}' ),
            'action'            => 'tec_tickets_commerce_order_status_flag_complete',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 3,
            'options'           => array(
                'post' => automatorwp_utilities_post_option( array(
                    'name' => __( 'Event:', 'automatorwp-pro' ),
                    'option_none_label' => __( 'any event', 'automatorwp-pro' ),
                    'post_type' => Tribe__Events__Main::POSTTYPE
                ) ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_utilities_post_tags(),
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param TECTicketsCommerceStatusStatus_Interface      $new_status     New post status
     * @param TECTicketsCommerceStatusStatus_Interface|null $old_status     Old post status
     * @param WP_Post   $post   Post object
     */
    public function listener( $new_status, $old_status, $post ) {

        $user_id = get_current_user_id();
        $order_data = tec_tc_get_order( $post );

        // Bail if order is not completed
        if ( $order_data->post_status !== 'tec-tc-completed' ) {
            return;
        }

        foreach( $order_data->items as $ticket_id => $value ) {
            $event_id = $value['event_id'];

            for ( $i = absint( $value['quantity'] ); $i > 0; $i -= 1 ){

                // Trigger the ticket purchase
                automatorwp_trigger_event( array(
                    'trigger'       => $this->trigger,
                    'user_id'       => $user_id,
                    'post_id'       => $event_id,
                    'ticket_id'     => $ticket_id,
                ) );
            }
            
        }

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Don't deserve if post is not received
        if( ! isset( $event['post_id'] ) ) {
            return false;
        }

        // Don't deserve if post doesn't match with the trigger option
        if( ! automatorwp_posts_matches( $event['post_id'], $trigger_options['post'] ) ) {
            return false;
        }

        return $deserves_trigger;

    }

}

new AutomatorWP_The_Events_Calendar_Purchase_Ticket();