<?php
/**
 * Functions
 *
 * @package     AutomatorWP\Integrations\Google_Calendar\Functions
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;


/**
 * Get the calendars
 *
 * @since 1.0.0
 *
 * @return array
 */
function automatorwp_google_calendar_get_calendars() {

    $calendars = array();

    $params = automatorwp_google_calendar_get_request_parameters( );

    // Bail if the authorization has not been setup from settings
    if( $params === false ) {
        return $calendars;
    }

    $url = 'https://www.googleapis.com/calendar/v3/users/me/calendarList';

    $response = wp_remote_get( $url, $params );

    $response = json_decode( wp_remote_retrieve_body( $response ), true );

    if( isset( $response['items'] ) && is_array( $response['items'] ) ) {

        foreach( $response['items'] as $key => $calendar ) {
            
            if ( $calendar['accessRole'] === 'owner' || $calendar['accessRole'] === 'writer' ) {

                $calendars[] = array(
                    'id' => $calendar['id'],
                    'name' => $calendar['summary'],
                );

            }
            
        } 
    }

    return $calendars;
}

/**
 * Options callback for select2 fields assigned to calendars
 *
 * @since 1.0.0
 *
 * @param stdClass $field
 *
 * @return array
 */
function automatorwp_google_calendar_options_cb_calendars( $field ) {

    // Setup vars
    $value = $field->escaped_value;
    $none_value = 'any';
    $none_label = __( 'any calendar', 'automatorwp-pro' );
    $options = automatorwp_options_cb_none_option( $field, $none_value, $none_label );

    if( ! empty( $value ) ) {
        if( ! is_array( $value ) ) {
            $value = array( $value );
        }

        foreach( $value as $calendar_id ) {

            // Skip option none
            if( $calendar_id === $none_value ) {
                continue;
            }

            $options[$calendar_id] = automatorwp_google_calendar_get_calendar_title( $calendar_id );
        }
    }

    return $options;

}

/**
 * Get the calendar title
 *
 * @since 1.0.0
 *
 * @param int $calendar_id
 *
 * @return string|null
 */
function automatorwp_google_calendar_get_calendar_title( $calendar_id ) {

    // Empty title if no ID provided
    if( empty( $calendar_id ) ) {
        return '';
    }

    $calendars = automatorwp_google_calendar_get_calendars();
    $calendars_name = '';

    foreach( $calendars as $calendar ) {

        if( $calendar['id'] === $calendar_id ) {
            $calendars_name = $calendar['name'];
            break;
        }
    }

    return $calendars_name;

}

/**
 * Get the events
 *
 * @since 1.0.0
 *
 * @return array
 */
function automatorwp_google_calendar_get_events( $calendar_id ) {

    $events = array();
    $params = automatorwp_google_calendar_get_request_parameters();

    // Bail if the authorization has not been setup from settings
    if( $params === false ) {
        return $events;
    }

    $date_min = date( 'c', current_time( 'timestamp' ) ); // ISO 8601 date

    $url = 'https://www.googleapis.com/calendar/v3/calendars/' . $calendar_id . '/events?timeMin=' . urlencode( $date_min );
      
    $response = wp_remote_get( $url, $params );

    $response = json_decode( wp_remote_retrieve_body( $response ), true );

    if( isset( $response['items'] ) && is_array( $response['items'] ) ) {

        foreach( $response['items'] as $key => $event ) {
            if ( !isset( $event['summary'] ) ){
                $event['summary'] = __( 'No title', 'automatorwp-pro' );
            }
            $events[] = array(
                'id' => $event['id'],
                'name' => $event['summary']
            );
        }
    }

    return $events;
}


/**
 * Options callback for select2 fields assigned to events
 *
 * @since 1.0.0
 *
 * @param stdClass $field
 *
 * @return array
 */
function automatorwp_google_calendar_options_cb_events( $field ) {

    // Setup vars
    $value = $field->escaped_value;
    $none_value = 'any';
    $none_label = __( 'any event', 'automatorwp-pro' );
    $options = automatorwp_options_cb_none_option( $field, $none_value, $none_label );

    $calendar_id = ct_get_object_meta( $field->object_id, 'calendar', true );

    if( ! is_array( $value ) ) {
        $value = array( $value );
    }

    foreach( $value as $event_id ) {

        // Skip option none
        if( $event_id === $none_value ) {
            continue;
        }

        $options[$event_id] = automatorwp_google_calendar_get_event_title( $calendar_id, $event_id, $field );

    }

    return $options;

}


/**
 * Get the event title
 *
 * @since 1.0.0
 *
 * @param int $calendar_id
 * @param int $event_id
 *
 * @return string|null
 */
function automatorwp_google_calendar_get_event_title( $calendar_id, $event_id, $field ) {

    $calendar = ct_get_object_meta( $field->object_id, 'calendar', true );

    $events = automatorwp_google_calendar_get_events( $calendar );
    
    $event_name = '';

    foreach( $events as $event ) {

        if( $event['id']  ==  $event_id ) {
            $event_name = $event['name'];
            break;
        }

    }
    return $event_name;

}

/**
 * Get the event dates
 *
 * @since 1.0.0
 *
 * @return array
 */
function automatorwp_google_calendar_get_current_event( $calendar_id, $event_id ) {

    $event = array();
    $params = automatorwp_google_calendar_get_request_parameters();

    // Bail if the authorization has not been setup from settings
    if( $params === false ) {
        return $events;
    }

    $url = 'https://www.googleapis.com/calendar/v3/calendars/' . $calendar_id . '/events/' . $event_id;
      
    $response = wp_remote_get( $url, $params );

    $event = json_decode( wp_remote_retrieve_body( $response ), true );

    return $event;
}

/**
 * Get the request parameters
 *
 * @since 1.0.0
 *
 * @param string $platform
 *
 * @return array|false
 */
function automatorwp_google_calendar_get_request_parameters( ) {

    $auth = get_option( 'automatorwp_google_calendar_auth' );
    
    if( ! is_array( $auth ) ) {
        return false;
    }

    return array(
        'user-agent'  => 'AutomatorWP; ' . home_url(),
        'timeout'     => 120,
        'httpversion' => '1.1',
        'headers'     => array(
            'Authorization' => 'Bearer ' . $auth['access_token'],
            'Content-Type'  => 'application/json',
            'Accept'        => 'application/json'
        )
    );
}

/**
 * Get the request parameters
 *
 * @since 1.0.0
 *
 * @param string $platform
 *
 * @return string|false|WP_Error
 */
function automatorwp_google_calendar_refresh_token( ) {

    $client_id = automatorwp_google_calendar_get_option( 'client_id', '' );
    $client_secret = automatorwp_google_calendar_get_option( 'client_secret', '' );

    if( empty( $client_id ) || empty( $client_secret ) ) {
        return false;
    }

    $auth = get_option( 'automatorwp_google_calendar_auth', false );

    if( ! is_array( $auth ) ) {
        return false;
    }

    $params = array(
        'headers' => array(
            'Content-Type'  => 'application/x-www-form-urlencoded; charset=utf-8',
            'Authorization' => 'Basic ' . base64_encode( $client_id . ':' . $client_secret ),
            'Accept'        => 'application/json',
        ),
        'body'  => array(
            'refresh_token' => $auth['refresh_token'],
            'grant_type'    => 'refresh_token',
        )
    );

    $response = wp_remote_post( 'https://www.googleapis.com/oauth2/v4/token', $params );

    if( is_wp_error( $response ) ) {
        return $response;
    }

    $response_code = wp_remote_retrieve_response_code( $response );

    if ( $response_code !== 200 ) {
        return false;
    }

    $body = json_decode( wp_remote_retrieve_body( $response ) );

    $ref_token = $auth['refresh_token'];

    $auth = array(
        'access_token'  => $body->access_token,
        'refresh_token' => $ref_token,
        'token_type'    => $body->token_type,
        'expires_in'    => $body->expires_in,
        'scope'         => $body->scope,
    );

    // Update the access and refresh tokens
    update_option( 'automatorwp_google_calendar_auth', $auth );

    return $body->access_token;

}

/**
 * Filters the HTTP API response immediately before the response is returned.
 *
 * @since 1.0.0
 *
 * @param array  $response    HTTP response.
 * @param array  $parsed_args HTTP request arguments.
 * @param string $url         The request URL.
 *
 * @return array
 */
function automatorwp_google_calendar_maybe_refresh_token( $response, $args, $url ) {
    
    // Ensure to only run this check to on Google API request
    if( strpos( $url, 'googleapis.com' ) !== false ) {

        $code = wp_remote_retrieve_response_code( $response );
        
        if( $code === 401 ) {

            $access_token = automatorwp_google_calendar_refresh_token( );

            // Send again the request if token gets refreshed successfully
            if( $access_token ) {

                $args['headers']['Authorization'] = 'Bearer ' . $access_token;

                $response = wp_remote_request( $url, $args );

            }

        }
    
    }

    return $response;

}
add_filter( 'http_response', 'automatorwp_google_calendar_maybe_refresh_token', 10, 3 );
