<?php
/**
 * Points Balance
 *
 * @package     AutomatorWP\Integrations\GamiPress\Triggers\Points_Balance
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_GamiPress_Points_Balance extends AutomatorWP_Integration_Trigger {

    public $integration = 'gamipress';
    public $trigger = 'gamipress_points_balance';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User points balance is greater than, less than or equal to a specific amount', 'automatorwp-pro' ),
            'select_option'     => __( 'User points balance is <strong>greater than, less than or equal</strong> to a specific amount', 'automatorwp-pro' ),
            /* translators: %1$s: Post title. %2$s: Condition. %3$s: Points amount. */
            'edit_label'        => sprintf( __( 'User %1$s balance %2$s %3$s', 'automatorwp-pro' ), '{points_type}', '{condition}', '{points}'  ),
            /* translators: %1$s: Points amount. %2$s: Post title. */
            'log_label'         => sprintf( __( 'User %1$s balance %2$s %3$s', 'automatorwp-pro' ), '{points_type}', '{condition}', '{points}' ),
            'action'            => 'gamipress_update_user_points',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 8,
            'options'           => array(
                'points_type' => array(
                    'from' => 'points_type',
                    'fields' => array(
                        'points_type' => array(
                            'name' => __( 'Points Type:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'option_none' => true,
                            'option_none_label' => __( 'points of any type', 'automatorwp-pro' ),
                            'options_cb' => 'automatorwp_gamipress_points_types_options_cb',
                            'default' => 'any',
                        )
                    )
                ),
                'condition' => automatorwp_utilities_condition_option(),
                'points' => array(
                    'from' => 'points',
                    'fields' => array(
                        'points' => array(
                            'name' => __( 'Points amount:', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => '1'
                        )
                    )
                ),
            ),
            'tags' => array_merge(
                automatorwp_utilities_post_tags()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param int       $user_id        The user ID
     * @param int       $new_points     Points amount earned
     * @param int       $total_points   User points balance
     * @param int       $admin_id       The admin that awarded those rank
     * @param int       $achievement_id The achievement ID
     * @param string    $points_type    The points type slug
     * @param string    $reason         Reason of this award
     * @param string    $log_type       The log type
     */
    public function listener( $user_id, $new_points, $total_points, $admin_id, $achievement_id, $points_type, $reason, $log_type ) {

        automatorwp_trigger_event( array(
            'trigger'       => $this->trigger,
            'user_id'       => $user_id,
            'points'        => $total_points,
            'points_type'   => $points_type,
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Don't deserve if points and points type are not received
        if( ! isset( $event['points'] ) && ! isset( $event['points_type'] ) ) {
            return false;
        }

        // Don't deserve if points type doesn't match with the trigger option
        if( $trigger_options['points_type'] !== 'any' && $trigger_options['points_type'] !== $event['points_type'] ) {
            return false;
        }

        // Ensure points amount required as integer
        $trigger_options['points'] = absint( $trigger_options['points'] );
        $event['points'] = absint( $event['points'] );

        // Don't deserve if points balance doesn't match with the trigger option
        if( ! automatorwp_number_condition_matches( $event['points'], $trigger_options['points'], $trigger_options['condition'] ) ) {
            return false;
        }

        return $deserves_trigger;

    }

}

new AutomatorWP_GamiPress_Points_Balance();