<?php
/**
 * Send Group Roles Message
 *
 * @package     AutomatorWP\Integrations\BuddyBoss\Actions\Send_Group_Roles_Message
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_BuddyBoss_Send_Group_Roles_Message extends AutomatorWP_Integration_Action {

    public $integration = 'buddyboss';
    public $action = 'buddyboss_send_group_roles_message';

    /**
     * Store the action result
     *
     * @since 1.0.0
     *
     * @var string $result
     */
    public $result = '';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_action( $this->action, array(
            'integration'       => $this->integration,
            'label'             => __( 'Send private message to members with a role of a group', 'automatorwp-pro' ),
            'select_option'     => __( 'Send <strong>private message</strong> to members with a role of a group', 'automatorwp-pro' ),
            /* translators: %1$s: Message. %2$s: Role. %3$s: Group Title. */
            'edit_label'        => sprintf( __( 'Send %1$s to %2$s of %3$s', 'automatorwp-pro' ), '{email}', '{role}', '{group}' ),
            /* translators: %1$s: Message. %2$s: Role. %3$s: Group Title. */
            'log_label'         => sprintf( __( 'Send %1$s to %2$s of %3$s', 'automatorwp-pro' ), '{email}', '{role}', '{group}' ),
            'options'           => array(
                'email' => array(
                    'default' => __( 'private message', 'automatorwp-pro' ),
                    'fields' => array(
                        'from' => array(
                            'name' => __( 'From:', 'automatorwp-pro' ),
                            'desc' => __( 'The user ID that sends the message. Leave blank to set the user who completed the automation.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                        'exclude_author' => array(
                            'name' => __( 'Exclude user who completes the automation:', 'automatorwp-pro' ),
                            'desc' => __( 'Do not send the email to user who completes the automation.', 'automatorwp-pro' ),
                            'type' => 'checkbox',
                            'classes' => 'cmb2-switch'
                        ),
                        'subject' => array(
                            'name' => __( 'Subject:', 'automatorwp-pro' ),
                            'desc' => __( 'Message\'s subject.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        ),
                        'content' => array(
                            'name' => __( 'Content:', 'automatorwp-pro' ),
                            'desc' => __( 'Message\'s content.', 'automatorwp-pro' ),
                            'type' => 'wysiwyg',
                            'default' => ''
                        ),
                    )
                ),
                'role' => array(
                    'default' => __( 'all members', 'automatorwp-pro' ),
                    'from' => 'role',
                    'fields' => array(
                        'role' => array(
                            'name' => __( 'Roles:', 'automatorwp-pro' ),
                            'desc' => __( 'Roles to send the message', 'automatorwp-pro' ),
                            'type' => 'multicheck',
                            'select_all_button' => false,
                            'options' => array(
                                'organizer' => 'Organizers',
                                'moderator' => 'Moderators',
                                'member' => 'Members',

                            ),
                        ),
                    )
                ),
                'group' => automatorwp_utilities_ajax_selector_option( array(
                    'field'             => 'group',
                    'option_default'    => __( 'group', 'automatorwp-pro' ),
                    'name'              => __( 'Group:', 'automatorwp-pro' ),
                    'option_none'       => false,
                    'option_custom'         => true,
                    'option_custom_desc'    => __( 'Group ID', 'automatorwp-pro' ),
                    'action_cb'         => 'automatorwp_buddyboss_get_groups',
                    'options_cb'        => 'automatorwp_buddyboss_options_cb_group',
                    'placeholder'       => __( 'Select a group', 'automatorwp-pro' ),
                    'default'           => '',
                ) )
            ),
        ) );

    }

    /**
     * Action execution function
     *
     * @since 1.0.0
     *
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     */
    public function execute( $action, $user_id, $action_options, $automation ) {

        $this->result = '';

        // Bail if groups component is not active
        if ( ! bp_is_active( 'groups' ) ) {
            $this->result = __( 'Groups component is not active.', 'automatorwp-pro' );
            return;
        }

        // Bail if messages component is not active
        if ( ! bp_is_active( 'messages' ) ) {
            $this->result = __( 'Private messaging component is not active.', 'automatorwp-pro' );
            return;
        }

        // Shorthand
        $from = $action_options['from'];
        $exclude_author = (bool) $action_options['exclude_author'];
        $subject = $action_options['subject'];
        $content = $action_options['content'];

        // Setup from
        if( empty( $from ) ) {
            $from = $user_id;
        }

        // Setup the group
        $group_id = $action_options['group'];

        if( absint( $group_id ) === 0 ) {
            $this->result = __( 'Invalid group provided on the action\'s setup.', 'automatorwp-pro' );
            return;
        }

        $members = BP_Groups_Member::get_group_member_ids( $group_id );

        // Check if group has members
        if ( count( $members) ) {

            foreach ( $members as $member_id ) {

                $member_id = absint( $member_id );

                // Do not send the email to the user who completes the automation if the option has been checked
                if( $exclude_author && $member_id === $user_id ) {
                    continue;
                }

                // Do not add the notification to users who do not meet the required role(s)
                if( ! $this->member_meets_role( $member_id, $group_id, $action_options ) ) {
                    continue;
                }

                add_filter( 'bp_force_friendship_to_message', 'automatorwp_buddyboss_disable_force_friendship_to_message' );

                // Send the message
                messages_new_message( array(
                    'sender_id'     => $from,
                    'thread_id'     => false,
                    'recipients'    => array( $member_id ),
                    'subject'       => $subject,
                    'content'       => $content,
                    'date_sent'     => bp_core_current_time(),
                    'append_thread' => true,
                    'is_hidden'     => false,
                    'mark_visible'  => false,
                    'group_thread'  => false,
                    'error_type'    => 'bool',
                ) );

                remove_filter( 'bp_force_friendship_to_message', 'automatorwp_buddyboss_disable_force_friendship_to_message' );

            }

        }

        $this->result = sprintf( __( '%d private messages sent successfully.', 'automatorwp-pro' ), count( $members ) );

    }

    /**
     * Check if member meets the role configured in action options
     *
     * @since 1.0.0
     *
     * @param int $member_id
     * @param int $group_id
     * @param array $action_options
     *
     * @return bool
     */
    private function member_meets_role( $member_id, $group_id, $action_options ) {

        // Get the selected roles
        $roles = $action_options['role'];

        // If none option marked,  then automatically member pass this condition
        if( empty( $roles ) ) {
            return true;
        }

        // If all options marked, then automatically member pass this condition
        if( in_array( 'organizer', $roles ) && in_array( 'moderator', $roles ) && in_array( 'member', $roles ) ) {
            return true;
        }

        // Check for organizers
        if ( in_array( 'organizer', $roles ) && BP_Groups_Member::check_is_admin( $member_id, $group_id ) ) {
            return true;
        }

        // Check for moderators
        if ( in_array( 'moderator', $roles ) && BP_Groups_Member::check_is_mod( $member_id, $group_id ) ) {
            return true;
        }

        // Check for members
        if ( in_array( 'member', $roles ) && ! BP_Groups_Member::check_is_admin( $member_id, $group_id )  && ! BP_Groups_Member::check_is_mod( $member_id, $group_id ) ) {
            return true;
        }

        return false;

    }

    /**
     * Register required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_action_log_meta', array( $this, 'log_meta' ), 10, 5 );

        // Log fields
        add_filter( 'automatorwp_log_fields', array( $this, 'log_fields' ), 10, 5 );

        parent::hooks();

    }

    /**
     * Action custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $action             The action object
     * @param int       $user_id            The user ID
     * @param array     $action_options     The action's stored options (with tags already passed)
     * @param stdClass  $automation         The action's automation object
     *
     * @return array
     */
    public function log_meta( $log_meta, $action, $user_id, $action_options, $automation ) {

        // Bail if action type don't match this action
        if( $action->type !== $this->action ) {
            return $log_meta;
        }

        $log_meta['result'] = $this->result;

        return $log_meta;

    }

    /**
     * Action custom log fields
     *
     * @since 1.0.0
     *
     * @param array     $log_fields The log fields
     * @param stdClass  $log        The log object
     * @param stdClass  $object     The trigger/action/automation object attached to the log
     *
     * @return array
     */
    public function log_fields( $log_fields, $log, $object ) {

        // Bail if log is not assigned to an action
        if( $log->type !== 'action' ) {
            return $log_fields;
        }

        // Bail if action type don't match this action
        if( $object->type !== $this->action ) {
            return $log_fields;
        }

        $log_fields['result'] = array(
            'name' => __( 'Result:', 'automatorwp-pro' ),
            'type' => 'text',
        );

        return $log_fields;

    }

}

new AutomatorWP_BuddyBoss_Send_Group_Roles_Message();