# Repository Guidelines

## Project Structure & Module Organization
- `ai-engine-pro.php`: WordPress plugin entry point and constants.
- `classes/`: PHP core (admin, API, engines, services, REST, modules).
- `app/js/`, `common/js/`, `premium/js/`: React/Preact admin UI and features (bundled by Webpack to `app/*.js` and `premium/forms.js`).
- `themes/`: Compiled CSS; source Sass in `themes/sass/`.
- `labs/tests/`: Node-based API test suite and utilities.
- `images/`, `archives/`, `premium/`, `constants/`: Assets and feature modules.

## Build, Test, and Development Commands

You can use either npm or yarn. The team commonly runs yarn during development.

- `yarn dev` / `npm run dev`: Watch and rebuild JS bundles for admin, chatbot, and forms.
- `yarn build` / `npm run build`: Production bundles; cleans old artifacts and strips LICENSE txts.
- `yarn analysis` / `npm run analysis`: Build with bundle analyzer for `chatbot` entry.
- `yarn zip`: Create a distributable plugin zip (if configured).
- `yarn install` to install deps (or `npm ci`).
- Sass: `npm run sass` still compiles `themes/sass` → `themes/*.css`.
- Tests (Node scripts under `labs/tests`):
  - `node labs/tests/test-utilities.js clear-logs`
  - `node labs/tests/test-general.js quick` (baseline)
  - `node labs/tests/test-embeddings.js count`

## Coding Style & Naming Conventions
- PHP: PSR-12 via `.php-cs-fixer.php` (2-space indent, single quotes, no unused imports). Run `php-cs-fixer fix` locally before PRs.
- JS/TSX: ESLint (`.eslintrc.js`) with React/Hooks; 2 spaces, semicolons, `eqeqeq`, `prefer-const`, no `var`.
- Naming: React components PascalCase (e.g., `screens/Settings`), variables camelCase, PHP files lowercase snake/hyphen per existing patterns.
- Paths: Use webpack aliases (`@app`, `@common`, `@premium`, `@root`).

## Testing Guidelines
- Framework: Node scripts under `labs/tests` (no install needed). See `labs/tests/README.md` for flows and expected warnings.
- Conventions: Test files prefixed `test-*.js` with small, composable commands.
- Config: Copy `labs/tests/query-api.conf` to `query-api.local.conf` for local overrides. Never commit secrets.

## Commit & Pull Request Guidelines
- Commits: Imperative, concise scope first line (e.g., "Fix drag-and-drop preview"), additional context in body as needed.
- PRs must include: clear description, linked issues, screenshots/GIFs for UI changes, reproduction steps, and a test plan (commands run and results).
- Quality checks: run `npm run build`, `npm run sass`, lint JS (`npx eslint app/js common/js premium/js`) and format PHP.
- Backward compatibility: avoid breaking public hooks/APIs; gate premium features under `premium/`.

## Security & Configuration Tips
- Do not commit API keys; configure providers in WP Admin or local conf files.
- Sanitize/escape all output and validate REST inputs; prefer existing helpers in `classes/` and `common/`.

---

## Project Overview (for Agents)

AI Engine Pro is a WordPress plugin that integrates multiple AI providers (OpenAI, Anthropic, Google, Hugging Face, OpenRouter, Perplexity, Replicate, etc.) and ships features like chatbots, forms, discussions, embeddings, assistants, realtime, and more.

### Architecture Highlights
- Core orchestrators and services under `classes/` (init, API, REST, engines, modules, services).
- Frontend apps under `app/js/` (admin UI, chatbot UI, reusable components); premium UI under `premium/js/`.
- Themes are Sass-first under `themes/sass/` and compiled to `themes/*.css`.
- Labs contain Node-based test tools under `labs/tests/` (no install required).
- Webpack aliases: `@root` → `/app/`, `@app` → `/app/js/`, `@common` → `/common/js/`, `@premium` → `/premium/js/`.

### Frontend Structure
- Entries: admin (`app/js/index.js` → `app/index.js`), chatbot (`app/js/chatbot.js` → `app/chatbot.js`), forms (`premium/js/forms.js` → `premium/forms.js`).
- Chatbot UI lives in `app/js/chatbot/` (header, body, messages, realtime, components registry).
- Styled-components are used in places; themes use Sass variables and mixins.

### Backend Structure
- Plugin bootstrap: `ai-engine-pro.php`, with init and core setup in `classes/`.
- Engines in `classes/engines/` implement provider specifics; modules in `classes/modules/` implement features like chatbot, discussions, files, forms, etc.
- Queries in `classes/query/` handle text, feedback (function results), images, assistants, transcribe, embed, etc.

## Function Calling & Feedback Loop

The plugin supports provider-agnostic function calling with a feedback loop:

1. User query is sent to a provider; the model may request one or more function calls.
2. Server detects function calls (`needFeedbacks` for PHP, `needClientActions` for client) and executes via WP filters.
3. Results are packaged in a Feedback query and sent back to the model.
4. The loop can repeat up to a max depth (default 5).

Key classes: `Meow_MWAI_Query_Text`, `Meow_MWAI_Query_Feedback`, `Meow_MWAI_Reply`, `Meow_MWAI_Engines_Core`.

Responses API support maintains state with `previous_response_id`; only new messages are sent between turns.

## Global Token Manager & Authentication

Long-lived sessions rely on a global REST nonce manager in the frontend (`app/js/helpers/tokenManager.js`).

- Server refreshes tokens when existing ones age (WordPress returns `2`).
- Clients subscribe to updates; React components use a ref for immediate token access.
- Admin UI integrates via the shared `nekoFetch` wrapper and window state.

## Testing & Debugging

- See `labs/tests/README.md` and scripts in `labs/tests/` for embeddings, AI features, audio, images, sessions, etc.
- Useful quick start:
  - `cd labs/tests`
  - `node test-embeddings.js help`
- Log monitoring (example path):
  - `tail -f ~/sites/ai/logs/php/error.log`
  - `grep -i "error\|exception" ~/sites/ai/logs/php/error.log | tail -20`
  - Enable “Queries Debug” in plugin settings for detailed traces.

## Commit Message Notes

- Keep messages short and direct; avoid mentioning AI agent names.
- Follow existing repo guidance for PR requirements and quality checks.
