<?php
/**
 * AFGF Plugin Bootstrap
 *
 * @package     AffiliateWP Affiliate Forms for Gravity Forms
 * @subpackage  Core
 * @copyright   Copyright (c) 2021, Sandhills Development, LLC
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       1.1
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) exit;

if ( ! class_exists( 'AffiliateWP_Affiliate_Forms_For_Gravity_Forms' ) ) {

	/**
	 * Affiliate Forms for Gravity Forms loader.
	 *
	 * @since 1.0
	 */
	final class AffiliateWP_Affiliate_Forms_For_Gravity_Forms {

		/**
		 * Holds the instance
		 *
		 * Ensures that only one instance of AffiliateWP_Affiliate_Forms_For_Gravity_Forms exists in memory at any one
		 * time and it also prevents needing to define globals all over the place.
		 *
		 * TL;DR This is a static property property that holds the singleton instance.
		 *
		 * @access private
		 * @since  1.0
		 * @var    AffiliateWP_Affiliate_Forms_For_Gravity_Forms
		 * @static
		 */
		private static $instance;

		/**
		 * The plugin's version number.
		 *
		 * @access private
		 * @since  1.0
		 * @var    string
		 * @static
		 */
		private static $version = '1.1.1';

		/**
		 * Main plugin file.
		 *
		 * @since 1.1
		 * @var   string
		 */
		private $file = '';

		/**
		 * The affiliate registration handler instance variable.
		 *
		 * @access public
		 * @since  1.0
		 * @var    Affiliate_WP_Gravity_forms_Register
		 */
		public $register;

		/**
		 * Fields component.
		 *
		 * @since 1.1
		 * @var   AffiliateWP_AFGF\Fields
		 */
		public $fields;

		/**
		 * Main AffiliateWP_Affiliate_Forms_For_Gravity_Forms Instance.
		 *
		 * Insures that only one instance of AffiliateWP_Affiliate_Forms_For_Gravity_Forms exists
		 * in memory at any one time. Also prevents needing to define globals all over the place.
		 *
		 * @access public
		 * @since  1.0
		 * @static
		 *
		 * @param string $file Main plugin file.
		 * @return \AffiliateWP_Affiliate_Forms_For_Gravity_Forms The one true plugin instance.
		 */
		public static function instance( $file = null ) {

			if ( ! isset( self::$instance ) && ! ( self::$instance instanceof AffiliateWP_Affiliate_Forms_For_Gravity_Forms ) ) {

				self::$instance       = new AffiliateWP_Affiliate_Forms_For_Gravity_Forms;
				self::$instance->file = $file;

				self::$instance->setup_constants();
				self::$instance->load_textdomain();
				self::$instance->includes();
				self::$instance->init();
				self::$instance->hooks();
				self::$instance->setup_objects();
			}

			return self::$instance;
		}

		/**
		 * Throws an error on object clone.
		 *
		 * The whole idea of the singleton design pattern is that there is a single
		 * object therefore, we don't want the object to be cloned.
		 *
		 * @access public
		 * @since  1.0
		 *
		 * @return void
		 */
		public function __clone() {
			// Cloning instances of the class is forbidden
			_doing_it_wrong( __FUNCTION__, __( 'Cheatin&#8217; huh?', 'affiliatewp-afgf' ), '1.0' );
		}

		/**
		 * Disables unserializing of the class.
		 *
		 * @access public
		 * @since  1.0
		 *
		 * @return void
		 */
		public function __wakeup() {
			// Unserializing instances of the class is forbidden
			_doing_it_wrong( __FUNCTION__, __( 'Cheatin&#8217; huh?', 'affiliatewp-afgf' ), '1.0' );
		}

		/**
		 * Sets up the instance.
		 *
		 * @access private
		 * @since  1.0
		 */
		private function __construct() {
			self::$instance = $this;
		}

		/**
		 * Resets the class instance.
		 *
		 * @access public
		 * @since  1.0
		 * @static
		 */
		public static function reset() {
			self::$instance = null;
		}

		/**
		 * Sets up the plugin constants.
		 *
		 * @access private
		 * @since  1.0
		 *
		 * @return void
		 */
		private function setup_constants() {

			// Plugin version
			if ( ! defined( 'AFFWP_AFGF_VERSION' ) ) {
				define( 'AFFWP_AFGF_VERSION', self::$version );
			}

			// Plugin Folder Path
			if ( ! defined( 'AFFWP_AFGF_PLUGIN_DIR' ) ) {
				define( 'AFFWP_AFGF_PLUGIN_DIR', plugin_dir_path( $this->file ) );
			}

			// Plugin Folder URL
			if ( ! defined( 'AFFWP_AFGF_PLUGIN_URL' ) ) {
				define( 'AFFWP_AFGF_PLUGIN_URL', plugin_dir_url( $this->file ) );
			}

			// Plugin Root File
			if ( ! defined( 'AFFWP_AFGF_PLUGIN_FILE' ) ) {
				define( 'AFFWP_AFGF_PLUGIN_FILE', $this->file );
			}

		}

		/**
		 * Loads the plugin language files.
		 *
		 * @access public
		 * @since  1.0
		 *
		 * @return void
		 */
		public function load_textdomain() {

			// Set filter for plugin's languages directory
			$lang_dir = dirname( plugin_basename( $this->file ) ) . '/languages/';
			$lang_dir = apply_filters( 'affiliatewp_afgf_languages_directory', $lang_dir );

			// Traditional WordPress plugin locale filter
			$locale   = apply_filters( 'plugin_locale',  get_locale(), 'affiliatewp-afgf' );
			$mofile   = sprintf( '%1$s-%2$s.mo', 'affiliatewp-afgf', $locale );

			// Setup paths to current locale file
			$mofile_local  = $lang_dir . $mofile;
			$mofile_global = WP_LANG_DIR . '/affiliatewp-afgf/' . $mofile;

			if ( file_exists( $mofile_global ) ) {
				// Look in global /wp-content/languages/affiliatewp-afgf/ folder
				load_textdomain( 'affiliatewp-afgf', $mofile_global );
			} elseif ( file_exists( $mofile_local ) ) {
				// Look in local /wp-content/plugins/affiliatewp-afgf/languages/ folder
				load_textdomain( 'affiliatewp-afgf', $mofile_local );
			} else {
				// Load the default language files
				load_plugin_textdomain( 'affiliatewp-afgf', false, $lang_dir );
			}
		}

		/**
		 * Includes the necessary plugin files.
		 *
		 * @access private
		 * @since  1.0
		 *
		 * @return void
		 */
		private function includes() {
			// Bring in the autoloader.
			require_once AFFWP_AFGF_PLUGIN_DIR . 'includes/lib/afgf-autoloader.php';

			require_once AFFWP_AFGF_PLUGIN_DIR . 'includes/functions.php';
			require_once AFFWP_AFGF_PLUGIN_DIR . 'includes/class-register.php';
			require_once AFFWP_AFGF_PLUGIN_DIR . 'includes/fields.php';
			require_once AFFWP_AFGF_PLUGIN_DIR . 'includes/email-tags.php';

			require_once AFFWP_AFGF_PLUGIN_DIR . 'includes/admin.php';
			require_once AFFWP_AFGF_PLUGIN_DIR . 'includes/shortcodes.php';
			require_once AFFWP_AFGF_PLUGIN_DIR . 'includes/class-upgrader.php';

			// must be an affiliate registration form set up
			if ( affwp_afgf_get_registration_form_id() ) {
				require_once AFFWP_AFGF_PLUGIN_DIR . 'includes/emails.php';
			}


		}

		/**
		 * Initializes the plugin.
		 *
		 * @access private
		 * @since  1.0
		 *
		 * @return void
		 */
		private function init() {

			if ( is_admin() ) {
				self::$instance->updater();
			}

		}

		/**
		 * Sets up the default actions and filters.
		 *
		 * @access private
		 * @since  1.0
		 *
		 * @return void
		 */
		private function hooks() {

			// Plugin meta.
			add_filter( 'plugin_row_meta', array( $this, 'plugin_meta' ), null, 2 );

			// Add the affiliate.
			add_action( 'gform_entry_created', array( $this, 'add_affiliate' ), 10, 2 );
		}

		/**
		 * Sets up objects against the plugin instance.
		 *
		 * @since 1.1
		 */
		private function setup_objects() {
			self::$instance->register = new Affiliate_WP_Gravity_forms_Register;
			self::$instance->fields   = new \AffiliateWP_AFGF\Fields;
		}

		/**
		 * Determines whether Gravity Forms 2.5+ is installed.
		 *
		 * Their new block-like editing experience was introduced in 2.5.
		 *
		 * @since 1.1
		 */
		public function has_gf_25() {
			return get_option( 'rg_form_version' );
		}

		/**
		 * Adds an affiliate account.
		 *
		 * @access public
		 * @since 1.0
		 *
		 * @param array $entry Entry data.
		 * @param array $form  Form data.
		 */
		public function add_affiliate( $entry, $form ) {

			$form_id = affwp_afgf_get_registration_form_id();

			// Bail if not the registration form.
			if ( ! $form_id || (int) $form['id'] !== $form_id ) {
				return;
			}

			// Register the user.
			$this->register->register_user( $entry, $form );
		}

		/**
		 * Modifies the plugin meta links in plugins list table.
		 *
		 * @access public
		 * @since  1.0
		 *
		 * @param array  $links The current links array.
		 * @param string $file  A specific plugin table entry.
		 * @return array The modified links array.
		 */
		public function plugin_meta( $links, $file ) {
			if ( $file == plugin_basename( $this->file ) ) {
				$plugins_link = array(
					'<a title="' . __( 'Get more add-ons for AffiliateWP', 'affiliatewp-afgf' ) . '" href="http://affiliatewp.com/addons/" target="_blank">' . __( 'Get add-ons', 'affiliatewp-afgf' ) . '</a>'
				);

				$links = array_merge( $links, $plugins_link );
			}

			return $links;
		}

		/**
		 * Loads the custom plugin updater.
		 *
		 * @access private
		 * @since  1.0
		 *
		 * @see AffWP_AddOn_Updater
		 *
		 * @return void
		 */
		public function updater() {

			if ( class_exists( 'AffWP_AddOn_Updater' ) ) {
				$updater = new AffWP_AddOn_Updater( 25083, $this->file, self::$version );
			}
		}
	}

	/**
	 * The main function responsible for returning the one true AffiliateWP_Affiliate_Forms_For_Gravity_Forms
	 * Instance to functions everywhere.
	 *
	 * Use this function like you would a global variable, except without needing
	 * to declare the global.
	 *
	 * Example:
	 *
	 *     $affiliatewp_afgf = affiliatewp_afgf();
	 *
	 * @since 1.0
	 *
	 * @return \AffiliateWP_Affiliate_Forms_For_Gravity_Forms The one true class instance.
	 */
	function affiliatewp_afgf() {
		return AffiliateWP_Affiliate_Forms_For_Gravity_Forms::instance();
	}

}
