<?php
/**
 * Core: Menu Links
 *
 * @package     AffiliateWP Affiliate Portal
 * @subpackage  Core/Components
 * @copyright   Copyright (c) 2021, Sandhills Development, LLC
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       1.0.8
 */
namespace AffiliateWP_Affiliate_Portal\Core;

/**
 * Manages retrieving and manipulating custom menu links.
 *
 * @since 1.0.8
 */
class Menu_Links {

	/**
	 * Sets sup hook callbacks needed by the component.
	 *
	 * @since 1.0.8
	 */
	public function init() {}

	/**
	 * Determines whether the given slug is a custom menu link.
	 *
	 * @since 1.0.8
	 *
	 * @param string $slug Menu link slug.
	 * @return bool True if the slug matches a custom menu link, otherwise false.
	 */
	public function is_menu_link( $slug ) {
		$menu_links = $this->get_menu_links();

		return array_key_exists( $slug, $menu_links );
	}

	/**
	 * Retrieves the list of custom menu links.
	 *
	 * @since 1.0.8
	 *
	 * @return array Array of the custom menu links.
	 */
	public function get_menu_links() {
		$affiliate_portal_settings = affiliate_wp()->settings->get( 'affiliate_portal', array() );

		$menu_links = isset( $affiliate_portal_settings['portal_menu_links'] ) ? $affiliate_portal_settings['portal_menu_links'] : array();

		// filter only menu links with a page.
		$menu_links = array_filter(
			$menu_links,
			function( $menu_link) {
				return ! empty( $menu_link['id'] );
			}
		);

		return $menu_links;
	}

	/**
	 * Retrieves a list of pages minus the Affiliate Area page.
	 *
	 * @since 1.0.8
	 *
	 * @return array Array of pages.
	 */
	public function get_pages() {
		$pages             = affwp_get_pages();
		$affiliate_area_id = affiliate_wp()->settings->get( 'affiliates_page' );

		if ( ! empty( $pages[ $affiliate_area_id ] ) ) {
			unset( $pages[ $affiliate_area_id ] );
		}

		return $pages;
	}

	/**
	 * Makes a slug for the menu link.
	 *
	 * @since 1.0.8
	 *
	 * @param string $title Menu link title.
	 * @return string Slug.
	 */
	public function make_slug( $title ) {

		$slug = rawurldecode( sanitize_title_with_dashes( $title ) );

		return $slug;
	}

	/**
	 * Get links slug/title pairs.
	 *
	 * @since 1.0.8
	 *
	 * @return array The array of links to show.
	 */
	public function get_link_pairs() {

		$links = array();

		$saved_links = $this->get_menu_links();

		if ( $saved_links ) {

			foreach ( $saved_links as $link ) {
				if ( isset( $link['slug'] ) ) {
					$links[ $link['slug'] ] = $link['label'];
				}
			}
		}

		return $links;
	}


}
