<?php

/**
 * YITH Wishlist plugin support
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

if (!class_exists('AWL_Yith_Wishlist')) :

    /**
     * Class for main plugin functions
     */
    class AWL_Yith_Wishlist
    {

        /**
         * @var AWL_Yith_Wishlist The single instance of the class
         */
        protected static $_instance = null;

        /**
         * Main AWL_Yith_Wishlist Instance
         *
         * Ensures only one instance of AWL_Yith_Wishlist is loaded or can be loaded.
         *
         * @static
         * @return AWL_Yith_Wishlist - Main instance
         */
        public static function instance() {
            if (is_null(self::$_instance)) {
                self::$_instance = new self();
            }
            return self::$_instance;
        }

        /*
         * Constructor
         */
        public function __construct() {

            add_filter( 'awl_labels_text_vars', array( $this, 'labels_text_vars' ), 1 );

            add_filter( 'awl_label_rules', array( $this, 'label_rules' ), 1 );

            add_filter( 'awl_labels_condition_rules', array( $this, 'condition_rules' ), 1 );

            add_filter( 'awl_labels_hooks', array( $this, 'awl_labels_hooks' ), 2 );

            add_filter( 'awl_js_container_selectors', array( $this, 'awl_js_container_selectors' ) );

            add_action( 'wp_footer', array( $this, 'add_styles' ) );

        }

        /*
         * Add new text variables
         */
        public function labels_text_vars( $variables ) {

            $variable['/{YITH_WISH_COUNT}/i'] = array(
                'func' => array( $this, 'wishlist_count_text_var' ),
                'desc' => 'total number of wishlists that contain current product',
                'example' => '{YITH_WISH_COUNT}',
            );

            $variables = array_merge( $variable, $variables );

            return $variables;

        }

        /*
         * Wishlist product count text variable
         */
        public function wishlist_count_text_var( $matches = '' ) {
            global $product;
            $value = '';
            if ( function_exists( 'yith_wcwl_count_add_to_wishlist' ) ) {
                $value = yith_wcwl_count_add_to_wishlist( $product->get_id() );
            }
            return $value;
        }

        /*
         * Add new label conditions for admin
         */
        public function label_rules( $options ) {

            $options['Special'][] = array(
                "name" => __( "YITH Wishlist: Is product in current user wishlist", "advanced-woo-labels" ),
                "id"   => "yith_wish_is_in_list",
                "type" => "bool",
                "operators" => "equals",
            );

            $options['Special'][] = array(
                "name" => __( "YITH Wishlist: Is wishlist page", "advanced-woo-labels" ),
                "id"   => "yith_wish_is_page",
                "type" => "bool",
                "operators" => "equals",
            );

            $options['Special'][] = array(
                "name" => __( "YITH Wishlist: Product number of wishlists", "advanced-woo-labels" ),
                "id"   => "yith_wish_in_wishlist",
                "type" => "number",
                "operators" => "equals_compare",
            );

            return $options;

        }

        /*
         * Add custom condition rule method
         */
        public function condition_rules( $rules ) {
            $rules['yith_wish_is_in_list'] = array( $this, 'yith_wish_is_in_list' );
            $rules['yith_wish_is_page'] = array( $this, 'yith_wish_is_page' );
            $rules['yith_wish_in_wishlist'] = array( $this, 'yith_wish_in_wishlist' );
            return $rules;
        }

        /*
         * Condition: Is product in users wishlist
         */
        public function yith_wish_is_in_list( $condition_rule ) {
            global $product;

            $match = false;
            $operator = $condition_rule['operator'];
            $value = $condition_rule['value'];
            $compare_value = '';

            if ( function_exists( 'YITH_WCWL' ) ) {
                $compare_value = YITH_WCWL()->is_product_in_wishlist( $product->get_id() );
            }

            if ( 'equal' == $operator ) {
                $match = ($compare_value == $value);
            } elseif ( 'not_equal' == $operator ) {
                $match = ($compare_value != $value);
            }

            return $match;

        }

        /*
         * Condition: Is wishlist page
         */
        public function yith_wish_is_page( $condition_rule ) {
            global $product;

            $match = false;
            $operator = $condition_rule['operator'];
            $value = $condition_rule['value'];
            $compare_value = '';

            if ( function_exists( 'yith_wcwl_is_wishlist_page' ) ) {
                $compare_value = yith_wcwl_is_wishlist_page();
            }

            if ( 'equal' == $operator ) {
                $match = ($compare_value == $value);
            } elseif ( 'not_equal' == $operator ) {
                $match = ($compare_value != $value);
            }

            return $match;

        }

        /*
         * Condition: Number of times product was added to wishlist
         */
        public function yith_wish_in_wishlist( $condition_rule ) {
            global $product;

            $match = false;
            $operator = $condition_rule['operator'];
            $value = $condition_rule['value'];
            $compare_value = '';

            if ( function_exists( 'yith_wcwl_count_add_to_wishlist' ) ) {
                $compare_value = yith_wcwl_count_add_to_wishlist( $product->get_id() );
            }

            if ( 'equal' == $operator ) {
                $match = ($compare_value == $value);
            } elseif ( 'not_equal' == $operator ) {
                $match = ($compare_value != $value);
            } elseif ( 'greater' == $operator ) {
                $match = ($compare_value >= $value);
            } elseif ( 'less' == $operator ) {
                $match = ($compare_value <= $value);
            }

            return $match;

        }

        /*
         * Change display hooks
         */
        public function awl_labels_hooks( $hooks ) {

            // standard
            $hooks['on_image']['archive']['yith_wcwl_table_after_product_thumbnail'] = array( 'priority' => 10 );
            $hooks['before_title']['archive']['yith_wcwl_table_before_product_name'] = array( 'priority' => 10 );

            // image
            $hooks['on_image']['archive']['yith_wcwl_wishlist_view_images_after_thumbnail'] = array( 'priority' => 10, 'js' =>  array( '.product-thumbnail', 'append' ) );

            // modern
            $hooks['on_image']['archive']['yith_wcwl_table_after_product_name'] = array( 'priority' => 10, 'js' =>  array( '.product-thumbnail', 'append' ) );
            $hooks['before_title']['archive']['yith_wcwl_table_after_product_name'] = array( 'priority' => 11, 'js' =>  array( '.item-details-wrapper .product-name', 'before' ) );

            return $hooks;

        }

        /*
         * Add custom container selector for js hooks
         */
        public function awl_js_container_selectors( $product_container ) {
            $product_container[] = '[data-row-id]';
            return $product_container;
        }

        /*
         * Add custom styles to wishlist page
         */
        public function add_styles() {

            if ( ! function_exists( 'yith_wcwl_is_wishlist_page' ) || ! yith_wcwl_is_wishlist_page() ) {
                return;
            }

            ?>

            <style>
                .shop_table .product-thumbnail {
                    position: relative;
                }
            </style>

        <?php }

    }

endif;

AWL_Yith_Wishlist::instance();