<?php

/**
 *  WooCommerce Memberships plugin integration
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

if (!class_exists('AWL_Woo_Memberships')) :

    /**
     * Class for main plugin functions
     */
    class AWL_Woo_Memberships {

        /**
         * @var AWL_Woo_Memberships Custom data
         */
        public $data = array();

        /**
         * @var AWL_Woo_Memberships The single instance of the class
         */
        protected static $_instance = null;

        /**
         * Main AWL_Woo_Memberships Instance
         *
         * Ensures only one instance of AWL_Woo_Memberships is loaded or can be loaded.
         *
         * @static
         * @return AWL_Woo_Memberships - Main instance
         */
        public static function instance()
        {
            if (is_null(self::$_instance)) {
                self::$_instance = new self();
            }
            return self::$_instance;
        }

        /**
         * Constructor
         */
        public function __construct() {

            add_filter( 'awl_label_rules', array( $this, 'label_rules' ), 1 );

            add_filter( 'awl_labels_condition_rules', array( $this, 'condition_rules' ), 1 );

            add_filter( 'awl_labels_text_vars', array( $this, 'awl_labels_text_vars' ), 1 );

        }

        /*
         * Add new label conditions for admin
         */
        public function label_rules( $options ) {

            $options['Woo Memberships'][] = array(
                "name" => __( "Current user membership", "advanced-woo-labels" ),
                "id"   => "woo_memberships_is_current_member",
                "type" => "callback",
                "operators" => "equals",
                "choices" => array(
                    'callback' => array($this, 'get_all_plans'),
                    'params'   => array()
                ),
            );

            $options['Woo Memberships'][] = array(
                "name" => __( "Current user can purchase", "advanced-woo-labels" ),
                "id"   => "woo_memberships_can_purchase",
                "type" => "bool",
                "operators" => "equals",
            );

            $options['Woo Memberships'][] = array(
                "name" => __( "Current user can view", "advanced-woo-labels" ),
                "id"   => "woo_memberships_can_view",
                "type" => "bool",
                "operators" => "equals",
            );

            $options['Woo Memberships'][] = array(
                "name" => __( "Current user has discount", "advanced-woo-labels" ),
                "id"   => "woo_memberships_has_discount",
                "type" => "bool",
                "operators" => "equals",
            );

            $options['Woo Memberships'][] = array(
                "name" => __( "Product has discount for", "advanced-woo-labels" ),
                "id"   => "woo_memberships_has_discount_for",
                "type" => "callback",
                "operators" => "equals",
                "choices" => array(
                    'callback' => array($this, 'get_all_plans'),
                    'params'   => array()
                ),
            );

            $options['Woo Memberships'][] = array(
                "name" => __( "Product grant access to", "advanced-woo-labels" ),
                "id"   => "woo_memberships_grant_access_to",
                "type" => "callback",
                "operators" => "equals",
                "choices" => array(
                    'callback' => array($this, 'get_all_plans'),
                    'params'   => array()
                ),
            );

            $options['Woo Memberships'][] = array(
                "name" => __( "Who can purchase product", "advanced-woo-labels" ),
                "id"   => "woo_memberships_who_can_purchase",
                "type" => "callback",
                "operators" => "equals",
                "choices" => array(
                    'callback' => array($this, 'get_all_plans'),
                    'params'   => array()
                ),
            );

            $options['Woo Memberships'][] = array(
                "name" => __( "Who can view product", "advanced-woo-labels" ),
                "id"   => "woo_memberships_who_can_view",
                "type" => "callback",
                "operators" => "equals",
                "choices" => array(
                    'callback' => array($this, 'get_all_plans'),
                    'params'   => array()
                ),
            );

            return $options;

        }

        /*
         * Add custom condition rule method
         */
        public function condition_rules( $rules ) {
            $rules['woo_memberships_is_current_member'] = array( $this, 'woo_memberships_is_current_member' );
            $rules['woo_memberships_can_purchase'] = array( $this, 'woo_memberships_can_purchase' );
            $rules['woo_memberships_can_view'] = array( $this, 'woo_memberships_can_view' );
            $rules['woo_memberships_has_discount'] = array( $this, 'woo_memberships_has_discount' );
            $rules['woo_memberships_has_discount_for'] = array( $this, 'woo_memberships_has_discount_for' );
            $rules['woo_memberships_grant_access_to'] = array( $this, 'woo_memberships_grant_access_to' );
            $rules['woo_memberships_who_can_purchase'] = array( $this, 'woo_memberships_who_can_purchase' );
            $rules['woo_memberships_who_can_view'] = array( $this, 'woo_memberships_who_can_view' );
            return $rules;
        }

        /*
         * Condition: Is user is WooCommerce Memberships member
         */
        public function woo_memberships_is_current_member( $condition_rule ) {

            global $product;

            $memberships = function_exists('wc_memberships_get_user_active_memberships') ? wc_memberships_get_user_active_memberships() : false;

            $match = false;
            $operator = $condition_rule['operator'];
            $value = $condition_rule['value'];

            $is_member = false;
            if ( $memberships && ! empty( $memberships ) ) {
                foreach ( $memberships as $membership ) {
                    if ( $membership && $membership->plan ) {
                        if ( $value === 'awl_any' || intval( $value ) === $membership->plan->get_id() ) {
                            $is_member = true;
                            break;
                        }
                    }
                }
            }

            $match = 'equal' == $operator ? $is_member : ! $is_member;

            return $match;

        }

        /*
         * Condition: Is user can purchase product
         */
        public function woo_memberships_can_purchase( $condition_rule ) {

            global $product;

            $compare_value = current_user_can( 'wc_memberships_purchase_restricted_product', $product->get_id() ) && current_user_can( 'wc_memberships_purchase_delayed_product', $product->get_id() ) ? 'true' : 'false';

            $match = false;
            $operator = $condition_rule['operator'];
            $value = $condition_rule['value'];

            if ( 'equal' == $operator ) {
                $match = ($compare_value == $value);
            } elseif ( 'not_equal' == $operator ) {
                $match = ($compare_value != $value);
            }

            return $match;

        }

        /*
         * Condition: Is user can view product
         */
        public function woo_memberships_can_view( $condition_rule ) {

            global $product;

            $compare_value = current_user_can( 'wc_memberships_view_restricted_product', $product->get_id() ) && current_user_can( 'wc_memberships_view_delayed_product', $product->get_id() ) ? 'true' : 'false';
            $match = false;
            $operator = $condition_rule['operator'];
            $value = $condition_rule['value'];

            if ( 'equal' == $operator ) {
                $match = ($compare_value == $value);
            } elseif ( 'not_equal' == $operator ) {
                $match = ($compare_value != $value);
            }

            return $match;

        }

        /*
         * Condition: Is user has product discount
         */
        public function woo_memberships_has_discount( $condition_rule ) {

            global $product;

            $compare_value = wc_memberships_user_has_member_discount( $product ) && ! wc_memberships_is_product_excluded_from_member_discounts( $product ) ? 'true' : 'false';
            $match = false;
            $operator = $condition_rule['operator'];
            $value = $condition_rule['value'];

            if ( 'equal' == $operator ) {
                $match = ($compare_value == $value);
            } elseif ( 'not_equal' == $operator ) {
                $match = ($compare_value != $value);
            }

            return $match;

        }

        /*
         * Condition: What memberships has discount for current product
         */
        public function woo_memberships_has_discount_for( $condition_rule ) {

            global $product;

            $match = false;
            $operator = $condition_rule['operator'];
            $value = $condition_rule['value'];

            $has_discount = false;
            $has_any_discount = wc_memberships_product_has_member_discount( $product->get_id() );

            if ( $has_any_discount ) {

                if ( $value === 'awl_any' ) {

                    $has_discount = true;

                } else {

                    $all_discount_rules = wc_memberships()->get_rules_instance()->get_product_purchasing_discount_rules( $product->get_id() );

                    if ( ! empty( $all_discount_rules ) ) {
                        foreach ( $all_discount_rules as $rule ) {
                            if ( $rule->is_active() && $rule->get_membership_plan_id() === intval( $value ) ) {
                                $has_discount = true;
                                break;
                            }
                        }
                    }

                }

            }

            $match = 'equal' == $operator ? $has_discount : ! $has_discount;

            return $match;

        }

        /*
         * Condition: To what memberships product purchase grant access to
         */
        public function woo_memberships_grant_access_to( $condition_rule ) {

            global $product;

            $match = false;
            $operator = $condition_rule['operator'];
            $value = $condition_rule['value'];

            $grant_access = false;
            $memberships = wc_memberships()->get_plans_instance()->get_membership_plans_for_product( $product );

            if ( $memberships && ! empty( $memberships ) ) {

                if ( $value === 'awl_any' ) {

                    $grant_access = true;

                } else {

                    foreach ( $memberships as $membership ) {
                        if ( $membership->get_id() === intval( $value ) ) {
                            $grant_access = true;
                            break;
                        }
                    }

                }

            }

            $match = 'equal' == $operator ? $grant_access : ! $grant_access;

            return $match;

        }

        /*
         * Condition: Memberships that can purchase current product
         */
        public function woo_memberships_who_can_purchase( $condition_rule ) {

            global $product;

            $match = false;
            $operator = $condition_rule['operator'];
            $value = $condition_rule['value'];

            $can_purchase = true;
            $rules = wc_memberships()->get_rules_instance()->get_product_restriction_rules( $product->get_id() );

            if ( ! empty( $rules ) ) {
                foreach ( $rules as $rule ) {

                    if ( 'purchase' === $rule->get_access_type() ) {

                        $can_purchase = false;

                        if ( $value === 'awl_any' ) {
                            break;
                        }

                        if ( $rule->get_membership_plan_id() === intval( $value ) ) {
                            $can_purchase = true;
                            break;
                        }

                    }
                }
            }

            $match = 'equal' == $operator ? $can_purchase : ! $can_purchase;

            return $match;

        }

        /*
         * Condition: Memberships that can view current product
         */
        public function woo_memberships_who_can_view( $condition_rule ) {

            global $product;

            $match = false;
            $operator = $condition_rule['operator'];
            $value = $condition_rule['value'];

            $can_view = true;
            $rules = wc_memberships()->get_rules_instance()->get_product_restriction_rules( $product->get_id() );

            if ( ! empty( $rules ) ) {
                foreach ( $rules as $rule ) {

                    if ( 'view' === $rule->get_access_type() ) {

                        $can_view = false;

                        if ( $value === 'awl_any' ) {
                            break;
                        }

                        if ( $rule->get_membership_plan_id() === intval( $value ) ) {
                            $can_view = true;
                            break;
                        }

                    }
                }
            }

            $match = 'equal' == $operator ? $can_view : ! $can_view;

            return $match;

        }

        /*
         * Get all membership plans
         */
        public function get_all_plans() {

            $options = array();

            $options['awl_any'] = __( "Any membership plan", "advanced-woo-labels" );

            $plans = wc_memberships_get_membership_plans();

            if ( ! empty( $plans ) ) {
                foreach ( $plans as $plan ) {
                    $options[$plan->id] = $plan->name;
                }
            }

            return $options;

        }

        /*
         * Set Dokan custom text variables
         */
        public function awl_labels_text_vars( $variables ) {

            $variable['/{WCMEMBER_USER_MEMBERSHIP}/i'] = array(
                'func' => array( $this, 'wcmember_text_user_membership' ),
                'desc' => __( "WooCommerce Memberships: Current user membership plan", "advanced-woo-labels" ),
                'example' => '{WCMEMBER_USER_MEMBERSHIP}',
            );

            $variable['/{WCMEMBER_DISCOUNT_AMOUNT}/i'] = array(
                'func' => array( $this, 'wcmember_text_discount_amount' ),
                'desc' => __( "WooCommerce Memberships: Current member product discount amount", "advanced-woo-labels" ),
                'example' => '{WCMEMBER_DISCOUNT_AMOUNT}',
            );

            $variable['/{WCMEMBER_DISCOUNT_PERCENT}/i'] = array(
                'func' => array( $this, 'wcmember_text_discount_percentage' ),
                'desc' => __( "WooCommerce Memberships: Current member product discount percentage", "advanced-woo-labels" ),
                'example' => '{WCMEMBER_DISCOUNT_PERCENT}',
            );

            $variable['/{WCMEMBER_GRANT_ACCESS_TO}/i'] = array(
                'func' => array( $this, 'wcmember_text_grant_access_to' ),
                'desc' => __( "WooCommerce Memberships: List of memberships that current product purchase grant access to", "advanced-woo-labels" ),
                'example' => '{WCMEMBER_GRANT_ACCESS_TO}',
            );

            $variable['/{WCMEMBER_WHO_CAN_PURCHASE}/i'] = array(
                'func' => array( $this, 'wcmember_text_who_can_purchase' ),
                'desc' => __( "WooCommerce Memberships: List of memberships with access to purchase current product", "advanced-woo-labels" ),
                'example' => '{WCMEMBER_WHO_CAN_PURCHASE}',
            );

            $variable['/{WCMEMBER_RESTRICTED_TO}/i'] = array(
                'func' => array( $this, 'wcmember_text_product_restricted_to' ),
                'desc' => __( "WooCommerce Memberships: List of memberships to which the current product is restricted", "advanced-woo-labels" ),
                'example' => '{WCMEMBER_RESTRICTED_TO}',
            );

            $variables = array_merge( $variable, $variables );

            return $variables;

        }

        /*
         * Text variable: WooCommerce Memberships current user membership plan
         */
        public function wcmember_text_user_membership( $matches ) {

            $plan = '';

            $memberships = function_exists('wc_memberships_get_user_active_memberships') ? wc_memberships_get_user_active_memberships() : false;

            if ( $memberships && ! empty( $memberships ) ) {
                $memberships_names = array();
                foreach ( $memberships as $membership ) {
                    if ( $membership && $membership->plan ) {
                        $memberships_names[] = $membership->plan->name;
                    }
                }
                $plan = implode(', ', $memberships_names);
            }

            return $plan;

        }

        /*
         * Text variable: WooCommerce Memberships list of memberships for which the current product is restricted
         */
        public function wcmember_text_product_restricted_to( $matches ) {

            global $product;

            $memberships_string = '';

            $memberships = $this->product_restricted_to_plans( $product->get_id() );

            if ( ! empty( $memberships ) ) {
                $memberships_string = implode(', ', $memberships );
            }

            return $memberships_string;

        }

        /*
         * Text variable: WooCommerce Memberships current user product discount amount
         */
        public function wcmember_text_discount_amount( $matches ) {

            global $product;

            $discount = 0;

            if ( wc_memberships_user_has_member_discount( $product ) && ! wc_memberships_is_product_excluded_from_member_discounts( $product ) ) {
                $discount = wc_memberships()->get_member_discounts_instance()->get_product_discount( $product );
            }

            return $discount;

        }

        /*
         * Text variable: WooCommerce Memberships current user product discount percentage
         */
        public function wcmember_text_discount_percentage( $matches ) {

            global $product;

            $percentage = 0;

            if ( wc_memberships_user_has_member_discount( $product ) && ! wc_memberships_is_product_excluded_from_member_discounts( $product ) ) {

                $discount_amount = wc_memberships()->get_member_discounts_instance()->get_product_discount( $product );

                $price   = $product instanceof \WC_Product ? (float) $product->get_price() : 0;
                $original_price = wc_memberships()->get_member_discounts_instance()->get_original_price( $price, $product );

                if ( $discount_amount <= 0 || $original_price <= 0 ) {
                    $percentage = 0;
                } else {
                    $percentage = $discount_amount / $original_price * 100;
                }

            }

            return $percentage;

        }

        /*
         * Text variable: WooCommerce Memberships list of memberships that current product purchase grant access to
         */
        public function wcmember_text_grant_access_to( $matches ) {

            global $product;

            $memberships_string = '';

            $memberships = wc_memberships()->get_plans_instance()->get_membership_plans_for_product( $product );

            if ( $memberships ) {
                $memberships_names = array();
                foreach ( $memberships as $membership ) {
                    $memberships_names[] = $membership->name;
                }
                $memberships_string = implode(', ', $memberships_names);
            }

            return $memberships_string;

        }

        /*
         * Text variable: WooCommerce Memberships list of memberships with access to purchase current product
         */
        public function wcmember_text_who_can_purchase( $matches ) {

            global $product;

            $memberships_string = '';

            $rules = wc_memberships()->get_rules_instance()->get_product_restriction_rules( $product->get_id() );

            if ( ! empty( $rules ) ) {
                $memberships_names = array();
                foreach ( $rules as $rule ) {
                    if ( 'purchase' === $rule->get_access_type() ) {
                        $membership = $rule->get_membership_plan();
                        $memberships_names[] = $membership->name;
                    }
                }
                $memberships_string = implode(', ', $memberships_names);
            }

            return $memberships_string;

        }

        /*
         * Return array of membership plans current product restricted to
         */
        private function product_restricted_to_plans( $product_id ) {

            $plans = array();

            $memberships = wc_memberships_get_membership_plans();

            if ( $memberships ) {
                foreach ( $memberships as $membership_id => $membership_data ) {
                    $plan = new WC_Memberships_Membership_Plan($membership_id );
                    if ( $plan ) {
                        foreach ( $plan->get_product_restriction_rules() as $rule ) {
                            $data = $rule->get_raw_data();
                            if ( $data ) {
                                if ( $data['object_ids'] && array_search($product_id, $data['object_ids']) !== false ) {
                                    $plans[$membership_id] = $membership_data->name;
                                }
                            }

                        }
                    }
                }
            }

            return $plans;

        }

    }

endif;

AWL_Woo_Memberships::instance();