<?php

/**
 * WCFM plugin integration
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

if (!class_exists('AWL_WCFM')) :

    /**
     * Class for main plugin functions
     */
    class AWL_WCFM {

        /**
         * @var AWL_WCFM Custom data
         */
        public $data = array();

        /**
         * @var AWL_WCFM The single instance of the class
         */
        protected static $_instance = null;

        /**
         * Main AWL_WCFM Instance
         *
         * Ensures only one instance of AWL_WCFM is loaded or can be loaded.
         *
         * @static
         * @return AWL_WCFM - Main instance
         */
        public static function instance()
        {
            if (is_null(self::$_instance)) {
                self::$_instance = new self();
            }
            return self::$_instance;
        }

        /**
         * Constructor
         */
        public function __construct() {

            add_filter( 'awl_label_rules', array( $this, 'label_rules' ), 1 );

            add_filter( 'awl_labels_condition_rules', array( $this, 'condition_rules' ), 1 );

            add_filter( 'awl_labels_text_vars', array( $this, 'awl_labels_text_vars' ), 1 );

        }

        /*
         * Add new label conditions for admin
         */
        public function label_rules( $options ) {

            $options['WCFM'][] = array(
                "name" => __( "Product sold by", "advanced-woo-labels" ),
                "id"   => "wcfm_sold_by",
                "type" => "callback",
                "operators" => "equals",
                "choices" => array(
                    'callback' => array($this, 'get_all_vendors'),
                    'params'   => array()
                ),
            );

            $options['WCFM'][] = array(
                "name" => __( "Is product sold by any vendor", "advanced-woo-labels" ),
                "id"   => "wcfm_is_sold_by_vendor",
                "type" => "bool",
                "operators" => "equals",
            );

            $options['WCFM'][] = array(
                "name" => __( "Is store page", "advanced-woo-labels" ),
                "id"   => "wcfm_is_store_page",
                "type" => "bool",
                "operators" => "equals",
            );

            $options['WCFM'][] = array(
                "name" => __( "Store rating", "advanced-woo-labels" ),
                "id"   => "wcfm_store_rating",
                "type" => "number",
                "step" => "0.01",
                "operators" => "equals_compare",
            );

            $options['WCFM'][] = array(
                "name" => __( "Store reviews count", "advanced-woo-labels" ),
                "id"   => "wcfm_store_reviews",
                "type" => "number",
                "operators" => "equals_compare",
            );

            return $options;

        }

        /*
         * Add custom condition rule method
         */
        public function condition_rules( $rules ) {
            $rules['wcfm_sold_by'] = array( $this, 'wcfm_sold_by' );
            $rules['wcfm_is_sold_by_vendor'] = array( $this, 'wcfm_is_sold_by_vendor' );
            $rules['wcfm_is_store_page'] = array( $this, 'wcfm_is_store_page' );
            $rules['wcfm_store_rating'] = array( $this, 'wcfm_store_rating' );
            $rules['wcfm_store_reviews'] = array( $this, 'wcfm_store_reviews' );
            return $rules;
        }

        /*
         * Condition: Is product sold by vendor
         */
        public function wcfm_sold_by( $condition_rule ) {
            global $product;

            $store_id = $this->get_store_id();

            $match = false;
            $operator = $condition_rule['operator'];
            $value = $condition_rule['value'];
            $compare_value = $store_id ? $store_id : 0;

            if ( ! $store_id ) {
                return false;
            }

            if ( 'equal' == $operator ) {
                $match = ($compare_value == $value);
            } elseif ( 'not_equal' == $operator ) {
                $match = ($compare_value != $value);
            }

            return $match;

        }

        /*
         * Condition: Is product sold by any available vendor
         */
        public function wcfm_is_sold_by_vendor( $condition_rule ) {
            global $product;

            $store_id = $this->get_store_id();

            $match = false;
            $operator = $condition_rule['operator'];
            $value = $condition_rule['value'];
            $compare_value = $store_id ? 'true' : 'false';

            if ( 'equal' == $operator ) {
                $match = ($compare_value == $value);
            } elseif ( 'not_equal' == $operator ) {
                $match = ($compare_value != $value);
            }

            return $match;

        }

        /*
         * Condition: Is inside store page
         */
        public function wcfm_is_store_page( $condition_rule ) {
            global $product;

            $is_shop = function_exists('wcfmmp_is_store_page') ? wcfmmp_is_store_page() : 'false';

            $match = false;
            $operator = $condition_rule['operator'];
            $value = $condition_rule['value'];
            $compare_value = $is_shop ? 'true' : 'false';

            if ( 'equal' == $operator ) {
                $match = ($compare_value == $value);
            } elseif ( 'not_equal' == $operator ) {
                $match = ($compare_value != $value);
            }

            return $match;

        }

        /*
         * Condition: Store rating
         */
        public function wcfm_store_rating( $condition_rule ) {
            global $product;

            $store_id   = $this->get_store_id();
            $store_user = wcfmmp_get_store( $store_id );

            if ( ! $store_user ) {
                return false;
            }

            $match = false;
            $operator = $condition_rule['operator'];
            $value = $condition_rule['value'];
            $compare_value = $store_user->get_avg_review_rating();

            if ( 'equal' == $operator ) {
                $match = ($compare_value == $value);
            } elseif ( 'not_equal' == $operator ) {
                $match = ($compare_value != $value);
            } elseif ( 'greater' == $operator ) {
                $match = ($compare_value >= $value);
            } elseif ( 'less' == $operator ) {
                $match = ($compare_value <= $value);
            }

            return $match;

        }

        /*
         * Condition: Store reviews count
         */
        public function wcfm_store_reviews( $condition_rule ) {
            global $product;

            $store_id   = $this->get_store_id();
            $store_user = wcfmmp_get_store( $store_id );

            if ( ! $store_user ) {
                return false;
            }

            $match = false;
            $operator = $condition_rule['operator'];
            $value = $condition_rule['value'];
            $compare_value = $store_user->get_total_review_count();

            if ( 'equal' == $operator ) {
                $match = ($compare_value == $value);
            } elseif ( 'not_equal' == $operator ) {
                $match = ($compare_value != $value);
            } elseif ( 'greater' == $operator ) {
                $match = ($compare_value >= $value);
            } elseif ( 'less' == $operator ) {
                $match = ($compare_value <= $value);
            }

            return $match;

        }

        /*
         * Condition callback: get all available vendors
         */
        public function get_all_vendors() {

            global $WCFMmp;

            $options = $WCFMmp ? $WCFMmp->wcfmmp_vendor->wcfmmp_search_vendor_list( true ) : array();

            return $options;

        }

        /*
         * Set WCFM custom text variables
         */
        public function awl_labels_text_vars( $variables ) {
            $variable['/{WCFM_SHOP_BADGE}/i'] = array(
                'func' => array( $this, 'wcfm_text_shop_badge' ),
                'desc' => __( "WCFM shop 'Sold by' badge", "advanced-woo-labels" ),
                'example' => '{WCFM_SHOP_BADGE}',
            );
            $variable['/{WCFM_SHOP_NAME}/i'] = array(
                'func' => array( $this, 'wcfm_text_shop_name' ),
                'desc' => __( "WCFM shop name for current product", "advanced-woo-labels" ),
                'example' => '{WCFM_SHOP_NAME}',
            );
            $variable['/{WCFM_SHOP_LOGO}/i'] = array(
                'func' => array( $this, 'wcfm_text_shop_logo' ),
                'desc' => __( "WCFM shop logo for current product", "advanced-woo-labels" ),
                'example' => '{WCFM_SHOP_LOGO}',
            );
            $variable['/{WCFM_SHOP_LINK}/i'] = array(
                'func' => array( $this, 'wcfm_text_shop_link' ),
                'desc' => __( "WCFM shop link for current product", "advanced-woo-labels" ),
                'example' => '{WCFM_SHOP_LINK}',
            );
            $variable['/{WCFM_SHOP_RATING}/i'] = array(
                'func' => array( $this, 'wcfm_text_shop_rating' ),
                'desc' => __( "WCFM shop avarage rating", "advanced-woo-labels" ),
                'example' => '{WCFM_SHOP_RATING}',
            );
            $variable['/{WCFM_SHOP_REVIEWS_NUM}/i'] = array(
                'func' => array( $this, 'wcfm_text_shop_reviews' ),
                'desc' => __( "WCFM shop reviews number", "advanced-woo-labels" ),
                'example' => '{WCFM_SHOP_REVIEWS_NUM}',
            );
            $variables = array_merge( $variable, $variables );
            return $variables;
        }

        /*
         * Text variable: WCFM shop badge
         */
        public function wcfm_text_shop_badge( $matches ) {
            global $product, $WCFMmp;
            $badge = do_shortcode('[wcfm_store_sold_by]');
            $badge = str_replace('class="wcfmmp_sold_by_container_advanced"', 'class="wcfmmp_sold_by_container_advanced" style="margin:0;min-width:auto;"', $badge);
            return $badge;
        }

        /*
         * Text variable: WCFM shop name
         */
        public function wcfm_text_shop_name( $matches ) {
            global $product;
            $store_id = $this->get_store_id();
            $name = $store_id ? do_shortcode('[wcfm_store_info id=' . $store_id . ' data="store_name"]') : '';
            $name = str_replace('<div', '<span', $name);
            $name = str_replace('</div>', '</span>', $name);
            return $name;
        }

        /*
         * Text variable: WCFM shop logo
         */
        public function wcfm_text_shop_logo( $matches ) {
            global $product;
            $store_id = $this->get_store_id();
            $logo = $store_id ? do_shortcode('[wcfm_store_info id=' . $store_id . ' data="store_gravatar"]') : '';
            $logo = str_replace('<img', '<img style="max-width:50px;"', $logo);
            return $logo;
        }

        /*
         * Text variable: WCFM shop link
         */
        public function wcfm_text_shop_link( $matches ) {
            global $product;
            $store_id = $this->get_store_id();
            $link = function_exists('wcfmmp_get_store_url') && $store_id ? wcfmmp_get_store_url( $store_id ) : '';
            return $link;
        }

        /*
         * Text variable: WCFM shop rating
         */
        public function wcfm_text_shop_rating( $matches ) {
            global $product;
            $store_id = $this->get_store_id();
            $store_user = wcfmmp_get_store( $store_id );
            if ( ! $store_user ) {
                return false;
            }
            $rating = $store_user ? $store_user->get_avg_review_rating() : '';
            return $rating;
        }

        /*
         * Text variable: WCFM shop rating
         */
        public function wcfm_text_shop_reviews( $matches ) {
            global $product;
            $store_id = $this->get_store_id();
            $store_user = wcfmmp_get_store( $store_id );
            if ( ! $store_user ) {
                return false;
            }
            $rating = $store_user ? $store_user->get_total_review_count() : '';
            return $rating;
        }

        /*
         * Get store ID for current product
         */
        private function get_store_id() {

            global $post;

            $store_id = 0;

            if( ! $post || ! function_exists('wcfm_is_vendor') || ! wcfm_is_vendor( $post->post_author ) ) {
                return $store_id;
            }

            $store_user = function_exists('wcfmmp_get_store') ? wcfmmp_get_store( $post->post_author ) : 0;
            $is_store_offline = $store_user ? get_user_meta( $store_user->get_id(), '_wcfm_store_offline', true ) : true;
            if ( $is_store_offline ) {
                return $store_id;
            }

            $store_id = $store_user->get_id();

            return $store_id;

        }

    }

endif;

AWL_WCFM::instance();