<?php

/**
 * MultiVendorX – WooCommerce multivendor marketplace plugin support
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

if (!class_exists('AWL_MultiVendorX')) :

    /**
     * Class for main plugin functions
     */
    class AWL_MultiVendorX {

        /**
         * @var AWL_MultiVendorX Custom data
         */
        public $data = array();

        /**
         * @var AWL_MultiVendorX The single instance of the class
         */
        protected static $_instance = null;

        /**
         * Main AWL_MultiVendorX Instance
         *
         * Ensures only one instance of AWL_MultiVendorX is loaded or can be loaded.
         *
         * @static
         * @return AWL_MultiVendorX - Main instance
         */
        public static function instance()
        {
            if (is_null(self::$_instance)) {
                self::$_instance = new self();
            }
            return self::$_instance;
        }

        /**
         * Constructor
         */
        public function __construct() {

            add_filter( 'awl_label_rules', array( $this, 'label_rules' ), 1 );

            add_filter( 'awl_labels_condition_rules', array( $this, 'condition_rules' ), 1 );

            add_filter( 'awl_labels_text_vars', array( $this, 'awl_labels_text_vars' ), 1 );

        }

        /*
         * Add new label conditions for admin
         */
        public function label_rules( $options ) {

            $options['MultiVendorX'][] = array(
                "name" => __( "Product sold by", "advanced-woo-labels" ),
                "id"   => "multivendorx_sold_by",
                "type" => "callback",
                "operators" => "equals",
                "choices" => array(
                    'callback' => array($this, 'get_all_vendors'),
                    'params'   => array()
                ),
            );

            $options['MultiVendorX'][] = array(
                "name" => __( "Is product sold by any vendor", "advanced-woo-labels" ),
                "id"   => "multivendorx_is_sold_by_vendor",
                "type" => "bool",
                "operators" => "equals",
            );

            $options['MultiVendorX'][] = array(
                "name" => __( "Is store page", "advanced-woo-labels" ),
                "id"   => "multivendorx_is_store_page",
                "type" => "bool",
                "operators" => "equals",
            );

            $options['MultiVendorX'][] = array(
                "name" => __( "Store rating", "advanced-woo-labels" ),
                "id"   => "multivendorx_store_rating",
                "type" => "number",
                "step" => "0.01",
                "operators" => "equals_compare",
            );

            $options['MultiVendorX'][] = array(
                "name" => __( "Store reviews count", "advanced-woo-labels" ),
                "id"   => "multivendorx_store_reviews",
                "type" => "number",
                "operators" => "equals_compare",
            );

            $options['MultiVendorX'][] = array(
                "name" => __( "Store products count", "advanced-woo-labels" ),
                "id"   => "multivendorx_store_products",
                "type" => "number",
                "operators" => "equals_compare",
            );

            $options['MultiVendorX'][] = array(
                "name" => __( "Store net sales", "advanced-woo-labels" ),
                "id"   => "multivendorx_store_sales",
                "type" => "number",
                "operators" => "equals_compare",
            );

            $options['MultiVendorX'][] = array(
                "name" => __( "Store items sold", "advanced-woo-labels" ),
                "id"   => "multivendorx_store_solds",
                "type" => "number",
                "operators" => "equals_compare",
            );

            return $options;

        }

        /*
         * Add custom condition rule method
         */
        public function condition_rules( $rules ) {
            $rules['multivendorx_sold_by'] = array( $this, 'multivendorx_sold_by' );
            $rules['multivendorx_is_sold_by_vendor'] = array( $this, 'multivendorx_is_sold_by_vendor' );
            $rules['multivendorx_is_store_page'] = array( $this, 'multivendorx_is_store_page' );
            $rules['multivendorx_store_rating'] = array( $this, 'multivendorx_store_rating' );
            $rules['multivendorx_store_reviews'] = array( $this, 'multivendorx_store_reviews' );
            $rules['multivendorx_store_products'] = array( $this, 'multivendorx_store_products' );
            $rules['multivendorx_store_sales'] = array( $this, 'multivendorx_store_sales' );
            $rules['multivendorx_store_solds'] = array( $this, 'multivendorx_store_solds' );
            return $rules;
        }

        /*
         * Condition: Is product sold by vendor
         */
        public function multivendorx_sold_by( $condition_rule ) {
            global $product;

            $vendor = $this->get_vendor();
            if ( ! $vendor ) {
                return false;
            }

            $operator = $condition_rule['operator'];
            $value = $condition_rule['value'];
            $compare_value = $vendor ? $vendor->get_id() : 0;

            if ( ! $vendor ) {
                return false;
            }

            return $this->compare( $operator, $value, $compare_value );

        }

        /*
         * Condition: Is product sold by any available vendor
         */
        public function multivendorx_is_sold_by_vendor( $condition_rule ) {
            global $product;

            $vendor = $this->get_vendor();

            $operator = $condition_rule['operator'];
            $value = $condition_rule['value'];
            $compare_value = $vendor ? 'true' : 'false';

            return $this->compare( $operator, $value, $compare_value );

        }

        /*
         * Condition: Is inside store page
         */
        public function multivendorx_is_store_page( $condition_rule ) {
            global $product;

            $compare_value = function_exists('mvx_is_store_page') && mvx_is_store_page() ? 'true' : 'false';
            $operator = $condition_rule['operator'];
            $value = $condition_rule['value'];

            return $this->compare( $operator, $value, $compare_value );

        }

        /*
         * Condition: Store rating
         */
        public function multivendorx_store_rating( $condition_rule ) {
            global $product;

            $vendor = $this->get_vendor();
            if ( ! $vendor ) {
                return false;
            }

            $store_info = function_exists('mvx_get_vendor_review_info') ? mvx_get_vendor_review_info($vendor->term_id) : false;
            $operator = $condition_rule['operator'];
            $value = $condition_rule['value'];
            $compare_value = $store_info ? round($store_info['avg_rating'], 2) : '0';

            return $this->compare( $operator, $value, $compare_value );

        }

        /*
         * Condition: Store reviews count
         */
        public function multivendorx_store_reviews( $condition_rule ) {
            global $product;

            $vendor = $this->get_vendor();
            if ( ! $vendor ) {
                return false;
            }

            $store_info = function_exists('mvx_get_vendor_review_info') ? mvx_get_vendor_review_info($vendor->term_id) : false;
            $operator = $condition_rule['operator'];
            $value = $condition_rule['value'];
            $compare_value = $store_info ? intval($store_info['total_rating']) : '0';

            return $this->compare( $operator, $value, $compare_value );

        }

        /*
         * Condition: Store products count
         */
        public function multivendorx_store_products( $condition_rule ) {
            global $product;

            $vendor = $this->get_vendor();
            if ( ! $vendor ) {
                return false;
            }

            $vendor_products = $vendor->get_products_ids();
            $operator = $condition_rule['operator'];
            $value = $condition_rule['value'];
            $compare_value = $vendor_products ? count( $vendor_products ) : '0';

            return $this->compare( $operator, $value, $compare_value );

        }

        /*
         * Condition: Store net sales
         */
        public function multivendorx_store_sales( $condition_rule ) {
            global $product;

            $vendor = $this->get_vendor();
            if ( ! $vendor ) {
                return false;
            }

            $store_stats = $vendor->get_vendor_orders_reports_of();
            $operator = $condition_rule['operator'];
            $value = $condition_rule['value'];
            $compare_value = $store_stats && isset( $store_stats['sales_total'] ) ? $store_stats['sales_total'] : '0';

            return $this->compare( $operator, $value, $compare_value );

        }

        /*
         * Condition: Store net sales
         */
        public function multivendorx_store_solds( $condition_rule ) {
            global $product;

            $vendor = $this->get_vendor();
            if ( ! $vendor ) {
                return false;
            }

            $store_stats = $vendor->get_vendor_orders_reports_of();
            $operator = $condition_rule['operator'];
            $value = $condition_rule['value'];
            $compare_value = $store_stats && isset( $store_stats['orders_no'] ) ? $store_stats['orders_no'] : '0';

            return $this->compare( $operator, $value, $compare_value );

        }

        /*
         * Set MultiVendorX custom text variables
         */
        public function awl_labels_text_vars( $variables ) {

            $variable['/{MULTIVENDORX_SHOP_BADGE}/i'] = array(
                'func' => array( $this, 'multivendorx_text_shop_badge' ),
                'desc' => __( "MultiVendorX shop 'Sold by' badge", "advanced-woo-labels" ),
                'example' => '{MULTIVENDORX_SHOP_BADGE}',
            );

            $variable['/{MULTIVENDORX_SHOP_LOGO}/i'] = array(
                'func' => array( $this, 'multivendorx_text_shop_logo' ),
                'desc' => __( "MultiVendorX shop logo for current product", "advanced-woo-labels" ),
                'example' => '{MULTIVENDORX_SHOP_LOGO}',
            );

            $variable['/{MULTIVENDORX_SHOP_NAME}/i'] = array(
                'func' => array( $this, 'multivendorx_text_shop_name' ),
                'desc' => __( "MultiVendorX shop name for current product", "advanced-woo-labels" ),
                'example' => '{MULTIVENDORX_SHOP_NAME}',
            );

            $variable['/{MULTIVENDORX_SHOP_LINK}/i'] = array(
                'func' => array( $this, 'multivendorx_text_shop_link' ),
                'desc' => __( "MultiVendorX shop link for current product", "advanced-woo-labels" ),
                'example' => '{MULTIVENDORX_SHOP_LINK}',
            );

            $variable['/{MULTIVENDORX_SHOP_RATING}/i'] = array(
                'func' => array( $this, 'multivendorx_text_shop_rating' ),
                'desc' => __( "MultiVendorX shop avarage rating", "advanced-woo-labels" ),
                'example' => '{MULTIVENDORX_SHOP_RATING}',
            );

            $variable['/{MULTIVENDORX_SHOP_REVIEWS_NUM}/i'] = array(
                'func' => array( $this, 'multivendorx_text_shop_reviews' ),
                'desc' => __( "MultiVendorX shop reviews number", "advanced-woo-labels" ),
                'example' => '{MULTIVENDORX_SHOP_REVIEWS_NUM}',
            );

            $variable['/{MULTIVENDORX_SHOP_PRODUCTS_NUM}/i'] = array(
                'func' => array( $this, 'multivendorx_text_shop_products' ),
                'desc' => __( "MultiVendorX shop products number", "advanced-woo-labels" ),
                'example' => '{MULTIVENDORX_SHOP_PRODUCTS_NUM}',
            );

            $variable['/{MULTIVENDORX_SHOP_NET_SALES}/i'] = array(
                'func' => array( $this, 'multivendorx_text_shop_net' ),
                'desc' => __( "MultiVendorX shop net sales", "advanced-woo-labels" ),
                'example' => '{MULTIVENDORX_SHOP_NET_SALES}',
            );

            $variable['/{MULTIVENDORX_SHOP_ITEMS_SOLD}/i'] = array(
                'func' => array( $this, 'multivendorx_text_shop_solds' ),
                'desc' => __( "MultiVendorX shop items sold", "advanced-woo-labels" ),
                'example' => '{MULTIVENDORX_SHOP_ITEMS_SOLD}',
            );

            $variables = array_merge( $variable, $variables );

            return $variables;

        }

        /*
         * Text variable: MultiVendorX shop badge
         */
        public function multivendorx_text_shop_badge( $matches ) {
            $vendor = $this->get_vendor();
            $badge = '';
            if ( $vendor ) {

                $store_name = $vendor->page_title;
                $store_url = $vendor->get_permalink();
                $store_image = $vendor->get_image() ? $vendor->get_image('image', array(125, 125)) : $MVX->plugin_url . 'assets/images/WP-stdavatar.png';

                $rating_info = function_exists('mvx_get_vendor_review_info') ? mvx_get_vendor_review_info($vendor->term_id) : false;
                $rating = $rating_info ? round( $rating_info['avg_rating'], 2 ) : '0';
                $count = $rating_info ? intval( $rating_info['total_rating'] ) : '0';

                $badge .= '<div class="awl-multivendorx-badge" style="max-width: 250px; display: flex;">';

                    $badge .= '<div style="margin-right: 10px" class="awl-multivendorx-vendor-image">';
                        $badge .= '<img style="border-radius: 50%;max-width: 30px;" src="' . esc_url( $store_image ) . '" alt="' . esc_attr( $store_name ) . '">';
                    $badge .= '</div>';

                    $badge .= '<div  class="awl-multivendorx-vendor-data">';

                        $badge .= '<div style="font-size:0.9em;margin-bottom:4px;" class="awl-multivendorx-vendor-name">';
                            $badge .= '<span>' . esc_html( $store_name ) . '</span>';
                        $badge .= '</div>';

                        if ( $count > 0) {

                            $badge .= '
                            <span class="awl-multivendorx-vendor-rating">
                                <span class="awl_votes" style="padding-top:4px;position:relative;display:inline-block;background-image: url(\'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAA0AAAANCAYAAABy6+R8AAAAGXRFWHRTb2Z0d2FyZQBBZG9iZSBJbWFnZVJlYWR5ccllPAAAAyZpVFh0WE1MOmNvbS5hZG9iZS54bXAAAAAAADw/eHBhY2tldCBiZWdpbj0i77u/IiBpZD0iVzVNME1wQ2VoaUh6cmVTek5UY3prYzlkIj8+IDx4OnhtcG1ldGEgeG1sbnM6eD0iYWRvYmU6bnM6bWV0YS8iIHg6eG1wdGs9IkFkb2JlIFhNUCBDb3JlIDUuNi1jMDE0IDc5LjE1Njc5NywgMjAxNC8wOC8yMC0wOTo1MzowMiAgICAgICAgIj4gPHJkZjpSREYgeG1sbnM6cmRmPSJodHRwOi8vd3d3LnczLm9yZy8xOTk5LzAyLzIyLXJkZi1zeW50YXgtbnMjIj4gPHJkZjpEZXNjcmlwdGlvbiByZGY6YWJvdXQ9IiIgeG1sbnM6eG1wPSJodHRwOi8vbnMuYWRvYmUuY29tL3hhcC8xLjAvIiB4bWxuczp4bXBNTT0iaHR0cDovL25zLmFkb2JlLmNvbS94YXAvMS4wL21tLyIgeG1sbnM6c3RSZWY9Imh0dHA6Ly9ucy5hZG9iZS5jb20veGFwLzEuMC9zVHlwZS9SZXNvdXJjZVJlZiMiIHhtcDpDcmVhdG9yVG9vbD0iQWRvYmUgUGhvdG9zaG9wIENDIDIwMTQgKFdpbmRvd3MpIiB4bXBNTTpJbnN0YW5jZUlEPSJ4bXAuaWlkOjU5NzBDNUJBREI0NzExRTVBMzA4RDg5QTYzM0I4NjJGIiB4bXBNTTpEb2N1bWVudElEPSJ4bXAuZGlkOjU5NzBDNUJCREI0NzExRTVBMzA4RDg5QTYzM0I4NjJGIj4gPHhtcE1NOkRlcml2ZWRGcm9tIHN0UmVmOmluc3RhbmNlSUQ9InhtcC5paWQ6NTk3MEM1QjhEQjQ3MTFFNUEzMDhEODlBNjMzQjg2MkYiIHN0UmVmOmRvY3VtZW50SUQ9InhtcC5kaWQ6NTk3MEM1QjlEQjQ3MTFFNUEzMDhEODlBNjMzQjg2MkYiLz4gPC9yZGY6RGVzY3JpcHRpb24+IDwvcmRmOlJERj4gPC94OnhtcG1ldGE+IDw/eHBhY2tldCBlbmQ9InIiPz7uu2KkAAABBUlEQVR42mL8//8/Azo4e/YsG5BiB+IvxsbGGAqYGLADAyDOA2IubJK4NJkBcTEQ8xOlCeg0FiDlBMSCQGyNTRPjmTNneIC0PBB/A2KQ+9WBeAUQCwDxNiDOgqplBmKQgQ9BBDcQFwGxJ8jjQMwD1QAC7kC8A6qBEYh7gXgZIyj0gE6SBXImAnEgDj/+BeIKIJ4EDM1fjLAghwZzHRBXo2l4BcRJQMVbMQICZAKQWg/1FzK4B8Tb8YVeINTtr6B+AQENILbCpykAiM8BsR8QhwFxNjSu7FCCHMlPIBOjgHgy0KmvkeLNGUgZAfE0oPhXdJteAnEXsgaoX/cCqQVA/AcmBhBgAFSOR2sRwnLNAAAAAElFTkSuQmCC\');background-position: 0 0;font-size: 0;background-size: 13px 13px;width: 65px;height: 13px;">
                                    <span class="awl_current_votes" style="position: absolute;left: 0;top: 0;background-image: url(\'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAA0AAAANCAYAAABy6+R8AAAAGXRFWHRTb2Z0d2FyZQBBZG9iZSBJbWFnZVJlYWR5ccllPAAAAyZpVFh0WE1MOmNvbS5hZG9iZS54bXAAAAAAADw/eHBhY2tldCBiZWdpbj0i77u/IiBpZD0iVzVNME1wQ2VoaUh6cmVTek5UY3prYzlkIj8+IDx4OnhtcG1ldGEgeG1sbnM6eD0iYWRvYmU6bnM6bWV0YS8iIHg6eG1wdGs9IkFkb2JlIFhNUCBDb3JlIDUuNi1jMDE0IDc5LjE1Njc5NywgMjAxNC8wOC8yMC0wOTo1MzowMiAgICAgICAgIj4gPHJkZjpSREYgeG1sbnM6cmRmPSJodHRwOi8vd3d3LnczLm9yZy8xOTk5LzAyLzIyLXJkZi1zeW50YXgtbnMjIj4gPHJkZjpEZXNjcmlwdGlvbiByZGY6YWJvdXQ9IiIgeG1sbnM6eG1wPSJodHRwOi8vbnMuYWRvYmUuY29tL3hhcC8xLjAvIiB4bWxuczp4bXBNTT0iaHR0cDovL25zLmFkb2JlLmNvbS94YXAvMS4wL21tLyIgeG1sbnM6c3RSZWY9Imh0dHA6Ly9ucy5hZG9iZS5jb20veGFwLzEuMC9zVHlwZS9SZXNvdXJjZVJlZiMiIHhtcDpDcmVhdG9yVG9vbD0iQWRvYmUgUGhvdG9zaG9wIENDIDIwMTQgKFdpbmRvd3MpIiB4bXBNTTpJbnN0YW5jZUlEPSJ4bXAuaWlkOjczN0NBQ0M4REI0NzExRTVBRkM4QjEwRTYzMEU5NzgwIiB4bXBNTTpEb2N1bWVudElEPSJ4bXAuZGlkOjczN0NBQ0M5REI0NzExRTVBRkM4QjEwRTYzMEU5NzgwIj4gPHhtcE1NOkRlcml2ZWRGcm9tIHN0UmVmOmluc3RhbmNlSUQ9InhtcC5paWQ6NzM3Q0FDQzZEQjQ3MTFFNUFGQzhCMTBFNjMwRTk3ODAiIHN0UmVmOmRvY3VtZW50SUQ9InhtcC5kaWQ6NzM3Q0FDQzdEQjQ3MTFFNUFGQzhCMTBFNjMwRTk3ODAiLz4gPC9yZGY6RGVzY3JpcHRpb24+IDwvcmRmOlJERj4gPC94OnhtcG1ldGE+IDw/eHBhY2tldCBlbmQ9InIiPz56rxCSAAABCklEQVR42mL8//8/AwZYzcgGJNmB+AtDKKYCJgbswACI84CYC5skLk1mQFwMxPzEaVrNyAIknYBYEIitsWli/L+KgQdIywPxNyAGuV8diFcAsQAQbwPiLKhaZiAGGfgQpEkcyGgDYk+wxxnAhkhCFf4F4ttQDYxA3AvEyxjBobeaURbImQjEgTj8CNJcAcSTgKH5ixEe5JBgrgPiajQNr4A4Cah4K8JPyNGwmtEYSJ6GOgUGToADJPT/P1yhFwjVADJ9B1RMA4it8AV5ABCfA2I/IA4D4mxoXNmhBjnCTyATo4B4MtApr5Gc7AwkjYB4GlD8K7omUGT+Bkp8wRLhokDyE1DuJ4gLEGAARw5K1iodv/cAAAAASUVORK5CYII=\');background-position: 0 0;font-size: 0;background-size: 13px 13px;height: 13px;width:' . (( $rating / 5 ) * 100) . '%"></span>
                                </span>
                                <span class="awl_review" style="opacity:0.7;font-size:0.75em;padding:4px 0 2px;display:block;">(' . sprintf( _n( '%s Review', '%s Reviews', $count, 'advanced-woo-labels' ), number_format_i18n( $count ) ) . ')</span>
                            </span>';

                        }

                    $badge .= '</div>';

                $badge .= '</div>';

            }
            return $badge;
        }

        /*
         * Text variable: MultiVendorX shop logo
         */
        public function multivendorx_text_shop_logo( $matches ) {
            $vendor = $this->get_vendor();
            $badge = '';
            if ( $vendor ) {
                $store_image = $vendor->get_image() ? $vendor->get_image('image', array(125, 125)) : $MVX->plugin_url . 'assets/images/WP-stdavatar.png';
                if ( $store_image ) {
                    $badge = '<img style="max-width:40px;" src="'. esc_url( $store_image ) . '"/>';
                }
            }
            return $badge;
        }

        /*
         * Text variable: MultiVendorX shop name
         */
        public function multivendorx_text_shop_name( $matches ) {
            $vendor = $this->get_vendor();
            $name = '';
            if ( $vendor ) {
                $name = $vendor->page_title;
            }
            return $name;
        }

        /*
         * Text variable: MultiVendorX shop link
         */
        public function multivendorx_text_shop_link( $matches ) {
            $vendor = $this->get_vendor();
            $link = '';
            if ( $vendor ) {
                $link = $vendor->get_permalink();
            }
            return $link;
        }

        /*
         * Text variable: MultiVendorX shop rating
         */
        public function multivendorx_text_shop_rating( $matches ) {
            $vendor = $this->get_vendor();
            $rating = '';
            if ( $vendor ) {
                $rating_info = function_exists('mvx_get_vendor_review_info') ? mvx_get_vendor_review_info($vendor->term_id) : false;
                $rating = $rating_info ? round( $rating_info['avg_rating'], 2 ) : '';
            }
            return $rating;
        }

        /*
         * Text variable: MultiVendorX shop reviews number
         */
        public function multivendorx_text_shop_reviews( $matches ) {
            $vendor = $this->get_vendor();
            $rating = '';
            if ( $vendor ) {
                $rating_info = function_exists('mvx_get_vendor_review_info') ? mvx_get_vendor_review_info($vendor->term_id) : false;
                $rating = $rating_info ? intval( $rating_info['total_rating'] ) : '0';
            }
            return $rating;
        }

        /*
         * Text variable: MultiVendorX shop products number
         */
        public function multivendorx_text_shop_products( $matches ) {
            $vendor = $this->get_vendor();
            $products = '';
            if ( $vendor ) {
                $vendor_products = $vendor->get_products_ids();
                $products = $vendor_products ? count( $vendor_products ) : '0';
            }
            return $products;
        }

        /*
         * Text variable: MultiVendorX shop net sales
         */
        public function multivendorx_text_shop_net( $matches ) {
            $vendor = $this->get_vendor();
            $sales = '';
            if ( $vendor ) {
                $store_stats = $vendor->get_vendor_orders_reports_of();
                $sales = $store_stats && isset( $store_stats['sales_total'] ) && $store_stats['sales_total'] ? $store_stats['sales_total'] : '0';
            }
            return $sales;
        }

        /*
         * Text variable: MultiVendorX shop items sold
         */
        public function multivendorx_text_shop_solds( $matches ) {
            $vendor = $this->get_vendor();
            $sales = '';
            if ( $vendor ) {
                $store_stats = $vendor->get_vendor_orders_reports_of();
                $sales = $store_stats && isset( $store_stats['orders_no'] ) && $store_stats['orders_no'] ? $store_stats['orders_no'] : '0';
            }
            return $sales;
        }

        /*
         * Get store vendor for current product
         */
        private function get_vendor() {

            global $product;

            $vendors = function_exists('get_mvx_product_vendors') ? get_mvx_product_vendors( $product->get_id() ) : false;
            $vendor = $vendors ? $vendors : false;

            return $vendor;

        }

        /*
         * Condition callback: get all available vendors
         */
        public function get_all_vendors( $args = array() ) {

            $options = array();

            if ( isset( $this->data['vendors'] ) ) {
                return $this->data['vendors'];
            }

            $vendors_list = function_exists('get_mvx_vendors') ? get_mvx_vendors() : false;

            if ( $vendors_list ) {
                foreach ( $vendors_list as $vendor ) {
                    $options[$vendor->get_id()] = $vendor->page_title ? $vendor->page_title : 'ID: ' . $vendor->get_id();
                }
            }

            $this->data['vendors'] = $options;

            return $options;

        }

        /*
         * Compare two values
         */
        private function compare( $operator, $value, $compare_value ) {

            $match = false;

            if ( 'equal' == $operator ) {
                $match = ($compare_value == $value);
            } elseif ( 'not_equal' == $operator ) {
                $match = ($compare_value != $value);
            } elseif ( 'greater' == $operator ) {
                $match = ($compare_value >= $value);
            } elseif ( 'less' == $operator ) {
                $match = ($compare_value <= $value);
            }

            return $match;

        }

    }

endif;

AWL_MultiVendorX::instance();