<?php

/**
 * Dokan – WooCommerce Multivendor Marketplace Solution plugin support
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

if (!class_exists('AWL_Dokan')) :

    /**
     * Class for main plugin functions
     */
    class AWL_Dokan {

        /**
         * @var AWL_Dokan Custom data
         */
        public $data = array();

        /**
         * @var AWL_Dokan The single instance of the class
         */
        protected static $_instance = null;

        /**
         * Main AWL_Dokan Instance
         *
         * Ensures only one instance of AWL_Dokan is loaded or can be loaded.
         *
         * @static
         * @return AWL_Dokan - Main instance
         */
        public static function instance()
        {
            if (is_null(self::$_instance)) {
                self::$_instance = new self();
            }
            return self::$_instance;
        }

        /**
         * Constructor
         */
        public function __construct() {

            add_filter( 'awl_label_rules', array( $this, 'label_rules' ), 1 );

            add_filter( 'awl_labels_condition_rules', array( $this, 'condition_rules' ), 1 );

            add_filter( 'awl_labels_text_vars', array( $this, 'awl_labels_text_vars' ), 1 );

        }

        /*
         * Add new label conditions for admin
         */
        public function label_rules( $options ) {

            $options['Dokan'][] = array(
                "name" => __( "Product sold by", "advanced-woo-labels" ),
                "id"   => "dokan_sold_by",
                "type" => "callback",
                "operators" => "equals",
                "choices" => array(
                    'callback' => array($this, 'get_all_vendors'),
                    'params'   => array()
                ),
            );

            $options['Dokan'][] = array(
                "name" => __( "Is product sold by any vendor", "advanced-woo-labels" ),
                "id"   => "dokan_is_sold_by_vendor",
                "type" => "bool",
                "operators" => "equals",
            );

            $options['Dokan'][] = array(
                "name" => __( "Is store page", "advanced-woo-labels" ),
                "id"   => "dokan_is_store_page",
                "type" => "bool",
                "operators" => "equals",
            );

            $options['Dokan'][] = array(
                "name" => __( "Is store featured", "advanced-woo-labels" ),
                "id"   => "dokan_store_featured",
                "type" => "bool",
                "operators" => "equals",
            );

            $options['Dokan'][] = array(
                "name" => __( "Store rating", "advanced-woo-labels" ),
                "id"   => "dokan_store_rating",
                "type" => "number",
                "step" => "0.01",
                "operators" => "equals_compare",
            );

            $options['Dokan'][] = array(
                "name" => __( "Store reviews count", "advanced-woo-labels" ),
                "id"   => "dokan_store_reviews",
                "type" => "number",
                "operators" => "equals_compare",
            );

            $options['Dokan'][] = array(
                "name" => __( "Store total sales count", "advanced-woo-labels" ),
                "id"   => "dokan_store_sales",
                "type" => "number",
                "operators" => "equals_compare",
            );

            $options['Dokan'][] = array(
                "name" => __( "Product views", "advanced-woo-labels" ),
                "id"   => "dokan_views",
                "type" => "number",
                "operators" => "equals_compare",
            );

            return $options;

        }

        /*
         * Add custom condition rule method
         */
        public function condition_rules( $rules ) {
            $rules['dokan_sold_by'] = array( $this, 'dokan_sold_by' );
            $rules['dokan_is_sold_by_vendor'] = array( $this, 'dokan_is_sold_by_vendor' );
            $rules['dokan_is_store_page'] = array( $this, 'dokan_is_store_page' );
            $rules['dokan_store_rating'] = array( $this, 'dokan_store_rating' );
            $rules['dokan_store_reviews'] = array( $this, 'dokan_store_reviews' );
            $rules['dokan_store_sales'] = array( $this, 'dokan_store_sales' );
            $rules['dokan_views'] = array( $this, 'dokan_views' );
            $rules['dokan_store_featured'] = array( $this, 'dokan_store_featured' );
            return $rules;
        }

        /*
        * Condition: Is product sold by vendor
        */
        public function dokan_sold_by( $condition_rule ) {
            global $product;

            $vendor = $this->get_vendor();
            if ( ! $vendor || ! $vendor->is_vendor() || ! $vendor->is_enabled() ) {
                return false;
            }

            $match = false;
            $operator = $condition_rule['operator'];
            $value = $condition_rule['value'];
            $compare_value = $vendor ? $vendor->get_id() : 0;

            if ( ! $vendor ) {
                return false;
            }

            if ( 'equal' == $operator ) {
                $match = ($compare_value == $value);
            } elseif ( 'not_equal' == $operator ) {
                $match = ($compare_value != $value);
            }

            return $match;

        }

        /*
         * Condition: Is product sold by any available vendor
         */
        public function dokan_is_sold_by_vendor( $condition_rule ) {
            global $product;

            $vendor = $this->get_vendor();

            $match = false;
            $operator = $condition_rule['operator'];
            $value = $condition_rule['value'];
            $compare_value = $vendor && $vendor->is_vendor() && $vendor->is_enabled() ? 'true' : 'false';

            if ( 'equal' == $operator ) {
                $match = ($compare_value == $value);
            } elseif ( 'not_equal' == $operator ) {
                $match = ($compare_value != $value);
            }

            return $match;

        }

        /*
         * Condition: Is inside store page
         */
        public function dokan_is_store_page( $condition_rule ) {
            global $product;

            $compare_value = $this->get_current_store_page_id() ? 'true' : 'false';
            $match = false;
            $operator = $condition_rule['operator'];
            $value = $condition_rule['value'];

            if ( 'equal' == $operator ) {
                $match = ($compare_value == $value);
            } elseif ( 'not_equal' == $operator ) {
                $match = ($compare_value != $value);
            }

            return $match;

        }

        /*
         * Condition: Store rating
         */
        public function dokan_store_rating( $condition_rule ) {
            global $product;

            $vendor = $this->get_vendor();
            if ( ! $vendor || ! $vendor->is_vendor() || ! $vendor->is_enabled() ) {
                return false;
            }

            $store_rating = $vendor->get_rating();

            $match = false;
            $operator = $condition_rule['operator'];
            $value = $condition_rule['value'];
            $compare_value = ! empty( $store_rating['count'] ) ? $store_rating['rating'] : '0';

            if ( 'equal' == $operator ) {
                $match = ($compare_value == $value);
            } elseif ( 'not_equal' == $operator ) {
                $match = ($compare_value != $value);
            } elseif ( 'greater' == $operator ) {
                $match = ($compare_value >= $value);
            } elseif ( 'less' == $operator ) {
                $match = ($compare_value <= $value);
            }

            return $match;

        }

        /*
         * Condition: Store reviews count
         */
        public function dokan_store_reviews( $condition_rule ) {
            global $product;

            $vendor = $this->get_vendor();
            if ( ! $vendor || ! $vendor->is_vendor() || ! $vendor->is_enabled() ) {
                return false;
            }

            $store_rating = $vendor->get_rating();
            $match = false;
            $operator = $condition_rule['operator'];
            $value = $condition_rule['value'];
            $compare_value = ! empty( $store_rating['count'] ) ? $store_rating['count'] : '0';

            if ( 'equal' == $operator ) {
                $match = ($compare_value == $value);
            } elseif ( 'not_equal' == $operator ) {
                $match = ($compare_value != $value);
            } elseif ( 'greater' == $operator ) {
                $match = ($compare_value >= $value);
            } elseif ( 'less' == $operator ) {
                $match = ($compare_value <= $value);
            }

            return $match;

        }

        /*
         * Condition: Store reviews count
         */
        public function dokan_store_sales( $condition_rule ) {
            global $product;

            $vendor = $this->get_vendor();
            if ( ! $vendor || ! $vendor->is_vendor() || ! $vendor->is_enabled() ) {
                return false;
            }

            $match = false;
            $operator = $condition_rule['operator'];
            $value = $condition_rule['value'];
            $compare_value = $vendor->get_total_sales();

            if ( 'equal' == $operator ) {
                $match = ($compare_value == $value);
            } elseif ( 'not_equal' == $operator ) {
                $match = ($compare_value != $value);
            } elseif ( 'greater' == $operator ) {
                $match = ($compare_value >= $value);
            } elseif ( 'less' == $operator ) {
                $match = ($compare_value <= $value);
            }

            return $match;

        }

        /*
         * Condition: Product views count
         */
        public function dokan_views( $condition_rule ) {
            global $product;

            $vendor = $this->get_vendor();
            if ( ! $vendor || ! $vendor->is_vendor() || ! $vendor->is_enabled() ) {
                return false;
            }

            $views = $vendor->get_product_views();
            $match = false;
            $operator = $condition_rule['operator'];
            $value = $condition_rule['value'];
            $compare_value = $views ? $views : '0';

            if ( 'equal' == $operator ) {
                $match = ($compare_value == $value);
            } elseif ( 'not_equal' == $operator ) {
                $match = ($compare_value != $value);
            } elseif ( 'greater' == $operator ) {
                $match = ($compare_value >= $value);
            } elseif ( 'less' == $operator ) {
                $match = ($compare_value <= $value);
            }

            return $match;

        }

        /*
         * Condition: Is product store featured
         */
        public function dokan_store_featured( $condition_rule ) {
            global $product;

            $vendor = $this->get_vendor();
            if ( ! $vendor || ! $vendor->is_vendor() || ! $vendor->is_enabled() ) {
                return false;
            }

            $match = false;
            $operator = $condition_rule['operator'];
            $value = $condition_rule['value'];
            $compare_value = $vendor->is_featured() ? 'true' : 'false';

            if ( 'equal' == $operator ) {
                $match = ($compare_value == $value);
            } elseif ( 'not_equal' == $operator ) {
                $match = ($compare_value != $value);
            }

            return $match;

        }

        /*
         * Set Dokan custom text variables
         */
        public function awl_labels_text_vars( $variables ) {
            $variable['/{DOKAN_SHOP_BADGE}/i'] = array(
                'func' => array( $this, 'dokan_text_shop_badge' ),
                'desc' => __( "Dokan shop 'Sold by' badge", "advanced-woo-labels" ),
                'example' => '{DOKAN_SHOP_BADGE}',
            );
            $variable['/{DOKAN_SHOP_LOGO}/i'] = array(
                'func' => array( $this, 'dokan_text_shop_logo' ),
                'desc' => __( "Dokan shop logo for current product", "advanced-woo-labels" ),
                'example' => '{DOKAN_SHOP_LOGO}',
            );
            $variable['/{DOKAN_SHOP_NAME}/i'] = array(
                'func' => array( $this, 'dokan_text_shop_name' ),
                'desc' => __( "Dokan shop name for current product", "advanced-woo-labels" ),
                'example' => '{DOKAN_SHOP_NAME}',
            );
            $variable['/{DOKAN_SHOP_LINK}/i'] = array(
                'func' => array( $this, 'dokan_text_shop_link' ),
                'desc' => __( "Dokan shop link for current product", "advanced-woo-labels" ),
                'example' => '{DOKAN_SHOP_LINK}',
            );
            $variable['/{DOKAN_SHOP_RATING}/i'] = array(
                'func' => array( $this, 'dokan_text_shop_rating' ),
                'desc' => __( "Dokan shop avarage rating", "advanced-woo-labels" ),
                'example' => '{DOKAN_SHOP_RATING}',
            );
            $variable['/{DOKAN_SHOP_REVIEWS_NUM}/i'] = array(
                'func' => array( $this, 'dokan_text_shop_reviews' ),
                'desc' => __( "Dokan shop reviews number", "advanced-woo-labels" ),
                'example' => '{DOKAN_SHOP_REVIEWS_NUM}',
            );
            $variable['/{DOKAN_PRODUCT_VIEWS_NUM}/i'] = array(
                'func' => array( $this, 'dokan_text_product_views' ),
                'desc' => __( "Dokan product views number", "advanced-woo-labels" ),
                'example' => '{DOKAN_PRODUCT_VIEWS_NUM}',
            );

            $variables = array_merge( $variable, $variables );

            return $variables;
        }

        /*
         * Text variable: Dokan shop logo
         */
        public function dokan_text_shop_logo( $matches ) {
            $vendor = $this->get_vendor();
            $badge = '';
            if ( $vendor && $vendor->is_vendor() && $vendor->is_enabled() ) {
                $avatar = $vendor->get_avatar();
                if ( $avatar ) {
                    $badge = '<img style="max-width:30px;" src="'.$avatar.'"/>';
                }
            }
            return $badge;
        }

        /*
         * Text variable: Dokan shop badge
         */
        public function dokan_text_shop_badge( $matches ) {
            $vendor = $this->get_vendor();
            $badge = '';
            if ( $vendor && $vendor->is_vendor() && $vendor->is_enabled() ) {
                $store_info   = $vendor->get_shop_info();
                $store_rating = $vendor->get_rating();
                if ( isset( $store_info['store_name'] ) && $store_info['store_name'] ) {

                    ob_start();

                    echo '<div class="aws-dokan-badge" style="max-width: 250px;">';

                    dokan_get_template_part(
                        'vendor-store-info',
                        '',
                        [
                            'vendor'       => $vendor,
                            'store_info'   => $store_info,
                            'store_rating' => $store_rating,
                        ]
                    );

                    echo '</div>';

                    $badge = ob_get_contents();
                    ob_end_clean();

                }
            }
            return $badge;
        }

        /*
         * Text variable: Dokan shop name
         */
        public function dokan_text_shop_name( $matches ) {
            $vendor = $this->get_vendor();
            $name = '';
            if ( $vendor && $vendor->is_vendor() && $vendor->is_enabled() ) {
                $store_info = $vendor->get_shop_info();
                if ( isset( $store_info['store_name'] ) && $store_info['store_name'] ) {
                    $name = $store_info['store_name'];
                }
            }
            return $name;
        }

        /*
         * Text variable: Dokan shop link
         */
        public function dokan_text_shop_link( $matches ) {
            $vendor = $this->get_vendor();
            $link = '';
            if ( $vendor && $vendor->is_vendor() && $vendor->is_enabled() ) {
                $link = $vendor->get_shop_url();
            }
            return $link;
        }

        /*
         * Text variable: Dokan shop rating
         */
        public function dokan_text_shop_rating( $matches ) {
            $vendor = $this->get_vendor();
            $rating = '';
            if ( $vendor && $vendor->is_vendor() && $vendor->is_enabled() ) {
                $store_rating = $vendor->get_rating();
                $rating = ! empty( $store_rating['count'] ) ? $store_rating['rating'] : '';
            }
            return $rating;
        }

        /*
         * Text variable: Dokan shop reviews number
         */
        public function dokan_text_shop_reviews( $matches ) {
            $vendor = $this->get_vendor();
            $rating = '';
            if ( $vendor && $vendor->is_vendor() && $vendor->is_enabled() ) {
                $store_rating = $vendor->get_rating();
                $rating = ! empty( $store_rating['count'] ) ? $store_rating['count'] : '0';
            }
            return $rating;
        }

        /*
         * Text variable: Dokan product views number
         */
        public function dokan_text_product_views( $matches ) {
            $vendor = $this->get_vendor();
            $views = '';
            if ( $vendor && $vendor->is_vendor() && $vendor->is_enabled() ) {
                $views = $vendor->get_product_views();
            }
            return $views;
        }

        /*
         * Condition callback: get all available vendors
         */
        public function get_all_vendors( $args = array() ) {

            $options = array();

            if ( isset( $this->data['vendors'] ) ) {
                return $this->data['vendors'];
            }

            if ( function_exists('dokan') ) {
                $vendors = dokan()->vendor;
                if ( $vendors ) {
                    foreach ( $vendors->get_vendors( $args ) as $vendor ) {
                        $store_name = get_user_meta( $vendor->id, 'dokan_store_name', true );
                        $options[$vendor->id] = $store_name ? $store_name : 'ID: ' . $vendor->id ;
                    }
                }
            }

            $this->data['vendors'] = $options;

            return $options;

        }

        /*
         * Get store vendor for current product
         */
        private function get_vendor() {

            global $product;

            $vendor = function_exists('dokan_get_vendor_by_product') ? dokan_get_vendor_by_product( $product ) : false;

            return $vendor;

        }

        /*
         * Get current vendor store ID
         */
        private function get_current_store_page_id() {

            $store_products = function_exists('dokan_get_option') ? dokan_get_option( 'store_products', 'dokan_appearance' ) : false;

            if ( ! empty( $store_products['hide_product_filter'] ) ) {
                return false;
            }

            $store_user = function_exists('dokan') ? dokan()->vendor->get( get_query_var( 'author' ) ) : false;
            $store_id  = $store_user ? $store_user->get_id() : false;

            return $store_id;

        }

    }

endif;

AWL_Dokan::instance();