<?php

/**
 * WooCommerce Product Table by Barn2 integration
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

if (!class_exists('AWL_Barn_Tables')) :

    /**
     * Class for main plugin functions
     */
    class AWL_Barn_Tables {

        /**
         * @var AWL_Barn_Tables Custom data
         */
        public $data = array();

        /**
         * @var AWL_Barn_Tables The single instance of the class
         */
        protected static $_instance = null;

        /**
         * Main AWL_Barn_Tables Instance
         *
         * Ensures only one instance of AWL_Barn_Tables is loaded or can be loaded.
         *
         * @static
         * @return AWL_Barn_Tables - Main instance
         */
        public static function instance()
        {
            if (is_null(self::$_instance)) {
                self::$_instance = new self();
            }
            return self::$_instance;
        }

        /**
         * Constructor
         */
        public function __construct() {

            add_action( 'wc_product_table_parse_args', array( $this, 'table_parse_args' ) );

            add_filter('wc_product_table_cell_data_image', array( $this, 'wc_product_table_cell_data_image' ) );
            add_filter('wc_product_table_data_image', array( $this, 'wc_product_table_cell_data_image' ) );

            add_filter('wc_product_table_cell_data_name', array( $this, 'wc_product_table_cell_data_name' ) );
            add_filter('wc_product_table_data_name', array( $this, 'wc_product_table_cell_data_name' ) );

            add_filter('wc_product_table_custom_data_labels', array( $this, 'labels_column' ) );

            add_filter('wc_product_table_shortcode_output', array( $this, 'shortcode_output' ) );

            add_action( 'wc_product_table_before_get_table', array( $this, 'wc_product_table_before_get_table' ) );
            add_action( 'wc_product_table_after_get_table', array( $this, 'wc_product_table_after_get_table' ) );

            add_filter( 'awl_label_rules', array( $this, 'label_rules' ), 1 );
            add_filter( 'awl_labels_condition_rules', array( $this, 'condition_rules' ), 1 );

        }

        /*
         * Parse current table args
         */
        public function table_parse_args( $args ) {

            $this->data['disable_labels'] = false;
            if ( property_exists( $args, 'columns' ) && in_array( 'labels', $args->columns ) !== false )  {
                $this->data['disable_labels'] = true;
            }

            /**
             * Enable/disable product labels inside table
             * @since 1.44
             * @param bool Show or not labels for products tables
             */
            $this->data['disable_labels'] = apply_filters( 'awl_hide_for_barn2_tables', $this->data['disable_labels'] );

        }

        /*
         * Show labels for table image column
         */
        public function wc_product_table_cell_data_image( $data ) {
            if ( ! ( isset( $this->data['disable_labels'] ) && $this->data['disable_labels'] ) ) {
                $data = '<div style="position: relative;">' . AWL_Label_Display::instance()->show_label( 'on_image' ) . $data . '</div>';
            }
            return $data;
        }

        /*
         * Show labels for table name column
         */
        public function wc_product_table_cell_data_name( $data ) {
            if ( ! ( isset( $this->data['disable_labels'] ) && $this->data['disable_labels'] ) ) {
                $data = AWL_Label_Display::instance()->show_label( 'before_title' ) . $data;
            }
            return $data;
        }

        /*
         * Custom column for labels
         */
        public function labels_column( $data ) {
            $data = '<div class="awl_barn_table_labels">' . AWL_Label_Display::instance()->show_label( 'on_image' ) . AWL_Label_Display::instance()->show_label( 'before_title' ) . '</div>' . $data;
            return $data;
        }

        /*
         * Filter table shortcode output. Add styles
         */
        public function shortcode_output( $data ) {
            $data .= '
                <style>
                    .wc-product-table .awl_barn_table_labels {
                        display: flex;
                        margin: 0 !important;
                        flex-wrap: wrap;
                        flex-direction: row;
                        text-align: left;
                        justify-content: flex-start;
                    }
                    .wc-product-table .awl_barn_table_labels .advanced-woo-labels {
                        position: relative !important;
                        width: auto !important;
                        margin: 0 5px 5px 0 !important;
                        transform: none !important;
                    }
                    .wc-product-table .awl_barn_table_labels .advanced-woo-labels:last-child {
                        margin-bottom: 0 !important;
                    }
                </style>
            ';
            return $data;
        }

        /*
         * Before product table rendering
         */
        public function wc_product_table_before_get_table( $table ) {
            $this->data['current_table'] = $table->id;
        }

        /*
         * After product table rendering
         */
        public function wc_product_table_after_get_table( $table ) {
            $this->data['current_table'] = false;
        }

        /*
         * Add new label conditions for admin
         */
        public function label_rules( $options ) {

            $options['Special'][] = array(
                "name" => __( "Product Table: Is inside table", "advanced-woo-labels" ),
                "id"   => "aws_is_inside_barn_table",
                "type" => "bool",
                "operators" => "equals",
            );

            $options['Special'][] = array(
                "name" => __( "Product Table: Is inside specific table", "advanced-woo-labels" ),
                "id"   => "aws_is_inside_specific_barn_table",
                "type" => "text",
                "placeholder" => __( "Table ID", "advanced-woo-labels" ),
                "operators" => "equals",
            );

            return $options;

        }

        /*
         * Add custom condition rule method
         */
        public function condition_rules( $rules ) {
            $rules['aws_is_inside_barn_table'] = array( $this, 'aws_is_inside_barn_table' );
            $rules['aws_is_inside_specific_barn_table'] = array( $this, 'aws_is_inside_specific_barn_table' );
            return $rules;
        }

        /*
         * Condition: Is inside product table
         */
        public function aws_is_inside_barn_table( $condition_rule ) {
            global $product;

            $match = false;
            $operator = $condition_rule['operator'];
            $value = $condition_rule['value'];
            $compare_value = 'false';

            if ( isset( $this->data['current_table'] ) && $this->data['current_table'] ) {
                $compare_value = 'true';
            }

            if ( 'equal' == $operator ) {
                $match = ($compare_value == $value);
            } elseif ( 'not_equal' == $operator ) {
                $match = ($compare_value != $value);
            }

            return $match;

        }

        /*
        * Condition: Is inside specific product table
        */
        public function aws_is_inside_specific_barn_table( $condition_rule ) {
            global $product;

            $match = false;
            $operator = $condition_rule['operator'];
            $value = $condition_rule['value'];
            $compare_value = '';

            if ( isset( $this->data['current_table'] ) && $this->data['current_table'] ) {
                $compare_value = $this->data['current_table'];
            }

            if ( 'equal' == $operator ) {
                $match = ($compare_value == $value);
            } elseif ( 'not_equal' == $operator ) {
                $match = ($compare_value != $value);
            }

            return $match;

        }

    }

endif;

AWL_Barn_Tables::instance();