<?php

/**
 * ACF plugin support
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

if (!class_exists('AWL_ACF')) :

    /**
     * Class for main plugin functions
     */
    class AWL_ACF {

        /**
         * @var AWL_ACF Custom data
         */
        public $data = array();

        /**
         * @var AWL_ACF The single instance of the class
         */
        protected static $_instance = null;

        /**
         * Main AWL_ACF Instance
         *
         * Ensures only one instance of AWL_ACF is loaded or can be loaded.
         *
         * @static
         * @return AWL_ACF - Main instance
         */
        public static function instance()
        {
            if (is_null(self::$_instance)) {
                self::$_instance = new self();
            }
            return self::$_instance;
        }

        /*
         * Constructor
         */
        public function __construct() {

            add_filter( 'awl_labels_text_vars', array( $this, 'labels_text_vars' ), 1 );

            add_filter( 'awl_label_rules', array( $this, 'label_rules' ), 1 );

            add_filter( 'awl_labels_condition_rules', array( $this, 'condition_rules' ), 1 );

        }

        /*
         * Add new text variables
         */
        public function labels_text_vars( $variables ) {

            $variable['/{ACF\:(.*?)}/i'] = array(
                'func' => array( $this, 'acf_field_text_var' ),
                'desc' => 'ACF field value, e.g {ACF:my_field}',
                'example' => '{ACF:field_name}',
            );

            $variables = array_merge( $variable, $variables );

            return $variables;

        }

        /*
         * ACF text variable match
         */
        public function acf_field_text_var( $matches = '' ) {
            global $product;

            if ( ! $matches || ! $matches[1] || ! function_exists( 'get_field_object' ) ) {
                return '';
            }

            $field_name = $matches[1];
            $field_object = get_field_object( $field_name, $product->get_id() );

            $value = $this->acf_field_value( $field_object );

            return $value;

        }

        /*
         * Add new label conditions for admin
         */
        public function label_rules( $options ) {

            $options['Special'][] = array(
                "name" => __( "ACF fields", "advanced-woo-labels" ),
                "id"   => "acf",
                "type" => "callback",
                "operators" => "equals",
                "choices" => array(
                    'callback' => array( $this, 'get_acf_field' ),
                    'params'   => array()
                ),
                "suboption" => array(
                    'callback' => array( $this, 'get_all_acf_fields' ),
                    'params'   => array()
                ),
            );

            return $options;

        }

        /*
         * Add custom condition rule method
         */
        public function condition_rules( $rules ) {
            $rules['acf'] = array( $this, 'match_acf' );
            return $rules;
        }

        /*
         * Custom match function for ACF condition rule
         */
        public function match_acf( $condition_rule ) {
            global $product;

            $match = false;
            $name = $condition_rule['suboption'];
            $operator = $condition_rule['operator'];
            $value = $condition_rule['value'];
            $compare_value = '';

            $field_object = get_field_object( $name, $product->get_id() );

            if ( $field_object ) {
                $compare_value = $this->acf_field_value( $field_object, 'compare' );
            }

            if ( $value === 'awl_any' && $field_object && isset( $field_object['value'] ) && $field_object['value'] ) {
                $match = 'equal' === $operator;
                return $match;
            }

            if ( is_array( $compare_value ) ) {

                if ( 'equal' == $operator ) {

                    foreach( $compare_value as $compare_i ) {
                        if ( $compare_i == $value ) {
                            $match = true;
                            break;
                        }
                    }

                } elseif ( 'not_equal' == $operator ) {

                    $match_temp = true;
                    foreach( $compare_value as $compare_i ) {
                        if ( $compare_i == $value ) {
                            $match_temp = false;
                            break;
                        }
                    }

                    $match = $match_temp;

                }

            } else {

                if ( 'equal' == $operator ) {
                    $match = ($compare_value == $value);
                } elseif ( 'not_equal' == $operator ) {
                    $match = ($compare_value != $value);
                }

            }

            return $match;

        }

        public function get_acf_field( $name = '' ) {

            global $wpdb;

            $options = array();

            $available_acf_fields = $this->get_available_acf_fields();
            $field_object = isset( $available_acf_fields[$name] ) ? $available_acf_fields[$name]['data'] : false;

            $acf_inner_fields = $wpdb->get_results("
                    SELECT DISTINCT meta_key
                    FROM $wpdb->postmeta
                    WHERE meta_value = '{$name}'
                    ORDER BY meta_key ASC
            ");

            if ( $field_object && is_array( $acf_inner_fields ) && ! empty( $acf_inner_fields ) ) {

                $acf_inner = $acf_inner_fields[0]->meta_key;
                $acf_inner = ltrim( $acf_inner, '_' );

                $acf_values = $wpdb->get_results("
                    SELECT DISTINCT meta_value
                    FROM $wpdb->postmeta
                    WHERE meta_key = '{$acf_inner}'
                    ORDER BY meta_value ASC
                ");

                if ( is_array( $acf_values ) && ! empty( $acf_values ) ) {
                    foreach( $acf_values as $acf_value ) {

                        $acf_field_value = maybe_unserialize( $acf_value->meta_value );
                        //$acf_field_value = $acf_value->meta_value;

                        if ( $acf_field_value ) {

                            $field_object['value'] = $acf_field_value;

                            $value = $this->acf_field_value( $field_object , 'admin' );

                            if ( is_array( $value ) ) {

                                foreach ( $value as $value_i ) {

                                    if ( is_array( $value_i ) && isset( $value_i['value'] ) ) {

                                        $options[sanitize_title($value_i['value'])] = $value_i;

                                    } else {

                                        $options[sanitize_title($value_i)] = array(
                                            'name'  => $value_i,
                                            'value' => $value_i
                                        );

                                    }

                                }

                            } else {

                                $options[sanitize_title($value)] = array(
                                    'name'  => $value,
                                    'value' => $value
                                );

                            }

                        }

                    }
                }

            }

            if ( empty( $options ) ) {
                $options[] = array(
                    'name'  => ' ',
                    'value' => ' '
                );
            }

            return $options;

        }


        /*
         * Get all available ACF fields for admin area
         */
        public function get_all_acf_fields() {

            $options = array();

            $available_acf_fields = $this->get_available_acf_fields();

            if ( $available_acf_fields ) {
                foreach( $available_acf_fields as $field_slug => $field_data ) {
                    $options[] = array(
                        'name'  => $field_data['title'],
                        'value' => $field_slug
                    );
                }
            }

            return $options;

        }

        /**
         * Get available ACF plugin fields
         * @return array $fields
         */
        private function get_available_acf_fields() {

            if ( isset( $this->data['acf_all_fields'] ) ) {
                return $this->data['acf_all_fields'];
            }

            $fields = array();

            $args = array(
                'posts_per_page' => -1,
                'post_type' => 'acf-field',
                'orderby' => 'menu_order',
                'order' => 'ASC',
                'suppress_filters' => true,
                'cache_results' => true,
                'update_post_meta_cache' => false,
                'update_post_term_cache' => false,
                'post_parent' => false,
                'post_status' => array( 'publish' ),
            );

            $posts = get_posts($args);

            if ( $posts && is_array( $posts ) ) {
                foreach ( $posts as $post ) {
                    $fields[$post->post_name] = array(
                        'type' => $post->post_excerpt,
                        'title' => $post->post_title,
                        'data' => maybe_unserialize( $post->post_content )
                    );
                }
            }

            $this->data['acf_all_fields'] = $fields;

            return $fields;

        }

        /*
         * Get ACF field value
         */
        private function acf_field_value( $field_object, $view = 'text' ) {

            $value = '';

            if ( $field_object && isset( $field_object['type'] ) && isset( $field_object['value'] ) && $field_object['value'] ) {

                switch( $field_object['type'] ) {

                    case 'image':
                        $value = $this->get_image_field( $field_object, $view );
                        break;

                    case 'gallery':
                        $value = $this->get_gallery_field( $field_object, $view );
                        break;

                    case 'link':
                        $value = $this->get_link_field( $field_object, $view );
                        break;

                    case 'post_object':
                        $value = $this->get_post_object_field( $field_object, $view );
                        break;

                    case 'page_link':
                        $value = $this->get_page_link_field( $field_object, $view );
                        break;

                    case 'relationship':
                        $value = $this->get_relationship_field( $field_object, $view );
                        break;

                    case 'taxonomy':
                        $value = $this->get_taxonomy_field( $field_object, $view );
                        break;

                    case 'user':
                        $value = $this->get_user_field( $field_object, $view );
                        break;

                    case 'select':
                        $value = $this->get_select_field( $field_object, $view );
                        break;

                    case 'checkbox':
                        $value = $this->get_checkbox_field( $field_object, $view );
                        break;

                    case 'radio':
                        $value = $this->get_radio_field( $field_object, $view );
                        break;

                    case 'google_map':
                        $value = $this->get_google_map_field( $field_object, $view );
                        break;

                    default:
                        $value = is_array( $field_object['value'] ) ? $field_object['value'] : strval( $field_object['value'] );

                }

            }

            return $value;

        }

        /*
         * Get posts by post types and taxonomies
         */
        private function get_posts( $post_type = array(), $taxonomies = array() ) {

            if ( empty( $post_type ) ) {
                $post_type = get_post_types();
            }

            $args = array(
                'posts_per_page'      => -1,
                'post_type'           => $post_type,
                'post_status'         => 'publish',
                'ignore_sticky_posts' => true,
                'suppress_filters'    => true,
                'has_password'        => false,
                'no_found_rows'       => 1,
                'orderby'             => 'ID',
                'order'               => 'DESC',
                'lang'                => ''
            );

            if ( $taxonomies && is_array( $taxonomies ) ) {
                $tax = array();
                foreach( $taxonomies as $taxonomy ) {
                    $current_tax = explode( ':', $taxonomy );
                    if ( $current_tax ) {
                        $tax[] = array(
                            'taxonomy' => $current_tax[0],
                            'field' => 'slug',
                            'terms' => array( $current_tax[1] )
                        );
                    }
                }
                $tax['relation'] = 'OR';
                $args['tax_query'] = $tax;
            }

            $posts = get_posts( $args );

            return $posts;

        }

        /*
         * Get value of image field type
         */
        private function get_image_field( $field_object, $view = 'text' ) {

            $value = '';
            $return_format = $field_object['return_format'];

            if ( $view === 'admin' ) {
                return wp_get_attachment_image_url( $field_object['value'], 'full' );
            }

            if ( $return_format === 'array' && is_array( $field_object['value'] ) ) {
                $value = $field_object['value']['url'];
            } elseif ( $return_format === 'id' && is_int( $field_object['value'] ) ) {
                $value = wp_get_attachment_image_url( $field_object['value'], 'full' );
            } else {
                $value = $field_object['value'];
            }

            return $value;

        }

        /*
         * Get value of gallery field type
         */
        private function get_gallery_field( $field_object, $view = 'text' ) {

            $value = '';

            if ( $view === 'admin' || $view === 'compare' ) {

                $value = array();

                if ( is_array( $field_object['value'] ) ) {
                    foreach( $field_object['value'] as $gallery_item ) {
                        $url = wp_get_attachment_image_url( $gallery_item, 'full' );
                        if ( $url ) {
                            $value[] = $url;
                        }
                    }
                }

                return $value;

            }
            else {

                if ( is_array( $field_object['value'] ) ) {
                    foreach( $field_object['value'] as $gallery_item ) {
                        $url = $gallery_item['url'];
                        if ( $url ) {
                            $value .= '<img style="width:20px;display:inline;margin:3px!important;padding:0!important;" src="'.$url.'">';
                        }
                    }
                }

            }

            return $value;

        }

        /*
         * Get value of link field type
         */
        private function get_link_field( $field_object, $view = 'text' ) {

            $value = '';

            if ( is_array( $field_object['value'] ) && isset( $field_object['value']['url'] ) ) {
                $value = $field_object['value']['url'];
            } else {
                $value = $field_object['value'];
            }

            return $value;

        }

        /*
         * Get value of post object field type
         */
        private function get_post_object_field( $field_object, $view = 'text' ) {

            $value = '';

            if ( $view === 'admin' ) {

                $value = array();

                $post_type = $field_object['post_type'];
                $taxonomies = $field_object['taxonomy'];

                $posts = $this->get_posts( $post_type, $taxonomies );

                if ( $posts && is_array( $posts ) ) {
                    foreach( $posts as $post ) {
                        $value[] = array(
                            'name'  => get_the_title( $post->ID ) . ' ( ID = ' . $post->ID . ' )',
                            'value' => $post->ID
                        );
                    }
                }

                return $value;

            }

            if ( $field_object['multiple'] && is_array( $field_object['value'] ) ) {
                $value_arr = array();
                foreach( $field_object['value'] as $posts ) {
                    if ( is_object( $posts ) ) {
                        $value_arr[] = ( $view === 'compare' ) ? $posts->ID : $posts->post_title;
                    } else {
                        $post = get_post( $posts );
                        if ( $post ) {
                            $value_arr[] = ( $view === 'compare' ) ? $post->ID : $post->post_title;
                        }
                    }
                }
                $value = ( $view === 'compare' ) ? $value_arr : implode(', ', $value_arr );
            } else {
                if ( is_object( $field_object['value'] ) ) {
                    $value = ( $view === 'compare' ) ? $field_object['value']->ID : $field_object['value']->post_title;
                } else {
                    $post = get_post( $field_object['value'] );
                    if ( $post ) {
                        $value = ( $view === 'compare' ) ? $post->ID : $post->post_title;
                    }
                }
            }

            return $value;

        }

        /*
         * Get value of page link field type
         */
        private function get_page_link_field( $field_object, $view = 'text' ) {

            $value = '';

            if ( $view === 'admin' ) {

                $value = array();

                $post_type = $field_object['post_type'];
                $taxonomies = $field_object['taxonomy'];

                $posts = $this->get_posts( $post_type, $taxonomies );

                if ( $posts && is_array( $posts ) ) {
                    foreach( $posts as $post ) {
                        $value[] = array(
                            'name'  => get_the_title( $post->ID ) . ' ( ID = ' . $post->ID . ' )',
                            'value' => get_permalink( $post->ID )
                        );
                    }
                }

                return $value;

            }

            if ( $field_object['multiple'] && is_array( $field_object['value'] ) ) {
                $value_arr = array();
                foreach( $field_object['value'] as $link ) {
                    $value_arr[] = $link;
                }
                $value = ( $view === 'compare' ) ? $value_arr : implode(', ', $value_arr );
            } else {
                $value = $field_object['value'];
            }

            return $value;

        }

        /*
         * Get value of page link field type
         */
        private function get_relationship_field( $field_object, $view = 'text' ) {

            $value = '';

            if ( $view === 'admin' ) {

                $value = array();

                $post_type = $field_object['post_type'];
                $taxonomies = $field_object['taxonomy'];

                $posts = $this->get_posts( $post_type, $taxonomies );

                if ( $posts && is_array( $posts ) ) {
                    foreach( $posts as $post ) {
                        $value[] = array(
                            'name'  => get_the_title( $post->ID ) . ' ( ID = ' . $post->ID . ' )',
                            'value' => $post->ID
                        );
                    }
                }

                return $value;

            }

            if ( is_array( $field_object['value'] ) ) {

                $value_arr = array();

                foreach( $field_object['value'] as $rel_post ) {
                    if ( is_object( $rel_post ) ) {
                        $value_arr[] = ( $view === 'compare' ) ? $rel_post->ID : $rel_post->post_title;
                    } else {
                        $post = get_post( $rel_post );
                        if ( $post ) {
                            $value_arr[] = ( $view === 'compare' ) ? $post->ID : $post->post_title;
                        }
                    }
                }

                $value = ( $view === 'compare' ) ? $value_arr : implode(', ', $value_arr );

            }

            return $value;

        }

        /*
         * Get value of taxonomy field type
         */
        private function get_taxonomy_field( $field_object, $view = 'text' ) {

            $value = '';

            if ( $view === 'admin' ) {

                $value = array();

                $terms = get_terms( array(
                    'taxonomy' => $field_object['taxonomy'],
                    'hide_empty' => false,
                ) );

                if ( $terms && is_array( $terms ) ) {
                    foreach( $terms as $term ) {
                        $value[] = array(
                            'name'  => $term->name,
                            'value' => $term->term_id
                        );
                    }
                }

                return $value;

            }

            if ( is_array( $field_object['value'] ) ) {
                $value_arr = array();
                foreach( $field_object['value'] as $terms ) {
                    if ( is_object( $terms ) ) {
                        $value_arr[] = ( $view === 'compare' ) ? $terms->term_id : $terms->name;
                    } else {
                        $term = get_term( $terms, $field_object['taxonomy'] );
                        if ( $term ) {
                            $value_arr[] = ( $view === 'compare' ) ? $term->term_id : $term->name;
                        }
                    }
                }
                $value = ( $view === 'compare' ) ? $value_arr : implode(', ', $value_arr );
            } else {
                if ( is_object( $field_object['value'] ) ) {
                    $value = ( $view === 'compare' ) ? $field_object['value']->term_id : $field_object['value']->name;
                } else {
                    $term = get_term( $field_object['value'], $field_object['taxonomy'] );
                    if ( $term ) {
                        $value = ( $view === 'compare' ) ? $term->term_id : $term->name;
                    }
                }
            }

            return $value;

        }

        /*
         * Get value of user field type
         */
        private function get_user_field( $field_object, $view = 'text' ) {

            $value = '';

            if ( $view === 'admin' ) {

                $value = array();

                $args = array(
                    'role__in' => $field_object['role'] ? $field_object['role'] : array()
                );

                $users = get_users( $args );

                if ( $users && is_array( $users ) ) {
                    foreach( $users as $user ) {
                        $value[] = array(
                            'name'  => $user->display_name . ' (' . $user->user_nicename . ')',
                            'value' => $user->ID
                        );
                    }
                }

                return $value;

            }

            if ( $field_object['multiple'] && is_array( $field_object['value'] ) ) {
                $value_arr = array();
                foreach( $field_object['value'] as $choice ) {
                    $value_arr[] = ( $view === 'compare' ) ? $choice['ID'] : $choice['display_name'];
                }
                $value = ( $view === 'compare' ) ? $value_arr : implode(', ', $value_arr );
            } else {
                $value = ( $view === 'compare' ) ? $field_object['value']['ID'] : $field_object['value']['display_name'];
            }

            return $value;

        }

        /*
         * Get value of select field type
         */
        private function get_select_field( $field_object, $view = 'text' ) {

            $value = '';

            if ( $view === 'admin' ) {
                $value = array();
                if ( isset( $field_object['choices'] ) && is_array( $field_object['choices'] ) ) {
                    foreach( $field_object['choices'] as $choice_val => $choice ) {
                        $value[] = $choice_val;
                    }
                }
                return $value;
            }
            else {

                if ( $field_object['multiple'] && is_array( $field_object['value'] ) ) {
                    $value_arr = array();
                    foreach( $field_object['value'] as $choice ) {
                        if ( is_array( $choice ) && isset( $choice['label'] ) ) {
                            $value_arr[] = $choice['label'];
                        } else {
                            $value_arr[] = $choice;
                        }
                    }
                    $value = ( $view === 'compare' ) ? $value_arr : implode(', ', $value_arr );
                } else {
                    if ( is_array( $field_object['value'] ) && isset( $field_object['value']['label'] ) ) {
                        $value = $field_object['value']['label'];
                    } else {
                        $value = $field_object['value'];
                    }
                }

            }

            if ( $view === 'compare' && $value && isset( $field_object['choices'] ) && is_array( $field_object['choices'] ) ) {
                $new_value = array();
                $value_arr = is_array( $value ) ? $value : array( $value );
                foreach( $value_arr as $value_k => $value_i ) {
                    if ( $k = array_search( $value_i, $field_object['choices'] ) ) {
                        $new_value[] = $k;
                    }
                }
                if ( ! empty( $new_value ) ) {
                    $value = $new_value;
                }
            }

            return $value;

        }

        /*
         * Get value of checkbox field type
         */
        private function get_checkbox_field( $field_object, $view = 'text' ) {

            if ( $view === 'admin' ) {

                $value = array();
                if ( isset( $field_object['choices'] ) && is_array( $field_object['choices']  ) ) {
                    foreach( $field_object['choices'] as $choice_val => $choice ) {
                        $value[] = $choice_val;
                    }
                }

                return $value;

            } else {

                $value = '';

                if ( is_array( $field_object['value'] ) ) {
                    $value_arr = array();
                    foreach( $field_object['value'] as $choice ) {
                        if ( is_array( $choice ) && isset( $choice['label'] ) ) {
                            $value_arr[] = $choice['label'];
                        } else {
                            $value_arr[] = $choice;
                        }
                    }
                    $value = $view === 'compare'  ? $value_arr : implode(', ', $value_arr );
                } else {
                    $value = $field_object['value'];
                }

                if ( $view === 'compare' && $value && isset( $field_object['choices'] ) && is_array( $field_object['choices'] ) ) {
                    $new_value = array();
                    $value_arr = is_array( $value ) ? $value : array( $value );
                    foreach( $value_arr as $value_k => $value_i ) {
                        if ( $k = array_search( $value_i, $field_object['choices'] ) ) {
                            $new_value[] = $k;
                        }
                    }
                    if ( ! empty( $new_value ) ) {
                        $value = $new_value;
                    }
                }

                return $value;

            }

        }

        /*
         * Get value of radio field type
         */
        private function get_radio_field( $field_object, $view = 'text' ) {

            $value = '';

            if ( $view === 'admin' ) {
                $value = array();
                if ( isset( $field_object['choices'] ) && is_array( $field_object['choices']  ) ) {
                    foreach( $field_object['choices'] as $choice_val => $choice ) {
                        $value[] = $choice_val;
                    }
                }
                return $value;
            }

            if ( is_array( $field_object['value'] ) && isset( $field_object['value']['label'] ) ) {
                $value = $field_object['value']['label'];
            } else {
                $value = $field_object['value'];
            }

            if ( $view === 'compare' && $value && isset( $field_object['choices'] ) && is_array( $field_object['choices'] ) ) {
                if ( $k = array_search( $value, $field_object['choices'] ) ) {
                    $value = $k;
                }
            }

            return $value;

        }

        /*
         * Get value of radio field type
         */
        private function get_google_map_field( $field_object, $view = 'text' ) {
            $value = $field_object['value']['address'];
            return $value;
        }

    }

endif;

AWL_ACF::instance();