<?php

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

if ( ! class_exists( 'AWL_Conditions_Check' ) ) :

    /**
     * AWL Conditions check class
     */
    class AWL_Conditions_Check {

        protected $conditions = null;
        protected $rule = null;


        /*
         * Constructor
         */
        public function __construct( $conditions ) {

            $this->conditions = $conditions;

        }


        /*
         * Match condition
         */
        public function match() {

            if ( empty( $this->conditions ) || ! is_array( $this->conditions ) ) {
                return false;
            }

            if ( ! isset( $GLOBALS['product'] ) ) {
                return false;
            }

            /**
             * Filter condition functions
             * @since 1.21
             * @param array Array of custom condition functions
             */
            $custom_match_functions = apply_filters( 'awl_labels_condition_rules', array() );

            $match = false;

            foreach ( $this->conditions as $condition_group ) {

                $rules_match = true;

                if ( $condition_group && ! empty( $condition_group ) ) {

                    foreach( $condition_group as $condition_rule ) {

                        /**
                         * Filter condition rule parameters
                         * @since 1.68
                         * @param array $condition_rule Condition parameters
                         */
                        $condition_rule = apply_filters( 'awl_label_condition_rule', $condition_rule );

                        $this->rule = $condition_rule;
                        $condition_name = $condition_rule['param'];

                        if ( isset( $custom_match_functions[$condition_name] ) ) {
                            $match_rule = call_user_func( $custom_match_functions[$condition_name], $condition_rule );
                        } elseif ( method_exists( $this, 'match_' . $condition_name ) ) {
                            $match_rule = call_user_func( array( $this, 'match_' . $condition_name ) );
                        } else {
                            $match_rule = true;
                        }

                        /**
                         * Filter the result of condition rule
                         * @since 1.68
                         * @param bool $match_rule Result of condition matching
                         * @param string $condition_name Condition name
                         * @param array $condition_rule Condition parameters
                         */
                        $match_rule = apply_filters( 'awl_label_condition_match_rule', $match_rule, $condition_name, $condition_rule );

                        if ( ! $match_rule ) {
                            $rules_match = false;
                            break;
                        }

                    }

                }

                if ( $rules_match ) {
                    $match = true;
                    break;
                }

            }


            return $match;

        }


        /*
         * Compare values
         * @param $value
         * @return bool
         */
        private function compare_values( $compare_value ) {

            global $product;

            /**
             * Filter condition value before compare
             * @since 1.23
             * @param string|integer $compare_value Value to compare with
             * @param array $this->rule Condition parameters
             * @param object $product Current product
             */
            $compare_value = apply_filters( 'awl_label_condition_compare_value', $compare_value, $this->rule, $product );

            $match = false;
            $value = isset( $this->rule['value'] ) ? $this->rule['value'] : '';
            $operator = $this->rule['operator'];

            if ( is_bool( $compare_value )  ) {
                $compare_value = $compare_value ? 'true' : 'false';
            }

            if ( 'equal' == $operator ) {
                $match = ($compare_value == $value);
            } elseif ( 'not_equal' == $operator ) {
                $match = ($compare_value != $value);
            } elseif ( 'greater' == $operator ) {
                $match = ($compare_value >= $value);
            } elseif ( 'less' == $operator ) {
                $match = ($compare_value <= $value);
            }

            return $match;

        }


        /*
         * Product stock status rule
         */
        public function match_stock_status() {

            global $product;
            $value = $product->get_stock_status();

            return call_user_func_array( array( $this, 'compare_values' ), array( $value ) );

        }


        /*
         * Product visibility rule
         */
        public function match_visibility() {

            global $product;

            if ( method_exists( $product, 'get_catalog_visibility' ) ) {
                $value = $product->get_catalog_visibility();
            } elseif ( method_exists( $product, 'get_visibility' ) ) {
                $value = $product->get_visibility();
            } else  {
                $value = $product->visibility;
            }

            return call_user_func_array( array( $this, 'compare_values' ), array( $value ) );

        }


        /*
         * Product price rule
         */
        public function match_price() {

            global $product;

            $this->rule['value'] = floatval( $this->rule['value'] );

            if ( isset( $this->rule['suboption'] ) && $this->rule['suboption'] === 'sale' ) {
                $value = $product->get_sale_price();
            } elseif( isset( $this->rule['suboption'] ) && $this->rule['suboption'] === 'regular' ) {
                $value = $product->get_regular_price();
            } else {
                $value = $product->get_price();
            }

            $value = floatval( $value );

            return call_user_func_array( array( $this, 'compare_values' ), array( $value ) );

        }


        /*
         * Product sale discount rule
         */
        public function match_sale_discount() {

            global $product;

            if ( ! isset( $this->rule['suboption'] ) ) {
                return false;
            }

            if ( $this->rule['suboption'] === 'percents' ) {
                $value = AWL_Product_Data::get_discount_percent( $product );
            }

            if ( $this->rule['suboption'] === 'amount' ) {
                $value = AWL_Product_Data::get_discount_amount( $product );
            }

            $decimals = strpos(strrev($this->rule['value']), ".");
            if ( ! $decimals ) {
                $decimals = 0;
            }

            $value = round( $value, $decimals );

            return call_user_func_array( array( $this, 'compare_values' ), array( $value ) );

        }


        /*
         * Product quantity rule
         */
        public function match_quantity() {

            global $product;

            $value = AWL_Product_Data::get_quantity( $product );

            if ( is_bool( $value ) && ! $value ) {
                return false;
            }

            return call_user_func_array( array( $this, 'compare_values' ), array( $value ) );

        }


        /*
         * Product shipping class rule
         */
        public function match_shipping_class() {
            global $product;
            $value = $product->get_shipping_class_id();
            if ( ! $value ) {
                $value = 'none';
            }
            return call_user_func_array( array( $this, 'compare_values' ), array( $value ) );
        }


        /*
         * Product type rule
         */
        public function match_product_type() {
            global $product;
            $value = $product->get_type();
            return call_user_func_array( array( $this, 'compare_values' ), array( $value ) );
        }


        /*
         * Product age rule
         */
        public function match_product_age() {
            global $post;

            $compared = date_i18n( 'Y-m-d-H', strtotime( $post->post_date ) );
            $time_frame = $this->rule['suboption'];

            $value = $this->rule['value'];
            $value = date_i18n( 'Y-m-d-H', strtotime( "-$value $time_frame", time() ) );

            $this->rule['value'] = $compared;

            return call_user_func_array( array( $this, 'compare_values' ), array( $value ) );
        }


        /*
         * Product sale date rule
         */
        public function match_sale_date() {
            global $product;

            if ( ! $product->is_on_sale() || ! $product->get_sale_price() ) {
                return false;
            }

            $time_frame = strpos( $this->rule['suboption'], 'hours' ) !== false ? 'hours' : 'days';
            $compared = 0;
            $value = 0;
            $value_set = $this->rule['value'];

            // Sale starts from rules
            if ( method_exists( $product, 'get_date_on_sale_from' ) && ( $this->rule['suboption'] === 'start_hours' || $this->rule['suboption'] === 'start_days' ) ) {
                $compared = date_i18n( 'Y-m-d-H', strtotime( $product->get_date_on_sale_from() ) );
                $value = date_i18n( 'Y-m-d-H', strtotime( "-$value_set $time_frame", time() ) );
            }

            // Sale ends in rules
            if ( method_exists( $product, 'get_date_on_sale_to' ) && ( $this->rule['suboption'] === 'end_hours' || $this->rule['suboption'] === 'end_days' ) ) {
                $value = date_i18n( 'Y-m-d-H', strtotime( $product->get_date_on_sale_to() ) );
                $compared = date_i18n( 'Y-m-d-H', strtotime( "+$value_set $time_frame", time() ) );
            }

            $this->rule['value'] = $compared;

            return call_user_func_array( array( $this, 'compare_values' ), array( $value ) );

        }


        /*
         * Product rating rule
         */
        public function match_rating() {
            global $product;

            $value = $product->get_average_rating();

            if ( ! $value ) {
                return false;
            }

            return call_user_func_array( array( $this, 'compare_values' ), array( $value ) );

        }


        /*
         * Product reviews count rule
         */
        public function match_reviews_count() {
            global $product;

            $date_query = 'all';

            if ( isset( $this->rule['suboption'] ) && $this->rule['suboption'] !== 'all' ) {

                $date_query = array();

                switch ( $this->rule['suboption'] ) {
                    case 'hour':
                        $date_query =  array( array( 'after' => '24 hours ago' ) );
                        break;
                    case 'week':
                        $date_query =  array( array( 'after' => '1 week ago' ) );
                        break;
                    case 'month':
                        $date_query =  array( array( 'after' => '30 days ago' ) );
                        break;
                    case 'year':
                        $date_query =  array( array( 'after' => '1 year ago' ) );
                        break;
                }

            }

            $value = AWL_Product_Data::get_reviews_count( $date_query, $product );

            return call_user_func_array( array( $this, 'compare_values' ), array( $value ) );

        }


        /*
         * Product sales rule
         */
        public function match_sales() {
            global $product;

            $date_query = 'all';

            if ( isset( $this->rule['suboption'] ) && $this->rule['suboption'] !== 'all' ) {

                switch ( $this->rule['suboption'] ) {
                    case 'hour':
                        $date_query = '-24 hours';
                        break;
                    case 'week':
                        $date_query = 'midnight -6 days';
                        break;
                    case 'month':
                        $date_query = '-1 month';
                        break;
                    case 'year':
                        $date_query = '-1 year';
                        break;
                }

            }

            $value = AWL_Product_Data::get_sales_count( $date_query, $product );

            return call_user_func_array( array( $this, 'compare_values' ), array( $value ) );

        }


        /*
         * Product sale status rule
         */
        public function match_sale_status() {
            global $product;
            $value = AWL_Product_Data::is_on_sale( $product );
            return call_user_func_array( array( $this, 'compare_values' ), array( $value ) );
        }


        /*
         * Product featured rule
         */
        public function match_featured() {
            global $product;
            $value = $product->is_featured();
            return call_user_func_array( array( $this, 'compare_values' ), array( $value ) );
        }


        /*
         * Product in cart rule
         */
        public function match_in_cart() {
            global $product;

            $product_ids = array();
            foreach ( WC()->cart->get_cart() as $item ) {
                $product_ids[] = $item['product_id'];
                if ( isset( $item['variation_id'] ) ) {
                    $product_ids[] = $item['variation_id'];
                }
            }

            $value = in_array( $product->get_id(), $product_ids );

            return call_user_func_array( array( $this, 'compare_values' ), array( $value ) );

        }


        /*
         * Product has image rule
         */
        public function match_has_image() {
            global $product;
            $value = !! $product->get_image_id();
            return call_user_func_array( array( $this, 'compare_values' ), array( $value ) );
        }


        /*
         * Product has gallery rule
         */
        public function match_has_gallery() {
            global $product;
            $value = !! $product->get_gallery_image_ids();
            return call_user_func_array( array( $this, 'compare_values' ), array( $value ) );
        }


        /*
         * Product rule
         */
        public function match_product() {
            global $product;
            $value = $product->is_type( 'variation' ) && method_exists( $product, 'get_parent_id' ) ? $product->get_parent_id() : $product->get_id();
            return call_user_func_array( array( $this, 'compare_values' ), array( $value ) );
        }


        /*
         * Product category rule
         */
        public function match_product_category() {
            global $product;

            $product_id = $product->is_type( 'variation' ) && method_exists( $product, 'get_parent_id' ) ? $product->get_parent_id() : $product->get_id();
            $value = has_term( $this->rule['value'], 'product_cat', $product_id );
            $operator = $this->rule['operator'];

            if ( 'equal' == $operator ) {
                return $value;
            } else {
                return !$value;
            }

        }


        /*
         * Product tag rule
         */
        public function match_product_tag() {
            global $product;

            $product_id = $product->is_type( 'variation' ) && method_exists( $product, 'get_parent_id' ) ? $product->get_parent_id() : $product->get_id();
            $value = has_term( $this->rule['value'], 'product_tag', $product_id );
            $operator = $this->rule['operator'];

            if ( 'equal' == $operator ) {
                return $value;
            } else {
                return !$value;
            }

        }


        /*
         * Product taxonomy rule
         */
        public function match_product_taxonomy() {
            global $product;

            $product_id = $product->is_type( 'variation' ) && method_exists( $product, 'get_parent_id' ) ? $product->get_parent_id() : $product->get_id();
            $term = $this->rule['value'] === 'awl_any' ? '' : $this->rule['value'];
            $value = has_term( $term, $this->rule['suboption'], $product_id );
            $operator = $this->rule['operator'];

            if ( 'equal' == $operator ) {
                return $value;
            } else {
                return !$value;
            }

        }


        /*
         * Product attributes rule
         */
        public function match_product_attributes() {
            global $product;

            $term = $this->rule['value'] === 'awl_any' ? '' : $this->rule['value'];
            $value = has_term( $term, $this->rule['suboption'], $product->get_id() );
            $operator = $this->rule['operator'];

            if ( 'equal' == $operator ) {
                return $value;
            } else {
                return !$value;
            }

        }


        /*
         * Product custom attributes rule
         */
        public function match_product_custom_attributes() {
            global $product;

            $term = $this->rule['value'] === 'awl_any' ? '' : $this->rule['value'];
            $operator = $this->rule['operator'];
            $tax = $this->rule['suboption'];
            $attributes = $product->get_attributes();
            $value = false;

            if ( $attributes && isset( $attributes[$tax] ) ) {

                if ( ! $term ) {
                    $value = true;
                } else {

                    $attribute = $attributes[$tax];
                    $attribute_options = $attribute->get_options();

                    if ( $attribute_options && ! empty( $attribute_options ) ) {
                        foreach( $attribute_options as $attribute_option ) {
                            if ( sanitize_key( strval( $attribute_option ) ) === $term ) {
                                $value = true;
                                break;
                            }
                        }
                    }

                }

            }

            if ( 'equal' == $operator ) {
                return $value;
            } else {
                return !$value;
            }

        }


        /*
         * Product meta rule
         */
        public function match_product_meta() {
            global $product;

            $meta_name = $this->rule['suboption'];
            $meta_value = $this->rule['value'] === 'awl_any' ? '' : $this->rule['value'];
            $value = false;

            $meta = get_post_meta( $product->get_id(), $meta_name );

            if ( ! $meta_value ) {
                $value = !! $meta;
            } else {
                if ( $meta ) {
                    $value = ( $meta_value === $meta[0] );
                } else {
                    return false;
                }
            }

            if ( 'equal' == $this->rule['operator'] ) {
                return $value;
            } else {
                return !$value;
            }

        }


        /*
         * User rule
         */
        public function match_user() {

            if ( ! is_user_logged_in() ) {
                return false;
            }

            $value = get_current_user_id();

            return call_user_func_array( array( $this, 'compare_values' ), array( $value ) );

        }


        /*
         * User role rule
         */
        public function match_user_role() {

            $role = $this->rule['value'];

            if ( is_user_logged_in() ) {
                global $current_user;
                $roles = (array) $current_user->roles;
            } else {
                $roles = array( 'non-logged' );
            }

            $value = array_search( $role, $roles ) !== false;

            if ( 'equal' == $this->rule['operator'] ) {
                return $value;
            } else {
                return !$value;
            }

        }


        /*
         * User country rule
         */
        public function match_user_country() {

            $value = WC()->customer->get_shipping_country();

            return call_user_func_array( array( $this, 'compare_values' ), array( $value ) );

        }


        /*
         * User language rule
         */
        public function match_user_language() {

            $value = substr( $_SERVER['HTTP_ACCEPT_LANGUAGE'], 0, 2 );

            return call_user_func_array( array( $this, 'compare_values' ), array( $value ) );

        }


        /*
         * User device rule
         */
        public function match_user_device() {

            $value = wp_is_mobile() ? 'mobile' : 'desktop';

            return call_user_func_array( array( $this, 'compare_values' ), array( $value ) );

        }

        /*
        * User cart rule
        */
        public function match_user_cart() {

            $this->rule['value'] = intval( $this->rule['value'] );

            if ( isset( $this->rule['suboption'] ) && $this->rule['suboption'] === 'number' ) {
                $value = count( WC()->cart->get_cart() );
            } elseif( isset( $this->rule['suboption'] ) && $this->rule['suboption'] === 'sum' ) {
                $value = WC()->cart->get_total( 'dis' );
            } elseif( isset( $this->rule['suboption'] ) && $this->rule['suboption'] === 'average' ) {
                $items = count( WC()->cart->get_cart() );
                $total = WC()->cart->get_total( 'dis' );
                $value = ( $items && $total ) ? $total / $items : 0;
            } else {
                return false;
            }

            $value = intval( $value );

            return call_user_func_array( array( $this, 'compare_values' ), array( $value ) );

        }


        /*
         * User shop stats rule
         */
        public function match_user_shop_stats() {

            $this->rule['value'] = intval( $this->rule['value'] );

            if ( isset( $this->rule['suboption'] ) && $this->rule['suboption'] === 'orders_number' ) {
                $value = wc_get_customer_order_count( get_current_user_id() );
            } elseif( isset( $this->rule['suboption'] ) && $this->rule['suboption'] === 'aov' ) {
                $orders = wc_get_customer_order_count( get_current_user_id() );
                $total_spend = wc_get_customer_total_spent( get_current_user_id() );
                $value = ( $orders && $total_spend ) ? $total_spend / $orders : 0;
            } elseif( isset( $this->rule['suboption'] ) && $this->rule['suboption'] === 'total_spend' ) {
                $value = wc_get_customer_total_spent( get_current_user_id() );
            } else {
                return false;
            }

            $value = intval( $value );

            return call_user_func_array( array( $this, 'compare_values' ), array( $value ) );

        }


        /*
         * Page rule
         */
        public function match_page() {

            global $wp_query;

            if ( is_shop() ) {
                $value = wc_get_page_id( 'shop' );
            } elseif ( is_cart() ) {
                $value = wc_get_page_id( 'cart' );
            } elseif ( is_checkout() ) {
                $value = wc_get_page_id( 'checkout' );
            } elseif ( is_account_page() ) {
                $value = wc_get_page_id( 'myaccount' );
            } else {
                $value = $wp_query->get_queried_object_id();
            }

            return call_user_func_array( array( $this, 'compare_values' ), array( $value ) );

        }


        /*
         * Page template rule
         */
        public function match_page_template() {

            if ( ! is_page() ) {
                return false;
            }

            $value = get_page_template_slug( get_queried_object_id() );

            if ( $value === '' ) {
                $value = 'default';
            }

            return call_user_func_array( array( $this, 'compare_values' ), array( $value ) );

        }


        /*
         * Page type rule
         */
        public function match_page_type() {

            $queried_obj = get_queried_object();

            $page_type = array();

            if ( is_singular( 'product' ) ) {
                $page_type[] = 'product';
            }
            if ( is_home() || is_front_page() ) {
                $page_type[] = 'front';
            }
            if ( is_shop() ) {
                $page_type[] = 'shop';
            }
            if ( is_cart() ) {
                $page_type[] = 'cart';
            }
            if ( is_checkout() ) {
                $page_type[] = 'checkout';
            }
            if ( is_account_page() ) {
                $page_type[] = 'account';
            }
            if ( is_search() ) {
                $page_type[] = 'search';
            }
            if ( is_tax( 'product_cat' ) ) {
                $page_type[] = 'category_page';
            }
            if ( is_tax( 'product_tag' ) ) {
                $page_type[] = 'tag_page';
            }
            if ( is_tax() && taxonomy_is_product_attribute( $queried_obj->taxonomy ) ) {
                $page_type[] = 'attribute_page';
            }
            if ( is_tax() ) {
                $page_type[] = 'tax_page';
            }
            if ( is_archive() ) {
                $page_type[] = 'archive';
            }
            if ( is_page() ) {
                $page_type[] = 'page';
            }

            $value = in_array( $this->rule['value'], $page_type );

            if ( 'equal' == $this->rule['operator'] ) {
                return $value;
            } else {
                return !$value;
            }

        }


        /*
         * Page archives rule
         */
        public function match_page_archives() {

            if ( ! is_tax() ) {
                return false;
            }

            $tax = $this->rule['suboption'];
            $term = $this->rule['value'] === 'awl_any' ? '' : $this->rule['value'];

            if ( 'attributes' === $tax ) {
                $queried_obj = get_queried_object();
                $value = $term ? is_tax( $term ) : ( is_tax() && taxonomy_is_product_attribute( $queried_obj->taxonomy ) );
            } else {
                $value = is_tax( $tax, $term );
            }

            if ( 'equal' == $this->rule['operator'] ) {
                return $value;
            } else {
                return !$value;
            }

        }


        /*
         * Page language rule
         */
        public function match_page_language() {

            if ( ! AWL_Helpers::is_lang_plugin_active() ) {
                return true;
            }

            $value = AWL_Helpers::get_current_lang();

            return call_user_func_array( array( $this, 'compare_values' ), array( $value ) );

        }


        /*
         * Date range rule
         */
        public function match_date_range() {

            $value = strtotime( date_i18n( 'd.m.Y' ) );

            if ( strpos( $this->rule['value'], '-' ) !== false ) {

                $dates = preg_split( '/-/', $this->rule['value'] );
                $date_from = trim( $dates[0] );
                $date_to = trim( $dates[1] );

                if ( $date_from && $date_to ) {

                    $date_from = str_replace( '/', '.', $date_from );
                    $date_to = str_replace( '/', '.', $date_to );

                    $date_from = strtotime( $date_from );
                    $date_to = strtotime( $date_to );

                    $compare = false;

                    switch( $this->rule['operator'] ) {
                        case 'equal':
                            $compare = (($value >= $date_from) && ($value <= $date_to));
                            break;
                        case 'not_equal':
                            $compare = (($value < $date_from) || ($value > $date_to));
                            break;
                        case 'greater':
                            $compare = ($value >= $date_from);
                            break;
                        case 'less':
                            $compare = ($value <= $date_to);
                            break;
                    }

                    return $compare;

                }

            }

            $this->rule['value'] = strtotime( str_replace( '/', '.', $this->rule['value'] ) );

            return call_user_func_array( array( $this, 'compare_values' ), array( $value ) );

        }


        /*
         * Time range rule
         */
        public function match_time_range() {

            $value = date_i18n( 'H:i' );

            if ( strpos( $this->rule['value'], '-' ) !== false ) {

                $dates = preg_split( '/-/', $this->rule['value'] );
                $date_from = trim( $dates[0] );
                $date_to = trim( $dates[1] );

                if ( $date_from && $date_to ) {

                    $compare = false;

                    switch( $this->rule['operator'] ) {
                        case 'equal':
                            $compare = (($value >= $date_from) && ($value <= $date_to));
                            break;
                        case 'not_equal':
                            $compare = (($value < $date_from) || ($value > $date_to));
                            break;
                        case 'greater':
                            $compare = ($value >= $date_from);
                            break;
                        case 'less':
                            $compare = ($value <= $date_to);
                            break;
                    }

                    return $compare;

                }

            }

            return call_user_func_array( array( $this, 'compare_values' ), array( $value ) );

        }


        /*
         * Date week day rule
         */
        public function match_date_week_days() {
            $value = strtolower( date('l') );
            return call_user_func_array( array( $this, 'compare_values' ), array( $value ) );
        }


    }

endif;