<?php

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}


if ( ! class_exists( 'AWL_Admin' ) ) :

    /**
     * Class for plugin admin panel
     */
    class AWL_Admin {

        /**
         * @var AWL_Admin The single instance of the class
         */
        protected static $_instance = null;

        /**
         * Main AWL_Admin Instance
         *
         * Ensures only one instance of AWL_Admin is loaded or can be loaded.
         *
         * @static
         * @return AWL_Admin - Main instance
         */
        public static function instance() {
            if ( is_null( self::$_instance ) ) {
                self::$_instance = new self();
            }
            return self::$_instance;
        }

        /*
         * Constructor
         */
        public function __construct() {

            if ( ! AWL_Admin_Options::check_settings() ) {
                $default_settings = AWL_Admin_Options::get_default_settings();
                update_option( 'awl_pro_settings', $default_settings );
            }

            add_action( 'admin_menu', array( $this, 'add_admin_page' ) );

            add_action( 'admin_init', array( $this, 'register_settings' ) );

            add_action( 'add_meta_boxes', array( $this, 'post_type_meta_box' ) );

            add_action( 'save_post', array( $this, 'save_meta_boxes' ), 10, 2 );

            add_action( 'admin_enqueue_scripts', array( $this, 'admin_enqueue_scripts' ) );

            add_action( 'admin_notices', array( $this, 'display_welcome_header' ), 1 );

        }

        /**
         * Add options page
         */
        public function add_admin_page() {

            $slug = 'edit.php?post_type=awl-labels';

            add_menu_page( esc_html__( 'Adv. Woo Labels', 'advanced-woo-labels' ), esc_html__( 'Adv. Woo Labels', 'advanced-woo-labels' ), 'edit_others_posts', $slug, false, 'dashicons-tag', 58);

            add_submenu_page( $slug, __( 'All Labels', 'advanced-woo-labels' ), __('All Labels', 'advanced-woo-labels'), 'edit_others_posts', $slug );
            add_submenu_page( $slug, __( 'Add New', 'advanced-woo-labels' ), __('Add New', 'advanced-woo-labels'), 'edit_others_posts', 'post-new.php?post_type=awl-labels' );
            add_submenu_page( $slug, __( 'Settings', 'advanced-woo-labels' ), __( 'Settings', 'advanced-woo-labels'), 'manage_options', 'awl-options', array( $this, 'display_admin_page' ) );

        }

        /*
         * Add meta boxes
         */
        public function post_type_meta_box() {

            add_meta_box( 'awl_label_conditions',  __( 'Label conditions', 'advanced-woo-labels' ), array( $this, 'metabox_show_rules' ), 'awl-labels', 'normal' );

            add_meta_box( 'awl_label_settings',  __( 'Label settings', 'advanced-woo-labels' ), array( $this, 'metabox_show_settings' ), 'awl-labels', 'normal' );

            add_meta_box( 'awl_label_status',  __( 'Status', 'advanced-woo-labels' ), array( $this, 'metabox_show_status' ), 'awl-labels', 'side' );

            add_meta_box( 'awl_label_link',  __( 'Link', 'advanced-woo-labels' ), array( $this, 'metabox_show_link' ), 'awl-labels', 'side' );

            add_meta_box( 'awl_label_priority',  __( 'Priority', 'advanced-woo-labels' ), array( $this, 'metabox_show_priority' ), 'awl-labels', 'side', 'low' );

        }

        /*
         * Save meta box settings
         */
        public function save_meta_boxes( $post_id, $post ) {

            if ( ! isset( $_POST['awl_label_meta_box_nonce'] ) || ! wp_verify_nonce( $_POST['awl_label_meta_box_nonce'], 'awl_label_meta_box' ) ) {
                return $post_id;
            }

            if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
                return $post_id;
            }

            if ( ! current_user_can( 'edit_posts' ) ) {
                return $post_id;
            }

            if ( wp_is_post_revision( $post_id ) ) {
                return $post_id;
            }

            if ( $post->post_type != 'awl-labels' ) {
                return $post_id;
            }

            if ( ! isset( $_POST['awl_label_params'] ) ) {
                return $post_id;
            }

            $label = $_POST['awl_label_params'];
            //$label['conditions'] = wpc_sanitize_conditions( $_POST['conditions'] );

            if ( isset( $label['awl_label_priority'] ) ) {
                update_post_meta( $post_id, '_awl_label_priority', sanitize_text_field( $label['awl_label_priority']['priority'] ) );
            }

            if ( ! isset( $label['awl_label_status'] ) ) {
                $label['awl_label_status']['status'] = '0';
            }

            AWL_PRO()->update_label_settings( $post_id, $label );

        }

        /*
         * Meta box settings for labels rules
         */
        public function metabox_show_rules() {

            global $post;

            wp_nonce_field( 'awl_label_meta_box', 'awl_label_meta_box_nonce' );

            $label = AWL_PRO()->get_label_settings( $post->ID );

            $html = AWL_Admin_Meta_Boxes::get_rules_meta_box( $label );

            echo $html;

        }

        /*
         * Meta box settings for labels
         */
        public function metabox_show_settings() {

            global $post;

            $label = AWL_PRO()->get_label_settings( $post->ID );

            $html = AWL_Admin_Meta_Boxes::get_settings_meta_box( $label );

            echo $html;

        }

        /*
         * Meta box status for labels
         */
        public function metabox_show_status() {

            global $post;

            $label = AWL_PRO()->get_label_settings( $post->ID );

            $html = AWL_Admin_Meta_Boxes::get_status_meta_box( $label );

            echo $html;

        }

        /*
         * Meta box link for labels
         */
        public function metabox_show_link() {

            global $post;

            $label = AWL_PRO()->get_label_settings( $post->ID );

            $html = AWL_Admin_Meta_Boxes::get_link_meta_box( $label );

            echo $html;

        }

        /*
         * Meta box priority for labels
         */
        public function metabox_show_priority() {

            global $post;

            $label = AWL_PRO()->get_label_settings( $post->ID );

            $html = AWL_Admin_Meta_Boxes::get_priority_meta_box( $post->ID );

            echo $html;

        }

        /**
         * Generate and display options page
         */
        public function display_admin_page() {

            $options = AWL_Admin_Options::options_array();
            $nonce = wp_create_nonce( 'plugin-settings' );

            $admin_page = new AWL_Admin_Page( $options, $nonce );

            $admin_page->generate_page();

        }

        /*
         * Register plugin settings
         */
        public function register_settings() {
            register_setting( 'awl_pro_settings', 'awl_pro_settings' );
        }

        /*
         * Enqueue admin scripts and styles
         */
        public function admin_enqueue_scripts() {

            $screen = get_current_screen();

            if ( ( isset( $_GET['page'] ) && $_GET['page'] == 'awl-options' ) || ( $screen && $screen->post_type && $screen->post_type === 'awl-labels' ) ) {

                wp_register_script( 'jquery-tiptip', AWL_PRO_URL . '/assets/js/jquery.tipTip.js', array( 'jquery' ), AWL_PRO_VERSION, true );

                wp_enqueue_style( 'wp-color-picker' );
                wp_enqueue_style( 'awl-select2', AWL_PRO_URL . '/assets/css/select2.min.css' );
                wp_enqueue_style( 'awl-admin-style', AWL_PRO_URL . '/assets/css/admin.css', array(), 'pro' . AWL_PRO_VERSION );

                wp_enqueue_script( 'jquery' );
                wp_enqueue_script( 'jquery-tiptip' );
                wp_enqueue_script( 'select2' );
                wp_enqueue_script( 'wp-color-picker' );
                wp_enqueue_media();
                //wp_enqueue_script( 'wp-color-picker-alpha', AWL_PRO_URL . '/assets/js/wp-color-picker-alpha.js', array('jquery', 'wp-color-picker' ), AWL_PRO_VERSION );
                wp_enqueue_script( 'awl-admin-scripts', AWL_PRO_URL . '/assets/js/admin.js', array( 'jquery', 'wp-color-picker', 'select2', 'jquery-tiptip' ), 'pro' . AWL_PRO_VERSION );
                wp_enqueue_script( 'awl-admin-preview', AWL_PRO_URL . '/assets/js/admin-preview.js', array('jquery', 'wp-color-picker' ), 'pro' . AWL_PRO_VERSION );
                wp_localize_script( 'awl-admin-scripts', 'awl_vars', array(
                    'ajaxurl' => admin_url( 'admin-ajax.php', 'relative' ),
                    'ajax_nonce' => wp_create_nonce( 'awl_admin_ajax_nonce' ),
                    'img_url' => AWL_PRO_IMG,
                    'copy_text' => __( 'Copied', 'advanced-woo-labels' ),
                    'confirm_hooks' => __( 'This action will generate the currently used display hooks. Please only confirm the action if you are sure of what you are doing. In most cases, this action is not required as these hooks are already in use.', 'advanced-woo-labels' ),
                ) );

            }

        }

        /*
         * Add welcome notice
         */
        public function display_welcome_header() {

            $screen = get_current_screen();

            if ( ! $screen || $screen->id !== 'edit-awl-labels' ) {
                return;
            }

            $labels = AWL_Helpers::get_awl_labels( array( 'post_status' => 'any', 'suppress_filters' => true ) );

            if ( $labels && count( $labels ) > 0 ) {
                return;
            }

            echo AWL_Admin_Meta_Boxes::get_welcome_notice();

        }

    }

endif;


add_action( 'init', 'AWL_Admin::instance' );