<?php

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

if ( ! class_exists( 'AWL_Admin_Helpers' ) ) :

    /**
     * Class for admin help methods
     */
    class AWL_Admin_Helpers {

        /*
         * Get available stock statuses
         * @return array
         */
        static public function get_stock_statuses() {

            $options = array();

            if ( function_exists( 'wc_get_product_stock_status_options' ) ) {
                $values = wc_get_product_stock_status_options();
            } else {
                $values = apply_filters(
                    'woocommerce_product_stock_status_options',
                    array(
                        'instock'     => __( 'In stock', 'woocommerce' ),
                        'outofstock'  => __( 'Out of stock', 'woocommerce' ),
                        'onbackorder' => __( 'On backorder', 'woocommerce' ),
                    )
                );
            }

            foreach ( $values as $value_val => $value_name ) {
                $options[$value_val] = $value_name;
            }

            return $options;

        }

        /*
         * Get available product visibilities
         * @return array
         */
        static public function get_visibilities() {

            $options = array();

            if ( function_exists( 'wc_get_product_visibility_options' ) ) {
                $values = wc_get_product_visibility_options();
            } else {
                $values = apply_filters(
                    'woocommerce_product_visibility_options',
                    array(
                        'visible' => __( 'Shop and search results', 'woocommerce' ),
                        'catalog' => __( 'Shop only', 'woocommerce' ),
                        'search'  => __( 'Search results only', 'woocommerce' ),
                        'hidden'  => __( 'Hidden', 'woocommerce' ),
                    )
                );
            }

            foreach ( $values as $value_val => $value_name ) {
                $options[$value_val] = $value_name;
            }

            return $options;

        }

        /*
         * Get available product types
         * @return array
         */
        static public function get_product_types() {

            $options = array();

            if ( function_exists( 'wc_get_product_types' ) ) {
                $values = wc_get_product_types();
            } else {
                $values = apply_filters(
                    'product_type_selector',
                    array(
                        'simple'   => __( 'Simple product', 'woocommerce' ),
                        'grouped'  => __( 'Grouped product', 'woocommerce' ),
                        'external' => __( 'External/Affiliate product', 'woocommerce' ),
                        'variable' => __( 'Variable product', 'woocommerce' ),
                    )
                );
            }

            $values['variation']  = __( 'Product variation', 'advanced-woo-labels' );

            foreach ( $values as $value_val => $value_name ) {
                $options[$value_val] = $value_name;
            }

            return $options;

        }

        /*
         * Get available products
         * @return array
         */
        static public function get_products() {

            $options = array();

            $args = array(
                'posts_per_page' => -1,
                'post_type'      => 'product'
            );

            $products = get_posts( $args );

            if ( ! empty( $products ) ) {
                foreach ( $products as $product ) {
                    $options[$product->ID] = $product->post_title;
                }
            }

            return $options;

        }

        /*
         * Get specific product
         * @return array
         */
        static public function get_product( $id = 0 ) {

            $options = array();

            if ( $id ) {
                $product_object = wc_get_product( $id );
                if ( $product_object ) {
                    $formatted_name = $product_object->get_formatted_name();
                    $options[$id] = rawurldecode( wp_strip_all_tags( $formatted_name ) );
                }
            }

            return $options;

        }

        /*
         * Get available taxonomies
         * @return array
         */
        static public function get_tax() {

            $taxonomy_objects = get_object_taxonomies( 'product', 'objects' );
            $options = array();

            foreach( $taxonomy_objects as $taxonomy_object ) {
                if ( in_array( $taxonomy_object->name, array( 'product_cat', 'product_tag', 'product_type', 'product_visibility', 'product_shipping_class' ) ) ) {
                    continue;
                }

                if ( strpos( $taxonomy_object->name, 'pa_' ) === 0 ) {
                    continue;
                }

                $options[] = array(
                    'name'  => $taxonomy_object->label,
                    'value' => $taxonomy_object->name
                );

            }

            return $options;

        }

        /*
         * Get available taxonomies_terms
         * @param $name string Tax name
         * @return array
         */
        static public function get_tax_terms( $name = false ) {

            if ( ! $name ) {
                return false;
            }

            $tax = get_terms( array(
                'taxonomy'   => $name,
                'hide_empty' => false,
            ) );

            $options = array();

            if ( $name && $name === 'product_shipping_class' ) {
                $options['none'] = __( "No shipping class", "advanced-woo-labels" );
            }

            if ( ! empty( $tax ) ) {
                foreach ( $tax as $tax_item ) {
                    $options[$tax_item->term_id] = $tax_item->name;
                }
            }

            /**
             * Filter options array of taxonomy terms
             * @since 1.68
             * @param array $options Terms array
             * @param string $name Taxonomy name
             */
            $options = apply_filters( 'awl_label_options_get_tax_terms', $options, $name );

            return $options;

        }

        /*
         * Get available product attributes
         * @return array
         */
        static public function get_attributes() {

            $options = array();

            if ( function_exists( 'wc_get_attribute_taxonomies' ) ) {
                $attributes = wc_get_attribute_taxonomies();
                if ( $attributes && ! empty( $attributes ) ) {
                    foreach( $attributes as $attribute ) {
                        $attribute_name = wc_attribute_taxonomy_name( $attribute->attribute_name );
                        $options[] = array(
                            'name'  => $attribute->attribute_label,
                            'value' => $attribute_name
                        );
                    }
                }

            }

            return $options;

        }

        /*
         * Get available product custom attributes
         * @return array
         */
        static public function get_custom_attributes( $name = '' ) {

            global $wpdb;

            $options = array();
            $attributes = array();
            $custom_attributes = $wpdb->get_results( "SELECT meta_value FROM $wpdb->postmeta WHERE meta_key = '_product_attributes'" );

            if ( ! empty( $custom_attributes ) && !is_wp_error( $custom_attributes ) ) {
                foreach ( $custom_attributes as $custom_attribute ) {
                    if ( $custom_attribute->meta_value ) {
                        $custom_attribute_array = maybe_unserialize( $custom_attribute->meta_value );

                        if ( $custom_attribute_array && is_array( $custom_attribute_array ) && ! empty( $custom_attribute_array ) ) {

                            foreach ($custom_attribute_array as $custom_attribute_key => $custom_attribute_val) {

                                if ( isset( $custom_attribute_val['is_taxonomy'] ) && $custom_attribute_val['is_taxonomy'] ) {
                                    continue;
                                }

                                $attributes[$custom_attribute_key]['name'] = $custom_attribute_val['name'];

                                $val_array = array_map( 'trim', explode( '|', $custom_attribute_val['value'] ) );

                                if ( $val_array && ! empty( $val_array ) ) {
                                    foreach( $val_array as $val_array_attr ) {
                                        $val_array_attr_key = sanitize_key( strval( $val_array_attr ) );
                                        $attributes[$custom_attribute_key]['val'][$val_array_attr_key] = $val_array_attr;
                                    }
                                }

                            }

                        }

                    }
                }
            }

            if ( ! empty( $attributes ) ) {

                foreach( $attributes as $attribute_slug => $attribute ) {

                    if ( $name ) {
                        if ( $name === $attribute_slug && isset( $attribute['val'] ) ) {
                            foreach( $attribute['val'] as $val_key => $val ) {
                                $options[] = array(
                                    'name'  => $val,
                                    'value' => $val_key
                                );
                            }
                        }
                    } else {
                        $options[] = array(
                            'name'  => $attribute['name'],
                            'value' => $attribute_slug
                        );
                    }

                }

            }

            return $options;

        }

        /*
         * Get available product custom fields
         * @return array
         */
        static public function get_custom_fields( $name = '' ) {

            global $wpdb;

            $query = "
                SELECT DISTINCT meta_key as val
                FROM $wpdb->postmeta
                WHERE meta_key NOT LIKE 'attribute_%'
                ORDER BY val ASC
            ";

            if ( $name ) {

                $query = "
                    SELECT DISTINCT meta_value as val
                    FROM $wpdb->postmeta
                    WHERE meta_key = '{$name}'
                    ORDER BY val ASC
                ";

            }

            $wp_es_fields = $wpdb->get_results( $query );
            $options = array();

            if ( is_array( $wp_es_fields ) && ! empty( $wp_es_fields ) ) {
                foreach ( $wp_es_fields as $field ) {
                    if ( isset( $field->val ) ) {
                        $options[] = array(
                            'name'  => $field->val,
                            'value' => $field->val
                        );
                    }
                }
            }

            return $options;

        }

        /*
         * Get all available pages
         * @return array
         */
        static public function get_pages() {

            $pages = get_pages( array( 'parent' => 0, 'hierarchical' => 0 ) );
            $options = array();

            if ( $pages && ! empty( $pages ) ) {

                foreach( $pages as $page ) {

                    $title = $page->post_title ? $page->post_title :  __( "(no title)", "advanced-woo-labels" );

                    $options[$page->ID] = $title;

                    $child_pages = get_pages( array( 'child_of' => $page->ID ) );

                    if ( $child_pages && ! empty( $child_pages ) ) {

                        foreach( $child_pages as $child_page ) {

                            $page_prefix = '';
                            $parents_number = sizeof( $child_page->ancestors );

                            if ( $parents_number && is_int( $parents_number ) ) {
                                $page_prefix = str_repeat( "-", $parents_number );
                            }

                            $title = $child_page->post_title ? $child_page->post_title :  __( "(no title)", "advanced-woo-labels" );
                            $title = $page_prefix . $title;

                            $options[$child_page->ID] = $title;

                        }

                    }

                }

            }

            return $options;

        }

        /*
         * Get all available page templates
         * @return array
         */
        static public function get_page_templates() {

            $page_templates = get_page_templates();
            $options = array();

            $options['default'] = __( 'Default template', 'advanced-woo-labels' );

            if ( $page_templates && ! empty( $page_templates ) ) {
                foreach( $page_templates as $page_template_name => $page_template_file ) {
                    $options[] = array(
                        'name'  => $page_template_name,
                        'value' => $page_template_file
                    );
                }
            }

            return $options;

        }

        /*
         * Get available pages types
         * @return array
         */
        static public function get_page_type() {

            $options = array();

            $types = array(
                'product' => __( 'Product single page', 'advanced-woo-labels' ),
                'front' => __( 'Front page', 'advanced-woo-labels' ),
                'shop' => __( 'Shop page', 'advanced-woo-labels' ),
                'cart' => __( 'Cart page', 'advanced-woo-labels' ),
                'checkout' => __( 'Checkout page', 'advanced-woo-labels' ),
                'account' => __( 'Account page', 'advanced-woo-labels' ),
                'search' => __( 'Search results page', 'advanced-woo-labels' ),
                'archive' => __( 'Archive page', 'advanced-woo-labels' ),
                'category_page' => __( 'Category archive page', 'advanced-woo-labels' ),
                'tag_page' => __( 'Tag archive page', 'advanced-woo-labels' ),
                'attribute_page' => __( 'Attributes archive page', 'advanced-woo-labels' ),
                'tax_page' => __( 'Custom taxonomy archive page', 'advanced-woo-labels' ),
                'page' => __( 'Simple page', 'advanced-woo-labels' ),
            );

            foreach( $types as $type_slug => $type_name ) {
                $options[$type_slug] = $type_name;
            }

            return $options;

        }

        /*
         * Get available archive pages
         * @return array
         */
        static public function get_page_archives() {

            $options = array();
            $taxonomy_objects = get_object_taxonomies( 'product', 'objects' );

            $types = array(
                'product_cat' => __( 'Category', 'advanced-woo-labels' ),
                'product_tag' => __( 'Tag', 'advanced-woo-labels' ),
                'attributes' => __( 'Attributes', 'advanced-woo-labels' ),
            );

            foreach( $types as $type_slug => $type_name ) {

                $options[] = array(
                    'name'  => $type_name,
                    'value' => $type_slug
                );

            }

            foreach( $taxonomy_objects as $taxonomy_object ) {
                if ( in_array( $taxonomy_object->name, array( 'product_cat', 'product_tag', 'product_type', 'product_visibility', 'product_shipping_class' ) ) ) {
                    continue;
                }

                if ( strpos( $taxonomy_object->name, 'pa_' ) === 0 ) {
                    continue;
                }

                $options[] = array(
                    'name'  => $taxonomy_object->label,
                    'value' => $taxonomy_object->name
                );

            }

            return $options;

        }

        /*
         * Get available archive pages terms
         * @return array
         */
        static public function get_page_archive_terms( $name = false ) {

            if ( ! $name ) {
                return false;
            }

            $options = array();

            switch( $name ) {

                case 'attributes':

                    if ( function_exists( 'wc_get_attribute_taxonomies' ) ) {
                        $attributes = wc_get_attribute_taxonomies();

                        if ( $attributes && ! empty( $attributes ) ) {
                            foreach( $attributes as $attribute ) {
                                if ( $attribute->attribute_public ) {

                                    $options[] = array(
                                        'name'  => $attribute->attribute_label,
                                        'value' => wc_attribute_taxonomy_name( $attribute->attribute_name )
                                    );

                                }
                            }
                        }

                    }

                    break;

                default:

                    $options = AWL_Admin_Helpers::get_tax_terms( $name );

            }

            return $options;

        }

        /*
         * Get all available users
         * @return array
         */
        static public function get_users() {

            $users = get_users();
            $options = array();

            if ( $users && ! empty( $users ) ) {
                foreach( $users as $user ) {
                    $options[$user->ID] = $user->display_name . ' (' . $user->user_nicename . ')';
                }
            }

            return $options;

        }

        /*
         * Get all available user roles
         * @return array
         */
        static public function get_user_roles() {

            global $wp_roles;

            $roles = $wp_roles->roles;
            $options = array();

            if ( $roles && ! empty( $roles ) ) {

                if ( is_multisite() ) {
                    $options['super_admin'] = __( 'Super Admin', 'advanced-woo-labels' );
                }

                foreach( $roles as $role_slug => $role ) {
                    $options[$role_slug] = $role['name'];
                }

                $options['non-logged'] = __( 'Visitor ( not logged-in )', 'advanced-woo-labels' );

            }

            return $options;

        }

        /*
         * Get all available user countries
         * @return array
         */
        static public function get_user_countries() {

            $options = array();

            $values = WC()->countries->get_allowed_countries() + WC()->countries->get_shipping_countries();

            foreach ( $values as $value_val => $value_name ) {
                $options[$value_val] = $value_name;
            }

            return $options;

        }

        /*
         * Get all available user languages
         * @return array
         */
        static public function get_user_languages() {

            $options = array();

            $values = include AWL_PRO_DIR . '/includes/admin/languages.php';

            foreach ( $values as $value_val => $value_name ) {
                $options[$value_val] = $value_name;
            }

            return $options;

        }

        /*
         * Get user devices
         * @return array
         */
        static public function get_user_devices() {

            $options = array();

            $values = array(
                'desktop' => __( 'Desktop', 'advanced-woo-labels' ),
                'mobile'  => __( 'Mobile', 'advanced-woo-labels' ),
            );

            foreach ( $values as $value_val => $value_name ) {
                $options[$value_val] = $value_name;
            }

            return $options;

        }

        /*
         * Get user cart
         * @return array
         */
        static public function get_user_cart() {

            $options = array();

            $values = array(
                'number'  => __( 'Number of items', 'advanced-woo-labels' ),
                'average' => __( 'Average items cost', 'advanced-woo-labels' ),
                'sum'     => __( 'Total sum of items', 'advanced-woo-labels' ),
            );

            foreach ( $values as $value_val => $value_name ) {
                $options[$value_val] = $value_name;
            }

            return $options;

        }

        /*
         * Get available price formats
         * @return array
         */
        static public function get_shop_stats() {

            $options = array();

            $values = array(
                'orders_number' => __( 'Orders number', 'advanced-woo-labels' ),
                'aov'           => __( 'Average order value', 'advanced-woo-labels' ),
                'total_spend'   => __( 'Total spend', 'advanced-woo-labels' ),
            );

            foreach ( $values as $value_val => $value_name ) {
                $options[$value_val] = $value_name;
            }

            return $options;

        }

        /*
         * Get available price formats
         * @return array
         */
        static public function get_price() {

            $options = array();

            $values = array(
                'current' => __( 'Current', 'advanced-woo-labels' ),
                'sale'    => __( 'Sale', 'advanced-woo-labels' ),
                'regular' => __( 'Regular', 'advanced-woo-labels' ),
            );

            foreach ( $values as $value_val => $value_name ) {
                $options[$value_val] = $value_name;
            }

            return $options;

        }

        /*
         * Get available sale discount formats
         * @return array
         */
        static public function get_sale_discount() {

            $options = array();

            $values = array(
                'percents' => __( 'percents', 'advanced-woo-labels' ),
                'amount'   => __( 'amount', 'advanced-woo-labels' ),
            );

            foreach ( $values as $value_val => $value_name ) {
                $options[$value_val] = $value_name;
            }

            return $options;

        }

        /*
         * Get available date formats
         * @return array
         */
        static public function get_date_formats() {

            $options = array();

            $values = array(
                'days'   => __( 'in days', 'advanced-woo-labels' ),
                'weeks'  => __( 'in weeks', 'advanced-woo-labels' ),
                'months' => __( 'in months', 'advanced-woo-labels' ),
                'hours'  => __( 'in hours', 'advanced-woo-labels' ),
            );

            foreach ( $values as $value_val => $value_name ) {
                $options[$value_val] = $value_name;
            }

            return $options;

        }

        /*
         * Get available sale date formats
         * @return array
         */
        static public function get_sale_date_formats() {

            $options = array();

            $values = array(
                'start_hours' => __( 'was started x hours ago', 'advanced-woo-labels' ),
                'start_days'  => __( 'was started x days ago', 'advanced-woo-labels' ),
                'end_hours'   => __( 'will ends in x hours', 'advanced-woo-labels' ),
                'end_days'    => __( 'will ends in x days', 'advanced-woo-labels' ),
            );

            foreach ( $values as $value_val => $value_name ) {
                $options[$value_val] = $value_name;
            }

            return $options;

        }

        /*
         * Get available sales number periods
         * @return array
         */
        static public function get_time_periods() {

            $options = array();

            $values = array(
                'all'   => __( 'all time', 'advanced-woo-labels' ),
                'hour'  => __( 'last 24 hours', 'advanced-woo-labels' ),
                'week'  => __( 'last 7 days', 'advanced-woo-labels' ),
                'month' => __( 'last month', 'advanced-woo-labels' ),
                'year'  => __( 'last year', 'advanced-woo-labels' ),
            );

            foreach ( $values as $value_val => $value_name ) {
                $options[$value_val] = $value_name;
            }

            return $options;

        }

        /*
         * Get available week days
         * @return array
         */
        static public function get_week_days() {

            $options = array();

            $values = array(
                'monday'    => __( 'Monday', 'advanced-woo-labels' ),
                'tuesday'   => __( 'Tuesday', 'advanced-woo-labels' ),
                'wednesday' => __( 'Wednesday', 'advanced-woo-labels' ),
                'thursday'  => __( 'Thursday', 'advanced-woo-labels' ),
                'friday'    => __( 'Friday', 'advanced-woo-labels' ),
                'saturday'  => __( 'Saturday', 'advanced-woo-labels' ),
                'sunday'    => __( 'Sunday', 'advanced-woo-labels' ),
            );

            foreach ( $values as $value_val => $value_name ) {
                $options[$value_val] = $value_name;
            }

            return $options;

        }

        /*
        * Get active site languages
        * @return array
        */
        static public function get_languages() {

            $options = array();

            if ( AWL_Helpers::is_lang_plugin_active() ) {

                if ( has_filter('wpml_active_languages') ) {

                    $languages = apply_filters( 'wpml_active_languages', NULL );
                    if ( $languages ) {

                        foreach ( $languages as $languages_code => $languages_arr ) {
                            $options[$languages_code] = isset( $languages_arr['native_name'] ) && $languages_arr['native_name'] ? $languages_arr['native_name'] : $languages_code;
                        }
                    }
                }

                elseif ( ( $enabled_languages = get_option('qtranslate_enabled_languages') ) && function_exists( 'qtranxf_getLanguage' ) ) {

                    foreach ( $enabled_languages as $enabled_languages_code ) {
                        $options[$enabled_languages_code] = $enabled_languages_code;
                    }

                }

            }

            return $options;

        }

        /*
         * Sanitize tag
         * @return string
         */
        static public function sanitize_tag( $text ) {
            $text = str_replace( array( '][', ']', '[', '_' ), '-', $text );
            $text = trim( sanitize_title( $text ), '-' );
            return $text;
        }

        /*
         * Remove script/style tags from the string
         * @return string
         */
        static public function remove_tags( $text ) {
            $text = preg_replace( '@<(script|style)[^>]*?>.*?</\\1>@si', '', $text );
            return $text;
        }

        /**
         * Get description of available text variables
         * @return string
         */
        static public function get_text_variables_info( $link = false ) {

            $variables = AWL_Admin_Options::include_text_vars();
            $info = '';

            if ( $link ) {
                $variables_link['{ADD_TO_CART_AJAX}'] = __( "product add to cart ajax link", "advanced-woo-labels" );
                $variables = array_merge( $variables_link, $variables );
            }

            if ( is_array( $variables ) && ! empty( $variables ) ) {

                $info .= "<div class='awl-content-vars'>";

                foreach ( $variables as $variabl_name => $variabl_desc ) {
                    $info .= "<div class='awl-content-var-item' data-text-var-tip='" . esc_attr( $variabl_desc ) . "'>";
                    $info .= "<span>" . $variabl_name . "</span>";
                    $info .= "</div>";
                }

                $info .= "</div>";

            }

            $info .= __( "for more info visit", "advanced-woo-labels" ) . '<a target="_blank" href="https://advanced-woo-labels.com/guide/text-variables/?utm_source=plugin&utm_medium=settings&utm_campaign=awl-pro-plugin"> ' . __( "guide page", "advanced-woo-labels" ) . '</a>';

            return $info;

        }

        /*
         * Get custom text variables description
         * @return string
         */
        static public function get_custom_text_variables() {

            $variables = apply_filters( 'awl_labels_text_vars', array() );
            $variables_arr = array();

            if ( $variables && is_array( $variables ) ) {
                foreach( $variables as $variable_name => $variable_params ) {
                    if ( isset( $variable_params['desc'] ) && is_string( $variable_params['desc'] ) ) {
                        $variables_name = isset( $variable_params['example'] ) && $variable_params['example'] ? $variable_params['example'] : $variable_name;
                        if ( is_string($variables_name) ) {
                            $variables_arr[$variables_name] = stripslashes( $variable_params['desc'] );
                        }
                    }
                }
            }

            return $variables_arr;

        }

        /*
         * Check for incorrect label display conditions and return them
         * @return string
         */
        static public function check_for_incorrect_display_rules( $label ) {

            $incorrect_rules_string = '';
            $check_rules = array( 'product', 'user', 'user_role', 'page', 'page_template' );

            if ( $label && isset( $label['conditions'] ) ) {
                foreach ( $label['conditions'] as $cond_group ) {

                    $maybe_wrong_rules = array();

                    foreach ( $cond_group as $cond_rule ) {
                        $rule_name = $cond_rule['param'];
                        if ( array_search( $rule_name, $check_rules ) !== false && $cond_rule['operator'] === 'equal' ) {
                            $maybe_wrong_rules[$rule_name][] = $cond_rule;
                        }
                        if ( isset( $maybe_wrong_rules[$rule_name] ) && count( $maybe_wrong_rules[$rule_name] ) > 1 ) {
                            foreach ( $maybe_wrong_rules[$rule_name] as $rule ) {
                                $rule_value = isset( $rule['value']  ) ? $rule['value'] : '';
                                $incorrect_rules_string .= $rule['param'] . ' -> ' . 'equal to' . ' -> ' . $rule_value .  '<br>';
                            }
                            break;
                        }
                    }

                    if ( $incorrect_rules_string ) {
                        break;
                    }

                }
            }

            return $incorrect_rules_string;

        }

    }

endif;