<?php

namespace ACPT\Tests;

use ACPT\Core\Models\Meta\MetaFieldModel;
use ACPT\Utils\Data\Sanitizer;

class SanitizerTest extends AbstractTestCase
{
    /**
     * @test
     */
    public function can_sanitize_text()
    {
        $sanitized_text = Sanitizer::sanitizeRawData(MetaFieldModel::TEXT_TYPE, "bla bla bla bla");

        $this->assertEquals('bla bla bla bla', $sanitized_text);
    }

    /**
     * @test
     */
    public function can_sanitize_base64_img()
    {
        if(function_exists("imagecreatefromstring")){
            $invalidString = '{"img":"data:image/png;base64,xxxxxxxxxxxxxxxxxxxxxx","resolution":"200", "colorLight":"#fff", "colorDark":"#000"}';
            $validString = '{"img":"data:image/png;base64,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","resolution":"400","colorDark":"#20aa0d","colorLight":"#ffffff"}';

            $sanitized_text = Sanitizer::sanitizeRawData(MetaFieldModel::TEXT_TYPE, $invalidString);

            $this->assertEquals('{"img":"","resolution":"200","colorLight":"#fff","colorDark":"#000"}', $sanitized_text);

            $sanitized_text = Sanitizer::sanitizeRawData(MetaFieldModel::TEXT_TYPE, $validString);

            $this->assertEquals($validString, $sanitized_text);
        } else {
            $this->markTestSkipped("Missing imagecreatefromstring");
        }
    }

    /**
     * @test
     */
    public function allow_complex_svg_and_xhtml()
    {
        $string = '<div class="mapdiv">
            <svg version="1.2" viewbox="0 0 500 375" xmlns:svg="http://www.w3.org/2000/svg">
              <path d="M 144.75,272.75 141.95,273.85 141.25,273.45 V 271.35 L 142.15,270.65 144.75,271.25 Z" id="AD" name="Andorra" style="fill:none" />
              <a xlink:title="Austria" xlink:href="/places/Austria">
              <path d="M 329.15,178.85 328.25,181.25 328.35,182.05 329.95,184.95 332.25,187.85 331.65,188.55 331.05,190.45 331.75,192.45 327.85,192.75 325.35,191.85 324.05,193.25 326.65,193.95 327.15,195.05 326.65,196.45 324.45,197.65 324.85,199.15 324.35,199.85 325.45,201.45 325.15,203.25 322.65,203.75 320.85,205.45 319.55,206.35 319.45,208.25 317.05,207.75 315.45,207.95 313.15,209.05 310.35,208.85 307.75,209.05 305.85,209.75 304.85,211.05 301.85,212.25 296.85,211.55 291.85,210.65 289.25,210.15 285.25,209.85 276.65,208.35 275.55,207.85 272.45,204.35 V 202.05 L 267.75,203.55 264.65,203.35 261.25,203.65 259.85,204.25 258.65,206.45 257.55,206.85 255.45,206.45 254.55,205.65 251.55,205.35 251.05,203.85 250.25,203.55 248.25,205.45 245.95,205.05 244.45,204.25 244.05,203.25 240.75,202.45 241.05,201.65 239.95,199.15 240.95,196.15 240.05,195.45 241.75,194.95 242.45,194.45 245.95,196.05 246.85,197.45 248.05,197.75 247.95,199.05 249.55,198.55 250.55,196.95 250.75,195.05 253.55,194.95 256.25,195.35 257.65,197.25 261.45,196.75 262.55,195.85 266.45,194.35 272.15,193.85 272.25,192.45 274.25,192.75 275.75,193.55 278.25,193.05 279.35,193.55 279.55,194.65 280.75,195.55 282.45,195.95 282.75,194.45 282.45,192.55 280.75,192.25 281.35,190.95 281.25,189.65 278.75,186.85 279.35,185.55 282.45,183.75 285.35,182.95 286.35,181.95 287.05,179.05 289.35,179.95 290.65,179.05 290.85,176.25 293.05,177.45 293.85,178.75 297.75,179.15 299.15,178.45 301.65,178.85 301.75,177.75 303.05,176.15 H 304.25 L 304.55,172.95 305.75,172.75 307.35,173.55 308.75,173.15 313.55,174.85 315.05,174.75 318.05,176.45 321.85,176.65 323.05,175.75 328.15,177.25 Z" id="AT" name="Austria" />
              </a>
              <circle cx="121.25" cy="134.75" id="0" r="0" />
              <circle cx="296.75" cy="155.95" id="1" r="0" />
              <circle cx="242.35" cy="10.55" id="2" r="0" />
            </svg>
            </div>

            turns into

            <div class="mapdiv">
            <svg viewbox="0 0 500 375">
              <path d="M 144.75,272.75 141.95,273.85 141.25,273.45 V 271.35 L 142.15,270.65 144.75,271.25 Z" />
              <a>
              <path d="M 329.15,178.85 328.25,181.25 328.35,182.05 329.95,184.95 332.25,187.85 331.65,188.55 331.05,190.45 331.75,192.45 327.85,192.75 325.35,191.85 324.05,193.25 326.65,193.95 327.15,195.05 326.65,196.45 324.45,197.65 324.85,199.15 324.35,199.85 325.45,201.45 325.15,203.25 322.65,203.75 320.85,205.45 319.55,206.35 319.45,208.25 317.05,207.75 315.45,207.95 313.15,209.05 310.35,208.85 307.75,209.05 305.85,209.75 304.85,211.05 301.85,212.25 296.85,211.55 291.85,210.65 289.25,210.15 285.25,209.85 276.65,208.35 275.55,207.85 272.45,204.35 V 202.05 L 267.75,203.55 264.65,203.35 261.25,203.65 259.85,204.25 258.65,206.45 257.55,206.85 255.45,206.45 254.55,205.65 251.55,205.35 251.05,203.85 250.25,203.55 248.25,205.45 245.95,205.05 244.45,204.25 244.05,203.25 240.75,202.45 241.05,201.65 239.95,199.15 240.95,196.15 240.05,195.45 241.75,194.95 242.45,194.45 245.95,196.05 246.85,197.45 248.05,197.75 247.95,199.05 249.55,198.55 250.55,196.95 250.75,195.05 253.55,194.95 256.25,195.35 257.65,197.25 261.45,196.75 262.55,195.85 266.45,194.35 272.15,193.85 272.25,192.45 274.25,192.75 275.75,193.55 278.25,193.05 279.35,193.55 279.55,194.65 280.75,195.55 282.45,195.95 282.75,194.45 282.45,192.55 280.75,192.25 281.35,190.95 281.25,189.65 278.75,186.85 279.35,185.55 282.45,183.75 285.35,182.95 286.35,181.95 287.05,179.05 289.35,179.95 290.65,179.05 290.85,176.25 293.05,177.45 293.85,178.75 297.75,179.15 299.15,178.45 301.65,178.85 301.75,177.75 303.05,176.15 H 304.25 L 304.55,172.95 305.75,172.75 307.35,173.55 308.75,173.15 313.55,174.85 315.05,174.75 318.05,176.45 321.85,176.65 323.05,175.75 328.15,177.25 Z" />
              </a>
            </svg>

            If you still have the solution could you please help? Thank you in advance, all the best wishes!
            mihai
        </div>';

        $sanitized_text = Sanitizer::sanitizeRawData(MetaFieldModel::HTML_TYPE, $string);

        $this->assertEquals($string, $sanitized_text);
    }

    /**
     * @test
     */
    public function can_strip_javascript()
    {
        $string = '<div>This is test</div><script>console.log("Malicious code");</script>';
        $sanitized_text = Sanitizer::sanitizeRawData(MetaFieldModel::HTML_TYPE, $string);

        $this->assertEquals('<div>This is test</div>console.log("Malicious code");', $sanitized_text);
    }
}
