<?php
// phpcs:ignoreFile
namespace AcademyProFluentCRM\Triggers;

use AcademyProFluentCRM\Helper;
use FluentCrm\App\Services\Funnel\BaseTrigger;
use FluentCrm\App\Services\Funnel\FunnelHelper;
use FluentCrm\App\Services\Funnel\FunnelProcessor;
use FluentCrm\Framework\Support\Arr;

class CourseEnrollTrigger extends BaseTrigger {
	public function __construct() {
		$this->triggerName  = 'academy/course/after_enroll';
		$this->priority     = 12;
		$this->actionArgNum = 3;
		parent::__construct();
	}

	public function getTrigger(): array {
		return [
			'category'    => __( 'Academy LMS', 'academy-pro' ),
			'label'       => __( 'Course Enrolled', 'academy-pro' ),
			'icon'        => 'dashicons dashicons-welcome-learn-more',
			'description' => __( 'This funnel runs when a student is enrolled in a course', 'academy-pro' )
		];
	}

	public function getFunnelSettingsDefaults(): array {
		return [
			'subscription_status' => 'subscribed'
		];
	}

	public function getSettingsFields( $funnel ): array {
		return [
			'title'     => __( 'Enrollment in a course in Academy LMS', 'academy-pro' ),
			'sub_title' => __( 'This Funnel will start when a student is enrolled in a course', 'academy-pro' ),
			'fields'    => [
				'subscription_status'      => [
					'type'        => 'option_selectors',
					'option_key'  => 'editable_statuses',
					'is_multiple' => false,
					'label'       => __( 'Subscription Status', 'academy-pro' ),
					'placeholder' => __( 'Select Status', 'academy-pro' )
				],
				'subscription_status_info' => [
					'type'       => 'html',
					'info'       => '<b>' . __( 'An Automated double-option email will be sent for new subscribers', 'academy-pro' ) . '</b>',
					'dependency' => [
						'depends_on' => 'subscription_status',
						'operator'   => '=',
						'value'      => 'pending'
					]
				]
			]
		];
	}

	public function getFunnelConditionDefaults( $funnel ): array {
		return [
			'update_type'  => 'update', // skip_all_actions, skip_update_if_exist
			'course_ids'   => [],
			'run_multiple' => 'no'
		];
	}

	public function getConditionFields( $funnel ): array {
		return [
			'update_type'  => [
				'type'    => 'radio',
				'label'   => __( 'If Contact Already Exist?', 'academy-pro' ),
				'help'    => __( 'Please specify what will happen if the subscriber already exist in the database', 'academy-pro' ),
				'options' => FunnelHelper::getUpdateOptions()
			],
			'course_ids'   => [
				'type'        => 'multi-select',
				'label'       => __( 'Target Courses', 'academy-pro' ),
				'help'        => __( 'Select for which Courses this automation will run', 'academy-pro' ),
				'options'     => Helper::get_courses(),
				'inline_help' => __( 'Keep it blank to run to any Course Enrollment', 'academy-pro' )
			],
			'run_multiple' => [
				'type'        => 'yes_no_check',
				'label'       => '',
				'check_label' => __( 'Restart the Automation Multiple times for a contact for this event. (Only enable if you want to restart automation for the same contact)', 'academy-pro' ),
				'inline_help' => __( 'If you enable, then it will restart the automation for a contact if the contact already in the automation. Otherwise, It will just skip if already exist', 'academy-pro' )
			]
		];
	}

	public function handle( $funnel, $originalArgs ) {
		$courseId = $originalArgs[0];
		$userId   = $originalArgs[2];

		$subscriberData = FunnelHelper::prepareUserData( $userId );

		$subscriberData['source'] = __( 'Academy LMS', 'academy-pro' );

		if ( empty( $subscriberData['email'] ) ) {
			return;
		}

		$willProcess = $this->isProcessable( $funnel, $courseId, $subscriberData );

		$willProcess = apply_filters( 'fluentcrm_funnel_will_process_' . $this->triggerName, $willProcess, $funnel, $subscriberData, $originalArgs );
		if ( ! $willProcess ) {
			return;
		}

		$subscriberData = wp_parse_args( $subscriberData, $funnel->settings );

		$subscriberData['status'] = $subscriberData['subscription_status'];
		unset( $subscriberData['subscription_status'] );

		( new FunnelProcessor() )->startFunnelSequence( $funnel, $subscriberData, [
			'source_trigger_name' => $this->triggerName,
			'source_ref_id'       => $courseId
		] );
	}

	private function isProcessable( $funnel, $courseId, $subscriberData ): bool {
		$conditions = $funnel->conditions;
		// check update_type
		$updateType = Arr::get( $conditions, 'update_type' );

		$subscriber = FunnelHelper::getSubscriber( $subscriberData['email'] );
		if ( $subscriber && $updateType == 'skip_all_if_exist' ) {
			return false;
		}

		// check the products ids
		if ( $conditions['course_ids'] ) {
			if ( ! in_array( $courseId, $conditions['course_ids'] ) ) {
				return false;
			}
		}

		// check run_only_one
		if ( $subscriber && FunnelHelper::ifAlreadyInFunnel( $funnel->id, $subscriber->id ) ) {
			$multipleRun = Arr::get( $conditions, 'run_multiple' ) == 'yes';
			if ( $multipleRun ) {
				FunnelHelper::removeSubscribersFromFunnel( $funnel->id, [ $subscriber->id ] );
			} else {
				return false;
			}
		}

		return true;
	}
}
