<?php
// phpcs:ignoreFile
namespace AcademyProFluentCRM\DeepIntegration;

use FluentCampaign\App\Services\Integrations\BaseImporter;
use FluentCrm\App\Models\Subscriber;
use FluentCrm\App\Models\Tag;
use FluentCrm\App\Services\Helper;
use FluentCrm\Framework\Support\Arr;
use WP_Error;
use WP_User_Query;

class AcademyImporter extends BaseImporter {
	public function __construct() {
		$this->importKey = 'academylms';
		parent::__construct();
	}

	public function getInfo() {
		return [
			'label'    => $this->getPluginName(),
			'logo'     => esc_url( ACADEMY_ASSETS_URI . 'images/logo.svg' ),
			'disabled' => false
		];
	}

	private function getPluginName() {
		return 'Academy LMS';
	}

	public function processUserDriver( $config, $request ) {
		$summary = $request->get( 'summary' );

		if ( $summary ) {
			$config = $request->get( 'config' );

			$type = Arr::get( $config, 'import_type' );

			if ( $type == 'course_type' ) {
				$courseIds = [];
				foreach ( $config['course_types_maps'] as $map ) {
					$courseIds[] = $map['field_key'];
				}
				$courseIds     = array_filter( array_unique( $courseIds ) );
				$selectedUsers = $this->getUserIdsByResourceIds( $courseIds, 5, 0 );
			} else {
				$selectedUsers['total'] = 0;
			}

			if ( ! $selectedUsers['total'] ) {
				return new WP_Error( 'not_found', 'Sorry no users found based on your filter' );
			}

			$userQuery = new WP_User_Query( [
				'include' => Arr::get( $selectedUsers, 'user_ids' ),
				'fields'  => [ 'ID', 'display_name', 'user_email' ],
			] );

			$users = $userQuery->get_results();
			$total = $selectedUsers['total'];

			$formattedUsers = [];

			foreach ( $users as $user ) {
				$formattedUsers[] = [
					'name'  => $user->display_name,
					'email' => $user->user_email
				];
			}

			return [
				'import_info' => [
					'subscribers'       => $formattedUsers,
					'total'             => $total,
					'has_list_config'   => true,
					'has_status_config' => true,
					'has_update_config' => true,
					'has_silent_config' => true
				]
			];
		}//end if

		$courses = \AcademyProFluentCRM\Helper::get_courses();

		$courseMaps = [];

		foreach ( $courses as $course ) {
			$courseMaps[ $course['id'] ] = [
				'label' => $course['title']
			];
		}

		$tags = Tag::orderBy( 'title', 'ASC' )->get();

		return [
			'config' => [
				'import_type'       => 'course_type',
				'course_types_maps' => [
					[
						'field_key'   => '',
						'field_value' => ''
					]
				]
			],
			'fields' => [
				'import_type'       => [
					'label'   => __( 'Import by', 'academy-pro' ),
					'help'    => __( 'Please select import by course enrollment', 'academy-pro' ),
					'type'    => 'input-radio',
					'options' => [
						[
							'id'    => 'course_type',
							'label' => __( 'Import By Courses', 'academy-pro' )
						]
					]
				],
				'course_types_maps' => [
					'label'              => __( 'Please map your Courses and associate FluentCRM Tags', 'academy-pro' ),
					'type'               => 'form-many-drop-down-mapper',
					'local_label'        => sprintf( __( 'Select %s Course', 'academy-pro' ), $this->getPluginName() ),
					'remote_label'       => __( 'Select FluentCRM Tag that will be applied', 'academy-pro' ),
					'local_placeholder'  => sprintf( __( 'Select %s Course', 'academy-pro' ), $this->getPluginName() ),
					'remote_placeholder' => __( 'Select FluentCRM Tag', 'academy-pro' ),
					'fields'             => $courseMaps,
					'value_options'      => $tags,
					'dependency'         => [
						'depends_on' => 'import_type',
						'operator'   => '=',
						'value'      => 'course_type'
					]
				]
			],
			'labels' => [
				'step_2' => __( 'Next [Review Data]', 'academy-pro' ),
				'step_3' => sprintf( __( 'Import %s Students Now', 'academy-pro' ), $this->getPluginName() )
			]
		];
	}

	private function getUserIdsByResourceIds( $courseIds, $limit, $offset ) {
		if ( ! $courseIds ) {
			return [
				'user_ids' => [],
				'total'    => 0
			];
		}

		$courseUsers = [];

		$enrollments = fluentCrmDb()->table( 'posts' )
									->select( [ 'post_author' ] )
									->where( 'post_type', 'academy_enrolled' )
									->whereIn( 'post_parent', $courseIds );

		$total = $enrollments->count();

		$enrollments = $enrollments->limit( $limit )
								   ->offset( $offset )
								   ->get();

		foreach ( $enrollments as $enrollment ) {
			$courseUsers[] = $enrollment->post_author;
		}

		return [
			'user_ids' => $courseUsers,
			'total'    => $total
		];
	}

	public function importData( $returnData, $config, $page ) {
		$type = Arr::get( $config, 'import_type' );

		if ( $type == 'course_type' ) {
			return $this->importByMemberCourses( $config, $page );
		}

		return new WP_Error( 'not_found', 'Invalid Request' );

	}

	protected function importByMemberCourses( $config, $page ) {
		$inputs = Arr::only( $config, [
			'lists',
			'update',
			'new_status',
			'double_optin_email',
			'import_silently'
		] );

		if ( Arr::get( $inputs, 'import_silently' ) == 'yes' ) {
			if ( ! defined( 'FLUENTCRM_DISABLE_TAG_LIST_EVENTS' ) ) {
				define( 'FLUENTCRM_DISABLE_TAG_LIST_EVENTS', true );
			}
		}

		$sendDoubleOptin = Arr::get( $inputs, 'double_optin_email' ) == 'yes';

		$courseTypesMaps = [];
		foreach ( $config['course_types_maps'] as $map ) {
			if ( ! absint( $map['field_value'] ) || ! $map['field_key'] ) {
				continue;
			}

			$typeSlug = $map['field_key'];
			if ( ! isset( $courseTypesMaps[ $typeSlug ] ) ) {
				$courseTypesMaps[ $typeSlug ] = [];
			}
			$courseTypesMaps[ $typeSlug ][] = absint( $map['field_value'] );
		}

		$limit  = 100;
		$offset = ( $page - 1 ) * $limit;

		$courseIds = array_keys( $courseTypesMaps );

		$userMaps = $this->getUserIdsByResourceIds( $courseIds, $limit, $offset );

		$userIds = $userMaps['user_ids'];

		foreach ( $userIds as $userId ) {
			// Create user data
			$subscriberData           = Helper::getWPMapUserInfo( $userId );
			$subscriberData['source'] = 'academylms';

			$inCourses = \AcademyProFluentCRM\Helper::get_user_courses( $userId );

			$tagIds = [];

			foreach ( $inCourses as $inCourse ) {
				if ( ! empty( $courseTypesMaps[ $inCourse ] ) ) {
					$tagIds = array_merge( $tagIds, $courseTypesMaps[ $inCourse ] );
				}
			}

			$tagIds = array_unique( $tagIds );

			Subscriber::import(
				[ $subscriberData ],
				$tagIds,
				$inputs['lists'],
				$inputs['update'],
				$inputs['new_status'],
				$sendDoubleOptin
			);
		}//end foreach

		return [
			'page_total'   => ceil( $userMaps['total'] / $limit ),
			'record_total' => $userMaps['total'],
			'has_more'     => $userMaps['total'] > ( $page * $limit ),
			'current_page' => $page,
			'next_page'    => $page + 1
		];

	}

}
