<?php
/**
 * WOO_VPF Hooks
 *
 * @class WOO_VPF_Hooks
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

class WOO_VPF_Hooks {

	/**
	 * @var page_title
	 */
	public $page_title = '';
	
	/**
	 * @var searched_term_id
	 */
	public $searched_term_id = '';
	
	/**
	 * @var parent_term_id
	 */
	public $parent_term_id = '';
	
	/**
	 * @var array $categories_counts 
	 */
	public $categories_counts = array ();

	/**
	 * Constructor
	 *
	 * @access public
	 */
	public function __construct() {
		// WPML: Init Language Cookie
		add_action( 'init', array($this, 'init_lang_cookie') );
		
		// Filter query hook to include WOO_VPF search criteria
		add_action( 'pre_get_posts', array($this, 'filter_posts'), apply_filters ( 'woo_vpf_filter_posts_priority', 50 ) );
		add_filter( 'posts_where', array($this, 'filter_posts_where'), apply_filters ( 'woo_vpf_filter_posts_where_priority', 10 ), 2 );
		
		// Speed Hack - Filter query not to create a cache for plugin custom taxonomy
		add_action( 'pre_get_posts', array($this, 'disable_post_term_cache') );
		add_action( 'get_terms_args', array($this, 'disable_get_terms_cache') );
		
		// Override search page title
		//add_filter( 'get_search_query', array($this, 'search_page_html_esc_title') );
		add_filter( 'woocommerce_page_title', array($this, 'search_page_title') );
		add_filter ( 'woocommerce_get_breadcrumb', array( $this, 'search_page_breadcrumb' ) );
		
		// Override WooCommerce Product Archive Template Option
		add_action( 'woocommerce_before_main_content', array($this, 'override_shop_page_display_option') );
		add_action( 'woocommerce_after_main_content', array($this, 'revert_shop_page_display_option') );
		
		// Add additional tab on product detail page
		add_filter( 'woocommerce_product_tabs', array($this, 'product_tab') );
		add_shortcode ( 'vpf_product_terms_list', array($this, 'product_terms_list_callback') );
		
		// Enable/Disable single result redirection to product details page
		add_filter( 'woocommerce_redirect_single_search_result', array($this, 'disable_woocommerce_single_result_redirection'), 99 );
		add_action ( 'template_redirect', array($this, 'redirect_single_search_result') );
		
		// Hide Empty Categories ( Display Mode: Categories ) AND Fix Categories Count
		add_filter ( 'woocommerce_product_subcategories_args', array ( $this, 'exclude_empty_categories' ), 99 );
		add_filter ( 'woocommerce_subcategory_count_html', array ( $this, 'category_count_html' ), 99, 2 );
		
		// Fix: Duplicate Products doesn't duplicate VPF Term Relations
		add_action ( 'woocommerce_product_duplicate', array( $this, 'duplicate_product_terms' ), 99, 2 );
		
		// Yoast SEO Plugin Hack: Speed issue conflict with Yoast SEO Plugin
		add_filter( 'wpseo_primary_term_taxonomies', array($this, 'wpseo_taxonomies') );
		
		// WooCommerce Layered Nav Filters: Conflict as This Widget doesn't Maintain VPF Search Results
		add_filter( 'woocommerce_layered_nav_link', array($this, 'woocommerce_layered_nav_link') );
		
		// Fix No Child Term Conflicts when having Parent Arg with get_terms
		add_filter( 'get_terms_args', array( $this, 'get_queried_parent_term_id' ), 99, 2 );
		
		if( WOO_VPF_Functions::is_wpml_activated() ) {
			$languages = icl_get_languages();
			
			if( ! empty( $languages ) ) {
				foreach( $languages as $language ) {
					add_filter( 'default_option_product_make_children_' . $language['code'], array( $this, 'fix_no_child_terms_conflict' ), 10, 2 );
					add_filter( 'option_product_make_children_' . $language['code'], array( $this, 'fix_no_child_terms_conflict' ), 10, 2 );
				}
			}
		}
		add_filter( 'default_option_product_make_children', array( $this, 'fix_no_child_terms_conflict' ), 10, 2 );
		add_filter( 'option_product_make_children', array( $this, 'fix_no_child_terms_conflict' ), 10, 2 );
	}
	
	/**
	 * WPML: Init Language Cookie
	 */
	function init_lang_cookie () {
		if( WOO_VPF_Functions::is_wpml_activated() ) {
			$current_lang = vpf_get_cookie ( 'current_lang' );
			
			if( $current_lang != ICL_LANGUAGE_CODE ) {
				vpf_set_cookie ( 'search', '' );
			}
			
			vpf_set_cookie ( 'current_lang', ICL_LANGUAGE_CODE );
		}
	}
	
	/**
	 * Include WOO_VP query params in the query
	 */
	function filter_posts( $query ) {
		if( ! is_admin() && $query->is_main_query() ) {
			
			$is_product_archive			= false;
			
			if ( is_product_category () || is_product_tag () || is_post_type_archive ( 'product' ) || WOO_VPF_Functions::is_search () ) {
				$is_product_archive		= true;
			} else if ( isset ( $query->query_vars['post_type'] ) ) {
				$post_types				= $query->query_vars['post_type'];
				
				if ( ! is_array ( $post_types ) ) {
					$post_types			= array ( $post_types );
				}
				
				if ( ! empty ( $post_types ) && in_array ( 'product', $post_types ) && ! is_singular () ) { // ! is_singlular = ! is_product () because is_product () didn't work here
					$is_product_archive	= true;
				}
			}
			
			if ( ! $is_product_archive ) {
				return;
			}
		
			$is_filter_enable = false;
			$term_id = ''; $category_id = ''; $s = '';
			$woo_vpf_activate_remember_search = WC_Admin_Settings::get_option( 'woo_vpf_activate_remember_search' );
			
			if( WOO_VPF_Functions::is_search() ) {
				$is_filter_enable = true;
				
				$make = 0;
				if( isset( $_REQUEST['make'] ) ) {
					$make = $_REQUEST['make'];
				}
				
				$model = 0;
				if( isset( $_REQUEST['model'] ) ) {
					$model = $_REQUEST['model'];
				}
				
				$year = 0;
				if( isset( $_REQUEST['year_id'] ) ) {
					$year = $_REQUEST['year_id'];
				}
				
				$engine = 0;
				if( isset( $_REQUEST['engine'] ) ) {
					$engine = $_REQUEST['engine'];
				}
				
				$category_id = '';
				if( isset($_REQUEST['category']) ) {
					$category_id = $_REQUEST['category'];
				}
				
				$s = '';
				if( isset($_REQUEST['s']) ) {
					$s = sanitize_text_field( $_REQUEST['s'] );
				}
				
				// Validate requested IDs
				$_error = false;
				
				if( ! $_error ) {
					if( $engine > 0 ) {
						$term_engine = get_term( $engine, 'product_make' );
						if( $term_engine && $term_engine->parent != $year ) {
							$_error = true;
						}
					}
				}
				
				if( ! $_error ) {
					if( $year > 0 ) {
						$term_year = get_term( $year, 'product_make' );
						if( $term_year && $term_year->parent != $model ) {
							$_error = true;
						}
					}
				}
				
				if( ! $_error ) {
					if( $model > 0 ) {
						$term_model = get_term( $model, 'product_make' );
						if( $term_model && $term_model->parent != $make ) {
							$_error = true;
						}
					}
				}
				
				$term_id = '';
				if( $_error ) {
					$term_id = -1;
				} else {
					if( $engine > 0 ) {
						$term_id = $engine;
					} else if( $year > 0 ) {
						$term_id = $year;
					} else if( $model > 0 ) {
						$term_id = $model;
					} else if( $make > 0 ) {
						$term_id = $make;
					}
				}
				
				// Remember User Search
				if( $woo_vpf_activate_remember_search == 'yes' ) {
					vpf_set_cookie ( 'search', array (
						'make'		=> $make,
						'model'		=> $model,
						'year_id'	=> $year,
						'engine'	=> $engine,
						'term_id'	=> $term_id
					) );
				}
			} else if( $woo_vpf_activate_remember_search == 'yes' ) {
				$search = vpf_get_cookie ( 'search' );
				
				if( ! empty ( $search ) && isset( $search['term_id'] ) ) {
					$is_filter_enable = true;
					$term_id	= $search['term_id'];
				
					$make		= isset( $search['make'] ) ? $search['make'] : 0;
					$model		= isset( $search['model'] ) ? $search['model'] : 0;
					$year		= isset( $search['year_id'] ) ? $search['year_id'] : 0;
					$engine		= isset( $search['engine'] ) ? $search['engine'] : 0;
				}
			}
			
			if( $is_filter_enable ) {
			
				do_action( 'woo_vpf_before_search_query', array(
					'make'		=> $make,
					'model'		=> $model,
					'year_id'	=> $year,
					'engine'	=> $engine,
					'category'	=> $category_id
				) );
				
				// Set Post Type
				$query->set('post_type', 'product');
				
				// Set Taxonomies Queries
				$tax_query = $query->get('tax_query');
				
				if( empty( $tax_query ) ) {
					$tax_query = array();
				}
				
				if( $term_id != '' ) {
					$tax_query[] = array(
						'taxonomy'			=> 'product_make',
						'field'				=> 'id',
						'terms'				=> array($term_id),
						'include_children'	=> false
					);
				}
				
				if( $category_id != '' ) {					
					$tax_query[] = array(
						'taxonomy'			=> 'product_cat',
						'field'				=> 'id',
						'terms'				=> array($category_id),
						'include_children'	=> true
					);
				}
				
				$query->set('tax_query', $tax_query);
				
				// Set Keyword Query
				if( $s != '' ) {					
					$query->set('s', $s);
				}
				
				$this->searched_term_id = $term_id;
				
				do_action( 'woo_vpf_after_search_query', array(
					'make'		=> $make,
					'model'		=> $model,
					'year_id'	=> $year,
					'engine'	=> $engine,
					'category'	=> $category_id
				) );
			}
		}
	}
	
	/**
	 * Inclulde Universal Products Always
	 */
	function filter_posts_where( $where, $query ) {
		if( ! is_admin() && $query->is_main_query() ) {
			if( $this->searched_term_id ) {
				global $wpdb;
				$term_taxonomy_id = $wpdb->get_var( "SELECT `term_taxonomy_id` FROM $wpdb->term_taxonomy WHERE `term_id`='{$this->searched_term_id}'" );
				if( $term_taxonomy_id ) {
					$woo_vpf_taxonomy_metabox_excluded_products = WC_Admin_Settings::get_option( 'woo_vpf_taxonomy_metabox_excluded_products' );
					if( ! empty( $woo_vpf_taxonomy_metabox_excluded_products ) ) {					
						if( is_array( $woo_vpf_taxonomy_metabox_excluded_products ) ) {
							$woo_vpf_taxonomy_metabox_excluded_products = implode( ',', $woo_vpf_taxonomy_metabox_excluded_products );
						}
						
						$where = preg_replace( '!\s+!', ' ', $where );
						$where = str_replace( "$wpdb->term_relationships.term_taxonomy_id IN ($term_taxonomy_id)", "( $wpdb->posts.ID IN($woo_vpf_taxonomy_metabox_excluded_products) OR $wpdb->term_relationships.term_taxonomy_id IN ($term_taxonomy_id) )", $where );
					}
				}
			}
		}
		
		return $where;
	}
	
	/**
	 * Fix - Speed up the taxonomy related queries by disabling post terms cache
	 */
	function disable_post_term_cache( $query ) {
		$post_types = $query->get('post_type');
		if( ! empty( $post_types ) ) {
			if( ! is_array( $post_types ) ) {
				$post_types = array( $post_types );
			}
			
			if( in_array( 'product', $post_types ) ) {
				$query->set('update_post_term_cache', false);
			}
		}
	}
	
	/**
	 * Fix - Speed up the taxonomy related queries by disabling post terms cache
	 */
	function disable_get_terms_cache( $args ) {
		if( isset( $args['taxonomy'] ) && ! empty( $args['taxonomy'] ) ) {
			if( in_array( 'product_make', $args['taxonomy'] ) ) {
				$args['update_term_meta_cache'] = false;
			}
		}

		return $args;
	}
	
	/**
	 * Override search page title with WOO_VPF_title ( Breadcrumbs / Page Meta Title )
	 */
	function search_page_html_esc_title( $title ) {
		if( WOO_VPF_Functions::is_search() ) {
			$title = WOO_VPF_Hooks::search_page_title( $title );
			$title = wp_strip_all_tags( $title );
			
			$title_search_results_label = WOO_VPF_Functions::get_search_results_label();
			if( ! empty( $title_search_results_label ) ) {
				$title = trim( str_replace( $title_search_results_label, '', $title ) );
			}
		}
		
		return $title;
	}
	
	/**
	 * Override search page title with WOO_VPF title
	 */
	function search_page_title( $title ) {
		if( WOO_VPF_Functions::is_search() ) {
		
			// If Title Already Generated
			if( $this->page_title != '' ) {
				return $this->page_title;
			}
		
			// Make Title
			$title_make = '';
			if( isset( $_REQUEST['make'] ) ) {
				$make = $_REQUEST['make'];
				if( $make > 0 ) {
					$term_make = get_term( $make, 'product_make' );
					if( !empty( $term_make ) ) {
						$title_make = sprintf( __( '<span class="search_col search_col_make">%s: <span class="search_col_val">%s</span></span>,', WOO_VPF_TEXT_DOMAIN ), WOO_VPF_Functions::get_make_label(), $term_make->name );
					}
				}
			}
			
			// Model Title
			$title_model = '';
			if( isset( $_REQUEST['model'] ) ) {
				$model = $_REQUEST['model'];
				if( $model > 0 ) {
					$term_model = get_term( $model, 'product_make' );
					if( !empty( $term_model ) ) {
						$title_model = sprintf( __( '<span class="search_col search_col_model">%s: <span class="search_col_val">%s</span></span>,', WOO_VPF_TEXT_DOMAIN ), WOO_VPF_Functions::get_model_label(), $term_model->name );
					}
				}
			}
			
			// Year Title
			$title_year = '';
			if( isset( $_REQUEST['year_id'] ) ) {
				$year = $_REQUEST['year_id'];
				if( $year > 0 ) {
					$term_year = get_term( $year, 'product_make' );
					if( !empty( $term_year ) ) {
						$title_year = sprintf( __( '<span class="search_col search_col_year">%s: <span class="search_col_val">%s</span></span>,', WOO_VPF_TEXT_DOMAIN ), WOO_VPF_Functions::get_year_label(), $term_year->name );
					}
				}
			}
			
			// Engine Title
			$title_engine = '';
			if( isset( $_REQUEST['engine'] ) ) {
				$engine = $_REQUEST['engine'];
				if( $engine > 0 ) {
					$term_engine = get_term( $engine, 'product_make' );
					if( !empty( $term_engine ) ) {
						$title_engine = sprintf( __( '<span class="search_col search_col_engine">%s: <span class="search_col_val">%s</span></span>,', WOO_VPF_TEXT_DOMAIN ), WOO_VPF_Functions::get_engine_label(), $term_engine->name );
					}
				}
			}
			
			// Category Title
			$title_category = '';
			if( isset( $_REQUEST['category'] ) ) {
				$category = $_REQUEST['category'];
				if( $category > 0 ) {
					$term_category = get_term( $category, 'product_cat' );
					if( !empty( $term_category ) ) {
						$title_category = sprintf( __( '<span class="search_col search_col_category">%s: <span class="search_col_val">%s</span></span>,', WOO_VPF_TEXT_DOMAIN ), WOO_VPF_Functions::get_category_label(), $term_category->name );
					}
				}
			}
			
			// Keyword Title
			$title_s = '';
			if( isset( $_REQUEST['s'] ) ) {
				$s = trim( stripslashes( sanitize_text_field( $_REQUEST['s'] ) ) );
				if( $s != '' ) {
					$title_s = sprintf( __( '<span class="search_col search_col_keyword">%s: <span class="search_col_val">%s</span></span>,', WOO_VPF_TEXT_DOMAIN ), WOO_VPF_Functions::get_keyword_label(), $s );
				}
			}
			
			// Complete Page Title
			$title_search_results_label = WOO_VPF_Functions::get_search_results_label();
			
			$title = sprintf( __( '%s %s %s %s %s %s %s', WOO_VPF_TEXT_DOMAIN ), $title_search_results_label, $title_make, $title_model, $title_year,  $title_engine, $title_category, $title_s );
			
			$title = trim( trim( trim( $title, ' ' ), ',' ), ' ' );
			$title = '<span class="woo-vpf-search-title">' . $title . '</span>';
			
			$title = apply_filters( 'woo_vpf_search_page_title', $title, $title_search_results_label, $title_make, $title_model, $title_year,  $title_engine, $title_category, $title_s );
			
			$this->page_title = $title;
		}
		
		return $title;
	}
	
	/**
	 * Append search page title to WooCommerce Breadcrumbs
	 */
	function search_page_breadcrumb( $crumbs ) {
		if( WOO_VPF_Functions::is_search() && ! empty ( $crumbs ) ) {
			$crumb_keys = array_keys ( $crumbs );
			$crumb_last_key = end ( $crumb_keys );
			$crumbs[$crumb_last_key][0] = wp_strip_all_tags( $this -> search_page_title ( $crumbs[ $crumb_last_key ][0] ) );
		}
		
		return $crumbs;
	}
	
	/**
	 * Override WooCommerce Product Archive Template Option
	 */
	function override_shop_page_display_option() {
		if( WOO_VPF_Functions::is_search() ) {
			add_filter( 'default_option_woocommerce_shop_page_display', array( $this, 'override_shop_page_template' ), 10, 2 );
			add_filter( 'option_woocommerce_shop_page_display', array( $this, 'override_shop_page_template' ), 10, 2 );
		}
	}
	
	/**
	 * Revert WooCommerce Categories List Template Option
	 */
	function revert_shop_page_display_option() {
		if( WOO_VPF_Functions::is_search() ) {
			remove_filter( 'default_option_woocommerce_shop_page_display', array( $this, 'override_shop_page_template' ), 10, 2 );
			remove_filter( 'option_woocommerce_shop_page_display', array( $this, 'override_shop_page_template' ), 10, 2 );
		}
	}
	
	/**
	 * Archive Template to Show Only Products Even When No Search Keyword
	 */
	function override_shop_page_template( $value, $option ) {
		if( WOO_VPF_Functions::is_search() ) {
			$value = '';
		}
		
		return $value;
	}
	
	/**
	 * Additional VPF tab on product details page
	 */
	function product_tab( $tabs ) {
		$woo_vpf_activate_tab = WC_Admin_Settings::get_option( 'woo_vpf_activate_tab' );
		if( $woo_vpf_activate_tab == 'yes' ) {
			global $post;
			$terms = wp_get_post_terms( $post->ID, 'product_make', array( 'fields' => 'ids', 'orderby' => 'parent', 'order' => 'ASC', 'parent' => 0 ) );
			if( ! empty( $terms ) ) {
				$woo_vpf_tab_title = WC_Admin_Settings::get_option( 'woo_vpf_tab_title' );
				if( $woo_vpf_tab_title == '' ) {
					$woo_vpf_tab_title = __( 'Fit for the Following Vehicles', WOO_VPF_TEXT_DOMAIN );
				}
				
				$tabs['vpf']		= array(
					'title'			=> apply_filters( 'woo_vpf_tab_title', $woo_vpf_tab_title ),
					'priority'		=> 50,
					'callback'		=> array($this, 'product_tab_content')
				);
			}
		}
		
		return $tabs;
	}
	
	/**
	 * Additional VPF tab callback
	 */
	function product_tab_content () {
		WOO_VPF_Functions::product_terms_list ();
	}
	
	/**
	 * Product Terms List Shortcode
	 */
	function product_terms_list_callback ( $args ) {
		$args = shortcode_atts( array(
			'id' => ''
		), $args );
		extract ( $args );
		
		ob_start ();
		WOO_VPF_Functions::product_terms_list ( $id );
		return ob_get_clean();
	}
	
	/**
	 * Disable WooCommerce Single Result Redirection
	 */
	function disable_woocommerce_single_result_redirection( $true ) {
		if( WOO_VPF_Functions::is_search() ) {
			$true = false;
		}
		
		return $true;
	}
	
	/**
	 * Enable/Disable Single Result Redirection to Product Details Page
	 */
	function redirect_single_search_result() {
		if( WOO_VPF_Functions::is_search() ) {
			$woo_vpf_disable_redirect_single_search_result = WC_Admin_Settings::get_option( 'woo_vpf_disable_redirect_single_search_result' );
			if( $woo_vpf_disable_redirect_single_search_result != 'yes' ) {
				global $wp_query;
				
				if( 1 === absint( $wp_query->found_posts ) ) {
					$product = wc_get_product( $wp_query->post );
				   
					if ( $product && $product->is_visible() ) {
						wp_safe_redirect( get_permalink( $product->get_id() ), 302 );
						exit;
					}
				}
			}
		}
	}
	
	/**
	 * Display Mode ( Categories ): Exclude Empty Categories
	 */
	public function exclude_empty_categories ( $args ) {
		if( $this->searched_term_id ) {
			$product_categories = get_categories ( array_merge ( $args, array ( 'hide_empty' => true ) ) );
			
			if ( ! empty ( $product_categories ) ) {
				foreach ( $product_categories as $product_category ) {
					$products = get_posts (
						array (
							'post_type'					=> 'product',
							'numberposts'				=> -1,
							'tax_query'					=> array (
								array (
									'taxonomy'			=> $product_category->taxonomy,
									'field'				=> 'term_id',
									'terms'				=> array ( $product_category->term_id ),
									'operator'			=> 'IN',
									'include_children'	=> true
								),
								
								array(
									'taxonomy'			=> 'product_make',
									'field'				=> 'id',
									'terms'				=> array ( $this->searched_term_id ),
									'include_children'	=> false
								)
							)
						)
					);
					
					if ( ! empty ( $products ) ) {
						$this->categories_counts[ $product_category->term_id ] = count ( $products );
					} else {
						$exclude_ids[] = $product_category->term_id;
					}
				}
				
				if ( ! empty ( $exclude_ids ) ) {
					if ( isset ( $args['exclude'] ) && ! empty ( $args['exclude'] ) ) {
						if ( ! is_array ( $args['exclude'] ) ) {
							$args['exclude'] = ( array ) $args['exclude'];
						}
						
						$args['exclude'] = array_merge ( $args['exclude'], $exclude_ids );
					} else {
						$args['exclude'] = $exclude_ids;
					}
				}
			}
		}
		
		return $args;
	}
	
	/**
	 * Display Mode ( Categories ): Update Categories Count
	 */
	public function category_count_html ( $html, $category ) {
		if ( ! empty ( $this->categories_counts ) && isset ( $this->categories_counts[ $category->term_id ] ) ) {
			$html = apply_filters ( 'woo_vpf_category_count_html', ' <mark class="count">(' . esc_html ( $this->categories_counts[ $category->term_id ] ) . ')</mark>', $category );
		}
		
		return $html;
	}
	
	/**
	 * Enable/Disable Single Result Redirection to Product Details Page
	 */
	function duplicate_product_terms( $duplicate, $product ) {
		if( empty( $product ) ) {
			return;
		}
	   
		if( empty( $duplicate ) ) {
			return;
		}
	   
		$duplicate_product_term_ids = wp_get_post_terms( $duplicate->get_id(), 'product_make', array( 'fields' => 'ids' ) );
		if( empty( $duplicate_product_term_ids ) ) {
			$term_ids = wp_get_post_terms( $product->get_id(), 'product_make', array( 'fields' => 'ids' ) );
			if( ! empty( $term_ids ) ) {
				$term_ids = array_map( 'intval', $term_ids );
				$term_ids = array_unique( $term_ids );
				wp_set_object_terms( $duplicate->get_id(), $term_ids, 'product_make', false );
			}
		}
	}
	
	/**
	 * Yoast SEO Plugin Hack: Speed issue conflict with Yoast SEO Plugin
	 */
	function wpseo_taxonomies( $taxonomies ) {
		if( isset( $taxonomies['product_make'] ) ) {
			unset( $taxonomies['product_make'] );
		}
		
		return $taxonomies;
	}
	
	/**
	 * WooCommerce Layered Nav Filters: Conflict as This Widget doesn't Maintain VPF Search Results
	 */
	function woocommerce_layered_nav_link( $link ) {
		if( WOO_VPF_Functions::is_search() ) {
			if( isset( $_REQUEST['make'] ) ) {
				$link = add_query_arg( 'make', $_REQUEST['make'], $link );
			}
			
			if( isset( $_REQUEST['model'] ) ) {
				$link = add_query_arg( 'model', $_REQUEST['model'], $link );
			}
			
			if( isset( $_REQUEST['year_id'] ) ) {
				$link = add_query_arg( 'year_id', $_REQUEST['year_id'], $link );
			}
		  
			if( isset( $_REQUEST['engine'] ) ) {
				$link = add_query_arg( 'engine', $_REQUEST['engine'], $link );
			}
		  
			if( isset( $_REQUEST['category'] ) ) {
				$link = add_query_arg( 'category', $_REQUEST['category'], $link );
			}
			
			if( isset( $_REQUEST['s'] ) ) {
				$link = add_query_arg( 's', sanitize_text_field( $_REQUEST['s'] ), $link );
			}
		  
			if( isset( $_REQUEST['post_type'] ) ) {
				$link = add_query_arg( 'post_type', $_REQUEST['post_type'], $link );
			}
			  
			if( isset( $_REQUEST['action'] ) ) {
				$link = add_query_arg( 'action', $_REQUEST['action'], $link );
			}
		}
	   
		return $link;
	}
	
	/**
	 * Set Queried Parent Term ID
	 */
	function get_queried_parent_term_id( $args, $taxonomies ) {
		$taxonomies = $args['taxonomy'];
		
		if ( ! empty( $taxonomies ) && in_array( 'product_make', $taxonomies ) ) {
			if( $args['parent'] ) {
				$this->parent_term_id = $args['parent'];
			}
		}
		
		return $args;
	}
	
	/**
	 * Fix Admin - Fixed Error Saying "Unsupported operand types when no children"
	 * Fix Global - When "No Child" term with get_terms
	 */
	public function fix_no_child_terms_conflict( $value, $option ) {
		if( $this->parent_term_id > 0 ) {
			$value[ $this->parent_term_id ] = '';
		}
		
		return $value;
	}
	
}

$GLOBALS['woo_vpf_hooks'] = new WOO_VPF_Hooks();