<?php
/**
 * CSV importer
 */

defined ( 'ABSPATH' ) || exit;

/**
 * Include dependencies
 */
if ( ! class_exists( 'WOO_VPF_Importer', false ) ) {
	include_once ( WOO_VPF_PLUGIN_PATH . '/includes/admin/importers/abstract-woo-vpf-importer.php' );
}

class WOO_VPF_CSV_Importer extends WOO_VPF_Importer {

	/**
	 * Tracks current row being parsed
	 */
	protected $parsing_raw_data_index = 0;

	/**
	 * Initialize importer.
	 */
	public function __construct ( $file, $params = array() ) {
		$default_args			= array(
			'start_pos'			=> 0, // File pointer start
			'end_pos'			=> -1, // File pointer end
			'lines'				=> -1, // Max lines to read
			'mapping'			=> array(), // Column mapping. csv_heading => schema_heading
			'parse'				=> false, // Whether to sanitize and format data
			'delete_existing'	=> false, // Whether to delete existing items
			'product_column'	=> 'sku', // Product matching column
			'delimiter'			=> ',', // CSV delimiter
			'prevent_timeouts'	=> true, // Check memory and time usage and abort if reaching limit
			'enclosure'			=> '"', // The character used to wrap text in the CSV
			'escape'			=> "\0", // PHP uses '\' as the default escape character. This is not RFC-4180 compliant. This disables the escape character
		);

		$this->params			= wp_parse_args ( $params, $default_args );
		$this->file				= $file;

		if ( isset ( $this->params['mapping']['from'], $this->params['mapping']['to'] ) ) {
			$this->params['mapping'] = array_combine ( $this->params['mapping']['from'], $this->params['mapping']['to'] );
		}
		
		$this->read_file();
	}

	/**
	 * Read file
	 */
	protected function read_file() {
		$handle = fopen ( $this->file, 'r' );

		if ( false !== $handle ) {
			$this->raw_keys = version_compare ( PHP_VERSION, '5.3', '>=' ) ? fgetcsv ( $handle, 0, $this->params['delimiter'], $this->params['enclosure'], $this->params['escape'] ) : fgetcsv ( $handle, 0, $this->params['delimiter'], $this->params['enclosure'] );

			// Remove BOM signature from the first item.
			if ( isset ( $this->raw_keys[0] ) ) {
				$this->raw_keys[0] = $this->remove_utf8_bom ( $this->raw_keys[0] );
			}
			
			if ( ! empty ( $this->raw_keys ) ) {
				$this->raw_keys = array_map ( 'trim', $this->raw_keys );
			}

			if ( 0 !== $this->params['start_pos'] ) {
				fseek( $handle, (int) $this->params['start_pos'] );
			}

			while ( 1 ) {
				$row = version_compare ( PHP_VERSION, '5.3', '>=' ) ? fgetcsv ( $handle, 0, $this->params['delimiter'], $this->params['enclosure'], $this->params['escape'] ) : fgetcsv ( $handle, 0, $this->params['delimiter'], $this->params['enclosure'] );

				if ( false !== $row ) {
					$this->raw_data[]									= $row;
					$this->file_positions[ count ( $this->raw_data ) ]	= ftell ( $handle );

					if ( ( $this->params['end_pos'] > 0 && ftell ( $handle ) >= $this->params['end_pos'] ) || 0 === --$this->params['lines'] ) {
						break;
					}
				} else {
					break;
				}
			}

			$this->file_position = ftell ( $handle );
		}

		if ( ! empty( $this->params['mapping'] ) ) {
			$this->set_mapped_keys ();
		}

		if ( $this->params['parse'] ) {
			$this->set_parsed_data ();
		}
	}

	/**
	 * Remove UTF-8 BOM signature
	 */
	protected function remove_utf8_bom ( $string ) {
		if ( 'efbbbf' === substr ( bin2hex ( $string ), 0, 6 ) ) {
			$string = substr ( $string, 3 );
		}

		return $string;
	}

	/**
	 * Set file mapped keys
	 */
	protected function set_mapped_keys () {
		$mapping = $this->params['mapping'];
		
		foreach ( $this->raw_keys as $key ) {
			$this->mapped_keys[] = isset ( $mapping[ $key ] ) ? $mapping[ $key ] : $key;
		}
	}
	
	/**
	 * Get formatting callback.
	 */
	protected function get_formating_callback () {

		/**
		 * Columns not mentioned here will get parsed with 'wc_clean'.
		 * column_name => callback.
		 */
		
		$data_formatting = array ();
		
		$levels			= WOO_VPF_Functions::get_levels ();
		if ( ! empty ( $levels ) ) {
			foreach ( $levels as $key => $level ) {
				$data_formatting[ $key ] = array ( $this, 'parse_comma_field' );
			}
		}

		$callbacks = array ();

		// Figure out the parse function for each column.
		foreach ( $this->get_mapped_keys () as $index => $heading ) {
			$callback = 'wc_clean';

			if ( isset( $data_formatting[ $heading ] ) ) {
				$callback = $data_formatting[ $heading ];
			}

			$callbacks[] = $callback;
		}
		
		return apply_filters ( 'woo_vpf_product_importer_formatting_callbacks', $callbacks, $this );
	}

	/**
	 * Parse a comma-delineated field from a CSV.
	 */
	public function parse_comma_field ( $value ) {
		if ( empty ( $value ) && '0' !== $value ) {
			return array();
		}

		$value = $this->unescape_data ( $value );
		return array_map ( 'wc_clean', $this->explode_values ( $value ) );
	}

	/**
	 * Map and format raw data to known fields
	 */
	protected function set_parsed_data () {
		$parse_functions	= $this->get_formating_callback ();
		$mapped_keys		= $this->get_mapped_keys ();
		$use_mb				= function_exists ( 'mb_convert_encoding' );
		
		// Parse the data.
		foreach ( $this->raw_data as $row_index => $row ) {
			// Skip empty rows.
			if ( ! count ( array_filter ( $row ) ) ) {
				continue;
			}

			$this->parsing_raw_data_index = $row_index;

			$data = array();

			foreach ( $row as $id => $value ) {
				// Skip ignored columns.
				if ( empty ( $mapped_keys[ $id ] ) ) {
					continue;
				}

				// Convert UTF8.
				if ( $use_mb ) {
					$encoding = mb_detect_encoding ( $value, mb_detect_order(), true );
					if ( $encoding ) {
						$value = mb_convert_encoding ( $value, 'UTF-8', $encoding );
					} else {
						$value = mb_convert_encoding ( $value, 'UTF-8', 'UTF-8' );
					}
				} else {
					$value = wp_check_invalid_utf8 ( $value, true );
				}
				
				$data[ $mapped_keys[ $id ] ] = call_user_func ( $parse_functions[ $id ], $value );
			}
			
			$this->parsed_data[] = apply_filters( 'woo_vpf_product_importer_parsed_data', $data, $this );
		}
	}

	/**
	 * Get a string to identify the row from parsed data
	 */
	protected function get_row_id ( $parsed_data ) {
		$row_data = array();
		
		foreach ( $parsed_data as $parsed_data_key => $parsed_data_val ) {
			$row_data[] = esc_attr ( $parsed_data_val );
		}

		return implode( ', ', $row_data );
	}

	/**
	 * Process importer
	 */
	public function import () {
		$this->start_time		= time();
		$index					= 0;
		
		$this->levels_schema	= array ();
		$levels					= WOO_VPF_Functions::get_levels ();
		
		if ( ! empty ( $levels ) ) {
			foreach ( $levels as $key => $level ) {
				$this->levels_schema[ $key ] = '';
			}
		}
		
		// Level Ranges
		$level_ranges	= array ();
		
		if ( ! empty ( $levels ) ) {
			$range_supported_cols		= WC_Admin_Settings::get_option ( 'woo_vpf_csv_range_supported_cols' );
			
			if ( ! empty ( $range_supported_cols ) ) {
				foreach ( $levels as $key => $level ) {
					if ( in_array ( $key, $range_supported_cols ) ) {
						$level_ranges[ $key ] = 'yes';
					}
				}
			}
		}
		
		$data				= array (
			'imported'		=> array (),
			'failed'		=> array (),
			'skipped'		=> array (),
		);
		
		foreach ( $this->parsed_data as $parsed_data_key => $parsed_data ) {
			$parsed_data = array_merge ( $this->levels_schema, $parsed_data );
			
			if ( ! empty ( $parsed_data ) ) {
				
				// Range
				if ( ! empty ( $level_ranges ) ) {
					foreach ( $level_ranges as $level => $level_range ) {
						if ( $level_range == 'yes' ) {
							if ( is_array ( $parsed_data[ $level ] ) ) {
								$parsed_data_level_vals = array ();
								
								foreach ( $parsed_data[ $level ] as $key => $value ) {
									$value = $this->get_range_to_comma_value ( $value );
									
									if ( is_array ( $value ) ) {
										$parsed_data_level_vals		= array_merge ( $parsed_data_level_vals, $value );
									} else {
										$parsed_data_level_vals[]	= $value;
									}
								}
								
								$parsed_data[ $level ] = $parsed_data_level_vals;
							} else {
								$parsed_data[ $level ] = $this->get_range_to_comma_value ( $parsed_data[ $level ] );
							}
						}
					}
				}
				
				// Comma
				$parsed_data = $this->get_comma_to_individual_rows ( $parsed_data );
				
				foreach ( $parsed_data as $row ) {
					$result = $this->process_item ( $row );
				}
				
				if ( is_wp_error ( $result ) ) {
					$result->add_data ( array( 'row' => $this->get_row_id ( $parsed_data ) ) );
					$data['failed'][] = $result;
				} else {
					$data['imported'][] = $result['id'];
				}
				
			}
			
			$index ++;

			if ( $this->params['prevent_timeouts'] && ( $this->time_exceeded () || $this->memory_exceeded () ) ) {
				$this->file_position = $this->file_positions[ $index ];
				break;
			}
		}

		return $data;
	}
	
	public function get_range_to_comma_value ( $value ) {
		if ( strstr ( $value, '-' ) !== false ) {
			$values_arr = explode ( '-', $value );
			
			if ( count ( $values_arr ) >= 2 ) {
				$values_arr = array_map ( 'wc_clean', $values_arr );
				
				if ( is_numeric ( $values_arr[0] ) && is_numeric ( $values_arr[1] ) ) {
					$value = range ( $values_arr[0], $values_arr[1] );
				}
			}
		}
		
		return $value;
	}
	
	public function get_comma_to_individual_rows ( $array ) {
		$result = array ( array () );
		
		foreach ( $array as $property => $property_values ) {
			$tmp = array ();
			
			foreach ( $result as $result_item ) {
				if ( is_array ( $property_values ) ) {
					foreach ( $property_values as $property_value ) {
						$tmp[] = array_merge ( $result_item, array ( $property => $property_value ) );
					}
				} else {
					$tmp[] = array_merge ( $result_item, array ( $property => $property_values ) );
				}
			}
			
			if ( ! empty ( $tmp) ) {
				$result = $tmp;
			}
		}
		
		return $result;
	}
}
