<?php
/**
 * CSV Importer Controller
 */

defined ( 'ABSPATH' ) || exit;

if ( ! class_exists( 'WP_Importer' ) ) {
	return;
}

/**
 * Importer Controller - handles file upload and forms in admin
 */
class WOO_VPF_CSV_Importer_Controller {
	
	/**
	 * The path to the current file.
	 */
	protected $file = '';

	/**
	 * The current import step.
	 */
	protected $step = '';

	/**
	 * Progress steps.
	 */
	protected $steps = array();

	/**
	 * Errors.
	 */
	protected $errors = array();

	/**
	 * The current delimiter for the file being read
	 */
	protected $delimiter = ',';

	/**
	 * Whether to skip existing terms
	 */
	protected $delete_existing = false;
	
	/**
	 * Product matching column
	 */
	protected $product_column = 'sku';

	/**
	 * Get importer instance
	 */
	public static function get_importer ( $file, $args = array() ) {
		$importer_class	= apply_filters ( 'woo_vpf_csv_importer_class', 'WOO_VPF_CSV_Importer' );
		$args			= apply_filters ( 'woo_vpf_csv_importer_args', $args, $importer_class );
		return new $importer_class ( $file, $args );
	}

	/**
	 * Constructor.
	 */
	public function __construct () {
		$default_steps		= array (
			'upload'		=> array (
				'name'		=> __( 'Upload CSV file', WOO_VPF_TEXT_DOMAIN ),
				'view'		=> array ( $this, 'upload_form' ),
				'handler'	=> array ( $this, 'upload_form_handler' ),
			),
			
			'mapping'		=> array (
				'name'		=> __( 'Column mapping', WOO_VPF_TEXT_DOMAIN ),
				'view'		=> array ( $this, 'mapping_form' ),
				'handler'	=> '',
			),
			
			'import'		=> array (
				'name'		=> __( 'Import', WOO_VPF_TEXT_DOMAIN ),
				'view'		=> array ( $this, 'import' ),
				'handler'	=> '',
			),
			
			'done'			=> array (
				'name'		=> __( 'Done!', WOO_VPF_TEXT_DOMAIN ),
				'view'		=> array ( $this, 'done' ),
				'handler'	=> '',
			),
		);

		$this->steps			= apply_filters ( 'woo_vpf_csv_importer_steps', $default_steps );

		$this->step				= isset ( $_REQUEST['step'] ) ? sanitize_key ( $_REQUEST['step'] ) : current ( array_keys ( $this->steps ) );
		$this->file				= isset ( $_REQUEST['file'] ) ? wc_clean ( wp_unslash ( $_REQUEST['file'] ) ) : '';
		$this->delete_existing	= isset ( $_REQUEST['delete_existing'] ) ? ( bool ) $_REQUEST['delete_existing'] : false;
		$this->product_column	= ! empty ( $_REQUEST['product_column'] ) ? wc_clean ( wp_unslash ( $_REQUEST['product_column'] ) ) : 'sku';
		$this->delimiter		= ! empty ( $_REQUEST['delimiter'] ) ? wc_clean ( wp_unslash ( $_REQUEST['delimiter'] ) ) : ',';
	}

	/**
	 * Get the URL for the next step's screen
	 */
	public function get_next_step_link ( $step = '' ) {
		if ( ! $step ) {
			$step = $this->step;
		}

		$keys = array_keys ( $this->steps );

		if ( end ( $keys ) === $step ) {
			return admin_url ();
		}

		$step_index = array_search ( $step, $keys, true );

		if ( false === $step_index ) {
			return '';
		}

		$params = array(
			'step'				=> $keys[ $step_index + 1 ],
			'file'				=> str_replace ( DIRECTORY_SEPARATOR, '/', $this->file ),
			'delimiter'			=> $this->delimiter,
			'delete_existing'	=> $this->delete_existing,
			'product_column'	=> $this->product_column,
			'_wpnonce'			=> wp_create_nonce ( 'woo-vpf-csv-importer' ),
		);

		return add_query_arg ( $params );
	}

	/**
	 * Output header view
	 */
	protected function output_header () {
		include dirname ( __FILE__ ) . '/views/html-csv-import-header.php';
	}

	/**
	 * Output steps view
	 */
	protected function output_steps () {
		include dirname ( __FILE__ ) . '/views/html-csv-import-steps.php';
	}

	/**
	 * Output footer view
	 */
	protected function output_footer () {
		include dirname ( __FILE__ ) . '/views/html-csv-import-footer.php';
	}

	/**
	 * Add error message
	 */
	protected function add_error ( $message, $actions = array() ) {
		$this->errors[]	= array(
			'message'	=> $message,
			'actions'	=> $actions,
		);
	}

	/**
	 * Add error message
	 */
	protected function output_errors () {
		if ( ! $this->errors ) {
			return;
		}

		foreach ( $this->errors as $error ) {
			echo '<div class="error inline">';
				echo '<p>' . esc_html ( $error['message'] ) . '</p>';

				if ( ! empty ( $error['actions'] ) ) {
					echo '<p>';
						foreach ( $error['actions'] as $action ) {
							echo '<a class="button button-primary" href="' . esc_url ( $action['url'] ) . '">' . esc_html ( $action['label'] ) . '</a> ';
						}
					echo '</p>';
				}
			echo '</div>';
		}
	}

	/**
	 * Dispatch current step and show correct view
	 */
	public function dispatch () {
		if ( ! empty ( $_POST['save_step'] ) && ! empty ( $this->steps[ $this->step ]['handler'] ) ) {
			call_user_func ( $this->steps[ $this->step ]['handler'], $this );
		}
		
		$this->output_header ();
		
		$this->output_steps ();
		$this->output_errors ();
		call_user_func ( $this->steps[ $this->step ]['view'], $this );
		
		$this->output_footer ();
	}

	/**
	 * Output information about the uploading process
	 */
	protected function upload_form () {
		$bytes		= apply_filters ( 'woo_vpf_import_upload_size_limit', wp_max_upload_size() );
		$size		= size_format ( $bytes );
		$upload_dir	= wp_upload_dir ();

		include dirname ( __FILE__ ) . '/views/html-csv-import-form.php';
	}

	/**
	 * Handle the upload form and store options.
	 */
	public function upload_form_handler () {
		check_admin_referer ( 'woo-vpf-csv-importer' );

		$file = $this->handle_upload ();

		if ( is_wp_error ( $file ) ) {
			$this->add_error ( $file->get_error_message () );
			return;
		} else {
			$this->file = $file;
		}
		
		wp_redirect ( esc_url_raw ( $this->get_next_step_link () ) );
		exit;
	}

	/**
	 * Handles the CSV upload and initial parsing of the file to prepare for displaying author import options
	 */
	public function handle_upload () {
		$valid_filetypes = apply_filters( 'woo_vpf_csv_import_valid_filetypes', array(
			'csv'	=> 'text/csv',
			'txt'	=> 'text/plain',
		) );

		$file_url = isset ( $_POST['file_url'] ) ? wc_clean ( wp_unslash ( $_POST['file_url'] ) ) : '';

		if ( empty ( $file_url ) ) {
			if ( ! isset ( $_FILES['import'] ) ) {
				return new WP_Error ( 'woo_vpf_csv_importer_upload_file_empty', __( 'File is empty. Please upload something more substantial. This error could also be caused by uploads being disabled in your php.ini or by post_max_size being defined as smaller than upload_max_filesize in php.ini.', WOO_VPF_TEXT_DOMAIN ) );
			}

			$filetype = wp_check_filetype ( wc_clean ( wp_unslash( $_FILES['import']['name'] ) ), $valid_filetypes );
			if ( ! in_array ( $filetype['type'], $valid_filetypes, true ) ) {
				return new WP_Error ( 'woo_vpf_csv_importer_upload_file_invalid', __( 'Invalid file type. The importer supports CSV and TXT file formats.', WOO_VPF_TEXT_DOMAIN ) );
			}

			$overrides = array(
				'test_form'	=> false,
				'mimes'		=> $valid_filetypes,
			);
			$import		= $_FILES['import'];
			$upload		= wp_handle_upload ( $import, $overrides );

			if ( isset( $upload['error'] ) ) {
				return new WP_Error ( 'woo_vpf_csv_importer_upload_error', $upload['error'] );
			}

			// Construct the object array.
			$object = array(
				'post_title'		=> basename ( $upload['file'] ),
				'post_content'		=> $upload['url'],
				'post_mime_type'	=> $upload['type'],
				'guid'				=> $upload['url'],
				'context'			=> 'import',
				'post_status'		=> 'private',
			);

			// Save the data.
			$id = wp_insert_attachment ( $object, $upload['file'] );

			/*
			 * Schedule a cleanup for one day from now in case of failed
			 * import or missing wp_import_cleanup() call
			 */
			wp_schedule_single_event ( time() + DAY_IN_SECONDS, 'importer_scheduled_cleanup', array ( $id ) );

			return $upload['file'];
		} elseif ( file_exists( ABSPATH . $file_url ) ) {
			$filetype = wp_check_filetype ( ABSPATH . $file_url, $valid_filetypes );
			
			if ( ! in_array ( $filetype['type'], $valid_filetypes, true ) ) {
				return new WP_Error ( 'woo_vpf_csv_importer_upload_file_invalid', __( 'Invalid file type. The importer supports CSV and TXT file formats.', WOO_VPF_TEXT_DOMAIN ) );
			}

			return ABSPATH . $file_url;
		}

		return new WP_Error ( 'woo_vpf_csv_importer_upload_invalid_file', __( 'Please upload or provide the link to a valid CSV file.', WOO_VPF_TEXT_DOMAIN ) );
	}

	/**
	 * Mapping step
	 */
	protected function mapping_form () {
		$args = array(
			'lines'		=> 1,
			'delimiter'	=> $this->delimiter,
		);

		$importer		= self::get_importer ( $this->file, $args );
		$headers		= $importer->get_raw_keys ();
		$mapped_items	= $this->auto_map_columns ( $headers );
		$sample			= current ( $importer->get_raw_data () );
		
		if ( empty ( $sample ) ) {
			$this->add_error (
				__( 'The file is empty or using a different encoding than UTF-8, please try again with a new file.', WOO_VPF_TEXT_DOMAIN ),
				array (
					array (
						'url'	=> admin_url ( 'admin.php?import=woo_product_makes_importer' ),
						'label'	=> __( 'Upload a new file', WOO_VPF_TEXT_DOMAIN ),
					),
				)
			);

			// Force output the errors in the same page
			$this->output_errors ();
			return;
		}

		include_once dirname ( __FILE__ ) . '/views/html-csv-import-mapping.php';
	}

	/**
	 * Import the file if it exists and is valid
	 */
	public function import () {
		if ( ! is_file ( $this->file ) ) {
			$this->add_error ( __( 'The file does not exist, please try again.', WOO_VPF_TEXT_DOMAIN ) );
			return;
		}
		
		if ( ! empty ( $_POST['map_from'] ) && ! empty ( $_POST['map_to'] ) ) {
			$mapping_from	= wc_clean ( wp_unslash ( $_POST['map_from'] ) );
			$mapping_to		= wc_clean ( wp_unslash ( $_POST['map_to'] ) );
		} else {
			wp_redirect ( esc_url_raw( $this->get_next_step_link ( 'upload' ) ) );
			exit;
		}

		wp_localize_script ( 'woo-vpf-admin-import', 'woo_vpf_import_params', array (
			'import_nonce'		=> wp_create_nonce ( 'woo-vpf-import' ),
			'mapping'			=> array(
				'from'			=> $mapping_from,
				'to'			=> $mapping_to,
			),
			'file'				=> $this->file,
			'delete_existing'	=> $this->delete_existing,
			'product_column'	=> $this->product_column,
			'delimiter'			=> $this->delimiter,
		) );
		wp_enqueue_script ( 'woo-vpf-admin-import' );

		include_once dirname ( __FILE__ ) . '/views/html-csv-import-progress.php';
	}

	/**
	 * Done step
	 */
	protected function done () {
		$imported	= isset ( $_GET['terms-imported'] ) ? absint ( $_GET['terms-imported'] ) : 0;
		$failed		= isset ( $_GET['terms-failed'] ) ? absint ( $_GET['terms-failed'] ) : 0;
		$skipped	= isset ( $_GET['terms-skipped'] ) ? absint ( $_GET['terms-skipped'] ) : 0;
		$errors		= array_filter ( (array) get_user_option ( 'woo_vpf_import_error_log' ) );
		
		include_once dirname ( __FILE__ ) . '/views/html-csv-import-done.php';
	}

	/**
	 * Auto map column names
	 */
	protected function auto_map_columns ( $raw_headers, $num_indexes = true ) {
		$levels_title = $this->get_mapping_options ();

		$headers = array();
		foreach ( $raw_headers as $key => $field ) {
			$field				= strtolower( $field );
			$index				= $num_indexes ? $key : $field;
			$headers[ $index ]	= $field;
			
			if ( isset ( $default_columns[ $field ] ) ) {
				$headers[ $index ] = $default_columns[ $field ];
			}
		}

		return apply_filters ( 'woo_vpf_csv_import_mapped_columns', $headers, $raw_headers );
	}

	/**
	 * Get mapping options
	 */
	protected function get_mapping_options () {
		$levels = WOO_VPF_Functions::get_levels ();
		return apply_filters ( 'woo_vpf_csv_import_mapping_options', $levels );
	}
}
