<?php
/**
 * Abstract Importer
 */

defined ( 'ABSPATH' ) || exit;

/**
 * Include dependencies.
 */
if ( ! class_exists( 'WC_Importer_Interface', false ) ) {
	include_once WC_ABSPATH . 'includes/interfaces/class-wc-importer-interface.php';
}

/**
 * WOO_VPF_Importer Class.
 */
abstract class WOO_VPF_Importer implements WC_Importer_Interface {

	/**
	 * CSV file.
	 */
	protected $file = '';

	/**
	 * The file position after the last read.
	 */
	protected $file_position = 0;

	/**
	 * Importer parameters.
	 */
	protected $params = array();

	/**
	 * Raw keys - CSV raw headers.
	 */
	protected $raw_keys = array();

	/**
	 * Mapped keys - CSV headers.
	 */
	protected $mapped_keys = array();

	/**
	 * Raw data.
	 */
	protected $raw_data = array();

	/**
	 * Raw data.
	 */
	protected $file_positions = array();

	/**
	 * Parsed data.
	 */
	protected $parsed_data = array();

	/**
	 * Start time of current import.
	 */
	protected $start_time = 0;
	
	/**
	 * Get file raw headers.
	 */
	public function get_raw_keys () {
		return $this->raw_keys;
	}

	/**
	 * Get file mapped headers.
	 */
	public function get_mapped_keys () {
		return ! empty( $this->mapped_keys ) ? $this->mapped_keys : $this->raw_keys;
	}

	/**
	 * Get raw data.
	 */
	public function get_raw_data () {
		return $this->raw_data;
	}

	/**
	 * Get parsed data.
	 */
	public function get_parsed_data () {
		return apply_filters( 'woo_vpf_importer_parsed_data', $this->parsed_data, $this->get_raw_data() );
	}

	/**
	 * Get importer parameters.
	 */
	public function get_params () {
		return $this->params;
	}

	/**
	 * Get file pointer position from the last read.
	 */
	public function get_file_position () {
		return $this->file_position;
	}

	/**
	 * Get file pointer position as a percentage of file size.
	 */
	public function get_percent_complete () {
		$size = filesize ( $this->file );
		if ( ! $size ) {
			return 0;
		}

		//return absint ( min ( round ( ( $this->file_position / $size ) * 100 ), 100 ) ); // It misses few rows from the last
		return absint ( min ( ( $this->file_position / $size ) * 100, 100 ) );
	}

	/**
	 * Process a single item and save.
	 */
	protected function process_item ( $data ) {
		try {
			$data = array_map ( 'trim', $data );
			
			// Get product ID from SKUs if created during the importation
			$product_ids		= array ();
			if ( ! empty ( $data['sku'] ) ) {
				$skus		= $data['sku'];
				$skus		= explode ( ',', $skus );
				$skus		= array_map ( 'trim', $skus );
				
				foreach ( $skus as $sku ) {
					if ( $this->params['product_column'] == 'slug' ) {
						$product_ids[] = $this->get_product_id_by_slug ( $sku );
					} else {
						$product_ids[] = wc_get_product_id_by_sku ( $sku );
					}
				}
			}
			unset ( $data['sku'] );
			
			// Create terms
			$term_ids = array ();
			
			$data = array_values ( $data );
			foreach ( $data as $term ) {
				if ( $term == '' ) {
					break;
				}
				
				// Term Parent ID
				$parent_id = 0;
				if ( ! empty ( $term_ids ) ) {
					$parent_id = end ( $term_ids );
				}
				
				$term_id = WOO_VPF_Functions::term_exists ( $term, $parent_id );
				if ( ! $term_id ) {
					$term_id = WOO_VPF_Functions::insert_term ( $term, $parent_id );
				}
				
				if ( is_wp_error( $term_id ) || ! $term_id ) {
					break;
				}
				
				$term_ids[] = $term_id;
			}
			
			if ( ! empty ( $term_ids ) && ! empty ( $product_ids ) ) {
				foreach ( $product_ids as $product_id ) {
					if ( $product_id ) {
						$term_ids = array_map ( 'intval', $term_ids );
						$term_ids = array_unique ( $term_ids );
						wp_set_object_terms ( $product_id, $term_ids, 'product_make', true );
					}
				}
			}
			
			return array (
				'id' => $term_ids[0]
			);
		} catch ( Exception $e ) {
			return new WP_Error( 'woo_vpf_importer_error', $e->getMessage(), array( 'status' => $e->getCode() ) );
		}
	}
	
	/**
	 * Get product ID from slug.
	 */
	protected function get_product_id_by_slug ( $slug ) {
		if ( ! empty ( $slug ) ) {
			global $wpdb;
			$post_id = $wpdb->get_var( "SELECT `ID` FROM ".$wpdb->posts." WHERE `post_name` like '".$slug."' && `post_type`='product'" );
			
			if ( WOO_VPF_Functions::is_wpml_activated() ) {
				$post_id = icl_object_id ( $post_id, 'product', false, ICL_LANGUAGE_CODE );
			}
			
			return $post_id;
		}
		
		return false;
	}

	/**
	 * Memory exceeded
	 *
	 * Ensures the batch process never exceeds 90%
	 * of the maximum WordPress memory.
	 */
	protected function memory_exceeded () {
		$memory_limit   = $this->get_memory_limit() * 0.9; // 90% of max memory
		$current_memory = memory_get_usage( true );
		$return         = false;
		if ( $current_memory >= $memory_limit ) {
			$return = true;
		}
		return apply_filters( 'woo_vpf_importer_memory_exceeded', $return );
	}

	/**
	 * Get memory limit
	 */
	protected function get_memory_limit () {
		if ( function_exists( 'ini_get' ) ) {
			$memory_limit = ini_get( 'memory_limit' );
		} else {
			// Sensible default.
			$memory_limit = '128M';
		}

		if ( ! $memory_limit || -1 === intval( $memory_limit ) ) {
			// Unlimited, set to 32GB.
			$memory_limit = '32000M';
		}
		return intval( $memory_limit ) * 1024 * 1024;
	}

	/**
	 * Time exceeded.
	 *
	 * Ensures the batch never exceeds a sensible time limit.
	 * A timeout limit of 30s is common on shared hosting.
	 */
	protected function time_exceeded () {
		$finish = $this->start_time + apply_filters( 'woo_vpf_importer_default_time_limit', 20 ); // 20 seconds
		$return = false;
		if ( time() >= $finish ) {
			$return = true;
		}
		return apply_filters( 'woo_vpf_importer_time_exceeded', $return );
	}

	/**
	 * Explode CSV cell values using commas by default, and handling escaped
	 * separators.
	 */
	protected function explode_values ( $value ) {
		$value  = str_replace( '\\,', '::separator::', $value );
		$values = explode( ',', $value );
		$values = array_map( array( $this, 'explode_values_formatter' ), $values );

		return $values;
	}

	/**
	 * Remove formatting and trim each value.
	 */
	protected function explode_values_formatter ( $value ) {
		return trim( str_replace( '::separator::', ',', $value ) );
	}

	/**
	 * The exporter prepends a ' to fields that start with a - which causes
	 * issues with negative numbers. This removes the ' if the input is still a valid
	 * number after removal.
	 */
	protected function unescape_negative_number ( $value ) {
		if ( 0 === strpos( $value, "'-" ) ) {
			$unescaped = trim( $value, "'" );
			if ( is_numeric( $unescaped ) ) {
				return $unescaped;
			}
		}
		return $value;
	}
	
	/**
	 * The exporter prepends a ' to escape fields that start with =, +, - or @.
	 * Remove the prepended ' character preceding those characters.
	 */
	protected function unescape_data( $value ) {
		$active_content_triggers = array( "'=", "'+", "'-", "'@" );

		if ( in_array( mb_substr( $value, 0, 2 ), $active_content_triggers, true ) ) {
			$value = mb_substr( $value, 1 );
		}

		return $value;
	}
}
