<?php
/**
 * Terms Importers
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

if ( ! class_exists( 'WOO_VPF_Importers' ) ) :

/**
 * WOO_VPF_Importers Class
 */
class WOO_VPF_Importers {

	/**
	 * Constructor
	 */
	public function __construct() {
		add_action ( 'init', array( __CLASS__, 'init' ) );
	}
	
	public static function init () {
		if ( ! self::import_allowed () ) {
			return;
		}
		
		if ( ! function_exists ( 'WC' ) ) {
			return;
		}
		
		add_action ( 'admin_init', array ( __CLASS__, 'register_importer' ) );
		add_action ( 'admin_enqueue_scripts', array ( __CLASS__, 'admin_scripts' ) );
		add_action ( 'wp_ajax_woo_vpf_import', array ( __CLASS__, 'do_ajax_import' ) );
	}

	/**
	 * Checks if imports are allowed for current user
	 */
	protected static function import_allowed () {
		return current_user_can ( 'manage_woocommerce' );
	}
	
	/**
	 * Register Importer
	 */
	public static function register_importer () {
		if ( defined( 'WP_LOAD_IMPORTERS' ) ) {
			register_importer ( 'woo_product_makes_importer', __( 'WooCommerce VPF Terms (CSV)', WOO_VPF_TEXT_DOMAIN ), __( 'Import <strong>VPF Terms</strong> to your store via a csv file.', WOO_VPF_TEXT_DOMAIN ), array ( __CLASS__, 'terms_importer' ) );
		}
	}
	
	/**
	 * Import Terms Page
	 */
	public static function terms_importer () {
		include_once ( WOO_VPF_PLUGIN_PATH . '/includes/admin/importers/class-woo-vpf-csv-importer.php' );
		include_once ( WOO_VPF_PLUGIN_PATH . '/includes/admin/importers/class-woo-vpf-csv-importer-controller.php' );

		$importer = new WOO_VPF_CSV_Importer_Controller ();
		$importer->dispatch ();
	}

	/**
	 * Register Assets
	 */
	public static function admin_scripts () {
		wp_register_script ( 'woo-vpf-admin-import', WOO_VPF_PLUGIN_URL . '/assets/js/admin/import.js', array ( 'jquery' ), WC_VERSION );
		
		wp_register_style ( 'woocommerce_admin_styles', WC ()->plugin_url () . '/assets/css/admin.css', array (), WC_VERSION );
		wp_enqueue_style ( 'woocommerce_admin_styles' );
	}

	/**
	 * Ajax callback for importing one batch of terms from a CSV.
	 */
	public static function do_ajax_import () {
		check_ajax_referer( 'woo-vpf-import', 'security' );

		if ( ! self::import_allowed () || ! isset ( $_POST['file'] ) ) {
			wp_send_json_error ( array( 'message' => __( 'Insufficient privileges to import terms.', WOO_VPF_TEXT_DOMAIN ) ) );
		}

		include_once ( WOO_VPF_PLUGIN_PATH . '/includes/admin/importers/class-woo-vpf-csv-importer.php' );
		include_once ( WOO_VPF_PLUGIN_PATH . '/includes/admin/importers/class-woo-vpf-csv-importer-controller.php' );
		
		$file				= wc_clean ( wp_unslash( $_POST['file'] ) );
		$delete_existing	= isset ( $_POST['delete_existing'] ) ? ( bool ) $_POST['delete_existing'] : false;
		
		$params = array (
			'delimiter'			=> ! empty ( $_POST['delimiter'] ) ? wc_clean ( wp_unslash ( $_POST['delimiter'] ) ) : ',',
			'start_pos'			=> isset ( $_POST['position'] ) ? absint ( $_POST['position'] ) : 0,
			'mapping'			=> isset ( $_POST['mapping'] ) ? ( array ) wc_clean ( wp_unslash( $_POST['mapping'] ) ) : array (),
			'delete_existing'	=> $delete_existing,
			'product_column'	=> ! empty ( $_POST['product_column'] ) ? wc_clean ( wp_unslash ( $_POST['product_column'] ) ) : 'sku',
			'lines'				=> apply_filters ( 'woo_vpf_import_batch_size', 30 ),
			'parse'				=> true,
		);

		// Log failures
		if ( 0 !== $params['start_pos'] ) {
			$error_log = array_filter ( ( array ) get_user_option ( 'woo_vpf_import_error_log' ) );
		} else {
			$error_log = array ();
		}
		
		// Delete existing terms
		if ( 0 === $params['start_pos'] && $delete_existing ) {
			WOO_VPF_Functions::delete_all_terms ();
		}

		$importer			= WOO_VPF_CSV_Importer_Controller::get_importer ( $file, $params );
		$results			= $importer->import ();
		$percent_complete	= $importer->get_percent_complete ();
		$error_log			= array_merge ( $error_log, $results['failed'], $results['skipped'] );

		update_user_option ( get_current_user_id (), 'woo_vpf_import_error_log', $error_log );

		if ( 100 === $percent_complete ) {
			// Send success
			wp_send_json_success (
				array (
					'position'		=> 'done',
					'percentage'	=> 100,
					'url'			=> add_query_arg ( array ( 'nonce' => wp_create_nonce ( 'product-csv' ) ), admin_url ( 'admin.php?import=woo_product_makes_importer&step=done' ) ),
					'imported'		=> count ( $results['imported'] ),
					'failed'		=> count ( $results['failed'] ),
					'skipped'		=> count ( $results['skipped'] ),
				)
			);
		} else {
			wp_send_json_success (
				array (
					'position'		=> $importer->get_file_position (),
					'percentage'	=> $percent_complete,
					'imported'		=> count ( $results['imported'] ),
					'failed'		=> count ( $results['failed'] ),
					'skipped'		=> count ( $results['skipped'] ),
				)
			);
		}
	}
}

endif;

new WOO_VPF_Importers();