<?php
/**
 * Plugin Name: UnGrabber
 * Plugin URI: https://1.envato.market/ungrabber
 * Description: Most effective way to protect your online content from being copy.
 * Author: پرشین اسکریپت
 * Version: 1.0.1
 * Author URI: https://www.persianscript.ir
 **/

/** 
 * UnGrabber - Most effective way to protect your online content from being copied.
 * Exclusively on Envato Market: https://1.envato.market/ungrabber
 * 
 * @encoding     UTF-8
 * @version      1.0.1
 * @copyright    Copyright (C) 2019 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license      Envato Standard License https://1.envato.market/KYbje
 * @author       Alexandr Khmelnytsky (info@alexander.khmelnitskiy.ua)
 * @support      dmitry@merkulov.design
 **/

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
    header( 'Status: 403 Forbidden' );
    header( 'HTTP/1.1 403 Forbidden' );
    exit;
}

if ( ! class_exists( 'UnGrabber' ) ) :
    
    /**
     * SINGLETON: Core class used to implement a UnGrabber plugin.
     *
     * This is used to define internationalization, admin-specific hooks, and
     * public-facing site hooks.
     *
     * @since 1.0.0
     */
    final class UnGrabber {
    
        /** Plugin version.
         *
         * @var Constant
         * @since 1.0.0
         **/
        const VERSION = '1.0.1';
        
        /**
         * UnGrabber Plugin settings.
         * 
	 * @var array()
	 * @since 1.0.0
	 **/
        public $options = array();
        
        /**
         * Use minified libraries if SCRIPT_DEBUG is turned off.
         * 
	 * @since 1.0.0
	 **/
        public $suffix;
        
        /**
         * URL (with trailing slash) to plugin folder.
         * 
	 * @var array()
	 * @since 1.0.0
	 **/
        public $url = '';
        
        /**
         * The one true UnGrabber.
         * 
	 * @var UnGrabber
	 * @since 1.0.0
	 **/
	private static $instance;
        
        /**
         * Main UnGrabber Instance.
         *
         * Insures that only one instance of UnGrabber exists in memory at any one time.
         *
         * @static
         * @return UnGrabber
         * @since 1.0.0
         **/
        public static function get_instance() {
            if ( ! isset( self::$instance ) && ! ( self::$instance instanceof UnGrabber ) ) {
                self::$instance = new UnGrabber;
            }

            return self::$instance;
        }

        /**
	 * Throw error on object clone.
	 *
	 * The whole idea of the singleton design pattern is that there is a single
	 * object therefore, we don't want the object to be cloned.
	 *
	 * @since 1.0.0
	 * @access protected
	 * @return void
	 **/
	public function __clone() {
            /** Cloning instances of the class is forbidden. */
            _doing_it_wrong( __FUNCTION__, esc_html__( 'The whole idea of the singleton design pattern is that there is a single object therefore, we don\'t want the object to be cloned.', 'ungrabber' ), self::VERSION );
	}

        /**
	 * Disable unserializing of the class.
         * 
         * The whole idea of the singleton design pattern is that there is a single
	 * object therefore, we don't want the object to be unserialized.
	 *
	 * @since 1.0.0
	 * @access protected
	 * @return void
	 **/
	public function __wakeup() {
            /** Unserializing instances of the class is forbidden. */
            _doing_it_wrong( __FUNCTION__, esc_html__( 'The whole idea of the singleton design pattern is that there is a single object therefore, we don\'t want the object to be unserialized.', 'ungrabber' ), self::VERSION );
	}
        
        /**
         * Sets up a new plugin instance.
         *
         * @since 1.0.0
         * @access public
         **/
        private function __construct() {
            
            /** Gets the plugin URL (with trailing slash). */
            $this->url = plugin_dir_url( __FILE__ );
            
            /** Use minified libraries if SCRIPT_DEBUG is turned off. */
            $this->suffix = ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ) ? '' : '.min';
            
            /** Load translation. */
            add_action( 'plugins_loaded', array( $this, 'load_textdomain' ) );
            
            /** Get plugin settings. */
            $this->options = $this->getOptions();
            
            /** Add plugin setting page. */
            add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
            add_action( 'admin_init', array( $this, 'settings_init' ) );

            /** Add plugin links. */
            add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), array( $this, 'add_links' ) );
            
            /** Add plugin meta. */
            add_filter( 'plugin_row_meta', array( $this, 'add_row_meta' ), 10, 2 );

            /** Add plugin script. */
            add_action( 'wp_enqueue_scripts', array( $this, 'load_scripts' ) );

            /** Add admin styles. */
            add_action( 'admin_enqueue_scripts', array( $this, 'load_admin_styles' ) );
            
            /** Add admin javascript. */
            add_action( 'admin_enqueue_scripts', array( $this, 'load_admin_scripts' ) );
            
            /** Add shortcode [disable_ungrabber] */
            add_shortcode( 'disable_ungrabber', array( $this, 'disable_ungrabber_shortcode' ) );
            
            /** JavaScript Required. */
            add_action( 'wp_footer', array( $this, 'javascript_required' ) );

        }
        
        /**
         * Protect Content if JavaScript is Disabled.
         *
         * @since 1.0.0
         * @access public
         **/
        public function javascript_required() {

            if ( $this->options['javascript'] != "true" ) { return; }
            
            ob_start();
            ?>
            <noscript>
                <div id='mdp-ungrabber-js-disabled'>
                    <div><?php echo wp_kses_post( $this->options['javascript_msg'] ); ?></div>
                </div>
                <style>
                    #mdp-ungrabber-js-disabled {
                        position: fixed;
                        top: 0;
                        left: 0;
                        height: 100%;
                        width: 100%;
                        z-index: 999999;
                        text-align: center;
                        background-color: #FFFFFF;
                        color: #000000;
                        font-size: 40px;
                        display: flex;
                        align-items: center;
                        justify-content: center;
                    }
                </style>
            </noscript>
            <?php
            $result = ob_get_clean();
            
            echo $result;
        }
        
        /**
         * Shortcode to disable plugin for certain pages.
         * Use [disable_ungrabber].
         *
         * @since 1.0.0
         * @access public
         **/
        public function disable_ungrabber_shortcode() {
            
            wp_enqueue_script( 'mdp-ungrabber-destroyer', $this->url . 'js/ungrabber-destroyer' . $this->suffix . '.js', array(), self::VERSION, true );
            
            return;
        }
        
        /**
         * Register the JavaScript for the public-facing side of the site.
         *
         * @since   1.0.0
         * @return void
         **/
        public function load_scripts() {
            
            /** Checks if plugin should work on this page. */
            if( $this->display() ) {
                
                wp_enqueue_script( 'mdp-ungrabber', $this->url . 'js/ungrabber' . $this->suffix . '.js', array(), self::VERSION, true );
                wp_localize_script( 'mdp-ungrabber', 'mdp_ungrabber', 
                    /** We use md_ prefix to avoid conflicts in JS. */
                    array( 
                        'md_select_all'     => $this->options['select_all'], // Disable Select All.
                        'md_copy'           => $this->options['сopy'], // Disable Copy.
                        'md_cut'            => $this->options['cut'], // Disable Cut.
                        'md_paste'          => $this->options['paste'], // Disable Paste.
                        'md_save'           => $this->options['save'], // Disable Save.
                        'md_view_source'    => $this->options['view_source'], // Disable View Source.
                        'md_print_page'     => $this->options['print_page'], // Disable Print Page.
                        'md_developer_tool' => $this->options['developer_tool'], // Disable Developer Tool.
                        'md_reader_mode' => $this->options['reader_mode'], // Disable Safari Reader Mode.
                        'md_right_click'    => $this->options['right_click'], // Disable Right Click.
                        'md_text_selection' => $this->options['text_selection'], // Disable Text Selection.
                        'md_image_dragging' => $this->options['image_dragging'], // Disable Image Dragging.
                    )
                );
                
            }
            
        }

        /**
         * Add admin menu for plugin settings.
         *
         * @since 1.0.0
         * @access public
         **/
        public function add_admin_menu() {
            add_submenu_page(
                'options-general.php',
                esc_html__( 'UnGrabber Settings', 'ungrabber' ),
                esc_html__( 'UnGrabber', 'ungrabber' ),
                'manage_options',
                'mdp_ungrabber_settings',
                array( $this, 'options_page' )
            );
        }

        /**
         * Plugin Settings Page.
         *
         * @since 1.0.0
         * @access public
         **/
        public function options_page() {
            
            /** User rights check. */
            if ( ! current_user_can('manage_options' ) ) { return; }
            
            global $pagenow; 
            ?>

            <div class="wrap">
                <h1><?php echo get_admin_page_title() ?></h1>
                <p><?php esc_html_e( 'UnGrabber - Most effective way to protect your online content from being copied.', 'ungrabber' ); ?></p>
                
                <?php 
                if ( isset( $_GET['tab'] ) ) {
                    $this->admin_tabs( $_GET['tab'] );
                } else {
                    $this->admin_tabs( 'general' );
                }

                if ( $pagenow == 'options-general.php' && $_GET['page'] == 'mdp_ungrabber_settings' ) { // DO We need this chek?

                    $tab = 'general';
                    if ( isset ( $_GET['tab'] ) ) {
                        $tab = $_GET['tab'];
                    }
        
                    /** General Tab. */
                    if( $tab == 'general' ) { 
                        
                        ?>
                        <form action='options.php' method='post'>
                            <?php settings_fields( 'UnGrabberOptionsGroup' ); ?>
                            <?php do_settings_sections( 'UnGrabberOptionsGroup' ); ?>
                            <?php submit_button(); ?>
                        </form>
                        <?php
                    
                    /** Assignments Tab. */
                    } elseif ( $tab == 'assignments' ) {
                        
                        ?>
                        <form action='options.php' method='post'>
                            <?php settings_fields( 'UnGrabberAssignmentsOptionsGroup' ); ?>
                            <?php do_settings_sections( 'UnGrabberAssignmentsOptionsGroup' ); ?>
                            <?php submit_button(); ?>
                        </form>
                        <?php
                        
                    }
        
                } // DO We need this chek?
            ?>
            </div>
            
            <?php
        }
        
        /**
         * Render Tabs.
         *
         * @since 1.0.0
         * @access public
         **/
        public function admin_tabs( $current = 'general' ) {
            
            /** Tabs array. */
            $tabs = array( 
                'general' => esc_html__( 'General', 'ungrabber' ), 
                'assignments' => esc_html__( 'Assignments', 'ungrabber' ) 
            );

            /** Render Tabs. */
            ?><div class="nav-tab-wrapper" style="margin-bottom: 10px;"><?php
            foreach ( $tabs as $tab => $name ) {
                $class = ( $tab == $current ) ? ' nav-tab-active' : '';
                echo "<a class='nav-tab $class' href='?page=mdp_ungrabber_settings&tab=$tab'>$name</a>";
            }
            ?></div><?php
        }

        /**
         * Generate Settings Page.
         *
         * @since 1.0.0
         * @access public
         **/
        public function settings_init() {
            
            /** General Tab. */
            register_setting( 'UnGrabberOptionsGroup', 'mdp_ungrabber_settings' );
            add_settings_section( 'mdp_ungrabber_settings_page_general_section', '', NULL, 'UnGrabberOptionsGroup' );
            
            /** Select All: */
            add_settings_field( 'select_all', esc_html__( 'Disable Select All:', 'ungrabber' ), array( $this, 'render_select_all' ), 'UnGrabberOptionsGroup', 'mdp_ungrabber_settings_page_general_section' );
            
            /** Copy: */
            add_settings_field( 'сopy', esc_html__( 'Disable Copy:', 'ungrabber' ), array( $this, 'render_сopy' ), 'UnGrabberOptionsGroup', 'mdp_ungrabber_settings_page_general_section' );
            
            /** Cut: */
            add_settings_field( 'cut', esc_html__( 'Disable Cut:', 'ungrabber' ), array( $this, 'render_cut' ), 'UnGrabberOptionsGroup', 'mdp_ungrabber_settings_page_general_section' );
            
            /** Paste: */
            add_settings_field( 'paste', esc_html__( 'Disable Paste:', 'ungrabber' ), array( $this, 'render_paste' ), 'UnGrabberOptionsGroup', 'mdp_ungrabber_settings_page_general_section' );
            
            /** Save: */
            add_settings_field( 'save', esc_html__( 'Disable Save:', 'ungrabber' ), array( $this, 'render_save' ), 'UnGrabberOptionsGroup', 'mdp_ungrabber_settings_page_general_section' );

            /** View Source: */
            add_settings_field( 'view_source', esc_html__( 'Disable View Source:', 'ungrabber' ), array( $this, 'render_view_source' ), 'UnGrabberOptionsGroup', 'mdp_ungrabber_settings_page_general_section' );

            /** Print Page: */
            add_settings_field( 'print_page', esc_html__( 'Disable Print Page:', 'ungrabber' ), array( $this, 'render_print_page' ), 'UnGrabberOptionsGroup', 'mdp_ungrabber_settings_page_general_section' );

            /** Developer Tool: */
            add_settings_field( 'developer_tool', esc_html__( 'Disable Developer Tool:', 'ungrabber' ), array( $this, 'render_developer_tool' ), 'UnGrabberOptionsGroup', 'mdp_ungrabber_settings_page_general_section' );

            /** Safari Reader mode: */
            add_settings_field( 'reader_mode', esc_html__( 'Disable Safari Reader Mode:', 'ungrabber' ), array( $this, 'render_reader_mode' ), 'UnGrabberOptionsGroup', 'mdp_ungrabber_settings_page_general_section' );

            /** Right Click: */
            add_settings_field( 'right_click', esc_html__( 'Disable Right Click:', 'ungrabber' ), array( $this, 'render_right_click' ), 'UnGrabberOptionsGroup', 'mdp_ungrabber_settings_page_general_section' );

            /** Text Selection: */
            add_settings_field( 'text_selection', esc_html__( 'Disable Text Selection:', 'ungrabber' ), array( $this, 'render_text_selection' ), 'UnGrabberOptionsGroup', 'mdp_ungrabber_settings_page_general_section' );

            /** Image Dragging: */
            add_settings_field( 'image_dragging', esc_html__( 'Disable Image Dragging:', 'ungrabber' ), array( $this, 'render_image_dragging' ), 'UnGrabberOptionsGroup', 'mdp_ungrabber_settings_page_general_section' );

            /** Block Page if JavaScript is Disabled: */
            add_settings_field( 'javascript', esc_html__( 'JavaScript Required:', 'ungrabber' ), array( $this, 'render_javascript' ), 'UnGrabberOptionsGroup', 'mdp_ungrabber_settings_page_general_section' );

            /** Message if JavaScript is Disabled: */
            add_settings_field( 'javascript_msg', esc_html__( 'JavaScript Required Message:', 'ungrabber' ), array( $this, 'render_javascript_msg' ), 'UnGrabberOptionsGroup', 'mdp_ungrabber_settings_page_general_section' );
            
            /** Assignment Tab. */
            register_setting( 'UnGrabberAssignmentsOptionsGroup', 'mdp_ungrabber_assignments_settings' );
            add_settings_section( 'mdp_ungrabber_settings_page_assignments_section', '', NULL, 'UnGrabberAssignmentsOptionsGroup' );
            
            /** Assignments. */
            add_settings_field( 'assignments', esc_html__( 'Assignments:', 'ungrabber' ), array( $this, 'render_assignments' ), 'UnGrabberAssignmentsOptionsGroup', 'mdp_ungrabber_settings_page_assignments_section' );
            
        }
        
        /**
         * Render Select All field.
         *
         * @since 1.0.0
         * @access public
         **/
        public function render_select_all() {
            
            ?>
            <label>
                <input type='checkbox' name='mdp_ungrabber_settings[select_all]' value='true' <?php checked( filter_var( $this->options['select_all'], FILTER_VALIDATE_BOOLEAN ) ); ?> >&nbsp;<?php esc_html_e( 'Disable HotKeys:', 'ungrabber' ); ?> <b>Ctrl+A</b> <?php esc_html_e( '(Windows and Linux),', 'ungrabber' ); ?> <b>⌘+A</b> <?php esc_html_e( '(macOS)', 'ungrabber' ); ?>
            </label>
            <p class="description"><?php esc_html_e( 'Protect Your Text from Being Copied by Select All HotKeys.', 'ungrabber' ); ?></p>
            <?php

        }
        
        /**
         * Render Copy field.
         *
         * @since 1.0.0
         * @access public
         **/
        public function render_сopy() {
            
            ?>
            <label>
                <input type='checkbox' name='mdp_ungrabber_settings[сopy]' value='true' <?php checked( filter_var( $this->options['сopy'], FILTER_VALIDATE_BOOLEAN ) ); ?> >&nbsp;<?php esc_html_e( 'Disable HotKeys:', 'ungrabber' ); ?> <b>Ctrl+C</b> <?php esc_html_e( '(Windows and Linux),', 'ungrabber' ); ?> <b>⌘+C</b> <?php esc_html_e( '(macOS)', 'ungrabber' ); ?>
            </label>
            <p class="description"><?php esc_html_e( 'Protect Your Text from Being Copied by Copy HotKeys.', 'ungrabber' ); ?></p>
            <?php

        }
        
        /**
         * Render Cut field.
         *
         * @since 1.0.0
         * @access public
         **/
        public function render_cut() {
            
            ?>
            <label>
                <input type='checkbox' name='mdp_ungrabber_settings[cut]' value='true' <?php checked( filter_var( $this->options['cut'], FILTER_VALIDATE_BOOLEAN ) ); ?> >&nbsp;<?php esc_html_e( 'Disable HotKeys:', 'ungrabber' ); ?> <b>Ctrl+X</b> <?php esc_html_e( '(Windows and Linux),', 'ungrabber' ); ?> <b>⌘+X</b> <?php esc_html_e( '(macOS)', 'ungrabber' ); ?>
            </label>
            <p class="description"><?php esc_html_e( 'Protect Your Text from Being Copied by Cut HotKeys.', 'ungrabber' ); ?></p>
            <?php

        }
        
        /**
         * Render Paste field.
         *
         * @since 1.0.0
         * @access public
         **/
        public function render_paste() {
            
            ?>
            <label>
                <input type='checkbox' name='mdp_ungrabber_settings[paste]' value='true' <?php checked( filter_var( $this->options['paste'], FILTER_VALIDATE_BOOLEAN ) ); ?> >&nbsp;<?php esc_html_e( 'Disable HotKeys:', 'ungrabber' ); ?> <b>Ctrl+V</b> <?php esc_html_e( '(Windows and Linux),', 'ungrabber' ); ?> <b>⌘+V</b> <?php esc_html_e( '(macOS)', 'ungrabber' ); ?>
            </label>
            <p class="description"><?php esc_html_e( 'Disable Peaste HotKeys.', 'ungrabber' ); ?></p>
            <?php
            
        }

        /**
         * Render Save field.
         *
         * @since 1.0.0
         * @access public
         **/
        public function render_save() {
            
            ?>
            <label>
                <input type='checkbox' name='mdp_ungrabber_settings[save]' value='true' <?php checked( filter_var( $this->options['save'], FILTER_VALIDATE_BOOLEAN ) ); ?> >&nbsp;<?php esc_html_e( 'Disable HotKeys:', 'ungrabber' ); ?> <b>Ctrl+S</b> <?php esc_html_e( '(Windows and Linux),', 'ungrabber' ); ?> <b>⌘+S</b> <?php esc_html_e( '(macOS)', 'ungrabber' ); ?>
            </label>
            <p class="description"><?php esc_html_e( 'Protect Your Text from Being Saved by Save HotKeys.', 'ungrabber' ); ?></p>
            <?php
            
        }

        /**
         * Render View Source field.
         *
         * @since 1.0.0
         * @access public
         **/
        public function render_view_source() {
            
            ?>
            <label>
                <input type='checkbox' name='mdp_ungrabber_settings[view_source]' value='true' <?php checked( filter_var( $this->options['view_source'], FILTER_VALIDATE_BOOLEAN ) ); ?> >&nbsp;<?php esc_html_e( 'Disable HotKeys:', 'ungrabber' ); ?> <b>Ctrl+U</b> <?php esc_html_e( '(Windows and Linux),', 'ungrabber' ); ?> <b>⌘+U</b> <?php esc_html_e( '(macOS)', 'ungrabber' ); ?>
            </label>
            <p class="description"><?php esc_html_e( 'Disable to View Source Code of Page by HotKeys.', 'ungrabber' ); ?></p>
            <?php
            
        }

        /**
         * Render Print Page field.
         *
         * @since 1.0.0
         * @access public
         **/
        public function render_print_page() {
            
            ?>
            <label>
                <input type='checkbox' name='mdp_ungrabber_settings[print_page]' value='true' <?php checked( filter_var( $this->options['print_page'], FILTER_VALIDATE_BOOLEAN ) ); ?> >&nbsp;<?php esc_html_e( 'Disable HotKeys:', 'ungrabber' ); ?> <b>Ctrl+P</b> <?php esc_html_e( '(Windows and Linux),', 'ungrabber' ); ?> <b>⌘+P</b> <?php esc_html_e( '(macOS)', 'ungrabber' ); ?>
            </label>
            <p class="description"><?php esc_html_e( 'Protect Your Page from Being Printed by HotKeys.', 'ungrabber' ); ?></p>
            <?php
            
        }

        /**
         * Render Developer Tool field.
         *
         * @since 1.0.0
         * @access public
         **/
        public function render_developer_tool() {
            
            ?>
            <label>
                <input type='checkbox' name='mdp_ungrabber_settings[developer_tool]' value='true' <?php checked( filter_var( $this->options['developer_tool'], FILTER_VALIDATE_BOOLEAN ) ); ?> >&nbsp;<?php esc_html_e( 'Disable HotKeys:', 'ungrabber' ); ?> <b>Ctrl+Shift+I</b> <?php esc_html_e( '(Windows and Linux),', 'ungrabber' ); ?> <b>⌘+⌥+I</b> <?php esc_html_e( '(macOS)', 'ungrabber' ); ?>
            </label>
            <p class="description"><?php esc_html_e( 'Disable to View Source Code of Page by Developer Tools.', 'ungrabber' ); ?></p>
            <?php
            
        }

        /**
         * Render Reader mode field.
         *
         * @since 1.0.0
         * @access public
         **/
        public function render_reader_mode() {

            ?>
            <label>
                <input type='checkbox' name='mdp_ungrabber_settings[reader_mode]' value='true' <?php checked( filter_var( $this->options['reader_mode'], FILTER_VALIDATE_BOOLEAN ) ); ?> >&nbsp;<?php esc_html_e( 'Disable Reader mode in Safari:', 'ungrabber' ); ?> <b>⌘+Shift+R</b> <?php esc_html_e( '(macOS)', 'ungrabber' ); ?>
            </label>
            <p class="description"><?php esc_html_e( 'Protect Your Text and Images from being copied in the Safari Reader mode.', 'ungrabber' ); ?></p>
            <?php

        }

        /**
         * Render Right Click field.
         *
         * @since 1.0.0
         * @access public
         **/
        public function render_right_click() {
            
            ?>
            <label>
                <input type='checkbox' name='mdp_ungrabber_settings[right_click]' value='true' <?php checked( filter_var( $this->options['right_click'], FILTER_VALIDATE_BOOLEAN ) ); ?> >&nbsp;<?php esc_html_e( 'Disable Mouse Right Click', 'ungrabber' ); ?> <b><span class="dashicons dashicons-menu"></span></b>
            </label>
            <p class="description"><?php esc_html_e( 'Protect Your Content from Being Copied by Context Menu.', 'ungrabber' ); ?></p>
            <?php
            
        }

        /**
         * Render Text Selection field.
         *
         * @since 1.0.0
         * @access public
         **/
        public function render_text_selection() {
            
            ?>
            <label>
                <input type='checkbox' name='mdp_ungrabber_settings[text_selection]' value='true' <?php checked( filter_var( $this->options['text_selection'], FILTER_VALIDATE_BOOLEAN ) ); ?> >&nbsp;<?php esc_html_e( 'Disable Text Selection', 'ungrabber' ); ?>
            </label>
            <p class="description"><?php esc_html_e( 'Disable Text Highlight (Text Selection) by Mouse.', 'ungrabber' ); ?></p>
            <?php
            
        }

        /**
         * Render Image Dragging field.
         *
         * @since 1.0.0
         * @access public
         **/
        public function render_image_dragging() {
            
            ?>
            <label>
                <input type='checkbox' name='mdp_ungrabber_settings[image_dragging]' value='true' <?php checked( filter_var( $this->options['image_dragging'], FILTER_VALIDATE_BOOLEAN ) ); ?> >&nbsp;<?php esc_html_e( 'Disable Image Dragging', 'ungrabber' ); ?> <b><span class="dashicons dashicons-move"></span></b>
            </label>
            <p class="description"><?php esc_html_e( 'Disable Image Dragging by Mouse.', 'ungrabber' ); ?></p>
            <?php
            
        }

        /**
         * Render JavaScript field.
         *
         * @since 1.0.0
         * @access public
         **/
        public function render_javascript() {
            
            ?>
            <label>
                <input type='checkbox' name='mdp_ungrabber_settings[javascript]' value='true' <?php checked( filter_var( $this->options['javascript'], FILTER_VALIDATE_BOOLEAN ) ); ?> >&nbsp;<?php esc_html_e( 'JavaScript Required', 'ungrabber' ); ?>
            </label>
            <p class="description"><?php esc_html_e( 'Protect Content if JavaScript is Disabled.', 'ungrabber' ); ?></p>
            <?php
            
        }
        
        /**
         * Render Content field.
         *
         * @since 1.0.0
         * @access public
         **/
        public function render_javascript_msg() {
            
            wp_editor( $this->options['javascript_msg'], 'mdpungrabbersettingsjavascriptmsg', array( 'textarea_rows' => 3, 'textarea_name' => 'mdp_ungrabber_settings[javascript_msg]' ) );
            ?><p class="description"><?php esc_html_e( 'Message to show if JavaScript is Disabled.', 'ungrabber' ); ?></p><?php
            
        }
        
        /**
         * Render Assignments field.
         * 
         * @since    1.0.0
         **/
        public function render_assignments() {

            $optval = get_option('mdp_ungrabber_assignments_settings');
            
            /**
            * Output options list for select
            */
            $options  = array();
            $defaults = array(
                'search'     => 'Search'
            );

            $selected = is_array($options) ? $options : array('*');

            if (count($selected) > 1 && in_array('*', $selected)) {
                $selected = array('*');
            }

            // set default options
            foreach ($defaults as $val => $label) {
                $attributes = in_array($val, $selected) ? array('value' => $val, 'selected' => 'selected') : array('value' => $val);
                $options[]  = sprintf('<option value="%s">%s</option>', $attributes["value"], $label);
            }

            // set pages
            if ($pages = get_pages()) {
                $options[] = '<optgroup label="Pages">';

                array_unshift($pages, (object) array('post_title' => 'Pages (All)'));

                foreach ($pages as $page) {
                    $val = isset($page->ID) ? 'page-'.$page->ID : 'page';
                    $attributes = in_array($val, $selected) ? array('value' => $val, 'selected' => 'selected') : array('value' => $val);
                    $options[]  = sprintf('<option value="%s">%s</option>', $attributes["value"], $page->post_title);
                }

                $options[] = '</optgroup>';
            }

            // set posts
            $options[] = '<optgroup label="Post">';
            foreach (array('home', 'single', 'archive') as $view) {
                $val = $view;
                $attributes = in_array($val, $selected) ? array('value' => $val, 'selected' => 'selected') : array('value' => $val);
                $options[] = sprintf('<option value="%s">%s (%s)</option>', $attributes["value"], 'Post', ucfirst($view));
            }
            $options[] = '</optgroup>';

            // set custom post types
            foreach (array_keys(get_post_types(array('_builtin' => false))) as $posttype) {
                $obj = get_post_type_object($posttype);
                $label = ucfirst($posttype);

                if ($obj->publicly_queryable) {
                    $options[] = '<optgroup label="'.$label.'">';

                    foreach (array('single', 'archive', 'search') as $view) {
                        $val = $posttype.'-'.$view;
                        $attributes = in_array($val, $selected) ? array('value' => $val, 'selected' => 'selected') : array('value' => $val);
                        $options[] = sprintf('<option value="%s">%s (%s)</option>', $attributes["value"], $label, ucfirst($view));
                    }

                    $options[] = '</optgroup>';
                }
            }

            // set categories
            foreach (array_keys(get_taxonomies()) as $tax) {

                if(in_array($tax, array("post_tag", "nav_menu"))) continue;

                if ($categories = get_categories(array( 'taxonomy' => $tax ))) {
                    $options[] = '<optgroup label="Categories ('.ucfirst(str_replace(array("_","-")," ",$tax)).')">';

                    foreach ($categories as $category) {
                        $val        = 'cat-'.$category->cat_ID;
                        $attributes = in_array($val, $selected) ? array('value' => $val, 'selected' => 'selected') : array('value' => $val);
                        $options[]  = sprintf('<option value="%s">%s</option>', $attributes["value"], $category->cat_name);
                    }

                    $options[] = '</optgroup>';
                }
            }
            
            // Set Default value 
            if( ! isset( $optval['mdp_ungrabber_assignments_field'] ) ) {
                //                                            {|matchingMethod|:1,|WPContent|:0,|WPContentVal|:||,|homePage|:0,|menuItems|:0,|menuItemsVal|:||,|userRolesVal|:||,|URL|:0,|URLVal|:||}
                $optval['mdp_ungrabber_assignments_field'] = "{|matchingMethod|:1,|WPContent|:0,|WPContentVal|:||,|homePage|:0,|menuItems|:0,|menuItemsVal|:||,|userRoles|:0,|userRolesVal|:||,|URL|:0,|URLVal|:||}";
            }
            ?>

            <input type='hidden' class="mdp-width-1-1" id="mdp-assignInput" name='mdp_ungrabber_assignments_settings[mdp_ungrabber_assignments_field]' value='<?php echo esc_attr( $optval['mdp_ungrabber_assignments_field'] ); ?>'>

            <div id="mdp-assign-box">

                <div class="mdp-all-fields">
                    <div class="mdp-alert"><?php esc_html_e('By selecting the specific assignments you can limit where plugin should or shouldn\'t be published. To have it published on all pages, simply do not specify any assignments.'); ?></div>
                    <div class="mdp-panel mdp-panel-box mdp-matching-method mdp-margin">
                        <h4><?php esc_html_e('Matching Method', 'ungrabber'); ?></h4>
                        <p><?php esc_html_e('Should all or any assignments be matched?', 'ungrabber'); ?></p>
                        <div class="mdp-button-group mdp-button-group mdp-matchingMethod" data-mdp-button-radio="">
                            <button class="mdp-button mdp-button-success mdp-button-small mdp-all mdp-active"><?php esc_html_e('All', 'ungrabber'); ?></button>
                            <button class="mdp-button mdp-button-success mdp-button-small mdp-any"><?php esc_html_e('ANY', 'ungrabber'); ?></button>
                        </div>
                        <p>
                            <strong><?php esc_html_e('All', 'ungrabber'); ?></strong><?php esc_html_e(' — Will be published if ', 'ungrabber'); ?><strong><?php esc_html_e('ALL', 'ungrabber'); ?></strong><?php esc_html_e(' of below assignments are matched.', 'ungrabber'); ?><br>
                            <strong><?php esc_html_e('ANY', 'ungrabber'); ?></strong><?php esc_html_e(' — Will be published if ', 'ungrabber'); ?><strong><?php esc_html_e('ANY', 'ungrabber'); ?></strong><?php esc_html_e(' (one or more) of below assignments are matched.', 'ungrabber'); ?><br>
                        </p>
                    </div>

                    <div class="mdp-panel mdp-panel-box mdp-wp-content mdp-margin">

                        <h4><?php esc_html_e('WordPress Content', 'ungrabber'); ?></h4>

                        <div class="mdp-button-group mdp-button-group" data-mdp-button-radio="">
                            <button class="mdp-button mdp-button-primary mdp-button-small mdp-ignore mdp-active"><?php esc_html_e('Ignore', 'ungrabber'); ?></button>
                            <button class="mdp-button mdp-button-success mdp-button-small mdp-include"><?php esc_html_e('Include', 'ungrabber'); ?></button>
                            <button class="mdp-button mdp-button-danger mdp-button-small mdp-exclude"><?php esc_html_e('Exclude', 'ungrabber'); ?></button>
                        </div>

                        <div class="mdp-wp-content-box">
                            <p class="mdp-margin-bottom-remove mdp-margin-top">
                                <?php esc_html_e('Select on what page types or categories the assignment should be active.', 'ungrabber'); ?>
                            </p>
                            <select class="wp-content chosen-select" multiple="multiple">
                                <option value=""></option>
                                <?php echo implode("", $options) ?>
                            </select>
                        </div>

                    </div>

                    <div class="mdp-panel mdp-panel-box mdp-home-page mdp-margin">

                        <h4><?php esc_html_e('Home Page', 'ungrabber'); ?></h4>

                        <div class="mdp-button-group mdp-button-group" data-mdp-button-radio="">
                            <button class="mdp-button mdp-button-primary mdp-button-small mdp-ignore mdp-active"><?php esc_html_e('Ignore', 'ungrabber'); ?></button>
                            <button class="mdp-button mdp-button-success mdp-button-small mdp-include"><?php esc_html_e('Include', 'ungrabber'); ?></button>
                            <button class="mdp-button mdp-button-danger mdp-button-small mdp-exclude"><?php esc_html_e('Exclude', 'ungrabber'); ?></button>
                        </div>

                    </div>

                    <div class="mdp-panel mdp-panel-box mdp-menu-items mdp-margin">
                        <h4><?php esc_html_e('Menu Items', 'ungrabber'); ?></h4>
                        <div class="mdp-button-group mdp-button-group" data-mdp-button-radio="">
                            <button class="mdp-button mdp-button-primary mdp-button-small mdp-ignore mdp-active"><?php esc_html_e('Ignore', 'ungrabber'); ?></button>
                            <button class="mdp-button mdp-button-success mdp-button-small mdp-include"><?php esc_html_e('Include', 'ungrabber'); ?></button>
                            <button class="mdp-button mdp-button-danger mdp-button-small mdp-exclude"><?php esc_html_e('Exclude', 'ungrabber'); ?></button>
                        </div>

                        <div class="mdp-menuitems-selection">
                            <p class="mdp-margin-bottom-remove mdp-margin-top"><?php esc_html_e('Select the menu items to assign to.', 'ungrabber'); ?></p>
                            <select class="menuitems chosen-select" multiple="">
                                    <option value=""></option>
                                    <?php

                                    /** Get all menus */
                                    $menus = get_terms('nav_menu');
                                    foreach ($menus as $menu) {
                                        ?><optgroup label="<?php echo esc_attr( $menu->name ); ?>"><?php
                                            $navMenuItems = wp_get_nav_menu_items($menu->slug);
                                            $menuTree = $this->menu_object_tree($navMenuItems);
                                            $this->printMenuTree($menuTree, $menu->slug, 0);
                                        ?></optgroup><?php
                                    }
                                    ?>
                            </select>
                        </div>

                    </div>
                    
                    <div class="mdp-panel mdp-panel-box mdp-user-roles mdp-margin">

                        <h4><?php esc_html_e( 'User Roles', 'ungrabber' ); ?></h4>
                        <div class="mdp-button-group mdp-button-group" data-mdp-button-radio="">
                            <button class="mdp-button mdp-button-primary mdp-button-small mdp-ignore mdp-active"><?php esc_html_e('Ignore', 'ungrabber'); ?></button>
                            <button class="mdp-button mdp-button-success mdp-button-small mdp-include"><?php esc_html_e('Include', 'ungrabber'); ?></button>
                            <button class="mdp-button mdp-button-danger mdp-button-small mdp-exclude"><?php esc_html_e('Exclude', 'ungrabber'); ?></button>
                        </div>

                        <div class="user-roles-box">
                            <p class="mdp-margin-remove-bottom mdp-margin-top"><?php esc_html_e( 'Select the user roles to assign to.', 'ungrabber' ); ?></p>
                            <select class="user-roles chosen-select" multiple="">
                                <option value=""></option>
                                <?php // Get user roles
                                $roles = get_editable_roles();
                                foreach ( $roles as $k => $role ) {
                                    ?><option value="<?php echo $k; ?>"><?php echo $role['name']; ?></option><?php
                                } ?>
                            </select>
                        </div>

                    </div>

                    <div class="mdp-panel mdp-panel-box mdp-url mdp-margin">

                        <h4><?php esc_html_e( 'URL', 'ungrabber' ); ?></h4>

                        <div class="mdp-button-group mdp-button-group" data-mdp-button-radio="">
                            <button class="mdp-button mdp-button-primary mdp-button-small mdp-ignore mdp-active"><?php esc_html_e('Ignore', 'ungrabber'); ?></button>
                            <button class="mdp-button mdp-button-success mdp-button-small mdp-include"><?php esc_html_e('Include', 'ungrabber'); ?></button>
                            <button class="mdp-button mdp-button-danger mdp-button-small mdp-exclude"><?php esc_html_e('Exclude', 'ungrabber'); ?></button>
                        </div>

                        <div class="mdp-url-box">
                            <p class="mdp-margin-bottom-remove mdp-margin-top">
                                <?php esc_html_e('Enter (part of) the URLs to match.', 'ungrabber'); ?><br>
                                <?php esc_html_e('Use a new line for each different match.', 'ungrabber'); ?>
                            </p>

                            <textarea class="mdp-url-field"></textarea>                       

                        </div>

                    </div>

                </div>

            </div>

            <?php   
        }
        
        /**
         * Get plugin settings with default values.
         *
         * @return array
         **/
        public function getOptions() {
            
            /** Options. */
            $options = get_option( 'mdp_ungrabber_settings' );
            
            /** Default values. */
            $defaults = array(
                'javascript_msg' => '<h3>' . esc_html__( 'Please Enable JavaScript in your Browser to Visit this Site.', 'ungrabber' ) . '<h3>', // JavaScript Required Message.
            );
            
            /** Default values for checkboxes. */
            if( $options === false ) {
                
                $defaults['select_all']     = 'true'; // Disable Select All.
                $defaults['сopy']           = 'true'; // Disable Copy.
                $defaults['cut']            = 'true'; // Disable Cut.
                $defaults['paste']          = 'false'; // Disable Paste.
                $defaults['save']           = 'true'; // Disable Save.
                $defaults['view_source']    = 'true'; // Disable View Source.
                $defaults['print_page']     = 'true'; // Disable Print Page.
                $defaults['developer_tool'] = 'true'; // Disable Developer Tool.
                $defaults['reader_mode']    = 'false'; // Disable Safari reader mode.
                $defaults['right_click']    = 'true'; // Disable Right Click.
                $defaults['text_selection'] = 'true'; // Disable Text Selection.
                $defaults['image_dragging'] = 'true'; // Disable Image Dragging.
                $defaults['javascript']     = 'false'; // JavaScript Required.
                
            } else {
                
                /** Set all to FALSE, if options have one, then wp_parse_args override false value. */
                $defaults['select_all']     = 'false'; // Disable Select All.
                $defaults['сopy']           = 'false'; // Disable Copy.
                $defaults['cut']            = 'false'; // Disable Cut.
                $defaults['paste']          = 'false'; // Disable Paste.
                $defaults['save']           = 'false'; // Disable Save.
                $defaults['view_source']    = 'false'; // Disable View Source.
                $defaults['print_page']     = 'false'; // Disable Print Page.
                $defaults['developer_tool'] = 'false'; // Disable Developer Tool.
                $defaults['reader_mode']    = 'false'; // Disable Safari reader mode.
                $defaults['right_click']    = 'false'; // Disable Right Click.
                $defaults['text_selection'] = 'false'; // Disable Text Selection.
                $defaults['image_dragging'] = 'false'; // Disable Image Dragging.
                $defaults['javascript']     = 'false'; // JavaScript Required.
                
            }

            $results = wp_parse_args( $options, $defaults );
            
            return $results;
        }

        /**
         * Loads the UnGrabber translated strings.
         *
         * @since 1.0.0
         * @access public
         **/
        public function load_textdomain() {
            load_plugin_textdomain( 'ungrabber', false, dirname( plugin_basename( __FILE__ ) ) . '/languages/' );
        }

        /**
         * Add "merkulov.design" and  "Envato Profile" links on plugin page.
         *
         * @since 1.0.0
         * @access public
         *
         * @param array $links Current links: Deactivate | Edit
         * @return array
         **/
        public function add_links($links) {

            array_unshift( $links, '<a title="' . esc_html__( 'Settings', 'ungrabber' ) . '" href="'. admin_url( 'options-general.php?page=mdp_ungrabber_settings' ) .'">' . esc_html__( 'Settings', 'ungrabber' ) . '</a>' );
            array_push( $links, '<a title="' . esc_html__( 'Documentation', 'ungrabber' ) . '" href="https://docs.merkulov.design/tag/ungrabber/" target="_blank">' . esc_html__( 'Documentation', 'ungrabber' ) . '</a>' );
            array_push( $links, '<a href="https://1.envato.market/cc-merkulove" target="_blank" class="cc-merkulove"><img src="data:image/svg+xml;base64,PD94bWwgdmVyc2lvbj0iMS4wIiBlbmNvZGluZz0iVVRGLTgiPz4KPHN2ZyB2aWV3Qm94PSIwIDAgMTE3Ljk5IDY3LjUxIiB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciPgo8ZGVmcz4KPHN0eWxlPi5jbHMtMSwuY2xzLTJ7ZmlsbDojMDA5ZWQ1O30uY2xzLTIsLmNscy0ze2ZpbGwtcnVsZTpldmVub2RkO30uY2xzLTN7ZmlsbDojMDA5ZWUyO308L3N0eWxlPgo8L2RlZnM+CjxjaXJjbGUgY2xhc3M9ImNscy0xIiBjeD0iMTUiIGN5PSI1Mi41MSIgcj0iMTUiLz4KPHBhdGggY2xhc3M9ImNscy0yIiBkPSJNMzAsMmgwQTE1LDE1LDAsMCwxLDUwLjQ4LDcuNUw3Miw0NC43NGExNSwxNSwwLDEsMS0yNiwxNUwyNC41LDIyLjVBMTUsMTUsMCwwLDEsMzAsMloiLz4KPHBhdGggY2xhc3M9ImNscy0zIiBkPSJNNzQsMmgwQTE1LDE1LDAsMCwxLDk0LjQ4LDcuNUwxMTYsNDQuNzRhMTUsMTUsMCwxLDEtMjYsMTVMNjguNSwyMi41QTE1LDE1LDAsMCwxLDc0LDJaIi8+Cjwvc3ZnPgo=" alt="' . esc_html__( 'Plugins', 'ungrabber' ) . '">' . esc_html__( 'Plugins', 'ungrabber' ) . '</a>');

            return $links;
        }
        
        /**
         * Return plugin slug.
         *
         * @since 1.0.0
         * @access public         
         **/
        public function get_slug() {

            $path_array = array_filter( explode ( '/', $this->url ) );
            $slug = end( $path_array );

            return $slug;
        }
        
        /**
         * Return plugin basename.
         *
         * @since 1.0.0
         * @access public         
         **/
        public function get_basename() {

            $basename = $this->get_slug() . '/' . basename(__FILE__);

            return $basename;
        }
        
        /**
         * Add "Rate us" link on plugin page.
         *
         * @since 1.0.0
         * @access public
         *
         * @param array $links Current links: Deactivate | Edit
         * @return array
         **/
        public function add_row_meta( $links, $file ) {
            
            if ( $this->get_basename() !== $file ) {
                return $links;
            }

            $links[] = esc_html__( 'Rate this plugin:', 'ungrabber' )
                    . "<span class='mdp-ungrabber-rating-stars'>"
                    . "     <a href='https://1.envato.market/cc-downloads' target='_blank'>"
                    . "         <span class='dashicons dashicons-star-filled'></span>"
                    . "     </a>"
                    
                    . "     <a href='https://1.envato.market/cc-downloads' target='_blank'>"
                    . "         <span class='dashicons dashicons-star-filled'></span>"
                    . "     </a>"
                    
                    . "     <a href='https://1.envato.market/cc-downloads' target='_blank'>"
                    . "         <span class='dashicons dashicons-star-filled'></span>"
                    . "     </a>"
                    
                    . "     <a href='https://1.envato.market/cc-downloads' target='_blank'>"
                    . "         <span class='dashicons dashicons-star-filled'></span>"
                    . "     </a>"
                    
                    . "     <a href='https://1.envato.market/cc-downloads' target='_blank'>"
                    . "         <span class='dashicons dashicons-star-filled'></span>"
                    . "     </a>"
                    . "<span>";

            return $links;
        }        
        
        /**
         * Add CSS for admin area.
         *
         * @since   1.0.0
         * @return void
         **/
        public function load_admin_styles( $hook ) {
            
            
            /** Add styles only on setting page */
            $screen = get_current_screen();
            
            if ( $screen->base == 'plugins' ) {
                
                wp_enqueue_style( 'mdp-ungrabber-plugin-page', $this->url . 'css/plugins' . $this->suffix . '.css', array(), self::VERSION );
                
            } elseif ( $screen->base == "settings_page_mdp_ungrabber_settings" ) {
                
                wp_enqueue_style( 'wp-color-picker' ); // Color Picker.
                wp_enqueue_style( 'uikit', $this->url . 'css/uikit' . $this->suffix . '.css', array(), self::VERSION );
                wp_enqueue_style( 'chosen', $this->url . 'css/chosen' . $this->suffix . '.css', array(), self::VERSION );
                wp_enqueue_style( 'mdp-ungrabber-admin', $this->url . 'css/admin' . $this->suffix . '.css', array(), self::VERSION );
                
            }
            
        }
        
        /**
         * Add JS for admin area.
         *
         * @since   1.0.0
         * @return void
         **/
        public function load_admin_scripts( $hook ) {
            
            /** Add styles only on setting page */
            $screen = get_current_screen();
            
            if ( $screen->base == 'plugins' ) {
                
                wp_enqueue_script( 'mdp-ungrabber-plugin-page', $this->url . 'js/plugin-page' . $this->suffix . '.js', array( 'jquery' ), self::VERSION, true );
                
            } elseif ( $screen->base == "settings_page_mdp_ungrabber_settings" ) {
                
                wp_enqueue_script( 'wp-color-picker' );
                wp_enqueue_script( 'wp-color-picker-alpha', $this->url . 'js/wp-color-picker-alpha.min.js', array( 'wp-color-picker' ), self::VERSION, true );
                wp_enqueue_script( 'uikit', $this->url . 'js/uikit' . $this->suffix . '.js', array( 'jquery' ), self::VERSION, true );
                wp_enqueue_script( 'chosen', $this->url . 'js/chosen.jquery' . $this->suffix . '.js', array( 'jquery' ), self::VERSION, true );
                wp_enqueue_script( 'mdp-ungrabber-admin', $this->url . 'js/admin' . $this->suffix . '.js', array( 'jquery', ), self::VERSION, true );
                
            }
        }
        
        /**
         * Checks if a plugin should work on current page.
         * 
         * @return boolean
         * 
         * @since 1.0.0
         * @access protected
         **/
        protected function display() {
            
            /** Get plugin options. */
            $options = get_option('mdp_ungrabber_assignments_settings');
            
            /** Assignments. */
            if( ! isset($options['mdp_ungrabber_assignments_field'] ) ) {
                $options['mdp_ungrabber_assignments_field'] = "{|matchingMethod|:1,|WPContent|:0,|WPContentVal|:||,|homePage|:0,|menuItems|:0,|menuItemsVal|:||,|URL|:0,|URLVal|:||}";
            }
            
            /** Get assignments for plugin. */
            $assignment = json_decode( str_replace('|', '"', $options['mdp_ungrabber_assignments_field']) );
            
            if( ! $assignment ) { return true; } // If no settings - Show plugin Everywhere

            /** WordPress Content. */
            $wordPressContent = $this->WordPressContent( $assignment );

            /** Home Page. */
            $homePage = $this->HomePage( $assignment );

            /** Menu Items. */
            $menuItems = $this->MenuItems( $assignment );
            
            /** User Roles. */
            $userRoles = $this->UserRoles( $assignment );

            /** URL. */
            $URL = $this->URL( $assignment );

            /** Matching Method. */
            $result = $this->MatchingMethod( $assignment, $wordPressContent, $homePage, $menuItems, $userRoles, $URL );

            return $result;
            
        }
        
        /**
         * Plugin assignments - WordPress Content.
         * 
         * @since 1.0.0
         * @access protected
         **/
        protected function WordPressContent($assignment) {

            $result = -1;

            switch ($assignment->WPContent) {
                case 0: // Ignore
                    $result = -1;
                    break;

                case 1: // Include
                    $result = FALSE;
                    if (!$assignment->WPContentVal) {
                        $result = -1;
                        return $result;
                    } // If no menu items - ignore

                    $query = $this->getQuery();
                    foreach ($query as $q) {
                        if (in_array($q, $assignment->WPContentVal)) {
                            $result = TRUE;
                            return $result;
                        }
                    }
                    break;

                case 2: // Exclude
                    $result = TRUE;
                    if (!$assignment->WPContentVal) {
                        $result = -1;
                        return $result;
                    } // If no menu items - ignore

                    $query = $this->getQuery();
                    foreach ($query as $q) {
                        if (in_array($q, $assignment->WPContentVal)) {
                            $result = FALSE;
                            return $result;
                        }
                    }
                    break;
            }

            return $result;
        }
        
        /**
         * Plugin assignments - Home Page.
         * 
         * @since 1.0.0
         * @access protected
         **/
        protected function HomePage($assignment) {
            switch ($assignment->homePage) {
                case 0: // Ignore
                    $result = -1;
                    break;

                case 1: // Include
                    $result = FALSE;
                    if (is_front_page()) {
                        $result = TRUE;
                    }
                    break;

                case 2: // Exclude
                    $result = TRUE;
                    if (is_front_page()) {
                        $result = FALSE;
                    }
                    break;
            }
            return $result;
        }

        /**
         * Plugin assignments - Menu Items.
         * 
         * @since 1.0.0
         * @access protected
         **/
        protected function MenuItems($assignment) {

            $result = -1;

            // If wrong input array - Ignore
            if (!is_array($assignment->menuItemsVal)) {
                $result = -1;
                return $result;
            }

            // Current URL
            $curUrl = "";
            if (!isset($_SERVER["HTTPS"]) || ($_SERVER["HTTPS"] != 'on')) {
                $curUrl = 'http://' . $_SERVER["SERVER_NAME"];
            } else {
                $curUrl = 'https://' . $_SERVER["SERVER_NAME"];
            }
            $curUrl .= $_SERVER["REQUEST_URI"];

            switch ($assignment->menuItems) {
                case 0: // Ignore
                    $result = -1;
                    break;

                case 1: // Include
                    $result = FALSE;
                    if (!$assignment->menuItemsVal) {
                        $result = -1;
                        return $result;
                    } // If no menu items - ignore

                    $menu_items_arr = array(); // Assignments menu items
                    foreach ($assignment->menuItemsVal as $val) {
                        if ($val == "") {
                            continue;
                        }
                        list($menuSlug, $menuItemID) = explode("+", $val);
                        $menu_items = wp_get_nav_menu_items($menuSlug);
                        $menu_item = wp_filter_object_list($menu_items, array('ID' => $menuItemID));
                        $menu_items_arr[] = reset($menu_item);
                    }

                    foreach ($menu_items_arr as $mItem) {
                        if ($curUrl == $mItem->url) {
                            $result = TRUE;
                            return $result;
                        }
                    }
                    break;

                case 2: // Exclude
                    $result = TRUE;
                    if (!$assignment->menuItemsVal) {
                        $result = -1;
                        return $result;
                    } // If no menu items - ignore

                    $menu_items_arr = array(); // Assignments menu items

                    foreach ($assignment->menuItemsVal as $val) {
                        list($menuSlug, $menuItemID) = explode("+", $val);
                        $menu_items = wp_get_nav_menu_items($menuSlug);
                        $menu_item = wp_filter_object_list($menu_items, array('ID' => $menuItemID));
                        $menu_items_arr[] = reset($menu_item);
                    }

                    foreach ($menu_items_arr as $mItem) {
                        if ($curUrl == $mItem->url) {
                            $result = FALSE;
                            return $result;
                        }
                    }
                    break;
            }
            
            return $result;
        }
        
        /**
         * Plugin assignments - User Roles.
         * 
         * @since 1.0.0
         * @access protected
         **/
        protected function UserRoles( $assignment ) {

            // If wrong input array - Ignore
            if (!is_array($assignment->userRolesVal)) {
                $result = -1;
                return $result;
            }

            switch ($assignment->userRoles) {
                case 0: // Ignore
                    $result = -1;
                    break;

                case 1: // Include
                    $result = FALSE;
                    $user = wp_get_current_user();
                    foreach ($user->roles as $role) {
                        if (in_array($role, $assignment->userRolesVal)) {
                            $result = TRUE;
                        }
                    }
                    break;

                case 2: // Exclude
                    $result = TRUE;
                    $user = wp_get_current_user();
                    foreach ($user->roles as $role) {
                        if (in_array($role, $assignment->userRolesVal)) {
                            $result = FALSE;
                        }
                    }
                    break;
            }
            return $result;
        }

        /**
         * Plugin assignments - URL.
         * 
         * @since 1.0.0
         * @access protected
         **/
        protected function URL($assignment) {

            // Current URL
            $curUrl = "";
            if (!isset($_SERVER["HTTPS"]) || ($_SERVER["HTTPS"] != 'on')) {
                $curUrl = 'http://' . $_SERVER["SERVER_NAME"];
            } else {
                $curUrl = 'https://' . $_SERVER["SERVER_NAME"];
            }
            $curUrl .= $_SERVER["REQUEST_URI"];

            $URLVal = (array) preg_split('/\r\n|[\r\n]/', $assignment->URLVal);
            $URLVal = array_filter($URLVal, function($value) {
                if (trim($value) != "") {
                    return $value;
                }
            });

            switch ($assignment->URL) {
                case 0: // Ignore
                    $result = -1;
                    break;

                case 1: // Include
                    $result = FALSE;
                    if (count($URLVal) == 0) {
                        $result = FALSE;
                    } // If no URLS to include - hide widget
                    foreach ($URLVal as $u) {
                        if (strpos($curUrl, $u) !== false) {
                            $result = TRUE;
                        }
                    }

                    break;

                case 2: // Exclude
                    $result = TRUE;
                    if (count($URLVal) == 0) {
                        $result = TRUE;
                    } // If no URLS to exclude - show widget
                    foreach ($URLVal as $u) {
                        if (strpos($curUrl, $u) !== false) {
                            $result = FALSE;
                        }
                    }
                    break;
            }
            return $result;
        }

        /**
         * Plugin assignments - Matching Method.
         * 
         * @since 1.0.0
         * @access protected
         **/
        protected function MatchingMethod($assignment, $wordPressContent, $homePage, $menuItems, $userRoles, $URL) {

            $arrCond = array(); // Add condition values

            // Ignore if -1 
            if ( $wordPressContent != -1 ) {
                $arrCond[] = $wordPressContent;
            }
            
            if ( $homePage != -1 ) {
                $arrCond[] = $homePage;
            }
            
            if ( $menuItems != -1 ) {
                $arrCond[] = $menuItems;
            }
            
            if ( $userRoles != -1 ) { 
                $arrCond[] = $userRoles;
            }
            
            if ( $URL != -1 ) {
                $arrCond[] = $URL;
            }

            if ( ! count( $arrCond ) ) {
                $arrCond[] = TRUE;
            } 
            
            // If all rules are Ignore - Show widget
            // Initialization
            $anytrue = false;
            $alltrue = true;

            // Processing
            foreach ($arrCond as $v) {
                $anytrue |= $v;
                $alltrue &= $v;
            }

            // Result
            if ($alltrue) {
                // All elements are TRUE
                $result = TRUE;
            } elseif (!$anytrue) {
                // All elements are FALSE
                $result = FALSE;
            } else {
                // Mixed values
                if ($assignment->matchingMethod == 0) { // ALL RULES
                    $result = FALSE;
                } else { // ANY OF RULES
                    $result = TRUE;
                }
            }

            return $result;
        }
        
        /**
         * Create tree view form menu items array.
         * 
         * @param type $nav_menu_items_array
         * @return type
         **/
        public function menu_object_tree($nav_menu_items_array) {
           foreach ($nav_menu_items_array as $key => $value) {
               $value->children = array();
               $nav_menu_items_array[$key] = $value;
           }

           $nav_menu_levels = array();
           $index = 0;
           if (!empty($nav_menu_items_array))
               do {
                   if ($index == 0) {
                       foreach ($nav_menu_items_array as $key => $obj) {
                           if ($obj->menu_item_parent == 0) {
                               $nav_menu_levels[$index][] = $obj;
                               unset($nav_menu_items_array[$key]);
                           }
                       }
                   } else {
                       foreach ($nav_menu_items_array as $key => $obj) {
                            if( is_array( $last_level_ids ) ){
                                if ( in_array( $obj->menu_item_parent, $last_level_ids ) ) {
                                    $nav_menu_levels[$index][] = $obj;
                                    unset($nav_menu_items_array[$key]);
                                }
                            }
                       }
                   }
                   $last_level_ids = wp_list_pluck($nav_menu_levels[$index], 'db_id');
                   $index++;
               } while (!empty($nav_menu_items_array));

           $nav_menu_levels_reverse = array_reverse($nav_menu_levels);

           $nav_menu_tree_build = array();
           $index = 0;
           if (!empty($nav_menu_levels_reverse))
               do {
                   if (count($nav_menu_levels_reverse) == 1) {
                       $nav_menu_tree_build = $nav_menu_levels_reverse;
                   }
                   $current_level = array_shift($nav_menu_levels_reverse);
                   if (isset($nav_menu_levels_reverse[$index])) {
                       $next_level = $nav_menu_levels_reverse[$index];
                       foreach ($next_level as $nkey => $nval) {
                           foreach ($current_level as $ckey => $cval) {
                               if ($nval->db_id == $cval->menu_item_parent) {
                                   $nval->children[] = $cval;
                               }
                           }
                       }
                   }
               } while (!empty($nav_menu_levels_reverse));

           $nav_menu_object_tree = $nav_menu_tree_build[0];
           return $nav_menu_object_tree;
        }       
       
        /**
         * Output options list for select.
         * 
         * @since    1.0.0
         **/
        public function printMenuTree($arr, $menuslug, $level = 0) {
            foreach ($arr as $item) {
                ?><option value="<?php echo esc_attr( $menuslug . "+" . $item->ID ); ?>"><?php echo str_repeat("-", $level) . " " . $item->title . " (" . $item->type_label . ")"; ?></option><?php
                if (count($item->children)) {
                    $this->printMenuTree($item->children, $menuslug, $level + 1);
                }
            }
        }
        
        /**
         * Get current query information.
         *
         * @global \WP_Query $wp_query
         *
         * @return string[]
         **/
        public function getQuery() {
            global $wp_query;

            // create, if not set
            if ( empty($this->query ) ) {

                // init vars
                $obj = $wp_query->get_queried_object();
                $type = get_post_type();
                $query = array();

                if (is_home()) {
                    $query[] = 'home';
                }

                if (is_front_page()) {
                    $query[] = 'front_page';
                }

                if ($type == 'post') {

                    if (is_single()) {
                        $query[] = 'single';
                    }

                    if (is_archive()) {
                        $query[] = 'archive';
                    }
                } else {
                    if (is_single()) {
                        $query[] = $type . '-single';
                    } elseif (is_archive()) {
                        $query[] = $type . '-archive';
                    }
                }

                if (is_search()) {
                    $query[] = 'search';
                }

                if (is_page()) {
                    $query[] = $type;
                    $query[] = $type . '-' . $obj->ID;
                }

                if (is_category()) {
                    $query[] = 'cat-' . $obj->term_id;
                }

                // WooCommerce
                include_once( ABSPATH.'wp-admin/includes/plugin.php' );
                if ( is_plugin_active( 'woocommerce/woocommerce.php' ) ) {

                    if (is_shop() && !is_search()) {
                        $query[] = 'page';
                        $query[] = 'page-' . wc_get_page_id('shop');
                    }

                    if (is_product_category() || is_product_tag()) {
                        $query[] = 'cat-' . $obj->term_id;
                    }
                }

                $this->query = $query;
            }

            return $this->query;
        }

    } // End Class UnGrabber.
endif; // End if class_exists check.

/** Run UnGrabber class. */
$UnGrabber = UnGrabber::get_instance();