<?php
/**
 * Payment Gateway API
 */
if ( ! defined( 'ABSPATH' ) )
    exit; // Exit if accessed directly

if ( ! class_exists( 'TC_Gateway_API' ) ) {

    class TC_Gateway_API {

        var $plugin_name = '';
        var $admin_name = '';
        var $public_name = '';
        var $method_img_url = '';
        var $admin_img_url = '';
        var $force_ssl = false;
        var $ipn_url;

        function cart_items( $sign = ', ' ) {

            global $tc;

            $session_order = $tc->session->get( 'tc_order' );
            $cart_contents = $this->cart_contents();
            $items = array();

            foreach ( $cart_contents as $item_id => $qty ) {
                $ticket = new TC_Ticket( $item_id );
                $items[] = sanitize_text_field( apply_filters( 'tc_cart_item_line', $ticket->details->post_title . ' x ' . $qty, $item_id, $qty, $this ) );
            }

            $items_result = sanitize_text_field( apply_filters( 'tc_cart_items_display', implode( apply_filters( 'tc_cart_items_sign', $sign ), $items ) ) );

            if ( !is_null( $session_order ) && apply_filters( 'tc_cart_items_prepend_order_id', false ) ) {
                $items_result = sprintf( __( 'Order #%s', 'tc' ), sanitize_text_field( $session_order ) ) . ': ' . $items_result;
            }

            return sanitize_text_field( apply_filters( 'tc_cart_items_display_result', $items_result ) );
        }

        function get_option( $option_name = '', $default_value = '', $gateway_name = false ) {

            $settings = get_option( 'tc_settings' );

            if ( false == $gateway_name ) {
                $gateway_name = $this->plugin_name;
            }

            if ( isset( $settings[ 'gateways' ][ $gateway_name ][ $option_name ] ) ) {

                if ( is_string( $settings[ 'gateways' ][ $gateway_name ][ $option_name ] ) ) {

                    if ( $settings[ 'gateways' ][ $gateway_name ][ $option_name ] ) {
                        $value = trim( $settings[ 'gateways' ][ $gateway_name ][ $option_name ] );

                    } else {
                        $value = trim( $default_value );
                    }

                } else {
                    $value = $settings[ 'gateways' ][ $gateway_name ][ $option_name ];
                }

            } else {
                $value = trim( $default_value );
            }

            return apply_filters( 'tc_gateway_option_value', $value, $gateway_name, $option_name, $default_value );
        }

        function get_global_currencies() {
            $settings = get_option( 'tc_settings' );
            $currencies = isset( $settings[ 'gateways' ][ 'currencies' ] ) ? $settings[ 'gateways' ][ 'currencies' ] : array();
            return $currencies;
        }

        function add_error( $error = '', $place = '' ) {
            global $tc;
            $tc->session->set( 'tc_gateway_error', $error );
            wp_redirect( $tc->get_payment_slug( true ) );
            exit;
        }

        function field_name( $field_name = '', $gateway_name = false ) {
            if ( $gateway_name == false ) {
                $gateway_name = $this->plugin_name;
            }
            return esc_attr( 'tc[gateways][' . $gateway_name . '][' . $field_name . ']' );
        }

        function on_creation() {}

        function init() {}

        function payment_form( $cart ) {}

        /**
         * Generate Order Confirmation Page upon success checkout
         *
         * @param $order
         * @param string $cart_info
         * @return string
         */
        function order_confirmation_message( $order, $cart_info = '' ) {
            global $tc;

            $order = tc_get_order_id_by_name( $order );

            $order = new TC_Order( $order->ID );

            $content = '';

            switch ( $order->details->post_status ) {

                case 'order_received':
                    $content .= '<p>' . sprintf( __( 'Your payment via %s for this order totaling <strong>%s</strong> is not yet complete.', 'tc' ), esc_html( $this->public_name ), tc_esc_html( apply_filters( 'tc_cart_currency_and_format', $order->details->tc_payment_info[ 'total' ] ) ) ) . '</p>';
                    $content .= '<p>' . __( 'Current order status:', 'tc' ) . ' <strong>' . __( 'Pending Payment', 'tc' ) . '</strong></p>';
                    break;

                case 'order_fraud':
                    $content .= '<p>' . __( 'Your payment is under review. We will back to you soon.', 'tc' ) . '</p>';
                    break;

                case 'order_paid':
                    $content .= '<p>' . sprintf( __( 'Your payment via %s for this order totaling <strong>%s</strong> is complete.', 'tc' ), esc_html( $this->public_name ), tc_esc_html( apply_filters( 'tc_cart_currency_and_format', $order->details->tc_payment_info[ 'total' ] ) ) ) . '</p>';
                    break;

                case 'order_cancelled':
                    $content .= '<p>' . sprintf( __( 'Your payment via %s for this order totaling <strong>%s</strong> is cancelled.', 'tc' ), esc_html( $this->public_name ), tc_esc_html( apply_filters( 'tc_cart_currency_and_format', $order->details->tc_payment_info[ 'total' ] ) ) ) . '</p>';
                    break;

                case 'order_refunded':
                    $content .= '<p>' . sprintf( __( 'Your payment via %s for this order totaling <strong>%s</strong> is refunded.', 'tc' ), esc_html( $this->public_name ), tc_esc_html( apply_filters( 'tc_cart_currency_and_format', $order->details->tc_payment_info[ 'total' ] ) ) ) . '</p>';
                    break;
            }

            $content = apply_filters( 'tc_order_confirmation_message_content_' . $this->plugin_name, $content );
            $content = apply_filters( 'tc_order_confirmation_message_content', $content, $order );

            $tc->remove_order_session_data();
            $tc->maybe_skip_confirmation_screen( $this, $order );
            return $content;
        }

        function maybe_start_session() {
            if ( ! session_id() || ! isset( $_SESSION ) ) {
                session_start();
            }
        }

        function is_payment_page() {
            global $wp;

            if ( isset( $wp->query_vars ) && ( array_key_exists( 'page_payment', $wp->query_vars ) || ( isset( $wp->query_vars[ 'pagename' ] ) && preg_match( '/' . tc_get_payment_page_slug() . '/', $wp->query_vars[ 'pagename' ], $matches, PREG_OFFSET_CAPTURE, 3 ) ) || ( isset( $wp->query_vars[ 'pagename' ] ) && $wp->query_vars[ 'pagename' ] == tc_get_payment_page_slug() ) ) ) {
                return true;

            } else {
                return false;
            }
        }

        function is_active() {

            $settings = get_option( 'tc_settings' );

            if ( ! isset( $settings[ 'gateways' ][ 'active' ] ) || ! is_array( $settings[ 'gateways' ][ 'active' ] ) ) {
                $settings[ 'gateways' ][ 'active' ] = array();
            }

            if ( in_array( $this->plugin_name, $settings[ 'gateways' ][ 'active' ] ) ) {
                return true;

            } else {
                return false;
            }
        }

        function subtotal() {
            global $tc;
            $session_cart_subtotal = $tc->session->get( 'tc_cart_subtotal' );
            return !is_null( $session_cart_subtotal ) ? (float) $session_cart_subtotal : 0;
        }

        function cart_info() {
            global $tc;
            $session_cart_info = $tc->session->get( 'cart_info' );
            return !is_null( $session_cart_info ) ? tc_sanitize_array2( $session_cart_info ) : [];
        }

        function buyer_info( $part ) {

            global $tc;
            $session = $tc->session->get();

            switch ( $part ) {

                case 'full_name':
                    $buyer_first_name = isset( $session[ 'cart_info' ][ 'buyer_data' ][ 'first_name_post_meta' ] ) ? sanitize_text_field( stripcslashes( $session[ 'cart_info' ][ 'buyer_data' ][ 'first_name_post_meta' ] ) ) : '';
                    $buyer_last_name = isset( $session[ 'cart_info' ][ 'buyer_data' ][ 'last_name_post_meta' ] ) ? sanitize_text_field( stripcslashes( $session[ 'cart_info' ][ 'buyer_data' ][ 'last_name_post_meta' ] ) ): '';
                    $info = $buyer_first_name . ' ' . $buyer_last_name;
                    break;

                case 'email':
                    $info = isset( $session[ 'cart_info' ][ 'buyer_data' ][ $part . '_post_meta' ] ) ? sanitize_email( stripcslashes( $session[ 'cart_info' ][ 'buyer_data' ][ $part . '_post_meta' ] ) ) : '';
                    break;

                default:
                    $info = isset( $session[ 'cart_info' ][ 'buyer_data' ][ $part . '_post_meta' ] ) ? sanitize_text_field( stripcslashes( $session[ 'cart_info' ][ 'buyer_data' ][ $part . '_post_meta' ] ) ) : '';
            }

            return $info;
        }

        function cart_contents() {
            global $tc;
            return $tc->get_cart_cookie();
        }

        function save_cart_info() {
            global $tc;
            $session = $tc->session->get();
            $session[ 'cart_info' ][ 'gateway' ] = $this->plugin_name;
            $session[ 'cart_info' ][ 'gateway_admin_name' ] = $this->admin_name;
            $session[ 'cart_info' ][ 'gateway_class' ] = get_class( $this );
            $tc->session->set( 'cart_info', $session[ 'cart_info' ] );
        }

        function save_payment_info( $payment_info_new = [] ) {

            global $tc;
            $payment_info = [];
            $payment_info[ 'gateway_public_name' ] = $this->public_name;
            $payment_info[ 'gateway_private_name' ] = $this->admin_name;
            $payment_info[ 'method' ] = $this->admin_name;
            $payment_info[ 'total' ] = $this->total();
            $payment_info[ 'subtotal' ] = $this->subtotal();
            $payment_info[ 'discount' ] = $this->discount();
            $payment_info[ 'fees_total' ] = $this->total_fees();
            $payment_info[ 'tax_total' ] = $this->total_taxes();
            $payment_info[ 'currency' ] = isset( $this->currency ) ? $this->currency : $tc->get_cart_currency();

            if ( ! empty( $payment_info_new ) ) {
                foreach ( $payment_info_new as $payment_info_key => $payment_info_value ) {
                    $payment_info[ $payment_info_key ] = $payment_info_value;
                }
            }

            $tc->session->set( 'tc_payment_info', $payment_info ); // Check Here
            return $payment_info;
        }

        function total() {

            global $tc, $cart_total;

            $session_discounted_total = $tc->session->get( 'discounted_total' );
            $discounted_total = !is_null( $session_discounted_total ) ? (float) $session_discounted_total : '';

            return ( isset( $discounted_total ) && is_numeric( $discounted_total ) )
                ? round( $discounted_total, 2 )
                : round( $cart_total, 2 );;
        }

        function discount() {
            global $tc;
            $session_discount_value_total = $tc->session->get( 'discount_value_total' );
            return !is_null( $session_discount_value_total ) ? (float) $session_discount_value_total : 0;
        }

        function total_fees() {
            global $tc;
            $session_total_fees = $tc->session->get( 'tc_total_fees' );
            return !is_null( $session_total_fees ) ? (float) $session_total_fees : 0;
        }

        function total_taxes() {
            global $tc;
            $session_tax_value = $tc->session->get( 'tc_tax_value' );
            return !is_null( $session_tax_value ) ? (float) $session_tax_value : 0;
        }

        function process_payment( $cart ) {
            wp_die( __( "You must override the process_payment() method in your {$this->admin_name} payment gateway plugin!", 'tc' ) );
        }

        function order_confirmation_email( $msg, $order ) {
            return $msg;
        }

        function gateway_admin_settings( $settings, $visible ) {}

        function ipn() {}

        function order_confirmation( $order, $payment_info = '', $cart_info = '' ) {}

        function _generate_ipn_url() {
            global $tc;

            if ( empty( $GLOBALS[ 'wp_rewrite' ] ) ) {
                $GLOBALS[ 'wp_rewrite' ] = new WP_Rewrite();
            }

            $this->ipn_url = trailingslashit( $tc->get_payment_gateway_return_slug( true ) ) . '?payment_gateway_return=' . $this->plugin_name;
        }

        function _generate_cancel_url() {

            global $tc;

            if ( empty( $GLOBALS[ 'wp_rewrite' ] ) ) {
                $GLOBALS[ 'wp_rewrite' ] = new WP_Rewrite();
            }

            if ( $tc->active_payment_gateways() == 1 && $this->skip_payment_screen == true ) {
                $this->cancel_url = $tc->get_cart_slug( true ) . '?' . $this->plugin_name . '_cancel';

            } else {
                $this->cancel_url = $tc->get_payment_slug( true ) . '?' . $this->plugin_name . '_cancel';
            }

            $this->cancel_slug = $this->plugin_name . '_cancel';
        }

        function _generate_failed_url() {

            global $tc;

            if ( empty( $GLOBALS[ 'wp_rewrite' ] ) ) {
                $GLOBALS[ 'wp_rewrite' ] = new WP_Rewrite();
            }

            if ( $tc->active_payment_gateways() == 1 && $this->skip_payment_screen == true ) {
                $this->failed_url = $tc->get_cart_slug( true ) . '?' . $this->plugin_name . '_failed';

            } else {
                $this->failed_url = $tc->get_payment_slug( true ) . '?' . $this->plugin_name . '_failed';

            }
            $this->failed_slug = $this->plugin_name . '_failed';
        }

        function _checkout_confirmation_hook() {
            global $tc, $wp_query;

            if ( $wp_query->query_vars[ 'pagename' ] == 'cart' ) {
                if ( isset( $wp_query->query_vars[ 'checkoutstep' ] ) && $wp_query->query_vars[ 'checkoutstep' ] == 'confirmation' ) {
                    $session_order = $tc->session->get( 'tc_order' );
                    $session_payment_method = $tc->session->get( 'tc_payment_method' );
                    do_action( 'tc_checkout_payment_pre_confirmation_' . sanitize_text_field( $session_payment_method ), $tc->get_order( sanitize_text_field( $session_order ) ) );
                }
            }
        }

        function isSSL() {

            if ( ! empty( $_SERVER[ 'https' ] ) ) {
                return true;
            }

            if ( ! empty( $_SERVER[ 'HTTP_X_FORWARDED_PROTO' ] ) && $_SERVER[ 'HTTP_X_FORWARDED_PROTO' ] == 'https' ) {
                return true;
            }

            return false;
        }

        function force_ssl() {
            if ( $this->is_payment_page() && $this->force_ssl && ! is_ssl() && $this->is_active() ) {
                if ( ! $this->isSSL() ) {
                    wp_redirect( 'https://' . $_SERVER[ 'HTTP_HOST' ] . $_SERVER[ 'REQUEST_URI' ] );
                    exit();
                }
            }
        }

        function show_cart_errors() {
            if ( isset( $_GET[ $this->cancel_slug ] ) || isset( $_GET[ $this->failed_slug ] ) ) {
                add_filter( 'tc_cart_errors', array( &$this, 'cart_error_content' ) );
            }
        }

        function cart_error_content( $content ) {
            $content = __( 'Your transaction has been canceled.', 'tc' );
            return $content;
        }

        /**
         * Return a json encoded country data
         * @return string
         */
        function get_country_data() {
            return '[{"id":"AFG","text":"Afghanistan | AFG"},{"id":"ALA","text":"\u00c5land Islands | ALA"},{"id":"ALB","text":"Albania | ALB"},{"id":"DZA","text":"Algeria | DZA"},{"id":"ASM","text":"American Samoa | ASM"},{"id":"AND","text":"Andorra | AND"},{"id":"AGO","text":"Angola | AGO"},{"id":"AIA","text":"Anguilla | AIA"},{"id":"ATA","text":"Antarctica | ATA"},{"id":"ATG","text":"Antigua and Barbuda | ATG"},{"id":"ARG","text":"Argentina | ARG"},{"id":"ARM","text":"Armenia | ARM"},{"id":"ABW","text":"Aruba | ABW"},{"id":"AUS","text":"Australia | AUS"},{"id":"AUT","text":"Austria | AUT"},{"id":"AZE","text":"Azerbaijan | AZE"},{"id":"BHS","text":"Bahamas | BHS"},{"id":"BHR","text":"Bahrain | BHR"},{"id":"BGD","text":"Bangladesh | BGD"},{"id":"BRB","text":"Barbados | BRB"},{"id":"BLR","text":"Belarus | BLR"},{"id":"BEL","text":"Belgium | BEL"},{"id":"BLZ","text":"Belize | BLZ"},{"id":"BEN","text":"Benin | BEN"},{"id":"BMU","text":"Bermuda | BMU"},{"id":"BTN","text":"Bhutan | BTN"},{"id":"BOL","text":"Bolivia | BOL"},{"id":"BES","text":"Bonaire, Sint Eustatius and Saba | BES"},{"id":"BIH","text":"Bosnia and Herzegovina | BIH"},{"id":"BWA","text":"Botswana | BWA"},{"id":"BVT","text":"Bouvet Island | BVT"},{"id":"BRA","text":"Brazil | BRA"},{"id":"IOT","text":"British Indian Ocean Territory | IOT"},{"id":"BRN","text":"Brunei Darussalam | BRN"},{"id":"BGR","text":"Bulgaria | BGR"},{"id":"BFA","text":"Burkina Faso | BFA"},{"id":"BDI","text":"Burundi | BDI"},{"id":"KHM","text":"Cambodia | KHM"},{"id":"CMR","text":"Cameroon | CMR"},{"id":"CAN","text":"Canada | CAN"},{"id":"CPV","text":"Cape Verde | CPV"},{"id":"CYM","text":"Cayman Islands | CYM"},{"id":"CAF","text":"Central African Republic | CAF"},{"id":"TCD","text":"Chad | TCD"},{"id":"CHL","text":"Chile | CHL"},{"id":"CHN","text":"China | CHN"},{"id":"CXR","text":"Christmas Island | CXR"},{"id":"CCK","text":"Cocos (Keeling) Islands | CCK"},{"id":"COL","text":"Colombia | COL"},{"id":"COM","text":"Comoros | COM"},{"id":"COG","text":"Congo, Republic of the (Brazzaville) | COG"},{"id":"COD","text":"Congo, the Democratic Republic of the (Kinshasa) | COD"},{"id":"COK","text":"Cook Islands | COK"},{"id":"CRI","text":"Costa Rica | CRI"},{"id":"CIV","text":"C\u00f4te d\'Ivoire, Republic of | CIV"},{"id":"HRV","text":"Croatia | HRV"},{"id":"CUB","text":"Cuba | CUB"},{"id":"CUW","text":"Cura\u00e7ao | CUW"},{"id":"CYP","text":"Cyprus | CYP"},{"id":"CZE","text":"Czech Republic | CZE"},{"id":"DNK","text":"Denmark | DNK"},{"id":"DJI","text":"Djibouti | DJI"},{"id":"DMA","text":"Dominica | DMA"},{"id":"DOM","text":"Dominican Republic | DOM"},{"id":"ECU","text":"Ecuador | ECU"},{"id":"EGY","text":"Egypt | EGY"},{"id":"SLV","text":"El Salvador | SLV"},{"id":"GNQ","text":"Equatorial Guinea | GNQ"},{"id":"ERI","text":"Eritrea | ERI"},{"id":"EST","text":"Estonia | EST"},{"id":"ETH","text":"Ethiopia | ETH"},{"id":"FLK","text":"Falkland Islands (Islas Malvinas) | FLK"},{"id":"FRO","text":"Faroe Islands | FRO"},{"id":"FJI","text":"Fiji | FJI"},{"id":"FIN","text":"Finland | FIN"},{"id":"FRA","text":"France | FRA"},{"id":"GUF","text":"French Guiana | GUF"},{"id":"PYF","text":"French Polynesia | PYF"},{"id":"ATF","text":"French Southern and Antarctic Lands | ATF"},{"id":"GAB","text":"Gabon | GAB"},{"id":"GMB","text":"Gambia, The | GMB"},{"id":"GEO","text":"Georgia | GEO"},{"id":"DEU","text":"Germany | DEU"},{"id":"GHA","text":"Ghana | GHA"},{"id":"GIB","text":"Gibraltar | GIB"},{"id":"GRC","text":"Greece | GRC"},{"id":"GRL","text":"Greenland | GRL"},{"id":"GRD","text":"Grenada | GRD"},{"id":"GLP","text":"Guadeloupe | GLP"},{"id":"GUM","text":"Guam | GUM"},{"id":"GTM","text":"Guatemala | GTM"},{"id":"GGY","text":"Guernsey | GGY"},{"id":"GIN","text":"Guinea | GIN"},{"id":"GNB","text":"Guinea-Bissau | GNB"},{"id":"GUY","text":"Guyana | GUY"},{"id":"HTI","text":"Haiti | HTI"},{"id":"HMD","text":"Heard Island and McDonald Islands | HMD"},{"id":"VAT","text":"Holy See (Vatican City) | VAT"},{"id":"HND","text":"Honduras | HND"},{"id":"HKG","text":"Hong Kong | HKG"},{"id":"HUN","text":"Hungary | HUN"},{"id":"ISL","text":"Iceland | ISL"},{"id":"IND","text":"India | IND"},{"id":"IDN","text":"Indonesia | IDN"},{"id":"IRN","text":"Iran, Islamic Republic of | IRN"},{"id":"IRQ","text":"Iraq | IRQ"},{"id":"IRL","text":"Ireland | IRL"},{"id":"IMN","text":"Isle of Man | IMN"},{"id":"ISR","text":"Israel | ISR"},{"id":"ITA","text":"Italy | ITA"},{"id":"JAM","text":"Jamaica | JAM"},{"id":"JPN","text":"Japan | JPN"},{"id":"JEY","text":"Jersey | JEY"},{"id":"JOR","text":"Jordan | JOR"},{"id":"KAZ","text":"Kazakhstan | KAZ"},{"id":"KEN","text":"Kenya | KEN"},{"id":"KIR","text":"Kiribati | KIR"},{"id":"PRK","text":"Korea, Democratic People\'s Republic of | PRK"},{"id":"KOR","text":"Korea, Republic of | KOR"},{"id":"KWT","text":"Kuwait | KWT"},{"id":"KGZ","text":"Kyrgyzstan | KGZ"},{"id":"LAO","text":"Laos | LAO"},{"id":"LVA","text":"Latvia | LVA"},{"id":"LBN","text":"Lebanon | LBN"},{"id":"LSO","text":"Lesotho | LSO"},{"id":"LBR","text":"Liberia | LBR"},{"id":"LBY","text":"Libya | LBY"},{"id":"LIE","text":"Liechtenstein | LIE"},{"id":"LTU","text":"Lithuania | LTU"},{"id":"LUX","text":"Luxembourg | LUX"},{"id":"MAC","text":"Macao | MAC"},{"id":"MKD","text":"Macedonia, Republic of | MKD"},{"id":"MDG","text":"Madagascar | MDG"},{"id":"MWI","text":"Malawi | MWI"},{"id":"MYS","text":"Malaysia | MYS"},{"id":"MDV","text":"Maldives | MDV"},{"id":"MLI","text":"Mali | MLI"},{"id":"MLT","text":"Malta | MLT"},{"id":"MHL","text":"Marshall Islands | MHL"},{"id":"MTQ","text":"Martinique | MTQ"},{"id":"MRT","text":"Mauritania | MRT"},{"id":"MUS","text":"Mauritius | MUS"},{"id":"MYT","text":"Mayotte | MYT"},{"id":"MEX","text":"Mexico | MEX"},{"id":"FSM","text":"Micronesia, Federated States of | FSM"},{"id":"MDA","text":"Moldova | MDA"},{"id":"MCO","text":"Monaco | MCO"},{"id":"MNG","text":"Mongolia | MNG"},{"id":"MNE","text":"Montenegro | MNE"},{"id":"MSR","text":"Montserrat | MSR"},{"id":"MAR","text":"Morocco | MAR"},{"id":"MOZ","text":"Mozambique | MOZ"},{"id":"MMR","text":"Myanmar | MMR"},{"id":"NAM","text":"Namibia | NAM"},{"id":"NRU","text":"Nauru | NRU"},{"id":"NPL","text":"Nepal | NPL"},{"id":"NLD","text":"Netherlands | NLD"},{"id":"NCL","text":"New Caledonia | NCL"},{"id":"NZL","text":"New Zealand | NZL"},{"id":"NIC","text":"Nicaragua | NIC"},{"id":"NER","text":"Niger | NER"},{"id":"NGA","text":"Nigeria | NGA"},{"id":"NIU","text":"Niue | NIU"},{"id":"NFK","text":"Norfolk Island | NFK"},{"id":"MNP","text":"Northern Mariana Islands | MNP"},{"id":"NOR","text":"Norway | NOR"},{"id":"OMN","text":"Oman | OMN"},{"id":"PAK","text":"Pakistan | PAK"},{"id":"PLW","text":"Palau | PLW"},{"id":"PSE","text":"Palestine, State of | PSE"},{"id":"PAN","text":"Panama | PAN"},{"id":"PNG","text":"Papua New Guinea | PNG"},{"id":"PRY","text":"Paraguay | PRY"},{"id":"PER","text":"Peru | PER"},{"id":"PHL","text":"Philippines | PHL"},{"id":"PCN","text":"Pitcairn | PCN"},{"id":"POL","text":"Poland | POL"},{"id":"PRT","text":"Portugal | PRT"},{"id":"PRI","text":"Puerto Rico | PRI"},{"id":"QAT","text":"Qatar | QAT"},{"id":"REU","text":"R\u00e9union | REU"},{"id":"ROU","text":"Romania | ROU"},{"id":"RUS","text":"Russian Federation | RUS"},{"id":"RWA","text":"Rwanda | RWA"},{"id":"BLM","text":"Saint Barth\u00e9lemy | BLM"},{"id":"SHN","text":"Saint Helena, Ascension and Tristan da Cunha | SHN"},{"id":"KNA","text":"Saint Kitts and Nevis | KNA"},{"id":"LCA","text":"Saint Lucia | LCA"},{"id":"MAF","text":"Saint Martin | MAF"},{"id":"SPM","text":"Saint Pierre and Miquelon | SPM"},{"id":"VCT","text":"Saint Vincent and the Grenadines | VCT"},{"id":"WSM","text":"Samoa | WSM"},{"id":"SMR","text":"San Marino | SMR"},{"id":"STP","text":"Sao Tome and Principe | STP"},{"id":"SAU","text":"Saudi Arabia | SAU"},{"id":"SEN","text":"Senegal | SEN"},{"id":"SRB","text":"Serbia | SRB"},{"id":"SYC","text":"Seychelles | SYC"},{"id":"SLE","text":"Sierra Leone | SLE"},{"id":"SGP","text":"Singapore | SGP"},{"id":"SXM","text":"Sint Maarten (Dutch part) | SXM"},{"id":"SVK","text":"Slovakia | SVK"},{"id":"SVN","text":"Slovenia | SVN"},{"id":"SLB","text":"Solomon Islands | SLB"},{"id":"SOM","text":"Somalia | SOM"},{"id":"ZAF","text":"South Africa | ZAF"},{"id":"SGS","text":"South Georgia and South Sandwich Islands | SGS"},{"id":"SSD","text":"South Sudan | SSD"},{"id":"ESP","text":"Spain | ESP"},{"id":"LKA","text":"Sri Lanka | LKA"},{"id":"SDN","text":"Sudan | SDN"},{"id":"SUR","text":"Suriname | SUR"},{"id":"SWZ","text":"Swaziland | SWZ"},{"id":"SWE","text":"Sweden | SWE"},{"id":"CHE","text":"Switzerland | CHE"},{"id":"SYR","text":"Syrian Arab Republic | SYR"},{"id":"TWN","text":"Taiwan | TWN"},{"id":"TJK","text":"Tajikistan | TJK"},{"id":"TZA","text":"Tanzania, United Republic of | TZA"},{"id":"THA","text":"Thailand | THA"},{"id":"TLS","text":"Timor-Leste | TLS"},{"id":"TGO","text":"Togo | TGO"},{"id":"TKL","text":"Tokelau | TKL"},{"id":"TON","text":"Tonga | TON"},{"id":"TTO","text":"Trinidad and Tobago | TTO"},{"id":"TUN","text":"Tunisia | TUN"},{"id":"TUR","text":"Turkey | TUR"},{"id":"TKM","text":"Turkmenistan | TKM"},{"id":"TCA","text":"Turks and Caicos Islands | TCA"},{"id":"TUV","text":"Tuvalu | TUV"},{"id":"UGA","text":"Uganda | UGA"},{"id":"UKR","text":"Ukraine | UKR"},{"id":"ARE","text":"United Arab Emirates | ARE"},{"id":"GBR","text":"United Kingdom | GBR"},{"id":"USA","text":"United States | USA"},{"id":"UMI","text":"United States Minor Outlying Islands | UMI"},{"id":"URY","text":"Uruguay | URY"},{"id":"UZB","text":"Uzbekistan | UZB"},{"id":"VUT","text":"Vanuatu | VUT"},{"id":"VEN","text":"Venezuela, Bolivarian Republic of | VEN"},{"id":"VNM","text":"Vietnam | VNM"},{"id":"VGB","text":"Virgin Islands, British | VGB"},{"id":"VIR","text":"Virgin Islands, U.S. | VIR"},{"id":"WLF","text":"Wallis and Futuna | WLF"},{"id":"ESH","text":"Western Sahara | ESH"},{"id":"YEM","text":"Yemen | YEM"},{"id":"ZMB","text":"Zambia | ZMB"},{"id":"ZWE","text":"Zimbabwe | ZWE"}]';
        }

        /**
         * Return a json encoded region data
         * @return string
         */
        function get_region_data() {
            return '[{"countryName":"Afghanistan","countryShortCode":"AFG","regions":[{"name":"Badakhshan","shortCode":"BDS"},{"name":"Badghis","shortCode":"BDG"},{"name":"Baghlan","shortCode":"BGL"},{"name":"Balkh","shortCode":"BAL"},{"name":"Bamyan","shortCode":"BAM"},{"name":"Daykundi","shortCode":"DAY"},{"name":"Farah","shortCode":"FRA"},{"name":"Faryab","shortCode":"FYB"},{"name":"Ghazni","shortCode":"GHA"},{"name":"Ghor","shortCode":"GHO"},{"name":"Helmand","shortCode":"HEL"},{"name":"Herat","shortCode":"HER"},{"name":"Jowzjan","shortCode":"JOW"},{"name":"Kabul","shortCode":"KAB"},{"name":"Kandahar","shortCode":"KAN"},{"name":"Kapisa","shortCode":"KAP"},{"name":"Khost","shortCode":"KHO"},{"name":"Kunar","shortCode":"KNR"},{"name":"Kunduz","shortCode":"KDZ"},{"name":"Laghman","shortCode":"LAG"},{"name":"Logar","shortCode":"LOW"},{"name":"Maidan Wardak","shortCode":"WAR"},{"name":"Nangarhar","shortCode":"NAN"},{"name":"Nimruz","shortCode":"NIM"},{"name":"Nuristan","shortCode":"NUR"},{"name":"Paktia","shortCode":"PIA"},{"name":"Paktika","shortCode":"PKA"},{"name":"Panjshir","shortCode":"PAN"},{"name":"Parwan","shortCode":"PAR"},{"name":"Samangan","shortCode":"SAM"},{"name":"Sar-e Pol","shortCode":"SAR"},{"name":"Takhar","shortCode":"TAK"},{"name":"Urozgan","shortCode":"ORU"},{"name":"Zabul","shortCode":"ZAB"}]},{"countryName":"\u00c5land Islands","countryShortCode":"ALA","regions":[{"name":"Br\u00e4nd\u00f6","shortCode":"BR"},{"name":"Ecker\u00f6","shortCode":"EC"},{"name":"Finstr\u00f6m","shortCode":"FN"},{"name":"F\u00f6gl\u00f6","shortCode":"FG"},{"name":"Geta","shortCode":"GT"},{"name":"Hammarland","shortCode":"HM"},{"name":"Jomala","shortCode":"JM"},{"name":"Kumlinge","shortCode":"KM"},{"name":"K\u00f6kar","shortCode":"KK"},{"name":"Lemland","shortCode":"LE"},{"name":"Lumparland","shortCode":"LU"},{"name":"Mariehamn","shortCode":"MH"},{"name":"Saltvik","shortCode":"SV"},{"name":"Sottunga","shortCode":"ST"},{"name":"Sund","shortCode":"SD"},{"name":"V\u00e5rd\u00f6","shortCode":"VR"}]},{"countryName":"Albania","countryShortCode":"ALB","regions":[{"name":"Berat","shortCode":"01"},{"name":"Dib\u00ebr","shortCode":"09"},{"name":"Durr\u00ebs","shortCode":"02"},{"name":"Elbasan","shortCode":"03"},{"name":"Fier","shortCode":"04"},{"name":"Gjirokast\u00ebr","shortCode":"05"},{"name":"Kor\u00e7\u00eb","shortCode":"06"},{"name":"Kuk\u00ebs","shortCode":"07"},{"name":"Lezh\u00eb","shortCode":"08"},{"name":"Shkod\u00ebr","shortCode":"10"},{"name":"Tirana","shortCode":"11"},{"name":"Vlor\u00eb","shortCode":"12"}]},{"countryName":"Algeria","countryShortCode":"DZA","regions":[{"name":"Adrar","shortCode":"01"},{"name":"A\u00efn Defla","shortCode":"44"},{"name":"A\u00efn T\u00e9mouchent","shortCode":"46"},{"name":"Algiers","shortCode":"16"},{"name":"Annaba","shortCode":"23"},{"name":"Batna","shortCode":"05"},{"name":"B\u00e9char","shortCode":"08"},{"name":"B\u00e9ja\u00efa","shortCode":"06"},{"name":"Biskra","shortCode":"07"},{"name":"Blida","shortCode":"09"},{"name":"Bordj Bou Arr\u00e9ridj","shortCode":"34"},{"name":"Bou\u00efra","shortCode":"10"},{"name":"Boumerd\u00e8s","shortCode":"35"},{"name":"Chlef","shortCode":"02"},{"name":"Constantine","shortCode":"25"},{"name":"Djelfa","shortCode":"17"},{"name":"El Bayadh","shortCode":"32"},{"name":"El Oued","shortCode":"39"},{"name":"El Tarf","shortCode":"36"},{"name":"Gharda\u00efa","shortCode":"47"},{"name":"Guelma","shortCode":"24"},{"name":"Illizi","shortCode":"33"},{"name":"Jijel","shortCode":"18"},{"name":"Khenchela","shortCode":"40"},{"name":"Laghouat","shortCode":"03"},{"name":"Mascara","shortCode":"29"},{"name":"M\u00e9d\u00e9a","shortCode":"26"},{"name":"Mila","shortCode":"43"},{"name":"Mostaganem","shortCode":"27"},{"name":"Msila","shortCode":"28"},{"name":"Na\u00e2ma","shortCode":"45"},{"name":"Oran","shortCode":"31"},{"name":"Ouargla","shortCode":"30"},{"name":"Oum el Bouaghi","shortCode":"04"},{"name":"Relizane","shortCode":"48"},{"name":"Sa\u00efda","shortCode":"20"},{"name":"S\u00e9tif","shortCode":"19"},{"name":"Sidi Bel Abb\u00e8s","shortCode":"22"},{"name":"Skikda","shortCode":"21"},{"name":"Souk Ahras","shortCode":"41"},{"name":"Tamanghasset","shortCode":"11"},{"name":"T\u00e9bessa","shortCode":"12"},{"name":"Tiaret","shortCode":"14"},{"name":"Tindouf","shortCode":"37"},{"name":"Tipaza","shortCode":"42"},{"name":"Tissemsilt","shortCode":"38"},{"name":"Tizi Ouzou","shortCode":"15"},{"name":"Tlemcen","shortCode":"13"}]},{"countryName":"American Samoa","countryShortCode":"ASM","regions":[{"name":"Tutuila","shortCode":"01"},{"name":"Aunu\'u","shortCode":"02"},{"name":"Ta\'\u016b","shortCode":"03"},{"name":"Ofu\u2011Olosega","shortCode":"04"},{"name":"Rose Atoll","shortCode":"21"},{"name":"Swains Island","shortCode":"22"}]},{"countryName":"Andorra","countryShortCode":"AND","regions":[{"name":"Andorra la Vella","shortCode":"07"},{"name":"Canillo","shortCode":"02"},{"name":"Encamp","shortCode":"03"},{"name":"Escaldes-Engordany","shortCode":"08"},{"name":"La Massana","shortCode":"04"},{"name":"Ordino","shortCode":"05"},{"name":"Sant Juli\u00e0 de L\u00f2ria","shortCode":"06"}]},{"countryName":"Angola","countryShortCode":"AGO","regions":[{"name":"Bengo","shortCode":"BGO"},{"name":"Benguela","shortCode":"BGU"},{"name":"Bi\u00e9","shortCode":"BIE"},{"name":"Cabinda","shortCode":"CAB"},{"name":"Cuando Cubango","shortCode":"CCU"},{"name":"Cuanza Norte","shortCode":"CNO"},{"name":"Cuanza Sul","shortCode":"CUS"},{"name":"Cunene","shortCode":"CNN"},{"name":"Huambo","shortCode":"HUA"},{"name":"Hu\u00edla","shortCode":"HUI"},{"name":"Luanda","shortCode":"LUA"},{"name":"Lunda Norte","shortCode":"LNO"},{"name":"Lunda Sul","shortCode":"LSU"},{"name":"Malanje","shortCode":"MAL"},{"name":"Moxico","shortCode":"MOX"},{"name":"Namibe","shortCode":"NAM"},{"name":"U\u00edge","shortCode":"UIG"},{"name":"Zaire","shortCode":"ZAI"}]},{"countryName":"Anguilla","countryShortCode":"AIA","regions":[{"name":"Anguilla","shortCode":"01"},{"name":"Anguillita Island","shortCode":"02"},{"name":"Blowing Rock","shortCode":"03"},{"name":"Cove Cay","shortCode":"04"},{"name":"Crocus Cay","shortCode":"05"},{"name":"Deadman\'s Cay","shortCode":"06"},{"name":"Dog Island","shortCode":"07"},{"name":"East Cay","shortCode":"08"},{"name":"Little Island","shortCode":"09"},{"name":"Little Scrub Island","shortCode":"10"},{"name":"Mid Cay","shortCode":"11"},{"name":"North Cay","shortCode":"12"},{"name":"Prickly Pear Cays","shortCode":"13"},{"name":"Rabbit Island","shortCode":"14"},{"name":"Sandy Island\/Sand Island","shortCode":"15"},{"name":"Scilly Cay","shortCode":"16"},{"name":"Scrub Island","shortCode":"17"},{"name":"Seal Island","shortCode":"18"},{"name":"Sombrero\/Hat Island","shortCode":"19"},{"name":"South Cay","shortCode":"20"},{"name":"South Wager Island","shortCode":"21"},{"name":"West Cay","shortCode":"22"}]},{"countryName":"Antarctica","countryShortCode":"ATA","regions":[{"name":"Antarctica","shortCode":"AQ"}]},{"countryName":"Antigua and Barbuda","countryShortCode":"ATG","regions":[{"name":"Antigua Island","shortCode":"01"},{"name":"Barbuda Island","shortCode":"02"},{"name":"Bird Island","shortCode":"04"},{"name":"Bishop Island","shortCode":"05"},{"name":"Blake Island","shortCode":"06"},{"name":"Crump Island","shortCode":"09"},{"name":"Dulcina Island","shortCode":"10"},{"name":"Exchange Island","shortCode":"11"},{"name":"Five Islands","shortCode":"12"},{"name":"Great Bird Island","shortCode":"13"},{"name":"Green Island","shortCode":"14"},{"name":"Guiana Island","shortCode":"15"},{"name":"Hawes Island","shortCode":"17"},{"name":"Hells Gate Island","shortCode":"16"},{"name":"Henry Island","shortCode":"18"},{"name":"Johnson Island","shortCode":"19"},{"name":"Kid Island","shortCode":"20"},{"name":"Lobster Island","shortCode":"22"},{"name":"Maiden Island","shortCode":"24"},{"name":"Moor Island","shortCode":"25"},{"name":"Nanny Island","shortCode":"26"},{"name":"Pelican Island","shortCode":"27"},{"name":"Prickly Pear Island","shortCode":"28"},{"name":"Rabbit Island","shortCode":"29"},{"name":"Red Head Island","shortCode":"31"},{"name":"Redonda Island","shortCode":"03"},{"name":"Sandy Island","shortCode":"32"},{"name":"Smith Island","shortCode":"33"},{"name":"The Sisters","shortCode":"34"},{"name":"Vernon Island","shortCode":"35"},{"name":"Wicked Will Island","shortCode":"36"},{"name":"York Island","shortCode":"37"}]},{"countryName":"Argentina","countryShortCode":"ARG","regions":[{"name":"Buenos Aires","shortCode":"B"},{"name":"Capital Federal","shortCode":"C"},{"name":"Catamarca","shortCode":"K"},{"name":"Chaco","shortCode":"H"},{"name":"Chubut","shortCode":"U"},{"name":"C\u00f3rdoba","shortCode":"X"},{"name":"Corrientes","shortCode":"W"},{"name":"Entre R\u00edos","shortCode":"E"},{"name":"Formosa","shortCode":"P"},{"name":"Jujuy","shortCode":"Y"},{"name":"La Pampa","shortCode":"L"},{"name":"La Rioja","shortCode":"F"},{"name":"Mendoza","shortCode":"M"},{"name":"Misiones","shortCode":"N"},{"name":"Neuqu\u00e9n","shortCode":"Q"},{"name":"R\u00edo Negro","shortCode":"R"},{"name":"Salta","shortCode":"A"},{"name":"San Juan","shortCode":"J"},{"name":"San Luis","shortCode":"D"},{"name":"Santa Cruz","shortCode":"Z"},{"name":"Santa Fe","shortCode":"S"},{"name":"Santiago del Estero","shortCode":"G"},{"name":"Tierra del Fuego","shortCode":"V"},{"name":"Tucum\u00e1n","shortCode":"T"}]},{"countryName":"Armenia","countryShortCode":"ARM","regions":[{"name":"Aragatsotn","shortCode":"AG"},{"name":"Ararat","shortCode":"AR"},{"name":"Armavir","shortCode":"AV"},{"name":"Gegharkunik","shortCode":"GR"},{"name":"Kotayk","shortCode":"KT"},{"name":"Lori","shortCode":"LO"},{"name":"Shirak","shortCode":"SH"},{"name":"Syunik","shortCode":"SU"},{"name":"Tavush","shortCode":"TV"},{"name":"Vayots Dzor","shortCode":"VD"},{"name":"Yerevan","shortCode":"ER"}]},{"countryName":"Aruba","countryShortCode":"ABW","regions":[{"name":"Aruba","shortCode":"AW"}]},{"countryName":"Australia","countryShortCode":"AUS","regions":[{"name":"Australian Capital Territory","shortCode":"ACT"},{"name":"New South Wales","shortCode":"NSW"},{"name":"Northern Territory","shortCode":"NT"},{"name":"Queensland","shortCode":"QLD"},{"name":"South Australia","shortCode":"SA"},{"name":"Tasmania","shortCode":"TAS"},{"name":"Victoria","shortCode":"VIC"},{"name":"Western Australia","shortCode":"WA"}]},{"countryName":"Austria","countryShortCode":"AUT","regions":[{"name":"Burgenland","shortCode":"1"},{"name":"K\u00e4rnten","shortCode":"2"},{"name":"Nieder\u00f6sterreich","shortCode":"3"},{"name":"Ober\u00f6sterreich","shortCode":"4"},{"name":"Salzburg","shortCode":"5"},{"name":"Steiermark","shortCode":"6"},{"name":"Tirol","shortCode":"7"},{"name":"Vorarlberg","shortCode":"8"},{"name":"Wien","shortCode":"9"}]},{"countryName":"Azerbaijan","countryShortCode":"AZE","regions":[{"name":"Ab\u015feron","shortCode":"ABS"},{"name":"A\u011fcab\u0259di","shortCode":"AGC"},{"name":"A\u011fdam","shortCode":"AGM"},{"name":"A\u011fda\u015f","shortCode":"AGS"},{"name":"A\u011fstafa","shortCode":"AGA"},{"name":"A\u011fsu","shortCode":"AGU"},{"name":"Astara","shortCode":"AST"},{"name":"Bak\u0131","shortCode":"BAK"},{"name":"Bab\u0259k","shortCode":"BAB"},{"name":"Balak\u0259n","shortCode":"BAL"},{"name":"B\u0259rd\u0259","shortCode":"BAR"},{"name":"Beyl\u0259qan","shortCode":"BEY"},{"name":"Bil\u0259suvar","shortCode":"BIL"},{"name":"C\u0259bray\u0131l","shortCode":"CAB"},{"name":"C\u0259lilabad","shortCode":"CAL"},{"name":"Culfa","shortCode":"CUL"},{"name":"Da\u015fk\u0259s\u0259n","shortCode":"DAS"},{"name":"F\u00fczuli","shortCode":"FUZ"},{"name":"G\u0259d\u0259b\u0259y","shortCode":"GAD"},{"name":"Goranboy","shortCode":"GOR"},{"name":"G\u00f6y\u00e7ay","shortCode":"GOY"},{"name":"G\u00f6yg\u00f6l","shortCode":"GYG"},{"name":"Hac\u0131qabul","shortCode":"HAC"},{"name":"\u0130mi\u015fli","shortCode":"IMI"},{"name":"\u0130smay\u0131ll\u0131","shortCode":"ISM"},{"name":"K\u0259lb\u0259c\u0259r","shortCode":"KAL"},{"name":"K\u01ddng\u01ddrli","shortCode":"KAN"},{"name":"K\u00fcrd\u0259mir","shortCode":"KUR"},{"name":"La\u00e7\u0131n","shortCode":"LAC"},{"name":"L\u0259nk\u0259ran","shortCode":"LAN"},{"name":"Lerik","shortCode":"LER"},{"name":"Masall\u0131","shortCode":"MAS"},{"name":"Neft\u00e7ala","shortCode":"NEF"},{"name":"O\u011fuz","shortCode":"OGU"},{"name":"Ordubad","shortCode":"ORD"},{"name":"Q\u0259b\u0259l\u0259","shortCode":"QAB"},{"name":"Qax","shortCode":"QAX"},{"name":"Qazax","shortCode":"QAZ"},{"name":"Qobustan","shortCode":"QOB"},{"name":"Quba","shortCode":"QBA"},{"name":"Qubadli","shortCode":"QBI"},{"name":"Qusar","shortCode":"QUS"},{"name":"Saatl\u0131","shortCode":"SAT"},{"name":"Sabirabad","shortCode":"SAB"},{"name":"\u015eabran","shortCode":"SBN"},{"name":"S\u0259d\u0259r\u0259k","shortCode":"SAD"},{"name":"\u015eahbuz","shortCode":"SAH"},{"name":"\u015e\u0259ki","shortCode":"SAK"},{"name":"Salyan","shortCode":"SAL"},{"name":"\u015eamax\u0131","shortCode":"SMI"},{"name":"\u015e\u0259mkir","shortCode":"SKR"},{"name":"Samux","shortCode":"SMX"},{"name":"\u015e\u0259rur","shortCode":"SAR"},{"name":"Siy\u0259z\u0259n","shortCode":"SIY"},{"name":"\u015eu\u015fa","shortCode":"SUS"},{"name":"T\u0259rt\u0259r","shortCode":"TAR"},{"name":"Tovuz","shortCode":"TOV"},{"name":"Ucar","shortCode":"UCA"},{"name":"Xa\u00e7maz","shortCode":"XAC"},{"name":"X\u0131z\u0131","shortCode":"XIZ"},{"name":"Xocal\u0131","shortCode":"XCI"},{"name":"Xocav\u0259nd","shortCode":"XVD"},{"name":"Yard\u0131ml\u0131","shortCode":"YAR"},{"name":"Yevlax","shortCode":"YEV"},{"name":"Z\u0259ngilan","shortCode":"ZAN"},{"name":"Zaqatala","shortCode":"ZAQ"},{"name":"Z\u0259rdab","shortCode":"ZAR"}]},{"countryName":"Bahamas","countryShortCode":"BHS","regions":[{"name":"Acklins Island","shortCode":"01"},{"name":"Berry Islands","shortCode":"22"},{"name":"Bimini","shortCode":"02"},{"name":"Black Point","shortCode":"23"},{"name":"Cat Island","shortCode":"03"},{"name":"Central Abaco","shortCode":"24"},{"name":"Crooked Island and Long Cay","shortCode":"28"},{"name":"East Grand Bahama","shortCode":"29"},{"name":"Exuma","shortCode":"04"},{"name":"Freeport","shortCode":"05"},{"name":"Fresh Creek","shortCode":"06"},{"name":"Governor\'s Harbour","shortCode":"07"},{"name":"Green Turtle Cay","shortCode":"08"},{"name":"Harbour Island","shortCode":"09"},{"name":"High Rock","shortCode":"10"},{"name":"Inagua","shortCode":"11"},{"name":"Kemps Bay","shortCode":"12"},{"name":"Long Island","shortCode":"13"},{"name":"Marsh Harbour","shortCode":"14"},{"name":"Mayaguana","shortCode":"15"},{"name":"Moore\u2019s Island","shortCode":"40"},{"name":"New Providence","shortCode":"16"},{"name":"Nichollstown and Berry Islands","shortCode":"17"},{"name":"North Abaco","shortCode":"42"},{"name":"North Andros","shortCode":"41"},{"name":"North Eleuthera","shortCode":"33"},{"name":"Ragged Island","shortCode":"18"},{"name":"Rock Sound","shortCode":"19"},{"name":"San Salvador and Rum Cay","shortCode":"20"},{"name":"Sandy Point","shortCode":"21"},{"name":"South Abaco","shortCode":"35"},{"name":"South Andros","shortCode":"36"},{"name":"South Eleuthera","shortCode":"37"},{"name":"West Grand Bahama","shortCode":"39"}]},{"countryName":"Bahrain","countryShortCode":"BHR","regions":[{"name":"Al Jan\u016bb\u012byah","shortCode":"14"},{"name":"Al Man\u0101mah","shortCode":"13"},{"name":"Al Mu\u1e29arraq","shortCode":"15"},{"name":"Al Wus\u0163\u00e1","shortCode":"16"},{"name":"Ash Sham\u0101l\u012byah","shortCode":"17"}]},{"countryName":"Bangladesh","countryShortCode":"BGD","regions":[{"name":"Barisal","shortCode":"A"},{"name":"Chittagong","shortCode":"B"},{"name":"Dhaka","shortCode":"C"},{"name":"Khulna","shortCode":"D"},{"name":"Mymensingh","shortCode":"M"},{"name":"Rajshahi","shortCode":"E"},{"name":"Rangpur","shortCode":"F"},{"name":"Sylhet","shortCode":"G"}]},{"countryName":"Barbados","countryShortCode":"BRB","regions":[{"name":"Christ Church","shortCode":"01"},{"name":"Saint Andrew","shortCode":"02"},{"name":"Saint George","shortCode":"03"},{"name":"Saint James","shortCode":"04"},{"name":"Saint John","shortCode":"05"},{"name":"Saint Joseph","shortCode":"06"},{"name":"Saint Lucy","shortCode":"07"},{"name":"Saint Michael","shortCode":"08"},{"name":"Saint Peter","shortCode":"09"},{"name":"Saint Philip","shortCode":"10"},{"name":"Saint Thomas","shortCode":"11"}]},{"countryName":"Belarus","countryShortCode":"BLR","regions":[{"name":"Brest voblast","shortCode":"BR"},{"name":"Gorod Minsk","shortCode":"HM"},{"name":"Homiel voblast","shortCode":"HO"},{"name":"Hrodna voblast","shortCode":"HR"},{"name":"Mahilyow voblast","shortCode":"MA"},{"name":"Minsk voblast","shortCode":"MI"},{"name":"Vitsebsk voblast","shortCode":"VI"}]},{"countryName":"Belgium","countryShortCode":"BEL","regions":[{"name":"Brussels","shortCode":"BRU"},{"name":"Flanders","shortCode":"VLG"},{"name":"Wallonia","shortCode":"WAL"}]},{"countryName":"Belize","countryShortCode":"BLZ","regions":[{"name":"Belize District","shortCode":"BZ"},{"name":"Cayo District","shortCode":"CY"},{"name":"Corozal District","shortCode":"CZL"},{"name":"Orange Walk District","shortCode":"OW"},{"name":"Stann Creek District","shortCode":"SC"},{"name":"Toledo District","shortCode":"TOL"}]},{"countryName":"Benin","countryShortCode":"BEN","regions":[{"name":"Alibori","shortCode":"AL"},{"name":"Atakora","shortCode":"AK"},{"name":"Atlantique","shortCode":"AQ"},{"name":"Borgou","shortCode":"BO"},{"name":"Collines Department","shortCode":"CO"},{"name":"Donga","shortCode":"DO"},{"name":"Kouffo","shortCode":"KO"},{"name":"Littoral Department","shortCode":"LI"},{"name":"Mono Department","shortCode":"MO"},{"name":"Ou\u00e9m\u00e9","shortCode":"OU"},{"name":"Plateau","shortCode":"PL"},{"name":"Zou","shortCode":"ZO"}]},{"countryName":"Bermuda","countryShortCode":"BMU","regions":[{"name":"City of Hamilton","shortCode":"03"},{"name":"Devonshire Parish","shortCode":"01"},{"name":"Hamilton Parish","shortCode":"02"},{"name":"Paget Parish","shortCode":"04"},{"name":"Pembroke Parish","shortCode":"05"},{"name":"Sandys Parish","shortCode":"08"},{"name":"Smith\'s Parish","shortCode":"09"},{"name":"Southampton Parish","shortCode":"10"},{"name":"St. George\'s Parish","shortCode":"07"},{"name":"Town of St. George","shortCode":"06"},{"name":"Warwick Parish","shortCode":"11"}]},{"countryName":"Bhutan","countryShortCode":"BTN","regions":[{"name":"Bumthang","shortCode":"33"},{"name":"Chhukha","shortCode":"12"},{"name":"Dagana","shortCode":"22"},{"name":"Gasa","shortCode":"GA"},{"name":"Haa","shortCode":"13"},{"name":"Lhuntse","shortCode":"44"},{"name":"Mongar","shortCode":"42"},{"name":"Paro","shortCode":"11"},{"name":"Pemagatshel","shortCode":"43"},{"name":"Punakha","shortCode":"23"},{"name":"Samdrup Jongkhar","shortCode":"45"},{"name":"Samtse","shortCode":"14"},{"name":"Sarpang","shortCode":"31"},{"name":"Thimphu","shortCode":"15"},{"name":"Trashigang","shortCode":"41"},{"name":"Trashiyangtse","shortCode":"TY"},{"name":"Trongsa","shortCode":"32"},{"name":"Tsirang","shortCode":"21"},{"name":"Wangdue Phodrang","shortCode":"24"},{"name":"Zhemgang","shortCode":"34"}]},{"countryName":"Bolivia","countryShortCode":"BOL","regions":[{"name":"Beni","shortCode":"B"},{"name":"Chuquisaca","shortCode":"H"},{"name":"Cochabamba","shortCode":"C"},{"name":"La Paz","shortCode":"L"},{"name":"Oruro","shortCode":"O"},{"name":"Pando","shortCode":"N"},{"name":"Potos\u00ed","shortCode":"P"},{"name":"Santa Cruz","shortCode":"S"},{"name":"Tarija","shortCode":"T"}]},{"countryName":"Bonaire, Sint Eustatius and Saba","countryShortCode":"BES","regions":[{"name":"Bonaire","shortCode":"BO"},{"name":"Saba Isand","shortCode":"SA"},{"name":"Sint Eustatius","shortCode":"SE"}]},{"countryName":"Bosnia and Herzegovina","countryShortCode":"BIH","regions":[{"name":"Br\u010dko Distrikt","shortCode":"BRC"},{"name":"Federacija Bosne i Hercegovine","shortCode":"BIH"},{"name":"Republika Srpska","shortCode":"SRP"}]},{"countryName":"Botswana","countryShortCode":"BWA","regions":[{"name":"Central","shortCode":"CE"},{"name":"Ghanzi","shortCode":"GH"},{"name":"Kgalagadi","shortCode":"KG"},{"name":"Kgatleng","shortCode":"KL"},{"name":"Kweneng","shortCode":"KW"},{"name":"North West","shortCode":"NW"},{"name":"North-East","shortCode":"NE"},{"name":"South East","shortCode":"SE"},{"name":"Southern","shortCode":"SO"}]},{"countryName":"Bouvet Island","countryShortCode":"BVT","regions":[{"name":"Bouvet Island","shortCode":"BV"}]},{"countryName":"Brazil","countryShortCode":"BRA","regions":[{"name":"Acre","shortCode":"AC"},{"name":"Alagoas","shortCode":"AL"},{"name":"Amap\u00e1","shortCode":"AP"},{"name":"Amazonas","shortCode":"AM"},{"name":"Bahia","shortCode":"BA"},{"name":"Cear\u00e1","shortCode":"CE"},{"name":"Distrito Federal","shortCode":"DF"},{"name":"Esp\u00edrito Santo","shortCode":"ES"},{"name":"Goi\u00e1s","shortCode":"GO"},{"name":"Maranh\u00e3o","shortCode":"MA"},{"name":"Mato Grosso","shortCode":"MT"},{"name":"Mato Grosso do Sul","shortCode":"MS"},{"name":"Minas Gerais","shortCode":"MG"},{"name":"Par\u00e1","shortCode":"PA"},{"name":"Para\u00edba","shortCode":"PB"},{"name":"Paran\u00e1","shortCode":"PR"},{"name":"Pernambuco","shortCode":"PE"},{"name":"Piau\u00ed","shortCode":"PI"},{"name":"Rio de Janeiro","shortCode":"RJ"},{"name":"Rio Grande do Norte","shortCode":"RN"},{"name":"Rio Grande do Sul","shortCode":"RS"},{"name":"Rond\u00f4nia","shortCode":"RO"},{"name":"Roraima","shortCode":"RR"},{"name":"Santa Catarina","shortCode":"SC"},{"name":"S\u00e3o Paulo","shortCode":"SP"},{"name":"Sergipe","shortCode":"SE"},{"name":"Tocantins","shortCode":"TO"}]},{"countryName":"British Indian Ocean Territory","countryShortCode":"IOT","regions":[{"name":"British Indian Ocean Territory","shortCode":"IO"}]},{"countryName":"Brunei Darussalam","countryShortCode":"BRN","regions":[{"name":"Belait","shortCode":"BE"},{"name":"Brunei Muara","shortCode":"BM"},{"name":"Temburong","shortCode":"TE"},{"name":"Tutong","shortCode":"TU"}]},{"countryName":"Bulgaria","countryShortCode":"BGR","regions":[{"name":"Blagoevgrad","shortCode":"01"},{"name":"Burgas","shortCode":"02"},{"name":"Dobrich","shortCode":"08"},{"name":"Gabrovo","shortCode":"07"},{"name":"Jambol","shortCode":"28"},{"name":"Khaskovo","shortCode":"26"},{"name":"Kjustendil","shortCode":"10"},{"name":"Kurdzhali","shortCode":"09"},{"name":"Lovech","shortCode":"11"},{"name":"Montana","shortCode":"12"},{"name":"Pazardzhik","shortCode":"13"},{"name":"Pernik","shortCode":"14"},{"name":"Pleven","shortCode":"15"},{"name":"Plovdiv","shortCode":"16"},{"name":"Razgrad","shortCode":"17"},{"name":"Ruse","shortCode":"18"},{"name":"Shumen","shortCode":"27"},{"name":"Silistra","shortCode":"19"},{"name":"Sliven","shortCode":"20"},{"name":"Smoljan","shortCode":"21"},{"name":"Sofija","shortCode":"23"},{"name":"Sofija-Grad","shortCode":"22"},{"name":"Stara Zagora","shortCode":"24"},{"name":"Turgovishhe","shortCode":"25"},{"name":"Varna","shortCode":"03"},{"name":"Veliko Turnovo","shortCode":"04"},{"name":"Vidin","shortCode":"05"},{"name":"Vraca","shortCode":"06"}]},{"countryName":"Burkina Faso","countryShortCode":"BFA","regions":[{"name":"Bal\u00e9","shortCode":"BAL"},{"name":"Bam\/Lake Bam","shortCode":"BAM"},{"name":"Banwa Province","shortCode":"BAN"},{"name":"Baz\u00e8ga","shortCode":"BAZ"},{"name":"Bougouriba","shortCode":"BGR"},{"name":"Boulgou Province","shortCode":"BLG"},{"name":"Boulkiemd\u00e9","shortCode":"BLK"},{"name":"Como\u00e9\/Komoe","shortCode":"COM"},{"name":"Ganzourgou Province","shortCode":"GAN"},{"name":"Gnagna","shortCode":"GNA"},{"name":"Gourma Province","shortCode":"GOU"},{"name":"Houet","shortCode":"HOU"},{"name":"Ioba","shortCode":"IOB"},{"name":"Kadiogo","shortCode":"KAD"},{"name":"K\u00e9n\u00e9dougou","shortCode":"KEN"},{"name":"Komondjari","shortCode":"KMD"},{"name":"Kompienga","shortCode":"KMP"},{"name":"Kossi Province","shortCode":"KOS"},{"name":"Koulp\u00e9logo","shortCode":"KOP"},{"name":"Kouritenga","shortCode":"KOT"},{"name":"Kourw\u00e9ogo","shortCode":"KOW"},{"name":"L\u00e9raba","shortCode":"LER"},{"name":"Loroum","shortCode":"LOR"},{"name":"Mouhoun","shortCode":"MOU"},{"name":"Namentenga","shortCode":"NAM"},{"name":"Naouri\/Nahouri","shortCode":"NAO"},{"name":"Nayala","shortCode":"NAY"},{"name":"Noumbiel","shortCode":"NOU"},{"name":"Oubritenga","shortCode":"OUB"},{"name":"Oudalan","shortCode":"OUD"},{"name":"Passor\u00e9","shortCode":"PAS"},{"name":"Poni","shortCode":"PON"},{"name":"Sangui\u00e9","shortCode":"SNG"},{"name":"Sanmatenga","shortCode":"SMT"},{"name":"S\u00e9no","shortCode":"SEN"},{"name":"Sissili","shortCode":"SIS"},{"name":"Soum","shortCode":"SOM"},{"name":"Sourou","shortCode":"SOR"},{"name":"Tapoa","shortCode":"TAP"},{"name":"Tui\/Tuy","shortCode":"TUI"},{"name":"Yagha","shortCode":"YAG"},{"name":"Yatenga","shortCode":"YAT"},{"name":"Ziro","shortCode":"ZIR"},{"name":"Zondoma","shortCode":"ZON"},{"name":"Zoundw\u00e9ogo","shortCode":"ZOU"}]},{"countryName":"Burundi","countryShortCode":"BDI","regions":[{"name":"Bubanza","shortCode":"BB"},{"name":"Bujumbura Mairie","shortCode":"BM"},{"name":"Bujumbura Rural","shortCode":"BL"},{"name":"Bururi","shortCode":"BR"},{"name":"Cankuzo","shortCode":"CA"},{"name":"Cibitoke","shortCode":"CI"},{"name":"Gitega","shortCode":"GI"},{"name":"Karuzi","shortCode":"KR"},{"name":"Kayanza","shortCode":"KY"},{"name":"Kirundo","shortCode":"KI"},{"name":"Makamba","shortCode":"MA"},{"name":"Muramvya","shortCode":"MU"},{"name":"Muyinga","shortCode":"MY"},{"name":"Mwaro","shortCode":"MW"},{"name":"Ngozi","shortCode":"NG"},{"name":"Rutana","shortCode":"RT"},{"name":"Ruyigi","shortCode":"RY"}]},{"countryName":"Cambodia","countryShortCode":"KHM","regions":[{"name":"Baat Dambang","shortCode":"2"},{"name":"Banteay Mean Chey","shortCode":"1"},{"name":"Kampong Chaam","shortCode":"3"},{"name":"Kampong Chhnang","shortCode":"4"},{"name":"Kampong Spueu","shortCode":"5"},{"name":"Kampong Thum","shortCode":"6"},{"name":"Kampot","shortCode":"7"},{"name":"Kandaal","shortCode":"8"},{"name":"Kaoh Kong","shortCode":"9"},{"name":"Kracheh","shortCode":"10"},{"name":"Krong Kaeb","shortCode":"23"},{"name":"Krong Pailin","shortCode":"24"},{"name":"Krong Preah Sihanouk","shortCode":"18"},{"name":"Mondol Kiri","shortCode":"11"},{"name":"Otdar Mean Chey","shortCode":"22"},{"name":"Phnom Penh","shortCode":"12"},{"name":"Pousaat","shortCode":"15"},{"name":"Preah Vihear","shortCode":"13"},{"name":"Prey Veaeng","shortCode":"14"},{"name":"Rotanah Kiri","shortCode":"16"},{"name":"Siem Reab","shortCode":"17"},{"name":"Stueng Treng","shortCode":"19"},{"name":"Svaay Rieng","shortCode":"20"},{"name":"Taakaev","shortCode":"21"},{"name":"Tbong Khmum","shortCode":"25"}]},{"countryName":"Cameroon","countryShortCode":"CMR","regions":[{"name":"Adamaoua","shortCode":"AD"},{"name":"Centre","shortCode":"CE"},{"name":"Est","shortCode":"ES"},{"name":"Extr\u00eame-Nord","shortCode":"EN"},{"name":"Littoral","shortCode":"LT"},{"name":"Nord","shortCode":"NO"},{"name":"Nord-Ouest","shortCode":"NW"},{"name":"Ouest","shortCode":"OU"},{"name":"Sud","shortCode":"SU"},{"name":"Sud-Ouest","shortCode":"SW"}]},{"countryName":"Canada","countryShortCode":"CAN","regions":[{"name":"Alberta","shortCode":"AB"},{"name":"British Columbia","shortCode":"BC"},{"name":"Manitoba","shortCode":"MB"},{"name":"New Brunswick","shortCode":"NB"},{"name":"Newfoundland and Labrador","shortCode":"NL"},{"name":"Northwest Territories","shortCode":"NT"},{"name":"Nova Scotia","shortCode":"NS"},{"name":"Nunavut","shortCode":"NU"},{"name":"Ontario","shortCode":"ON"},{"name":"Prince Edward Island","shortCode":"PE"},{"name":"Quebec","shortCode":"QC"},{"name":"Saskatchewan","shortCode":"SK"},{"name":"Yukon","shortCode":"YT"}]},{"countryName":"Cape Verde","countryShortCode":"CPV","regions":[{"name":"Boa Vista","shortCode":"BV"},{"name":"Brava","shortCode":"BR"},{"name":"Calheta de S\u00e3o Miguel","shortCode":"CS"},{"name":"Maio","shortCode":"MA"},{"name":"Mosteiros","shortCode":"MO"},{"name":"Pa\u00fal","shortCode":"PA"},{"name":"Porto Novo","shortCode":"PN"},{"name":"Praia","shortCode":"PR"},{"name":"Ribeira Brava","shortCode":"RB"},{"name":"Ribeira Grande","shortCode":"RG"},{"name":"Sal","shortCode":"SL"},{"name":"Santa Catarina","shortCode":"CA"},{"name":"Santa Cruz","shortCode":"CR"},{"name":"S\u00e3o Domingos","shortCode":"SD"},{"name":"S\u00e3o Filipe","shortCode":"SF"},{"name":"S\u00e3o Nicolau","shortCode":"SN"},{"name":"S\u00e3o Vicente","shortCode":"SV"},{"name":"Tarrafal","shortCode":"TA"},{"name":"Tarrafal de S\u00e3o Nicolau","shortCode":"TS"}]},{"countryName":"Cayman Islands","countryShortCode":"CYM","regions":[{"name":"Creek"},{"name":"Eastern"},{"name":"Midland"},{"name":"South Town"},{"name":"Spot Bay"},{"name":"Stake Bay"},{"name":"West End"},{"name":"Western"}]},{"countryName":"Central African Republic","countryShortCode":"CAF","regions":[{"name":"Bamingui-Bangoran","shortCode":"BB"},{"name":"Bangui","shortCode":"BGF"},{"name":"Basse-Kotto","shortCode":"BK"},{"name":"Haute-Kotto","shortCode":"HK"},{"name":"Haut-Mbomou","shortCode":"HM"},{"name":"K\u00e9mo","shortCode":"KG"},{"name":"Lobaye","shortCode":"LB"},{"name":"Mamb\u00e9r\u00e9-Kad\u00e9\u00ef","shortCode":"HS"},{"name":"Mbomou","shortCode":"MB"},{"name":"Nana-Grebizi","shortCode":"10"},{"name":"Nana-Mamb\u00e9r\u00e9","shortCode":"NM"},{"name":"Ombella-M\'Poko","shortCode":"MP"},{"name":"Ouaka","shortCode":"UK"},{"name":"Ouham","shortCode":"AC"},{"name":"Ouham P\u00e9nd\u00e9","shortCode":"OP"},{"name":"Sangha-Mba\u00e9r\u00e9","shortCode":"SE"},{"name":"Vakaga","shortCode":"VK"}]},{"countryName":"Chad","countryShortCode":"TCD","regions":[{"name":"Bahr el Ghazal","shortCode":"BG"},{"name":"Batha","shortCode":"BA"},{"name":"Borkou","shortCode":"BO"},{"name":"Chari-Baguirmi","shortCode":"CB"},{"name":"Ennedi-Est","shortCode":"EE"},{"name":"Ennedi-Ouest","shortCode":"EO"},{"name":"Gu\u00e9ra","shortCode":"GR"},{"name":"Hadjer Lamis","shortCode":"HL"},{"name":"Kanem","shortCode":"KA"},{"name":"Lac","shortCode":"LC"},{"name":"Logone Occidental","shortCode":"LO"},{"name":"Logone Oriental","shortCode":"LR"},{"name":"Mondoul","shortCode":"MA"},{"name":"Mayo-K\u00e9bbi-Est","shortCode":"ME"},{"name":"Moyen-Chari","shortCode":"MC"},{"name":"Ouaddai","shortCode":"OD"},{"name":"Salamat","shortCode":"SA"},{"name":"Sila","shortCode":"SI"},{"name":"Tandjil\u00e9","shortCode":"TA"},{"name":"Tibesti","shortCode":"TI"},{"name":"Ville de Ndjamena","shortCode":"ND"},{"name":"Wadi Fira","shortCode":"WF"}]},{"countryName":"Chile","countryShortCode":"CHL","regions":[{"name":"Ais\u00e9n del General Carlos Ib\u00e1\u00f1ez del Campo","shortCode":"AI"},{"name":"Antofagasta","shortCode":"AN"},{"name":"Araucan\u00eda","shortCode":"AR"},{"name":"Arica y Parinacota","shortCode":"AP"},{"name":"Atacama","shortCode":"AT"},{"name":"B\u00edo-B\u00edo","shortCode":"BI"},{"name":"Coquimbo","shortCode":"CO"},{"name":"Libertador General Bernardo O\'Higgins","shortCode":"LI"},{"name":"Los Lagos","shortCode":"LL"},{"name":"Los R\u00edos","shortCode":"LR"},{"name":"Magallanes y Antartica Chilena","shortCode":"MA"},{"name":"Marga-Marga","shortCode":""},{"name":"Maule","shortCode":"ML"},{"name":"\u00d1uble","shortCode":"NB"},{"name":"Regi\u00f3n Metropolitana de Santiago","shortCode":"RM"},{"name":"Tarapac\u00e1","shortCode":"TA"},{"name":"Valpara\u00edso","shortCode":"VS"}]},{"countryName":"China","countryShortCode":"CHN","regions":[{"name":"Anhui","shortCode":"34"},{"name":"Beijing","shortCode":"11"},{"name":"Chongqing","shortCode":"50"},{"name":"Fujian","shortCode":"35"},{"name":"Gansu","shortCode":"62"},{"name":"Guangdong","shortCode":"44"},{"name":"Guangxi","shortCode":"45"},{"name":"Guizhou","shortCode":"52"},{"name":"Hainan","shortCode":"46"},{"name":"Hebei","shortCode":"13"},{"name":"Heilongjiang","shortCode":"23"},{"name":"Henan","shortCode":"41"},{"name":"Hong Kong","shortCode":"91"},{"name":"Hubei","shortCode":"42"},{"name":"Hunan","shortCode":"43"},{"name":"Inner Mongolia","shortCode":"15"},{"name":"Jiangsu","shortCode":"32"},{"name":"Jiangxi","shortCode":"36"},{"name":"Jilin","shortCode":"22"},{"name":"Liaoning","shortCode":"21"},{"name":"Macau","shortCode":"92"},{"name":"Ningxia","shortCode":"64"},{"name":"Qinghai","shortCode":"63"},{"name":"Shaanxi","shortCode":"61"},{"name":"Shandong","shortCode":"37"},{"name":"Shanghai","shortCode":"31"},{"name":"Shanxi","shortCode":"14"},{"name":"Sichuan","shortCode":"51"},{"name":"Tianjin","shortCode":"12"},{"name":"Tibet","shortCode":"54"},{"name":"Xinjiang","shortCode":"65"},{"name":"Yunnan","shortCode":"53"},{"name":"Zhejiang","shortCode":"33"}]},{"countryName":"Christmas Island","countryShortCode":"CXR","regions":[{"name":"Christmas Island","shortCode":"CX"}]},{"countryName":"Cocos (Keeling) Islands","countryShortCode":"CCK","regions":[{"name":"Direction Island","shortCode":"DI"},{"name":"Home Island","shortCode":"HM"},{"name":"Horsburgh Island","shortCode":"HR"},{"name":"North Keeling Island","shortCode":"NK"},{"name":"South Island","shortCode":"SI"},{"name":"West Island","shortCode":"WI"}]},{"countryName":"Colombia","countryShortCode":"COL","regions":[{"name":"Amazonas","shortCode":"AMA"},{"name":"Antioquia","shortCode":"ANT"},{"name":"Arauca","shortCode":"ARA"},{"name":"Archipi\u00e9lago de San Andr\u00e9s","shortCode":"SAP"},{"name":"Atl\u00e1ntico","shortCode":"ATL"},{"name":"Bogot\u00e1 D.C.","shortCode":"DC"},{"name":"Bol\u00edvar","shortCode":"BOL"},{"name":"Boyac\u00e1","shortCode":"BOY"},{"name":"Caldas","shortCode":"CAL"},{"name":"Caquet\u00e1","shortCode":"CAQ"},{"name":"Casanare","shortCode":"CAS"},{"name":"Cauca","shortCode":"CAU"},{"name":"Cesar","shortCode":"CES"},{"name":"Choc\u00f3","shortCode":"CHO"},{"name":"C\u00f3rdoba","shortCode":"COR"},{"name":"Cundinamarca","shortCode":"CUN"},{"name":"Guain\u00eda","shortCode":"GUA"},{"name":"Guaviare","shortCode":"GUV"},{"name":"Huila","shortCode":"HUI"},{"name":"La Guajira","shortCode":"LAG"},{"name":"Magdalena","shortCode":"MAG"},{"name":"Meta","shortCode":"MET"},{"name":"Nari\u00f1o","shortCode":"NAR"},{"name":"Norte de Santander","shortCode":"NSA"},{"name":"Putumayo","shortCode":"PUT"},{"name":"Quind\u00edo","shortCode":"QUI"},{"name":"Risaralda","shortCode":"RIS"},{"name":"Santander","shortCode":"SAN"},{"name":"Sucre","shortCode":"SUC"},{"name":"Tolima","shortCode":"TOL"},{"name":"Valle del Cauca","shortCode":"VAC"},{"name":"Vaup\u00e9s","shortCode":"VAU"},{"name":"Vichada","shortCode":"VID"}]},{"countryName":"Comoros","countryShortCode":"COM","regions":[{"name":"Andjaz\u00eedja","shortCode":"G"},{"name":"Andjou\u00e2n","shortCode":"A"},{"name":"Mo\u00fbh\u00eel\u00ee","shortCode":"M"}]},{"countryName":"Congo, Republic of the (Brazzaville)","countryShortCode":"COG","regions":[{"name":"Bouenza","shortCode":"11"},{"name":"Brazzaville","shortCode":"BZV"},{"name":"Cuvette","shortCode":"8"},{"name":"Cuvette-Ouest","shortCode":"15"},{"name":"Kouilou","shortCode":"5"},{"name":"L\u00e9koumou","shortCode":"2"},{"name":"Likouala","shortCode":"7"},{"name":"Niari","shortCode":"9"},{"name":"Plateaux","shortCode":"14"},{"name":"Pointe-Noire","shortCode":"16"},{"name":"Pool","shortCode":"12"},{"name":"Sangha","shortCode":"13"}]},{"countryName":"Congo, the Democratic Republic of the (Kinshasa)","countryShortCode":"COD","regions":[{"name":"Bandundu","shortCode":"BN"},{"name":"Bas-Congo","shortCode":"BC"},{"name":"\u00c9quateur","shortCode":"EQ"},{"name":"Kasa\u00ef-Occidental","shortCode":"KE"},{"name":"Kasa\u00ef-Oriental","shortCode":"KW"},{"name":"Katanga","shortCode":"KA"},{"name":"Kinshasa","shortCode":"KN"},{"name":"Maniema","shortCode":"MA"},{"name":"Nord-Kivu","shortCode":"NK"},{"name":"Orientale","shortCode":"OR"},{"name":"Sud-Kivu","shortCode":"SK"}]},{"countryName":"Cook Islands","countryShortCode":"COK","regions":[{"name":"Aitutaki"},{"name":"Atiu"},{"name":"Avarua"},{"name":"Mangaia"},{"name":"Manihiki"},{"name":"Ma\'uke"},{"name":"Mitiaro"},{"name":"Nassau"},{"name":"Palmerston"},{"name":"Penrhyn"},{"name":"Pukapuka"},{"name":"Rakahanga"}]},{"countryName":"Costa Rica","countryShortCode":"CRI","regions":[{"name":"Alajuela","shortCode":"2"},{"name":"Cartago","shortCode":"3"},{"name":"Guanacaste","shortCode":"5"},{"name":"Heredia","shortCode":"4"},{"name":"Lim\u00f3n","shortCode":"7"},{"name":"Puntarenas","shortCode":"6"},{"name":"San Jos\u00e9","shortCode":"1"}]},{"countryName":"C\u00f4te d\'Ivoire, Republic of","countryShortCode":"CIV","regions":[{"name":"Agn\u00e9by","shortCode":"16"},{"name":"Bafing","shortCode":"17"},{"name":"Bas-Sassandra","shortCode":"09"},{"name":"Dengu\u00e9l\u00e9","shortCode":"10"},{"name":"Dix-Huit Montagnes","shortCode":"06"},{"name":"Fromager","shortCode":"18"},{"name":"Haut-Sassandra","shortCode":"02"},{"name":"Lacs","shortCode":"07"},{"name":"Lagunes","shortCode":"01"},{"name":"Marahou\u00e9","shortCode":"12"},{"name":"Moyen-Cavally","shortCode":"19"},{"name":"Moyen-Como\u00e9","shortCode":"05"},{"name":"N\'zi-Como\u00e9","shortCode":"11"},{"name":"Savanes","shortCode":"03"},{"name":"Sud-Bandama","shortCode":"15"},{"name":"Sud-Como\u00e9","shortCode":"13"},{"name":"Vall\u00e9e du Bandama","shortCode":"04"},{"name":"Worodougou","shortCode":"14"},{"name":"Zanzan","shortCode":"08"}]},{"countryName":"Croatia","countryShortCode":"HRV","regions":[{"name":"Bjelovarsko-Bilogorska \u017dupanija","shortCode":"07"},{"name":"Brodsko-Posavska \u017dupanija","shortCode":"12"},{"name":"Dubrova\u010dko-Neretvanska \u017dupanija","shortCode":"19"},{"name":"Grad Zagreb","shortCode":"21"},{"name":"Istarska \u017dupanija","shortCode":"18"},{"name":"Karlova\u010dka \u017dupanija","shortCode":"04"},{"name":"Koprivni\u010dko-Krizeva\u010dka \u017dupanija","shortCode":"06"},{"name":"Krapinsko-Zagorska \u017dupanija","shortCode":"02"},{"name":"Li\u010dko-Senjska \u017dupanija","shortCode":"09"},{"name":"Me\u0111imurska \u017dupanija","shortCode":"20"},{"name":"Osje\u010dko-Baranjska \u017dupanija","shortCode":"14"},{"name":"Po\u017ee\u0161ko-Slavonska \u017dupanija","shortCode":"11"},{"name":"Primorsko-Goranska \u017dupanija","shortCode":"08"},{"name":"Sisa\u010dko-Moslava\u010dka \u017dupanija","shortCode":"03"},{"name":"Splitsko-Dalmatinska \u017dupanija","shortCode":"17"},{"name":"Sibensko-Kninska \u017dupanija","shortCode":"15"},{"name":"Vara\u017edinska \u017dupanija","shortCode":"05"},{"name":"Viroviti\u010dko-Podravska \u017dupanija","shortCode":"10"},{"name":"Vukovarsko-Srijemska \u017dupanija","shortCode":"16"},{"name":"Zadarska \u017dupanija","shortCode":"13"},{"name":"Zagrebacka Zupanija","shortCode":"01"}]},{"countryName":"Cuba","countryShortCode":"CUB","regions":[{"name":"Artemisa","shortCode":"15"},{"name":"Camag\u00fcey","shortCode":"09"},{"name":"Ciego de \u00c1vila","shortCode":"08"},{"name":"Cienfuegos","shortCode":"06"},{"name":"Granma","shortCode":"12"},{"name":"Guant\u00e1namo","shortCode":"14"},{"name":"Holgu\u00edn","shortCode":"11"},{"name":"Isla de la Juventud","shortCode":"99"},{"name":"La Habana","shortCode":"03"},{"name":"Las Tunas","shortCode":"10"},{"name":"Matanzas","shortCode":"04"},{"name":"Mayabeque","shortCode":"16"},{"name":"Pinar del R\u00edo","shortCode":"01"},{"name":"Sancti Sp\u00edritus","shortCode":"07"},{"name":"Santiago de Cuba","shortCode":"13"},{"name":"Villa Clara","shortCode":"05"}]},{"countryName":"Cura\u00e7ao","countryShortCode":"CUW","regions":[{"name":"Cura\u00e7ao","shortCode":"CW"}]},{"countryName":"Cyprus","countryShortCode":"CYP","regions":[{"name":"Ammochostos","shortCode":"04"},{"name":"Keryneia","shortCode":"05"},{"name":"Larnaka","shortCode":"03"},{"name":"Lefkosia","shortCode":"01"},{"name":"Lemesos","shortCode":"02"},{"name":"Pafos","shortCode":"05"}]},{"countryName":"Czech Republic","countryShortCode":"CZE","regions":[{"name":"Hlavn\u00ed m\u011bsto Praha","shortCode":"PR"},{"name":"Jiho\u010desk\u00fd kraj","shortCode":"JC"},{"name":"Jihomoravsk\u00fd kraj","shortCode":"JM"},{"name":"Karlovarsk\u00fd kraj","shortCode":"KA"},{"name":"Kr\u00e1lov\u00e9hradeck\u00fd kraj","shortCode":"KR"},{"name":"Libereck\u00fd kraj","shortCode":"LI"},{"name":"Moravskoslezsk\u00fd kraj","shortCode":"MO"},{"name":"Olomouck\u00fd kraj","shortCode":"OL"},{"name":"Pardubick\u00fd kraj","shortCode":"PA"},{"name":"Plze\u0148sk\u00fd kraj","shortCode":"PL"},{"name":"St\u0159edo\u010desk\u00fd kraj","shortCode":"ST"},{"name":"\u00dasteck\u00fd kraj","shortCode":"US"},{"name":"Vyso\u010dina","shortCode":"VY"},{"name":"Zl\u00ednsk\u00fd kraj","shortCode":"ZL"}]},{"countryName":"Denmark","countryShortCode":"DNK","regions":[{"name":"Hovedstaden","shortCode":"84"},{"name":"Kujalleq","shortCode":"GL-KU"},{"name":"Midtjylland","shortCode":"82"},{"name":"Norder\u00f8erne","shortCode":"FO-01"},{"name":"Nordjylland","shortCode":"81"},{"name":"\u00d8ster\u00f8","shortCode":"FO-06"},{"name":"Qaasuitsup","shortCode":"GL-QA"},{"name":"Qeqqata","shortCode":"GL-QE"},{"name":"Sand\u00f8","shortCode":"FO-02"},{"name":"Sermersooq","shortCode":"GL-SM"},{"name":"Sj\u00e6lland","shortCode":"85"},{"name":"Str\u00f8m\u00f8","shortCode":"FO-03"},{"name":"Suder\u00f8","shortCode":"FO-04"},{"name":"Syddanmark","shortCode":"83"},{"name":"V\u00e5g\u00f8","shortCode":"FO-05"}]},{"countryName":"Djibouti","countryShortCode":"DJI","regions":[{"name":"Ali Sabieh","shortCode":"AS"},{"name":"Arta","shortCode":"AR"},{"name":"Dikhil","shortCode":"DI"},{"name":"Obock","shortCode":"OB"},{"name":"Tadjourah","shortCode":"TA"}]},{"countryName":"Dominica","countryShortCode":"DMA","regions":[{"name":"Saint Andrew Parish","shortCode":"02"},{"name":"Saint David Parish","shortCode":"03"},{"name":"Saint George Parish","shortCode":"04"},{"name":"Saint John Parish","shortCode":"05"},{"name":"Saint Joseph Parish","shortCode":"06"},{"name":"Saint Luke Parish","shortCode":"07"},{"name":"Saint Mark Parish","shortCode":"08"},{"name":"Saint Patrick Parish","shortCode":"09"},{"name":"Saint Paul Parish","shortCode":"10"},{"name":"Saint Peter Parish","shortCode":"11"}]},{"countryName":"Dominican Republic","countryShortCode":"DOM","regions":[{"name":"Cibao Central","shortCode":"02"},{"name":"Del Valle","shortCode":"37"},{"name":"Distrito Nacional","shortCode":"01"},{"name":"Enriquillo","shortCode":"38"},{"name":"Norcentral","shortCode":"04"},{"name":"Nordeste","shortCode":"34"},{"name":"Noroeste","shortCode":"34"},{"name":"Norte","shortCode":"35"},{"name":"Valdesia","shortCode":"42"}]},{"countryName":"Ecuador","countryShortCode":"ECU","regions":[{"name":"Azuay","shortCode":"A"},{"name":"Bol\u00edvar","shortCode":"B"},{"name":"Ca\u00f1ar","shortCode":"F"},{"name":"Carchi","shortCode":"C"},{"name":"Chimborazo","shortCode":"H"},{"name":"Cotopaxi","shortCode":"X"},{"name":"El Oro","shortCode":"O"},{"name":"Esmeraldas","shortCode":"E"},{"name":"Gal\u00e1pagos","shortCode":"W"},{"name":"Guayas","shortCode":"G"},{"name":"Imbabura","shortCode":"I"},{"name":"Loja","shortCode":"L"},{"name":"Los R\u00edos","shortCode":"R"},{"name":"Manab\u00ed","shortCode":"M"},{"name":"Morona-Santiago","shortCode":"S"},{"name":"Napo","shortCode":"N"},{"name":"Orellana","shortCode":"D"},{"name":"Pastaza","shortCode":"Y"},{"name":"Pichincha","shortCode":"P"},{"name":"Santa Elena","shortCode":"SE"},{"name":"Santo Domingo de los Ts\u00e1chilas","shortCode":"SD"},{"name":"Sucumb\u00edos","shortCode":"U"},{"name":"Tungurahua","shortCode":"T"},{"name":"Zamora-Chinchipe","shortCode":"Z"}]},{"countryName":"Egypt","countryShortCode":"EGY","regions":[{"name":"Alexandria","shortCode":"ALX"},{"name":"Aswan","shortCode":"ASN"},{"name":"Asyout","shortCode":"AST"},{"name":"Bani Sueif","shortCode":"BNS"},{"name":"Beheira","shortCode":"BH"},{"name":"Cairo","shortCode":"C"},{"name":"Daqahlia","shortCode":"DK"},{"name":"Dumiat","shortCode":"DT"},{"name":"El Bahr El Ahmar","shortCode":"BA"},{"name":"El Ismailia","shortCode":"IS"},{"name":"El Suez","shortCode":"SUZ"},{"name":"El Wadi El Gedeed","shortCode":"WAD"},{"name":"Fayoum","shortCode":"FYM"},{"name":"Gharbia","shortCode":"GH"},{"name":"Giza","shortCode":"SUZ"},{"name":"Helwan","shortCode":"HU"},{"name":"Kafr El Sheikh","shortCode":"KFS"},{"name":"Luxor","shortCode":"LX"},{"name":"Matrouh","shortCode":"MT"},{"name":"Menia","shortCode":"MN"},{"name":"Menofia","shortCode":"MNF"},{"name":"North Sinai","shortCode":"SIN"},{"name":"Port Said","shortCode":"PTS"},{"name":"Qalubia","shortCode":"KB"},{"name":"Qena","shortCode":"KN"},{"name":"Sharqia","shortCode":"SHR"},{"name":"Sixth of October","shortCode":"SU"},{"name":"Sohag","shortCode":"SHG"},{"name":"South Sinai","shortCode":"JS"}]},{"countryName":"El Salvador","countryShortCode":"SLV","regions":[{"name":"Ahuachap\u00e1n","shortCode":"AH"},{"name":"Caba\u00f1as","shortCode":"CA"},{"name":"Cuscatl\u00e1n","shortCode":"CU"},{"name":"Chalatenango","shortCode":"CH"},{"name":"La Libertad","shortCode":"LI"},{"name":"La Paz","shortCode":"PA"},{"name":"La Uni\u00f3n","shortCode":"UN"},{"name":"Moraz\u00e1n","shortCode":"MO"},{"name":"San Miguel","shortCode":"SM"},{"name":"San Salvador","shortCode":"SS"},{"name":"Santa Ana","shortCode":"SA"},{"name":"San Vicente","shortCode":"SV"},{"name":"Sonsonate","shortCode":"SO"},{"name":"Usulut\u00e1n","shortCode":"US"}]},{"countryName":"Equatorial Guinea","countryShortCode":"GNQ","regions":[{"name":"Annob\u00f3n","shortCode":"AN"},{"name":"Bioko Norte","shortCode":"BN"},{"name":"Bioko Sur","shortCode":"BS"},{"name":"Centro Sur","shortCode":"CS"},{"name":"Ki\u00e9-Ntem","shortCode":"KN"},{"name":"Litoral","shortCode":"LI"},{"name":"Wele-Nzas","shortCode":"WN"}]},{"countryName":"Eritrea","countryShortCode":"ERI","regions":[{"name":"Anseba","shortCode":"AN"},{"name":"Debub","shortCode":"DU"},{"name":"Debub-Keih-Bahri","shortCode":"DK"},{"name":"Gash-Barka","shortCode":"GB"},{"name":"Maekel","shortCode":"MA"},{"name":"Semien-Keih-Bahri","shortCode":"SK"}]},{"countryName":"Estonia","countryShortCode":"EST","regions":[{"name":"Harjumaa (Tallinn)","shortCode":"37"},{"name":"Hiiumaa (Kardla)","shortCode":"39"},{"name":"Ida-Virumaa (Johvi)","shortCode":"44"},{"name":"J\u00e4rvamaa (Paide)","shortCode":"41"},{"name":"J\u00f5gevamaa (Jogeva)","shortCode":"49"},{"name":"L\u00e4\u00e4nemaa","shortCode":"57"},{"name":"L\u00e4\u00e4ne-Virumaa (Rakvere)","shortCode":"59"},{"name":"P\u00e4rnumaa (Parnu)","shortCode":"67"},{"name":"P\u00f5lvamaa (Polva)","shortCode":"65"},{"name":"Raplamaa (Rapla)","shortCode":"70"},{"name":"Saaremaa (Kuessaare)","shortCode":"74"},{"name":"Tartumaa (Tartu)","shortCode":"78"},{"name":"Valgamaa (Valga)","shortCode":"82"},{"name":"Viljandimaa (Viljandi)","shortCode":"84"},{"name":"V\u00f5rumaa (Voru)","shortCode":"86"}]},{"countryName":"Ethiopia","countryShortCode":"ETH","regions":[{"name":"Addis Ababa","shortCode":"AA"},{"name":"Afar","shortCode":"AF"},{"name":"Amhara","shortCode":"AM"},{"name":"Benshangul-Gumaz","shortCode":"BE"},{"name":"Dire Dawa","shortCode":"DD"},{"name":"Gambela","shortCode":"GA"},{"name":"Harari","shortCode":"HA"},{"name":"Oromia","shortCode":"OR"},{"name":"Somali","shortCode":"SO"},{"name":"Southern Nations Nationalities and People\'s Region","shortCode":"SN"},{"name":"Tigray","shortCode":"TI"}]},{"countryName":"Falkland Islands (Islas Malvinas)","countryShortCode":"FLK","regions":[{"name":"Falkland Islands (Islas Malvinas)"}]},{"countryName":"Faroe Islands","countryShortCode":"FRO","regions":[{"name":"Bordoy"},{"name":"Eysturoy"},{"name":"Mykines"},{"name":"Sandoy"},{"name":"Skuvoy"},{"name":"Streymoy"},{"name":"Suduroy"},{"name":"Tvoroyri"},{"name":"Vagar"}]},{"countryName":"Fiji","countryShortCode":"FJI","regions":[{"name":"Ba","shortCode":"01"},{"name":"Bua","shortCode":"01"},{"name":"Cakaudrove","shortCode":"03"},{"name":"Kadavu","shortCode":"04"},{"name":"Lau","shortCode":"05"},{"name":"Lomaiviti","shortCode":"06"},{"name":"Macuata","shortCode":"07"},{"name":"Nadroga and Navosa","shortCode":"08"},{"name":"Naitasiri","shortCode":"09"},{"name":"Namosi","shortCode":"10"},{"name":"Ra","shortCode":"011"},{"name":"Rewa","shortCode":"12"},{"name":"Rotuma","shortCode":"R"},{"name":"Serua","shortCode":"12"},{"name":"Tailevu","shortCode":"14"}]},{"countryName":"Finland","countryShortCode":"FIN","regions":[{"name":"Ahvenanmaan maakunta","shortCode":"FI-01"},{"name":"Etel\u00e4-Karjala","shortCode":"FI-02"},{"name":"Etel\u00e4-Pohjanmaa","shortCode":"FI-03"},{"name":"Etel\u00e4-Savo","shortCode":"FI-04"},{"name":"Kainuu","shortCode":"FI-05"},{"name":"Kanta-H\u00e4me","shortCode":"FI-06"},{"name":"Keski-Pohjanmaa","shortCode":"FI-07"},{"name":"Keski-Suomi","shortCode":"FI-08"},{"name":"Kymenlaakso","shortCode":"FI-09"},{"name":"Lappi","shortCode":"FI-10"},{"name":"Pirkanmaa","shortCode":"FI-11"},{"name":"Pohjanmaa","shortCode":"FI-12"},{"name":"Pohjois-Karjala","shortCode":"FI-13"},{"name":"Pohjois-Pohjanmaa","shortCode":"FI-14"},{"name":"Pohjois-Savo","shortCode":"FI-15"},{"name":"P\u00e4ij\u00e4t-H\u00e4me","shortCode":"FI-16"},{"name":"Satakunta","shortCode":"FI-17"},{"name":"Uusimaa","shortCode":"FI-18"},{"name":"Varsinais-Suomi","shortCode":"FI-19"}]},{"countryName":"France","countryShortCode":"FRA","regions":[{"name":"Auvergne-Rh\u00f4ne-Alpes","shortCode":"ARA"},{"name":"Bourgogne-Franche-Comt\u00e9","shortCode":"BFC"},{"name":"Bretagne","shortCode":"BRE"},{"name":"Centre-Val de Loire","shortCode":"CVL"},{"name":"Corse","shortCode":"COR"},{"name":"Grand Est","shortCode":"GES"},{"name":"Hauts-de-France","shortCode":"HDF"},{"name":"\u00cele-de-France","shortCode":"IDF"},{"name":"Normandie","shortCode":"NOR"},{"name":"Nouvelle-Aquitaine","shortCode":"NAQ"},{"name":"Occitanie","shortCode":"OCC"},{"name":"Pays de la Loire","shortCode":"PDL"},{"name":"Provence-Alpes-Cote d\'Azur","shortCode":"PAC"},{"name":"Clipperton","shortCode":"CP"},{"name":"Guadeloupe","shortCode":"GP"},{"name":"Guyane","shortCode":"GF"},{"name":"Martinique","shortCode":"MQ"},{"name":"Mayotte","shortCode":"YT"},{"name":"Novelle-Cal\u00e9donie","shortCode":"NC"},{"name":"Polyn\u00e9sie","shortCode":"PF"},{"name":"Saint-Pierre-et-Miquelon","shortCode":"PM"},{"name":"Saint Barth\u00e9lemy","shortCode":"BL"},{"name":"Saint Martin","shortCode":"MF"},{"name":"R\u00e9union","shortCode":"RE"},{"name":"Terres Australes Fran\u00e7aises","shortCode":"TF"},{"name":"Wallis-et-Futuna","shortCode":"WF"}]},{"countryName":"French Guiana","countryShortCode":"GUF","regions":[{"name":"French Guiana"}]},{"countryName":"French Polynesia","countryShortCode":"PYF","regions":[{"name":"Archipel des Marquises"},{"name":"Archipel des Tuamotu"},{"name":"Archipel des Tubuai"},{"name":"Iles du Vent"},{"name":"Iles Sous-le-Vent"}]},{"countryName":"French Southern and Antarctic Lands","countryShortCode":"ATF","regions":[{"name":"Adelie Land"},{"name":"Ile Crozet"},{"name":"Iles Kerguelen"},{"name":"Iles Saint-Paul et Amsterdam"}]},{"countryName":"Gabon","countryShortCode":"GAB","regions":[{"name":"Estuaire","shortCode":"1"},{"name":"Haut-Ogoou\u00e9","shortCode":"2"},{"name":"Moyen-Ogoou\u00e9","shortCode":"3"},{"name":"Ngouni\u00e9","shortCode":"4"},{"name":"Nyanga","shortCode":"5"},{"name":"Ogoou\u00e9-Ivindo","shortCode":"6"},{"name":"Ogoou\u00e9-Lolo","shortCode":"7"},{"name":"Ogoou\u00e9-Maritime","shortCode":"8"},{"name":"Woleu-Ntem","shortCode":"9"}]},{"countryName":"Gambia, The","countryShortCode":"GMB","regions":[{"name":"Banjul","shortCode":"B"},{"name":"Central River","shortCode":"M"},{"name":"Lower River","shortCode":"L"},{"name":"North Bank","shortCode":"N"},{"name":"Upper River","shortCode":"U"},{"name":"Western","shortCode":"W"}]},{"countryName":"Georgia","countryShortCode":"GEO","regions":[{"name":"Abkhazia (Sokhumi)","shortCode":"AB"},{"name":"Ajaria (Bat\'umi)","shortCode":"AJ"},{"name":"Guria","shortCode":"GU"},{"name":"Imereti","shortCode":"IM"},{"name":"K\'akheti","shortCode":"KA"},{"name":"Kvemo Kartli","shortCode":"KK"},{"name":"Mtshkheta-Mtianeti","shortCode":"MM"},{"name":"Rach\'a-Lexhkumi-KvemoSvaneti","shortCode":"RL"},{"name":"Samegrelo-Zemo Svaneti","shortCode":"SZ"},{"name":"Samtskhe-Javakheti","shortCode":"SJ"},{"name":"Shida Kartli","shortCode":"SK"},{"name":"Tbilisi","shortCode":"TB"}]},{"countryName":"Germany","countryShortCode":"DEU","regions":[{"name":"Baden-W\u00fcrttemberg","shortCode":"BW"},{"name":"Bayern","shortCode":"BY"},{"name":"Berlin","shortCode":"BE"},{"name":"Brandenburg","shortCode":"BB"},{"name":"Bremen","shortCode":"HB"},{"name":"Hamburg","shortCode":"HH"},{"name":"Hessen","shortCode":"HE"},{"name":"Mecklenburg-Vorpommern","shortCode":"MV"},{"name":"Niedersachsen","shortCode":"NI"},{"name":"Nordrhein-Westfalen","shortCode":"NW"},{"name":"Rheinland-Pfalz","shortCode":"RP"},{"name":"Saarland","shortCode":"SL"},{"name":"Sachsen","shortCode":"SN"},{"name":"Sachsen-Anhalt","shortCode":"ST"},{"name":"Schleswig-Holstein","shortCode":"SH"},{"name":"Th\u00fcringen","shortCode":"TH"}]},{"countryName":"Ghana","countryShortCode":"GHA","regions":[{"name":"Ahafo"},{"name":"Ashanti","shortCode":"AH"},{"name":"Bono"},{"name":"Bono East"},{"name":"Central","shortCode":"CP"},{"name":"Eastern","shortCode":"EP"},{"name":"Greater Accra","shortCode":"AA"},{"name":"Northern","shortCode":"NP"},{"name":"North East"},{"name":"Oti"},{"name":"Savannah"},{"name":"Upper East","shortCode":"UE"},{"name":"Upper West","shortCode":"UW"},{"name":"Volta","shortCode":"TV"},{"name":"Western","shortCode":"WP"},{"name":"Western North"}]},{"countryName":"Gibraltar","countryShortCode":"GIB","regions":[{"name":"Gibraltar"}]},{"countryName":"Greece","countryShortCode":"GRC","regions":[{"name":"Anatolik\u00ed Makedon\u00eda kai Thr\u00e1ki","shortCode":"A"},{"name":"Attik\u1e2f","shortCode":"I"},{"name":"Dytik\u00ed Ell\u00e1da","shortCode":"G"},{"name":"Dytik\u00ed Makedon\u00eda","shortCode":"C"},{"name":"Ion\u00eda N\u00edsia","shortCode":"F"},{"name":"Kentrik\u00ed Makedon\u00eda","shortCode":"B"},{"name":"Kr\u00edt\u00ed","shortCode":"M"},{"name":"Not\u00edo Aiga\u00edo","shortCode":"L"},{"name":"Peloponn\u00edsos","shortCode":"J"},{"name":"Stere\u00e1 Ell\u00e1da","shortCode":"H"},{"name":"Thessal\u00eda","shortCode":"E"},{"name":"Vore\u00edo Aiga\u00edo","shortCode":"K"},{"name":"\u00cdpeiros","shortCode":"D"},{"name":"\u00c1gion \u00d3ros","shortCode":"69"}]},{"countryName":"Greenland","countryShortCode":"GRL","regions":[{"name":"Kommune Kujalleq","shortCode":"KU"},{"name":"Kommuneqarfik Sermersooq","shortCode":"SM"},{"name":"Qaasuitsup Kommunia","shortCode":"QA"},{"name":"Qeqqata Kommunia","shortCode":"QE"}]},{"countryName":"Grenada","countryShortCode":"GRD","regions":[{"name":"Saint Andrew","shortCode":"01"},{"name":"Saint David","shortCode":"02"},{"name":"Saint George","shortCode":"03"},{"name":"Saint John","shortCode":"04"},{"name":"Saint Mark","shortCode":"05"},{"name":"Saint Patrick","shortCode":"06"},{"name":"Southern Grenadine Islands","shortCode":"10"}]},{"countryName":"Guadeloupe","countryShortCode":"GLP","regions":[{"name":"Guadeloupe"}]},{"countryName":"Guam","countryShortCode":"GUM","regions":[{"name":"Guam"}]},{"countryName":"Guatemala","countryShortCode":"GTM","regions":[{"name":"Alta Verapaz","shortCode":"AV"},{"name":"Baja Verapaz","shortCode":"BV"},{"name":"Chimaltenango","shortCode":"CM"},{"name":"Chiquimula","shortCode":"CQ"},{"name":"El Progreso","shortCode":"PR"},{"name":"Escuintla","shortCode":"ES"},{"name":"Guatemala","shortCode":"GU"},{"name":"Huehuetenango","shortCode":"HU"},{"name":"Izabal","shortCode":"IZ"},{"name":"Jalapa","shortCode":"JA"},{"name":"Jutiapa","shortCode":"JU"},{"name":"Pet\u00e9n","shortCode":"PE"},{"name":"Quetzaltenango","shortCode":"QZ"},{"name":"Quich\u00e9","shortCode":"QC"},{"name":"Retalhuleu","shortCode":"Re"},{"name":"Sacatep\u00e9quez","shortCode":"SA"},{"name":"San Marcos","shortCode":"SM"},{"name":"Santa Rosa","shortCode":"SR"},{"name":"Solol\u00e1","shortCode":"SO"},{"name":"Suchitep\u00e9quez","shortCode":"SU"},{"name":"Totonicap\u00e1n","shortCode":"TO"},{"name":"Zacapa","shortCode":"ZA"}]},{"countryName":"Guernsey","countryShortCode":"GGY","regions":[{"name":"Castel"},{"name":"Forest"},{"name":"St. Andrew"},{"name":"St. Martin"},{"name":"St. Peter Port"},{"name":"St. Pierre du Bois"},{"name":"St. Sampson"},{"name":"St. Saviour"},{"name":"Torteval"},{"name":"Vale"}]},{"countryName":"Guinea","countryShortCode":"GIN","regions":[{"name":"Bok\u00e9","shortCode":"B"},{"name":"Conakry","shortCode":"C"},{"name":"Faranah","shortCode":"F"},{"name":"Kankan","shortCode":"K"},{"name":"Kindia","shortCode":"D"},{"name":"Lab\u00e9","shortCode":"L"},{"name":"Mamou","shortCode":"M"},{"name":"Nz\u00e9r\u00e9kor\u00e9","shortCode":"N"}]},{"countryName":"Guinea-Bissau","countryShortCode":"GNB","regions":[{"name":"Bafat\u00e1","shortCode":"BA"},{"name":"Biombo","shortCode":"BM"},{"name":"Bissau","shortCode":"BS"},{"name":"Bolama-Bijagos","shortCode":"BL"},{"name":"Cacheu","shortCode":"CA"},{"name":"Gab\u00fa","shortCode":"GA"},{"name":"Oio","shortCode":"OI"},{"name":"Quinara","shortCode":"QU"},{"name":"Tombali","shortCode":"TO"}]},{"countryName":"Guyana","countryShortCode":"GUY","regions":[{"name":"Barima-Waini","shortCode":"BA"},{"name":"Cuyuni-Mazaruni","shortCode":"CU"},{"name":"Demerara-Mahaica","shortCode":"DE"},{"name":"East Berbice-Corentyne","shortCode":"EB"},{"name":"Essequibo Islands-West Demerara","shortCode":"ES"},{"name":"Mahaica-Berbice","shortCode":"MA"},{"name":"Pomeroon-Supenaam","shortCode":"PM"},{"name":"Potaro-Siparuni","shortCode":"PT"},{"name":"Upper Demerara-Berbice","shortCode":"UD"},{"name":"Upper Takutu-Upper Essequibo","shortCode":"UT"}]},{"countryName":"Haiti","countryShortCode":"HTI","regions":[{"name":"Artibonite","shortCode":"AR"},{"name":"Centre","shortCode":"CE"},{"name":"Grand\'Anse","shortCode":"GA"},{"name":"Nippes","shortCode":"NI"},{"name":"Nord","shortCode":"ND"},{"name":"Nord-Est","shortCode":"NE"},{"name":"Nord-Ouest","shortCode":"NO"},{"name":"Ouest","shortCode":"OU"},{"name":"Sud","shortCode":"SD"},{"name":"Sud-Est","shortCode":"SE"}]},{"countryName":"Heard Island and McDonald Islands","countryShortCode":"HMD","regions":[{"name":"Heard Island and McDonald Islands"}]},{"countryName":"Holy See (Vatican City)","countryShortCode":"VAT","regions":[{"name":"Holy See (Vatican City)","shortCode":"01"}]},{"countryName":"Honduras","countryShortCode":"HND","regions":[{"name":"Atl\u00e1ntida","shortCode":"AT"},{"name":"Choluteca","shortCode":"CH"},{"name":"Col\u00f3n","shortCode":"CL"},{"name":"Comayagua","shortCode":"CM"},{"name":"Cop\u00e1n","shortCode":"CP"},{"name":"Cort\u00e9s","shortCode":"CR"},{"name":"El Para\u00edso","shortCode":"EP"},{"name":"Francisco Morazan","shortCode":"FM"},{"name":"Gracias a Dios","shortCode":"GD"},{"name":"Intibuc\u00e1","shortCode":"IN"},{"name":"Islas de la Bah\u00eda","shortCode":"IB"},{"name":"La Paz","shortCode":"LP"},{"name":"Lempira","shortCode":"LE"},{"name":"Ocotepeque","shortCode":"OC"},{"name":"Olancho","shortCode":"OL"},{"name":"Santa B\u00e1rbara","shortCode":"SB"},{"name":"Valle","shortCode":"VA"},{"name":"Yoro","shortCode":"YO"}]},{"countryName":"Hong Kong","countryShortCode":"HKG","regions":[{"name":"Hong Kong","shortCode":"HK"}]},{"countryName":"Hungary","countryShortCode":"HUN","regions":[{"name":"B\u00e1cs-Kiskun","shortCode":"BK"},{"name":"Baranya","shortCode":"BA"},{"name":"B\u00e9k\u00e9s","shortCode":"BE"},{"name":"B\u00e9k\u00e9scsaba","shortCode":"BC"},{"name":"Borsod-Abauj-Zemplen","shortCode":"BZ"},{"name":"Budapest","shortCode":"BU"},{"name":"Csongr\u00e1d","shortCode":"CS"},{"name":"Debrecen","shortCode":"DE"},{"name":"Duna\u00fajv\u00e1ros","shortCode":"DU"},{"name":"Eger","shortCode":"EG"},{"name":"\u00c9rd","shortCode":"ER"},{"name":"Fej\u00e9r","shortCode":"FE"},{"name":"Gy\u0151r","shortCode":"GY"},{"name":"Gy\u0151r-Moson-Sopron","shortCode":"GS"},{"name":"Hajd\u00fa-Bihar","shortCode":"HB"},{"name":"Heves","shortCode":"HE"},{"name":"H\u00f3dmez\u0151v\u00e1s\u00e1rhely","shortCode":"HV"},{"name":"J\u00e1sz-Nagykun-Szolnok","shortCode":"N"},{"name":"Kaposv\u00e1r","shortCode":"KV"},{"name":"Kecskem\u00e9t","shortCode":"KM"},{"name":"Kom\u00e1rom-Esztergom","shortCode":"KE"},{"name":"Miskolc","shortCode":"MI"},{"name":"Nagykanizsa","shortCode":"NK"},{"name":"N\u00f3gr\u00e1d","shortCode":"NO"},{"name":"Ny\u00edregyh\u00e1za","shortCode":"NY"},{"name":"P\u00e9cs","shortCode":"PS"},{"name":"Pest","shortCode":"PE"},{"name":"Salg\u00f3tarj\u00e1n","shortCode":"ST"},{"name":"Somogy","shortCode":"SO"},{"name":"Sopron","shortCode":"SN"},{"name":"Szabolcs-\u00e1-Bereg","shortCode":"SZ"},{"name":"Szeged","shortCode":"SD"},{"name":"Sz\u00e9kesfeh\u00e9rv\u00e1r","shortCode":"SF"},{"name":"Szeksz\u00e1rd","shortCode":"SS"},{"name":"Szolnok","shortCode":"SK"},{"name":"Szombathely","shortCode":"SH"},{"name":"Tatab\u00e1nya","shortCode":"TB"},{"name":"Tolna","shortCode":"TO"},{"name":"Vas","shortCode":"VA"},{"name":"Veszpr\u00e9m","shortCode":"VE"},{"name":"Veszpr\u00e9m (City)","shortCode":"VM"},{"name":"Zala","shortCode":"ZA"},{"name":"Zalaegerszeg","shortCode":"ZE"}]},{"countryName":"Iceland","countryShortCode":"ISL","regions":[{"name":"Austurland","shortCode":"7"},{"name":"H\u00f6fu\u00f0borgarsv\u00e6\u00f0i utan Reykjav\u00edkur","shortCode":"1"},{"name":"Nor\u00f0urland eystra","shortCode":"6"},{"name":"Nor\u00f0urland vestra","shortCode":"5"},{"name":"Su\u00f0urland","shortCode":"8"},{"name":"Su\u00f0urnes","shortCode":"2"},{"name":"Vestfir\u00f0ir","shortCode":"4"},{"name":"Vesturland","shortCode":"3"}]},{"countryName":"India","countryShortCode":"IND","regions":[{"name":"Andaman and Nicobar Islands","shortCode":"AN"},{"name":"Andhra Pradesh","shortCode":"AP"},{"name":"Arunachal Pradesh","shortCode":"AR"},{"name":"Assam","shortCode":"AS"},{"name":"Bihar","shortCode":"BR"},{"name":"Chandigarh","shortCode":"CH"},{"name":"Chhattisgarh","shortCode":"CT"},{"name":"Dadra and Nagar Haveli","shortCode":"DN"},{"name":"Daman and Diu","shortCode":"DD"},{"name":"Delhi","shortCode":"DL"},{"name":"Goa","shortCode":"GA"},{"name":"Gujarat","shortCode":"GJ"},{"name":"Haryana","shortCode":"HR"},{"name":"Himachal Pradesh","shortCode":"HP"},{"name":"Jammu and Kashmir","shortCode":"JK"},{"name":"Jharkhand","shortCode":"JH"},{"name":"Karnataka","shortCode":"KA"},{"name":"Kerala","shortCode":"KL"},{"name":"Lakshadweep","shortCode":"LD"},{"name":"Madhya Pradesh","shortCode":"MP"},{"name":"Maharashtra","shortCode":"MH"},{"name":"Manipur","shortCode":"MN"},{"name":"Meghalaya","shortCode":"ML"},{"name":"Mizoram","shortCode":"MZ"},{"name":"Nagaland","shortCode":"NL"},{"name":"Odisha","shortCode":"OR"},{"name":"Puducherry","shortCode":"PY"},{"name":"Punjab","shortCode":"PB"},{"name":"Rajasthan","shortCode":"RJ"},{"name":"Sikkim","shortCode":"WK"},{"name":"Tamil Nadu","shortCode":"TN"},{"name":"Telangana","shortCode":"TG"},{"name":"Tripura","shortCode":"TR"},{"name":"Uttarakhand","shortCode":"UT"},{"name":"Uttar Pradesh","shortCode":"UP"},{"name":"West Bengal","shortCode":"WB"}]},{"countryName":"Indonesia","countryShortCode":"IDN","regions":[{"name":"Aceh","shortCode":"AC"},{"name":"Bali","shortCode":"BA"},{"name":"Bangka Belitung","shortCode":"BB"},{"name":"Banten","shortCode":"BT"},{"name":"Bengkulu","shortCode":"BE"},{"name":"Gorontalo","shortCode":"GO"},{"name":"Jakarta Raya","shortCode":"JK"},{"name":"Jambi","shortCode":"JA"},{"name":"Jawa Barat","shortCode":"JB"},{"name":"Jawa Tengah","shortCode":"JT"},{"name":"Jawa Timur","shortCode":"JI"},{"name":"Kalimantan Barat","shortCode":"KB"},{"name":"Kalimantan Selatan","shortCode":"KS"},{"name":"Kalimantan Tengah","shortCode":"KT"},{"name":"Kalimantan Timur","shortCode":"KI"},{"name":"Kalimantan Utara","shortCode":"KU"},{"name":"Kepulauan Riau","shortCode":"KR"},{"name":"Lampung","shortCode":"LA"},{"name":"Maluku","shortCode":"MA"},{"name":"Maluku Utara","shortCode":"MU"},{"name":"Nusa Tenggara Barat","shortCode":"NB"},{"name":"Nusa Tenggara Timur","shortCode":"NT"},{"name":"Papua","shortCode":"PA"},{"name":"Papua Barat","shortCode":"PB"},{"name":"Riau","shortCode":"RI"},{"name":"Sulawesi Selatan","shortCode":"SR"},{"name":"Sulawesi Tengah","shortCode":"ST"},{"name":"Sulawesi Tenggara","shortCode":"SG"},{"name":"Sulawesi Utara","shortCode":"SA"},{"name":"Sumatera Barat","shortCode":"SB"},{"name":"Sumatera Selatan","shortCode":"SS"},{"name":"Sumatera Utara","shortCode":"SU"},{"name":"Yogyakarta","shortCode":"YO"}]},{"countryName":"Iran, Islamic Republic of","countryShortCode":"IRN","regions":[{"name":"Alborz","shortCode":"32"},{"name":"Ardab\u012bl","shortCode":"03"},{"name":"\u0100z\u0304arb\u0101yj\u0101n-e Gharb\u012b","shortCode":"02"},{"name":"\u0100z\u0304arb\u0101yj\u0101n-e Sharq\u012b","shortCode":"01"},{"name":"B\u016bshehr","shortCode":"06"},{"name":"Chah\u0101r Ma\u1e29\u0101l va Bakht\u012b\u0101r\u012b","shortCode":"08"},{"name":"E\u015ffah\u0101n","shortCode":"04"},{"name":"F\u0101rs","shortCode":"14"},{"name":"G\u012bl\u0101n","shortCode":"19"},{"name":"Golest\u0101n","shortCode":"27"},{"name":"Hamad\u0101n","shortCode":"24"},{"name":"Hormozg\u0101n","shortCode":"23"},{"name":"\u012al\u0101m","shortCode":"05"},{"name":"Kerm\u0101n","shortCode":"15"},{"name":"Kerm\u0101nsh\u0101h","shortCode":"17"},{"name":"Khor\u0101s\u0101n-e Jon\u016bb\u012b","shortCode":"29"},{"name":"Khor\u0101s\u0101n-e Ra\u1e95av\u012b","shortCode":"30"},{"name":"Khor\u0101s\u0101n-e Shom\u0101l\u012b","shortCode":"61"},{"name":"Kh\u016bzest\u0101n","shortCode":"10"},{"name":"Kohg\u012bl\u016byeh va Bowyer A\u1e29mad","shortCode":"18"},{"name":"Kordest\u0101n","shortCode":"16"},{"name":"Lorest\u0101n","shortCode":"20"},{"name":"Markazi","shortCode":"22"},{"name":"M\u0101zandar\u0101n","shortCode":"21"},{"name":"Qazv\u012bn","shortCode":"28"},{"name":"Qom","shortCode":"26"},{"name":"Semn\u0101n","shortCode":"12"},{"name":"S\u012bst\u0101n va Bal\u016bchest\u0101n","shortCode":"13"},{"name":"Tehr\u0101n","shortCode":"07"},{"name":"Yazd","shortCode":"25"},{"name":"Zanj\u0101n","shortCode":"11"}]},{"countryName":"Iraq","countryShortCode":"IRQ","regions":[{"name":"Al Anb\u0101r","shortCode":"AN"},{"name":"Al Ba\u015frah","shortCode":"BA"},{"name":"Al Muthann\u00e1","shortCode":"MU"},{"name":"Al Q\u0101dis\u012byah","shortCode":"QA"},{"name":"An Najaf","shortCode":"NA"},{"name":"Arb\u012bl","shortCode":"AR"},{"name":"As Sulaym\u0101n\u012byah","shortCode":"SU"},{"name":"B\u0101bil","shortCode":"BB"},{"name":"Baghd\u0101d","shortCode":"BG"},{"name":"Dohuk","shortCode":"DA"},{"name":"Dh\u012b Q\u0101r","shortCode":"DQ"},{"name":"Diy\u0101l\u00e1","shortCode":"DI"},{"name":"Karbal\u0101\'","shortCode":"KA"},{"name":"Kirkuk","shortCode":"KI"},{"name":"Mays\u0101n","shortCode":"MA"},{"name":"N\u012bnaw\u00e1","shortCode":"NI"},{"name":"\u015eal\u0101\u1e29 ad D\u012bn","shortCode":"SD"},{"name":"W\u0101si\u0163","shortCode":"WA"}]},{"countryName":"Ireland","countryShortCode":"IRL","regions":[{"name":"Carlow","shortCode":"CW"},{"name":"Cavan","shortCode":"CN"},{"name":"Clare","shortCode":"CE"},{"name":"Cork","shortCode":"CO"},{"name":"Donegal","shortCode":"DL"},{"name":"Dublin","shortCode":"D"},{"name":"Galway","shortCode":"G"},{"name":"Kerry","shortCode":"KY"},{"name":"Kildare","shortCode":"KE"},{"name":"Kilkenny","shortCode":"KK"},{"name":"Laois","shortCode":"LS"},{"name":"Leitrim","shortCode":"LM"},{"name":"Limerick","shortCode":"LK"},{"name":"Longford","shortCode":"LD"},{"name":"Louth","shortCode":"LH"},{"name":"Mayo","shortCode":"MO"},{"name":"Meath","shortCode":"MH"},{"name":"Monaghan","shortCode":"MN"},{"name":"Offaly","shortCode":"OY"},{"name":"Roscommon","shortCode":"RN"},{"name":"Sligo","shortCode":"SO"},{"name":"Tipperary","shortCode":"TA"},{"name":"Waterford","shortCode":"WD"},{"name":"Westmeath","shortCode":"WH"},{"name":"Wexford","shortCode":"WX"},{"name":"Wicklow","shortCode":"WW"}]},{"countryName":"Isle of Man","countryShortCode":"IMN","regions":[{"name":"Isle of Man"}]},{"countryName":"Israel","countryShortCode":"ISR","regions":[{"name":"HaDarom","shortCode":"D"},{"name":"HaMerkaz","shortCode":"M"},{"name":"HaTsafon","shortCode":"Z"},{"name":"H\u0331efa","shortCode":"HA"},{"name":"Tel-Aviv","shortCode":"TA"},{"name":"Yerushalayim","shortCode":"JM"}]},{"countryName":"Italy","countryShortCode":"ITA","regions":[{"name":"Abruzzo","shortCode":"65"},{"name":"Basilicata","shortCode":"77"},{"name":"Calabria","shortCode":"78"},{"name":"Campania","shortCode":"72"},{"name":"Emilia-Romagna","shortCode":"45"},{"name":"Friuli-Venezia Giulia","shortCode":"36"},{"name":"Lazio","shortCode":"62"},{"name":"Liguria","shortCode":"42"},{"name":"Lombardia","shortCode":"25"},{"name":"Marche","shortCode":"57"},{"name":"Molise","shortCode":"67"},{"name":"Piemonte","shortCode":"21"},{"name":"Puglia","shortCode":"75"},{"name":"Sardegna","shortCode":"88"},{"name":"Sicilia","shortCode":"82"},{"name":"Toscana","shortCode":"52"},{"name":"Trentino-Alto Adige","shortCode":"32"},{"name":"Umbria","shortCode":"55"},{"name":"Valle d\'Aosta","shortCode":"23"},{"name":"Veneto","shortCode":"34"}]},{"countryName":"Jamaica","countryShortCode":"JAM","regions":[{"name":"Clarendon","shortCode":"13"},{"name":"Hanover","shortCode":"09"},{"name":"Kingston","shortCode":"01"},{"name":"Manchester","shortCode":"12"},{"name":"Portland","shortCode":"04"},{"name":"Saint Andrew","shortCode":"02"},{"name":"Saint Ann","shortCode":"06"},{"name":"Saint Catherine","shortCode":"14"},{"name":"Saint Elizabeth","shortCode":"11"},{"name":"Saint James","shortCode":"08"},{"name":"Saint Mary","shortCode":"05"},{"name":"Saint Thomas","shortCode":"03"},{"name":"Trelawny","shortCode":"07"},{"name":"Westmoreland","shortCode":"10"}]},{"countryName":"Japan","countryShortCode":"JPN","regions":[{"name":"Aichi","shortCode":"23"},{"name":"Akita","shortCode":"05"},{"name":"Aomori","shortCode":"02"},{"name":"Chiba","shortCode":"12"},{"name":"Ehime","shortCode":"38"},{"name":"Fukui","shortCode":"18"},{"name":"Fukuoka","shortCode":"40"},{"name":"Fukushima","shortCode":"07"},{"name":"Gifu","shortCode":"21"},{"name":"Gunma","shortCode":"10"},{"name":"Hiroshima","shortCode":"34"},{"name":"Hokkaido","shortCode":"01"},{"name":"Hyogo","shortCode":"28"},{"name":"Ibaraki","shortCode":"08"},{"name":"Ishikawa","shortCode":"17"},{"name":"Iwate","shortCode":"03"},{"name":"Kagawa","shortCode":"37"},{"name":"Kagoshima","shortCode":"46"},{"name":"Kanagawa","shortCode":"14"},{"name":"Kochi","shortCode":"39"},{"name":"Kumamoto","shortCode":"43"},{"name":"Kyoto","shortCode":"26"},{"name":"Mie","shortCode":"24"},{"name":"Miyagi","shortCode":"04"},{"name":"Miyazaki","shortCode":"45"},{"name":"Nagano","shortCode":"20"},{"name":"Nagasaki","shortCode":"42"},{"name":"Nara","shortCode":"29"},{"name":"Niigata","shortCode":"15"},{"name":"Oita","shortCode":"44"},{"name":"Okayama","shortCode":"33"},{"name":"Okinawa","shortCode":"47"},{"name":"Osaka","shortCode":"27"},{"name":"Saga","shortCode":"41"},{"name":"Saitama","shortCode":"11"},{"name":"Shiga","shortCode":"25"},{"name":"Shimane","shortCode":"32"},{"name":"Shizuoka","shortCode":"22"},{"name":"Tochigi","shortCode":"09"},{"name":"Tokushima","shortCode":"36"},{"name":"Tokyo","shortCode":"13"},{"name":"Tottori","shortCode":"31"},{"name":"Toyama","shortCode":"16"},{"name":"Wakayama","shortCode":"30"},{"name":"Yamagata","shortCode":"06"},{"name":"Yamaguchi","shortCode":"35"},{"name":"Yamanashi","shortCode":"19"}]},{"countryName":"Jersey","countryShortCode":"JEY","regions":[{"name":"Jersey"}]},{"countryName":"Jordan","countryShortCode":"JOR","regions":[{"name":"\u2018Ajl\u016bn","shortCode":"AJ"},{"name":"Al \'Aqabah","shortCode":"AQ"},{"name":"Al Balq\u0101\u2019","shortCode":"BA"},{"name":"Al Karak","shortCode":"KA"},{"name":"Al Mafraq","shortCode":"MA"},{"name":"Al \u2018A\u0305\u015fimah","shortCode":"AM"},{"name":"A\u0163 \u0162af\u012blah","shortCode":"AT"},{"name":"Az Zarq\u0101\u2019","shortCode":"AZ"},{"name":"Irbid","shortCode":"IR"},{"name":"Jarash","shortCode":"JA"},{"name":"Ma\u2018\u0101n","shortCode":"MN"},{"name":"M\u0101dab\u0101","shortCode":"MD"}]},{"countryName":"Kazakhstan","countryShortCode":"KAZ","regions":[{"name":"Almaty","shortCode":"ALA"},{"name":"Aqmola","shortCode":"AKM"},{"name":"Aqtobe","shortCode":"AKT"},{"name":"Astana","shortCode":"AST"},{"name":"Atyrau","shortCode":"ATY"},{"name":"Batys Qazaqstan","shortCode":"ZAP"},{"name":"Bayqongyr"},{"name":"Mangghystau","shortCode":"MAN"},{"name":"Ongtustik Qazaqstan","shortCode":"YUZ"},{"name":"Pavlodar","shortCode":"PAV"},{"name":"Qaraghandy","shortCode":"KAR"},{"name":"Qostanay","shortCode":"KUS"},{"name":"Qyzylorda","shortCode":"KZY"},{"name":"Shyghys Qazaqstan","shortCode":"VOS"},{"name":"Soltustik Qazaqstan","shortCode":"SEV"},{"name":"Zhambyl","shortCode":"ZHA"}]},{"countryName":"Kenya","countryShortCode":"KEN","regions":[{"name":"Baringo","shortCode":"01"},{"name":"Bomet","shortCode":"02"},{"name":"Bungoma","shortCode":"03"},{"name":"Busia","shortCode":"04"},{"name":"Eleyo\/Marakwet","shortCode":"05"},{"name":"Embu","shortCode":"06"},{"name":"Garissa","shortCode":"07"},{"name":"Homa Bay","shortCode":"08"},{"name":"Isiolo","shortCode":"09"},{"name":"Kajiado","shortCode":"10"},{"name":"Kakamega","shortCode":"11"},{"name":"Kericho","shortCode":"12"},{"name":"Kiambu","shortCode":"13"},{"name":"Kilifi","shortCode":"14"},{"name":"Kirinyaga","shortCode":"15"},{"name":"Kisii","shortCode":"16"},{"name":"Kisumu","shortCode":"17"},{"name":"Kitui","shortCode":"18"},{"name":"Kwale","shortCode":"19"},{"name":"Laikipia","shortCode":"20"},{"name":"Lamu","shortCode":"21"},{"name":"Machakos","shortCode":"22"},{"name":"Makueni","shortCode":"23"},{"name":"Mandera","shortCode":"24"},{"name":"Marsabit","shortCode":"25"},{"name":"Meru","shortCode":"26"},{"name":"Migori","shortCode":"27"},{"name":"Mombasa","shortCode":"28"},{"name":"Murang\'a","shortCode":"29"},{"name":"Nairobi City","shortCode":"30"},{"name":"Nakuru","shortCode":"31"},{"name":"Nandi","shortCode":"32"},{"name":"Narok","shortCode":"33"},{"name":"Nyamira","shortCode":"34"},{"name":"Nyandarua","shortCode":"35"},{"name":"Nyeri","shortCode":"36"},{"name":"Samburu","shortCode":"37"},{"name":"Siaya","shortCode":"38"},{"name":"Taita\/Taveta","shortCode":"39"},{"name":"Tana River","shortCode":"40"},{"name":"Tharaka-Nithi","shortCode":"41"},{"name":"Trans Nzoia","shortCode":"42"},{"name":"Turkana","shortCode":"43"},{"name":"Uasin Gishu","shortCode":"44"},{"name":"Vihiga","shortCode":"45"},{"name":"Wajir","shortCode":"46"},{"name":"West Pokot","shortCode":"47"}]},{"countryName":"Kiribati","countryShortCode":"KIR","regions":[{"name":"Abaiang"},{"name":"Abemama"},{"name":"Aranuka"},{"name":"Arorae"},{"name":"Banaba"},{"name":"Beru"},{"name":"Butaritari"},{"name":"Central Gilberts"},{"name":"Gilbert Islands","shortCode":"G"},{"name":"Kanton"},{"name":"Kiritimati"},{"name":"Kuria"},{"name":"Line Islands","shortCode":"L"},{"name":"Maiana"},{"name":"Makin"},{"name":"Marakei"},{"name":"Nikunau"},{"name":"Nonouti"},{"name":"Northern Gilberts"},{"name":"Onotoa"},{"name":"Phoenix Islands","shortCode":"P"},{"name":"Southern Gilberts"},{"name":"Tabiteuea"},{"name":"Tabuaeran"},{"name":"Tamana"},{"name":"Tarawa"},{"name":"Teraina"}]},{"countryName":"Korea, Democratic People\'s Republic of","countryShortCode":"PRK","regions":[{"name":"Chagang-do (Chagang Province)","shortCode":"04"},{"name":"Hamgyong-bukto (North Hamgyong Province)","shortCode":"09"},{"name":"Hamgyong-namdo (South Hamgyong Province)","shortCode":"08"},{"name":"Hwanghae-bukto (North Hwanghae Province)","shortCode":"06"},{"name":"Hwanghae-namdo (South Hwanghae Province)","shortCode":"05"},{"name":"Kangwon-do (Kangwon Province)","shortCode":"07"},{"name":"Nas\u014fn (Najin-S\u014fnbong)","shortCode":"13"},{"name":"P\'yongan-bukto (North P\'yongan Province)","shortCode":"03"},{"name":"P\'yongan-namdo (South P\'yongan Province)","shortCode":"02"},{"name":"P\'yongyang-si (P\'yongyang City)","shortCode":"01"},{"name":"Yanggang-do (Yanggang Province)","shortCode":"10"}]},{"countryName":"Korea, Republic of","countryShortCode":"KOR","regions":[{"name":"Ch\'ungch\'ongbuk-do","shortCode":"43"},{"name":"Ch\'ungch\'ongnam-do","shortCode":"44"},{"name":"Cheju-do","shortCode":"49"},{"name":"Chollabuk-do","shortCode":"45"},{"name":"Chollanam-do","shortCode":"46"},{"name":"Inch\'on-Kwangyokhi","shortCode":"28"},{"name":"Kang-won-do","shortCode":"42"},{"name":"Kwangju-Kwangyokshi","shortCode":"28"},{"name":"Kyonggi-do","shortCode":"41"},{"name":"Kyongsangbuk-do","shortCode":"47"},{"name":"Kyongsangnam-do","shortCode":"48"},{"name":"Pusan-Kwangyokshi","shortCode":"26"},{"name":"Seoul-T\'ukpyolshi","shortCode":"11"},{"name":"Sejong","shortCode":"50"},{"name":"Taegu-Kwangyokshi","shortCode":"27"},{"name":"Taejon-Kwangyokshi","shortCode":"30"},{"name":"Ulsan-Kwangyokshi","shortCode":"31"}]},{"countryName":"Kuwait","countryShortCode":"KWT","regions":[{"name":"Al A\u1e29madi","shortCode":"AH"},{"name":"Al Farw\u0101n\u012byah","shortCode":"FA"},{"name":"Al Jahr\u0101\u2019","shortCode":"JA"},{"name":"Al \u2018\u0100\u015fimah","shortCode":"KU"},{"name":"\u1e28awall\u012b","shortCode":"HA"},{"name":"Mub\u0101rak al Kabir","shortCode":"MU"}]},{"countryName":"Kyrgyzstan","countryShortCode":"KGZ","regions":[{"name":"Batken Oblasty","shortCode":"B"},{"name":"Bishkek Shaary","shortCode":"GB"},{"name":"Chuy Oblasty (Bishkek)","shortCode":"C"},{"name":"Jalal-Abad Oblasty","shortCode":"J"},{"name":"Naryn Oblasty","shortCode":"N"},{"name":"Osh Oblasty","shortCode":"O"},{"name":"Talas Oblasty","shortCode":"T"},{"name":"Ysyk-Kol Oblasty (Karakol)","shortCode":"Y"}]},{"countryName":"Laos","countryShortCode":"LAO","regions":[{"name":"Attapu","shortCode":"AT"},{"name":"Bok\u00e8o","shortCode":"BK"},{"name":"Bolikhamxai","shortCode":"BL"},{"name":"Champasak","shortCode":"CH"},{"name":"Houaphan","shortCode":"HO"},{"name":"Khammouan","shortCode":"KH"},{"name":"Louang Namtha","shortCode":"LM"},{"name":"Louangphabang","shortCode":"LP"},{"name":"Oud\u00f4mxai","shortCode":"OU"},{"name":"Ph\u00f4ngsali","shortCode":"PH"},{"name":"Salavan","shortCode":"SL"},{"name":"Savannakh\u00e9t","shortCode":"SV"},{"name":"Vientiane","shortCode":"VI"},{"name":"Xaignabouli","shortCode":"XA"},{"name":"X\u00e9kong","shortCode":"XE"},{"name":"Xaisomboun","shortCode":"XS"},{"name":"Xiangkhouang","shortCode":"XI"}]},{"countryName":"Latvia","countryShortCode":"LVA","regions":[{"name":"Aglona","shortCode":"001"},{"name":"Aizkraukle","shortCode":"002"},{"name":"Aizpute","shortCode":"003"},{"name":"Akn\u012bste","shortCode":"004"},{"name":"Aloja","shortCode":"005"},{"name":"Alsunga","shortCode":"06"},{"name":"Al\u016bksne","shortCode":"007"},{"name":"Amata","shortCode":"008"},{"name":"Ape","shortCode":"009"},{"name":"Auce","shortCode":"010"},{"name":"\u0100da\u017ei","shortCode":"011"},{"name":"Bab\u012bte","shortCode":"012"},{"name":"Baldone","shortCode":"013"},{"name":"Baltinava","shortCode":"014"},{"name":"Balvi","shortCode":"015"},{"name":"Bauska","shortCode":"016"},{"name":"Bever\u012bna","shortCode":"017"},{"name":"Broc\u0113ni","shortCode":"018"},{"name":"Burtnieki","shortCode":"019"},{"name":"Carnikava","shortCode":"020"},{"name":"Cesvaine","shortCode":"021"},{"name":"C\u0113sis","shortCode":"022"},{"name":"Cibla","shortCode":"023"},{"name":"Dagda","shortCode":"024"},{"name":"Daugavpils","shortCode":"025"},{"name":"Daugavpils (City)","shortCode":"DGV"},{"name":"Dobele","shortCode":"026"},{"name":"Dundaga","shortCode":"027"},{"name":"Durbe","shortCode":"028"},{"name":"Engure","shortCode":"029"},{"name":"\u0112rg\u013ci","shortCode":"030"},{"name":"Garkalne","shortCode":"031"},{"name":"Grobi\u0146a","shortCode":"032"},{"name":"Gulbene","shortCode":"033"},{"name":"Iecava","shortCode":"034"},{"name":"Ik\u0161\u0137ile","shortCode":"035"},{"name":"Il\u016bkste","shortCode":"036"},{"name":"In\u010dukalns","shortCode":"037"},{"name":"Jaunjelgava","shortCode":"038"},{"name":"Jaunpiebalga","shortCode":"039"},{"name":"Jaunpils","shortCode":"040"},{"name":"Jelgava","shortCode":"041"},{"name":"Jelgava (City)","shortCode":"JEL"},{"name":"J\u0113kabpils","shortCode":"042"},{"name":"J\u0113kabpils (City)","shortCode":"JKB"},{"name":"J\u016brmala (City)","shortCode":"JUR"},{"name":"Kandava","shortCode":"043"},{"name":"K\u0101rsava","shortCode":"044"},{"name":"Koc\u0113ni","shortCode":"045"},{"name":"Koknese","shortCode":"046"},{"name":"Kr\u0101slava","shortCode":"047"},{"name":"Krimulda","shortCode":"048"},{"name":"Krustpils","shortCode":"049"},{"name":"Kuld\u012bga","shortCode":"050"},{"name":"\u0136egums","shortCode":"051"},{"name":"\u0136ekava","shortCode":"052"},{"name":"Lielv\u0101rde","shortCode":"053"},{"name":"Liep\u0101ja","shortCode":"LPX"},{"name":"Limba\u017ei","shortCode":"054"},{"name":"L\u012bgatne","shortCode":"055"},{"name":"L\u012bv\u0101ni","shortCode":"056"},{"name":"Lub\u0101na","shortCode":"057"},{"name":"Ludza","shortCode":"058"},{"name":"Madona","shortCode":"059"},{"name":"Mazsalaca","shortCode":"060"},{"name":"M\u0101lpils","shortCode":"061"},{"name":"M\u0101rupe","shortCode":"062"},{"name":"M\u0113rsrags","shortCode":"063"},{"name":"Nauk\u0161\u0113ni","shortCode":"064"},{"name":"Nereta","shortCode":"065"},{"name":"N\u012bca","shortCode":"066"},{"name":"Ogre","shortCode":"067"},{"name":"Olaine","shortCode":"068"},{"name":"Ozolnieki","shortCode":"069"},{"name":"P\u0101rgauja","shortCode":"070"},{"name":"P\u0101vilosta","shortCode":"071"},{"name":"P\u013cavi\u0146as","shortCode":"072"},{"name":"Prei\u013ci","shortCode":"073"},{"name":"Priekule","shortCode":"074"},{"name":"Prieku\u013ci","shortCode":"075"},{"name":"Rauna","shortCode":"076"},{"name":"R\u0113zekne","shortCode":"077"},{"name":"R\u0113zekne (City)","shortCode":"REZ"},{"name":"Riebi\u0146i","shortCode":"078"},{"name":"R\u012bga","shortCode":"RIX"},{"name":"Roja","shortCode":"079"},{"name":"Ropa\u017ei","shortCode":"080"},{"name":"Rucava","shortCode":"081"},{"name":"Rug\u0101ji","shortCode":"082"},{"name":"Rund\u0101le","shortCode":"083"},{"name":"R\u016bjiena","shortCode":"084"},{"name":"Sala","shortCode":"085"},{"name":"Salacgr\u012bva","shortCode":"086"},{"name":"Salaspils","shortCode":"087"},{"name":"Saldus","shortCode":"088"},{"name":"Saulkrasti","shortCode":"089"},{"name":"S\u0113ja","shortCode":"090"},{"name":"Sigulda","shortCode":"091"},{"name":"Skr\u012bveri","shortCode":"092"},{"name":"Skrunda","shortCode":"093"},{"name":"Smiltene","shortCode":"094"},{"name":"Stopi\u0146i","shortCode":"095"},{"name":"Stren\u010di","shortCode":"096"},{"name":"Talsi","shortCode":"097"},{"name":"T\u0113rvete","shortCode":"098"},{"name":"Tukums","shortCode":"099"},{"name":"Vai\u0146ode","shortCode":"100"},{"name":"Valka","shortCode":"101"},{"name":"Valmiera","shortCode":"VMR"},{"name":"Varak\u013c\u0101ni","shortCode":"102"},{"name":"V\u0101rkava","shortCode":"103"},{"name":"Vecpiebalga","shortCode":"104"},{"name":"Vecumnieki","shortCode":"105"},{"name":"Ventspils","shortCode":"106"},{"name":"Ventspils (City)","shortCode":"VEN"},{"name":"Vies\u012bte","shortCode":"107"},{"name":"Vi\u013caka","shortCode":"108"},{"name":"Vi\u013c\u0101ni","shortCode":"109"},{"name":"Zilupe","shortCode":"110"}]},{"countryName":"Lebanon","countryShortCode":"LBN","regions":[{"name":"Aakk\u00e2r","shortCode":"AK"},{"name":"Baalbelk-Hermel","shortCode":"BH"},{"name":"B\u00e9qaa","shortCode":"BI"},{"name":"Beyrouth","shortCode":"BA"},{"name":"Liban-Nord","shortCode":"AS"},{"name":"Liban-Sud","shortCode":"JA"},{"name":"Mont-Liban","shortCode":"JL"},{"name":"Nabat\u00eey\u00e9","shortCode":"NA"}]},{"countryName":"Lesotho","countryShortCode":"LSO","regions":[{"name":"Berea","shortCode":"D"},{"name":"Butha-Buthe","shortCode":"B"},{"name":"Leribe","shortCode":"C"},{"name":"Mafeteng","shortCode":"E"},{"name":"Maseru","shortCode":"A"},{"name":"Mohales Hoek","shortCode":"F"},{"name":"Mokhotlong","shortCode":"J"},{"name":"Qacha\'s Nek","shortCode":"H"},{"name":"Quthing","shortCode":"G"},{"name":"Thaba-Tseka","shortCode":"K"}]},{"countryName":"Liberia","countryShortCode":"LBR","regions":[{"name":"Bomi","shortCode":"BM"},{"name":"Bong","shortCode":"BG"},{"name":"Gbarpolu","shortCode":"GP"},{"name":"Grand Bassa","shortCode":"GB"},{"name":"Grand Cape Mount","shortCode":"CM"},{"name":"Grand Gedeh","shortCode":"GG"},{"name":"Grand Kru","shortCode":"GK"},{"name":"Lofa","shortCode":"LO"},{"name":"Margibi","shortCode":"MG"},{"name":"Maryland","shortCode":"MY"},{"name":"Montserrado","shortCode":"MO"},{"name":"Nimba","shortCode":"NI"},{"name":"River Cess","shortCode":"RI"},{"name":"River Geee","shortCode":"RG"},{"name":"Sinoe","shortCode":"SI"}]},{"countryName":"Libya","countryShortCode":"LBY","regions":[{"name":"Al Bu\u0163n\u0101n","shortCode":"BU"},{"name":"Al Jabal al Akh\u1e11ar","shortCode":"JA"},{"name":"Al Jabal al Gharb\u012b","shortCode":"JG"},{"name":"Al Jaf\u0101rah","shortCode":"JA"},{"name":"Al Jufrah","shortCode":"JU"},{"name":"Al Kufrah","shortCode":"FK"},{"name":"Al Marj","shortCode":"MJ"},{"name":"Al Marquab","shortCode":"MB"},{"name":"Al W\u0101\u1e29\u0101t","shortCode":"WA"},{"name":"An Nuqa\u0163 al Khams","shortCode":"NQ"},{"name":"Az Z\u0101wiyah","shortCode":"ZA"},{"name":"Bangh\u0101z\u012b","shortCode":"BA"},{"name":"Darnah","shortCode":"DR"},{"name":"Gh\u0101t","shortCode":"GH"},{"name":"Mi\u015fr\u0101tah","shortCode":"MI"},{"name":"Murzuq","shortCode":"MQ"},{"name":"N\u0101l\u016bt","shortCode":"NL"},{"name":"Sabh\u0101","shortCode":"SB"},{"name":"Surt","shortCode":"SR"},{"name":"\u0162ar\u0101bulus","shortCode":"TB"},{"name":"Yafran","shortCode":"WD"},{"name":"W\u0101d\u012b ash Sh\u0101\u0163i\u02be","shortCode":"WS"}]},{"countryName":"Liechtenstein","countryShortCode":"LIE","regions":[{"name":"Balzers","shortCode":"01"},{"name":"Eschen","shortCode":"02"},{"name":"Gamprin","shortCode":"03"},{"name":"Mauren","shortCode":"04"},{"name":"Planken","shortCode":"05"},{"name":"Ruggell","shortCode":"06"},{"name":"Schaan","shortCode":"07"},{"name":"Schellenberg","shortCode":"08"},{"name":"Triesen","shortCode":"09"},{"name":"Triesenberg","shortCode":"10"},{"name":"Vaduz","shortCode":"11"}]},{"countryName":"Lithuania","countryShortCode":"LTU","regions":[{"name":"Alytaus","shortCode":"AL"},{"name":"Kauno","shortCode":"KU"},{"name":"Klaip\u0117dos","shortCode":"KL"},{"name":"Marijampol\u0117s","shortCode":"MR"},{"name":"Panev\u0117\u017eio","shortCode":"PN"},{"name":"\u0160iauli\u0173","shortCode":"SA"},{"name":"Taurag\u0117s","shortCode":"TA"},{"name":"Tel\u0161i\u0173","shortCode":"TE"},{"name":"Utenos","shortCode":"UT"},{"name":"Vilniaus","shortCode":"VL"}]},{"countryName":"Luxembourg","countryShortCode":"LUX","regions":[{"name":"Capellen","shortCode":"CA"},{"name":"Clevaux","shortCode":"CL"},{"name":"Diekirch","shortCode":"DI"},{"name":"Echternach","shortCode":"EC"},{"name":"Esch-sur-Alzette","shortCode":"ES"},{"name":"Grevenmacher","shortCode":"GR"},{"name":"Luxembourg","shortCode":"LU"},{"name":"Mersch","shortCode":"ME"},{"name":"Redange","shortCode":"RD"},{"name":"Remich","shortCode":"RM"},{"name":"Vianden","shortCode":"VD"},{"name":"Wiltz","shortCode":"WI"}]},{"countryName":"Macao","countryShortCode":"MAC","regions":[{"name":"Macao"}]},{"countryName":"Macedonia, Republic of","countryShortCode":"MKD","regions":[{"name":"Ara\u010dinovo","shortCode":"02"},{"name":"Berovo","shortCode":"03"},{"name":"Bitola","shortCode":"04"},{"name":"Bogdanci","shortCode":"05"},{"name":"Bogovinje","shortCode":"06"},{"name":"Bosilovo","shortCode":"07"},{"name":"Brvenica","shortCode":"08"},{"name":"Centar \u017dupa","shortCode":"78"},{"name":"\u010ca\u0161ka","shortCode":"08"},{"name":"\u010ce\u0161inovo-Oble\u0161evo","shortCode":"81"},{"name":"\u010cu\u010der Sandevo","shortCode":"82"},{"name":"Debar","shortCode":"21"},{"name":"Debarca","shortCode":"22"},{"name":"Del\u010devo","shortCode":"23"},{"name":"Demir Hisar","shortCode":"25"},{"name":"Demir Kapija","shortCode":"24"},{"name":"Doran","shortCode":"26"},{"name":"Dolneni","shortCode":"27"},{"name":"Gevgelija","shortCode":"18"},{"name":"Gostivar","shortCode":"19"},{"name":"Gradsko","shortCode":"20"},{"name":"Ilinden","shortCode":"34"},{"name":"Jegunovce","shortCode":"35"},{"name":"Karbinci","shortCode":"37"},{"name":"Kavadarci","shortCode":"36"},{"name":"Ki\u010devo","shortCode":"40"},{"name":"Ko\u010dani","shortCode":"42"},{"name":"Kon\u010de","shortCode":"41"},{"name":"Kratovo","shortCode":"43"},{"name":"Kriva Palanka","shortCode":"44"},{"name":"Krivoga\u0161tani","shortCode":"45"},{"name":"Kru\u0161evo","shortCode":"46"},{"name":"Kumanovo","shortCode":"47"},{"name":"Lipkovo","shortCode":"48"},{"name":"Lozovo","shortCode":"49"},{"name":"Makedonska Kamenica","shortCode":"51"},{"name":"Makedonski Brod","shortCode":"52"},{"name":"Mavrovo i Rostu\u0161a","shortCode":"50"},{"name":"Mogila","shortCode":"53"},{"name":"Negotino","shortCode":"54"},{"name":"Novaci","shortCode":"55"},{"name":"Novo Selo","shortCode":"56"},{"name":"Ohrid","shortCode":"58"},{"name":"Peh\u010devo","shortCode":"60"},{"name":"Petrovec","shortCode":"59"},{"name":"Plasnica","shortCode":"61"},{"name":"Prilep","shortCode":"62"},{"name":"Probi\u0161tip","shortCode":"63"},{"name":"Radovi\u0161","shortCode":""},{"name":"Rankovce","shortCode":"65"},{"name":"Resen","shortCode":"66"},{"name":"Rosoman","shortCode":"67"},{"name":"Skopje","shortCode":"85"},{"name":"Sopi\u0161te","shortCode":"70"},{"name":"Staro Nagori\u010dane","shortCode":"71"},{"name":"Struga","shortCode":"72"},{"name":"Strumica","shortCode":"73"},{"name":"Studeni\u010dani","shortCode":"74"},{"name":"Sveti Nikole","shortCode":"69"},{"name":"\u0160tip","shortCode":"83"},{"name":"Tearce","shortCode":"75"},{"name":"Tetovo","shortCode":"76"},{"name":"Valandovo","shortCode":"10"},{"name":"Vasilevo","shortCode":"11"},{"name":"Veles","shortCode":"13"},{"name":"Vev\u010dani","shortCode":"12"},{"name":"Vinica","shortCode":"14"},{"name":"Vrap\u010di\u0161te","shortCode":"16"},{"name":"Zelenikovo","shortCode":"32"},{"name":"Zrnovci","shortCode":"33"},{"name":"\u017delino","shortCode":"30"}]},{"countryName":"Madagascar","countryShortCode":"MDG","regions":[{"name":"Antananarivo","shortCode":"T"},{"name":"Antsiranana","shortCode":"D"},{"name":"Fianarantsoa","shortCode":"F"},{"name":"Mahajanga","shortCode":"M"},{"name":"Toamasina","shortCode":"A"},{"name":"Toliara","shortCode":"U"}]},{"countryName":"Malawi","countryShortCode":"MWI","regions":[{"name":"Balaka","shortCode":"BA"},{"name":"Blantyre","shortCode":"BL"},{"name":"Chikwawa","shortCode":"CK"},{"name":"Chiradzulu","shortCode":"CR"},{"name":"Chitipa","shortCode":"CT"},{"name":"Dedza","shortCode":"DE"},{"name":"Dowa","shortCode":"DO"},{"name":"Karonga","shortCode":"KR"},{"name":"Kasungu","shortCode":"KS"},{"name":"Likoma","shortCode":"LK"},{"name":"Lilongwe","shortCode":"LI"},{"name":"Machinga","shortCode":"MH"},{"name":"Mangochi","shortCode":"MG"},{"name":"Mchinji","shortCode":"MC"},{"name":"Mulanje","shortCode":"MU"},{"name":"Mwanza","shortCode":"MW"},{"name":"Mzimba","shortCode":"MZ"},{"name":"Nkhata Bay","shortCode":"NE"},{"name":"Nkhotakota","shortCode":"NB"},{"name":"Nsanje","shortCode":"NS"},{"name":"Ntcheu","shortCode":"NU"},{"name":"Ntchisi","shortCode":"NI"},{"name":"Phalombe","shortCode":"PH"},{"name":"Rumphi","shortCode":"RU"},{"name":"Salima","shortCode":"SA"},{"name":"Thyolo","shortCode":"TH"},{"name":"Zomba","shortCode":"ZO"}]},{"countryName":"Malaysia","countryShortCode":"MYS","regions":[{"name":"Johor","shortCode":"01"},{"name":"Kedah","shortCode":"02"},{"name":"Kelantan","shortCode":"03"},{"name":"Melaka","shortCode":"04"},{"name":"Negeri Sembilan","shortCode":"05"},{"name":"Pahang","shortCode":"06"},{"name":"Perak","shortCode":"08"},{"name":"Perlis","shortCode":"09"},{"name":"Pulau Pinang","shortCode":"07"},{"name":"Sabah","shortCode":"12"},{"name":"Sarawak","shortCode":"13"},{"name":"Selangor","shortCode":"10"},{"name":"Terengganu","shortCode":"11"},{"name":"Wilayah Persekutuan (Kuala Lumpur)","shortCode":"14"},{"name":"Wilayah Persekutuan (Labuan)","shortCode":"15"},{"name":"Wilayah Persekutuan (Putrajaya)","shortCode":"16"}]},{"countryName":"Maldives","countryShortCode":"MDV","regions":[{"name":"Alifu Alifu","shortCode":"02"},{"name":"Alifu Dhaalu","shortCode":"00"},{"name":"Baa","shortCode":"20"},{"name":"Dhaalu","shortCode":"17"},{"name":"Faafu","shortCode":"14"},{"name":"Gaafu Alifu","shortCode":"27"},{"name":"Gaafu Dhaalu","shortCode":"28"},{"name":"Gnaviyani","shortCode":"29"},{"name":"Haa Alifu","shortCode":"07"},{"name":"Haa Dhaalu","shortCode":"23"},{"name":"Kaafu","shortCode":"29"},{"name":"Laamu","shortCode":"05"},{"name":"Lhaviyani","shortCode":"03"},{"name":"Mal\u00e9","shortCode":"MLE"},{"name":"Meemu","shortCode":"12"},{"name":"Noonu","shortCode":"25"},{"name":"Raa","shortCode":"13"},{"name":"Seenu","shortCode":"01"},{"name":"Shaviyani","shortCode":"24"},{"name":"Thaa","shortCode":"08"},{"name":"Vaavu","shortCode":"04"}]},{"countryName":"Mali","countryShortCode":"MLI","regions":[{"name":"Bamako","shortCode":"BKO"},{"name":"Gao","shortCode":"7"},{"name":"Kayes","shortCode":"1"},{"name":"Kidal","shortCode":"8"},{"name":"Koulikoro","shortCode":"2"},{"name":"Mopti","shortCode":"5"},{"name":"Segou","shortCode":"4"},{"name":"Sikasso","shortCode":"3"},{"name":"Tombouctou","shortCode":"6"}]},{"countryName":"Malta","countryShortCode":"MLT","regions":[{"name":"Attard","shortCode":"01"},{"name":"Balzan","shortCode":"02"},{"name":"Birgu","shortCode":"03"},{"name":"Birkirkara","shortCode":"04"},{"name":"Bir\u017cebbu\u0121a","shortCode":"05"},{"name":"Bormla","shortCode":"06"},{"name":"Dingli","shortCode":"07"},{"name":"Fgura","shortCode":"08"},{"name":"Floriana","shortCode":"09"},{"name":"Fontana","shortCode":"10"},{"name":"Guda","shortCode":"11"},{"name":"G\u017cira","shortCode":"12"},{"name":"G\u0127ajnsielem","shortCode":"13"},{"name":"G\u0127arb","shortCode":"14"},{"name":"G\u0127arg\u0127ur","shortCode":"15"},{"name":"G\u0127asri","shortCode":"16"},{"name":"G\u0127axaq","shortCode":"17"},{"name":"\u0126amrun","shortCode":"18"},{"name":"Iklin","shortCode":"19"},{"name":"Isla","shortCode":"20"},{"name":"Kalkara","shortCode":"21"},{"name":"Ker\u010bem","shortCode":"22"},{"name":"Kirkop","shortCode":"23"},{"name":"Lija","shortCode":"24"},{"name":"Luqa","shortCode":"25"},{"name":"Marsa","shortCode":"26"},{"name":"Marsaskala","shortCode":"27"},{"name":"Marsaxlokk","shortCode":"28"},{"name":"Mdina","shortCode":"29"},{"name":"Mellie\u0127a","shortCode":"30"},{"name":"M\u0121arr","shortCode":"31"},{"name":"Mosta","shortCode":"32"},{"name":"Mqabba","shortCode":"33"},{"name":"Msida","shortCode":"34"},{"name":"Mtarfa","shortCode":"35"},{"name":"Munxar","shortCode":"36"},{"name":"Nadur","shortCode":"37"},{"name":"Naxxar","shortCode":"38"},{"name":"Paola","shortCode":"39"},{"name":"Pembroke","shortCode":"40"},{"name":"Piet\u00e0","shortCode":"41"},{"name":"Qala","shortCode":"42"},{"name":"Qormi","shortCode":"43"},{"name":"Qrendi","shortCode":"44"},{"name":"Rabat G\u0127awdex","shortCode":"45"},{"name":"Rabat Malta","shortCode":"46"},{"name":"Safi","shortCode":"47"},{"name":"San \u0120iljan","shortCode":"48"},{"name":"San \u0120wann","shortCode":"49"},{"name":"San Lawrenz","shortCode":"50"},{"name":"San Pawl il-Ba\u0127ar","shortCode":"51"},{"name":"Sannat","shortCode":"52"},{"name":"Santa Lu\u010bija","shortCode":"53"},{"name":"Santa Venera","shortCode":"54"},{"name":"Si\u0121\u0121iewi","shortCode":"55"},{"name":"Sliema","shortCode":"56"},{"name":"Swieqi","shortCode":"57"},{"name":"Tai Xbiex","shortCode":"58"},{"name":"Tarzien","shortCode":"59"},{"name":"Valletta","shortCode":"60"},{"name":"Xag\u0127ra","shortCode":"61"},{"name":"Xewkija","shortCode":"62"},{"name":"Xg\u0127ajra","shortCode":"63"},{"name":"\u017babbar","shortCode":"64"},{"name":"\u017bebbu\u0121 G\u0127awde","shortCode":"65"},{"name":"\u017bebbu\u0121 Malta","shortCode":"66"},{"name":"\u017bejtun","shortCode":"67"},{"name":"\u017burrieq","shortCode":"68"}]},{"countryName":"Marshall Islands","countryShortCode":"MHL","regions":[{"name":"Ailinglaplap","shortCode":"ALL"},{"name":"Ailuk","shortCode":"ALK"},{"name":"Arno","shortCode":"ARN"},{"name":"Aur","shortCode":"AUR"},{"name":"Bikini and Kili","shortCode":"KIL"},{"name":"Ebon","shortCode":"EBO"},{"name":"Jabat","shortCode":"JAB"},{"name":"Jaluit","shortCode":"JAL"},{"name":"Kwajalein","shortCode":"KWA"},{"name":"Lae","shortCode":"LAE"},{"name":"Lib","shortCode":"LIB"},{"name":"Likiep","shortCode":"LIK"},{"name":"Majuro","shortCode":"MAJ"},{"name":"Maloelap","shortCode":"MAL"},{"name":"Mejit","shortCode":"MEJ"},{"name":"Namdrik","shortCode":"NMK"},{"name":"Namu","shortCode":"NMU"},{"name":"Rongelap","shortCode":"RON"},{"name":"Ujae","shortCode":"UJA"},{"name":"Utrik","shortCode":"UTI"},{"name":"Wotho","shortCode":"WTH"},{"name":"Wotje","shortCode":"WTJ"}]},{"countryName":"Martinique","countryShortCode":"MTQ","regions":[{"name":"Martinique"}]},{"countryName":"Mauritania","countryShortCode":"MRT","regions":[{"name":"Adrar","shortCode":"07"},{"name":"Assaba","shortCode":"03"},{"name":"Brakna","shortCode":"05"},{"name":"Dakhlet Nouadhibou","shortCode":"08"},{"name":"Gorgol","shortCode":"04"},{"name":"Guidimaka","shortCode":"10"},{"name":"Hodh Ech Chargui","shortCode":"01"},{"name":"Hodh El Gharbi","shortCode":"02"},{"name":"Inchiri","shortCode":"12"},{"name":"Nouakchott Nord","shortCode":"14"},{"name":"Nouakchott Ouest","shortCode":"13"},{"name":"Nouakchott Sud","shortCode":"15"},{"name":"Tagant","shortCode":"09"},{"name":"Tiris Zemmour","shortCode":"11"},{"name":"Trarza","shortCode":"06"}]},{"countryName":"Mauritius","countryShortCode":"MUS","regions":[{"name":"Agalega Islands","shortCode":"AG"},{"name":"Beau Bassin-Rose Hill","shortCode":"BR"},{"name":"Black River","shortCode":"BL"},{"name":"Cargados Carajos Shoals","shortCode":"CC"},{"name":"Curepipe","shortCode":"CU"},{"name":"Flacq","shortCode":"FL"},{"name":"Grand Port","shortCode":"GP"},{"name":"Moka","shortCode":"MO"},{"name":"Pamplemousses","shortCode":"PA"},{"name":"Plaines Wilhems","shortCode":"PW"},{"name":"Port Louis (City)","shortCode":"PU"},{"name":"Port Louis","shortCode":"PL"},{"name":"Riviere du Rempart","shortCode":"RR"},{"name":"Rodrigues Island","shortCode":"RO"},{"name":"Savanne","shortCode":"SA"},{"name":"Vacoas-Phoenix","shortCode":"CP"}]},{"countryName":"Mayotte","countryShortCode":"MYT","regions":[{"name":"Dzaoudzi","shortCode":"01"},{"name":"Pamandzi","shortCode":"02"},{"name":"Mamoudzou","shortCode":"03"},{"name":"Dembeni","shortCode":"04"},{"name":"Bandr\u00e9l\u00e9","shortCode":"05"},{"name":"Kani-K\u00e9li","shortCode":"06"},{"name":"Bou\u00e9ni","shortCode":"07"},{"name":"Chirongui","shortCode":"08"},{"name":"Sada","shortCode":"09"},{"name":"Ouangani","shortCode":"10"},{"name":"Chiconi","shortCode":"11"},{"name":"Tsingoni","shortCode":"12"},{"name":"M\'Tsangamouji","shortCode":"13"},{"name":"Acoua","shortCode":"14"},{"name":"Mtsamboro","shortCode":"15"},{"name":"Bandraboua","shortCode":"16"},{"name":"Koungou","shortCode":"17"}]},{"countryName":"Mexico","countryShortCode":"MEX","regions":[{"name":"Aguascalientes","shortCode":"AGU"},{"name":"Baja California","shortCode":"BCN"},{"name":"Baja California Sur","shortCode":"BCS"},{"name":"Campeche","shortCode":"CAM"},{"name":"Ciudad de M\u00e9xico","shortCode":"DIF"},{"name":"Chiapas","shortCode":"CHP"},{"name":"Chihuahua","shortCode":"CHH"},{"name":"Coahuila de Zaragoza","shortCode":"COA"},{"name":"Colima","shortCode":"COL"},{"name":"Durango","shortCode":"DUR"},{"name":"Estado de M\u00e9xico","shortCode":"MEX"},{"name":"Guanajuato","shortCode":"GUA"},{"name":"Guerrero","shortCode":"GRO"},{"name":"Hidalgo","shortCode":"HID"},{"name":"Jalisco","shortCode":"JAL"},{"name":"Michoac\u00e1n de Ocampo","shortCode":"MIC"},{"name":"Morelos","shortCode":"MOR"},{"name":"Nayarit","shortCode":"NAY"},{"name":"Nuevo Le\u00f3n","shortCode":"NLE"},{"name":"Oaxaca","shortCode":"OAX"},{"name":"Puebla","shortCode":"PUE"},{"name":"Quer\u00e9taro de Arteaga","shortCode":"QUE"},{"name":"Quintana Roo","shortCode":"ROO"},{"name":"San Luis Potos\u00ed","shortCode":"SLP"},{"name":"Sinaloa","shortCode":"SIN"},{"name":"Sonora","shortCode":"SON"},{"name":"Tabasco","shortCode":"TAB"},{"name":"Tamaulipas","shortCode":"TAM"},{"name":"Tlaxcala","shortCode":"TLA"},{"name":"Veracruz","shortCode":"VER"},{"name":"Yucat\u00e1n","shortCode":"YUC"},{"name":"Zacatecas","shortCode":"ZAC"}]},{"countryName":"Micronesia, Federated States of","countryShortCode":"FSM","regions":[{"name":"Chuuk (Truk)","shortCode":"TRK"},{"name":"Kosrae","shortCode":"KSA"},{"name":"Pohnpei","shortCode":"PNI"},{"name":"Yap","shortCode":"YAP"}]},{"countryName":"Moldova","countryShortCode":"MDA","regions":[{"name":"Aenii Noi","shortCode":"AN"},{"name":"Basarabeasca","shortCode":"BS"},{"name":"B\u0103l\u021bi","shortCode":"BA"},{"name":"Bender","shortCode":"BD"},{"name":"Briceni","shortCode":"BR"},{"name":"Cahul","shortCode":"CA"},{"name":"Cantemir","shortCode":"CT"},{"name":"C\u0103l\u0103ra\u0219i","shortCode":"CL"},{"name":"C\u0103u\u0219eni","shortCode":"CS"},{"name":"Chi\u0219in\u0103u","shortCode":"CU"},{"name":"Cimi\u0219lia","shortCode":"CM"},{"name":"Criuleni","shortCode":"CR"},{"name":"Dondu\u0219eni","shortCode":"DO"},{"name":"Drochia","shortCode":"DR"},{"name":"Dub\u0103sari","shortCode":"DU"},{"name":"Edine\u021b","shortCode":"ED"},{"name":"F\u0103le\u0219ti","shortCode":"FA"},{"name":"Flore\u0219ti","shortCode":"FL"},{"name":"G\u0103g\u0103uzia","shortCode":"GA"},{"name":"Glodeni","shortCode":"GL"},{"name":"H\u00eence\u0219ti","shortCode":"HI"},{"name":"Ialoveni","shortCode":"IA"},{"name":"Leova","shortCode":"LE"},{"name":"Nisporeni","shortCode":"NI"},{"name":"Ocni\u021ba","shortCode":"OC"},{"name":"Orhei","shortCode":"OR"},{"name":"Rezina","shortCode":"RE"},{"name":"R\u00ee\u0219cani","shortCode":"RI"},{"name":"S\u00eengerei","shortCode":"SI"},{"name":"Soroca","shortCode":"SO"},{"name":"St\u00eenga Nistrului","shortCode":"SN"},{"name":"Str\u0103\u0219eni","shortCode":"ST"},{"name":"\u0218old\u0103ne\u0219ti","shortCode":"SD"},{"name":"\u0218tefan Vod\u0103","shortCode":"SV"},{"name":"Taraclia","shortCode":"TA"},{"name":"Telene\u0219ti","shortCode":"TE"},{"name":"Ungheni","shortCode":"UN"}]},{"countryName":"Monaco","countryShortCode":"MCO","regions":[{"name":"Colle","shortCode":"CL"},{"name":"Condamine","shortCode":"CO"},{"name":"Fontvieille","shortCode":"FO"},{"name":"Gare","shortCode":"GA"},{"name":"Jardin Exotique","shortCode":"JE"},{"name":"Larvotto","shortCode":"LA"},{"name":"Malbousquet","shortCode":"MA"},{"name":"Monaco-Ville","shortCode":"MO"},{"name":"Moneghetti","shortCode":"MG"},{"name":"Monte-Carlo","shortCode":"MC"},{"name":"Moulins","shortCode":"MU"},{"name":"Port-Hercule","shortCode":"PH"},{"name":"Saint-Roman","shortCode":"SR"},{"name":"Sainte-D\u00e9vote","shortCode":"SD"},{"name":"Source","shortCode":"SO"},{"name":"Sp\u00e9lugues","shortCode":"SP"},{"name":"Vallon de la Rousse","shortCode":"VR"}]},{"countryName":"Mongolia","countryShortCode":"MNG","regions":[{"name":"Arhangay","shortCode":"073"},{"name":"Bayan-Olgiy","shortCode":"071"},{"name":"Bayanhongor","shortCode":"069"},{"name":"Bulgan","shortCode":"067"},{"name":"Darhan","shortCode":"037"},{"name":"Dornod","shortCode":"061"},{"name":"Dornogovi","shortCode":"063"},{"name":"Dundgovi","shortCode":"059"},{"name":"Dzavhan","shortCode":"065"},{"name":"Govi-Altay","shortCode":"065"},{"name":"Govi-Sumber","shortCode":"064"},{"name":"Hovd","shortCode":"043"},{"name":"Hovsgol","shortCode":"041"},{"name":"Omnogovi","shortCode":"053"},{"name":"Ovorhangay","shortCode":"055"},{"name":"Selenge","shortCode":"049"},{"name":"Suhbaatar","shortCode":"051"},{"name":"Tov","shortCode":"047"},{"name":"Ulaanbaatar","shortCode":"1"},{"name":"Uvs","shortCode":"046"}]},{"countryName":"Montenegro","countryShortCode":"MNE","regions":[{"name":"Andrijevica","shortCode":"01"},{"name":"Bar","shortCode":"02"},{"name":"Berane","shortCode":"03"},{"name":"Bijelo Polje","shortCode":"04"},{"name":"Budva","shortCode":"05"},{"name":"Cetinje","shortCode":"06"},{"name":"Danilovgrad","shortCode":"07"},{"name":"Gusinje","shortCode":"22"},{"name":"Herceg Novi","shortCode":"08"},{"name":"Kola\u0161in","shortCode":"09"},{"name":"Kotor","shortCode":"10"},{"name":"Mojkovac","shortCode":"11"},{"name":"Nik\u0161i\u0107","shortCode":"12"},{"name":"Petnica","shortCode":"23"},{"name":"Plav","shortCode":"13"},{"name":"Plu\u017eine","shortCode":"14"},{"name":"Pljevlja","shortCode":"15"},{"name":"Podgorica","shortCode":"16"},{"name":"Ro\u017eaje","shortCode":"17"},{"name":"\u0160avnik","shortCode":"18"},{"name":"Tivat","shortCode":"19"},{"name":"Ulcinj","shortCode":"20"},{"name":"\u017dabljak","shortCode":"21"}]},{"countryName":"Montserrat","countryShortCode":"MSR","regions":[{"name":"Saint Anthony"},{"name":"Saint Georges"},{"name":"Saint Peter\'s"}]},{"countryName":"Morocco","countryShortCode":"MAR","regions":[{"name":"Chaouia-Ouardigha","shortCode":"09"},{"name":"Doukhala-Abda","shortCode":"10"},{"name":"F\u00e8s-Boulemane","shortCode":"05"},{"name":"Gharb-Chrarda-Beni Hssen","shortCode":"02"},{"name":"Grand Casablanca","shortCode":"08"},{"name":"Guelmim-Es Semara","shortCode":"14"},{"name":"La\u00e2youne-Boujdour-Sakia el Hamra","shortCode":"15"},{"name":"Marrakech-Tensift-Al Haouz","shortCode":"11"},{"name":"Mekn\u00e8s-Tafilalet","shortCode":"06"},{"name":"Oriental","shortCode":"04"},{"name":"Oued ed Dahab-Lagouira","shortCode":"16"},{"name":"Souss-Massa-Dr\u00e2a","shortCode":"13"},{"name":"Tadla-Azilal","shortCode":"12"},{"name":"Tanger-T\u00e9touan","shortCode":"01"},{"name":"Taza-Al Hoceima-Taounate","shortCode":"03"}]},{"countryName":"Mozambique","countryShortCode":"MOZ","regions":[{"name":"Cabo Delgado","shortCode":"P"},{"name":"Gaza","shortCode":"G"},{"name":"Inhambane","shortCode":"I"},{"name":"Manica","shortCode":"B"},{"name":"Maputo","shortCode":"L"},{"name":"Maputo (City)","shortCode":"MPM"},{"name":"Nampula","shortCode":"N"},{"name":"Niassa","shortCode":"A"},{"name":"Sofala","shortCode":"S"},{"name":"Tete","shortCode":"T"},{"name":"Zambezia","shortCode":"Q"}]},{"countryName":"Myanmar","countryShortCode":"MMR","regions":[{"name":"Ayeyarwady","shortCode":"07"},{"name":"Bago","shortCode":"02"},{"name":"Chin","shortCode":"14"},{"name":"Kachin","shortCode":"11"},{"name":"Kayah","shortCode":"12"},{"name":"Kayin","shortCode":"13"},{"name":"Magway","shortCode":"03"},{"name":"Mandalay","shortCode":"04"},{"name":"Mon","shortCode":"15"},{"name":"Nay Pyi Taw","shortCode":"18"},{"name":"Rakhine","shortCode":"16"},{"name":"Sagaing","shortCode":"01"},{"name":"Shan","shortCode":"17"},{"name":"Tanintharyi","shortCode":"05"},{"name":"Yangon","shortCode":"06"}]},{"countryName":"Namibia","countryShortCode":"NAM","regions":[{"name":"Erongo","shortCode":"ER"},{"name":"Hardap","shortCode":"HA"},{"name":"Kavango East","shortCode":"KE"},{"name":"Kavango West","shortCode":"KW"},{"name":"Karas","shortCode":"KA"},{"name":"Khomas","shortCode":"KH"},{"name":"Kunene","shortCode":"KU"},{"name":"Ohangwena","shortCode":"OW"},{"name":"Omaheke","shortCode":"OH"},{"name":"Omusati","shortCode":"OS"},{"name":"Oshana","shortCode":"ON"},{"name":"Oshikoto","shortCode":"OT"},{"name":"Otjozondjupa","shortCode":"OD"},{"name":"Zambezi","shortCode":"CA"}]},{"countryName":"Nauru","countryShortCode":"NRU","regions":[{"name":"Aiwo","shortCode":"01"},{"name":"Anabar","shortCode":"02"},{"name":"Anetan","shortCode":"03"},{"name":"Anibare","shortCode":"04"},{"name":"Baiti","shortCode":"05"},{"name":"Boe","shortCode":"06"},{"name":"Buada","shortCode":"07"},{"name":"Denigomodu","shortCode":"08"},{"name":"Ewa","shortCode":"09"},{"name":"Ijuw","shortCode":"10"},{"name":"Meneng","shortCode":"11"},{"name":"Nibok","shortCode":"12"},{"name":"Uaboe","shortCode":"13"},{"name":"Yaren","shortCode":"14"}]},{"countryName":"Nepal","countryShortCode":"NPL","regions":[{"name":"Bagmati","shortCode":"BA"},{"name":"Bheri","shortCode":"BH"},{"name":"Dhawalagiri","shortCode":"DH"},{"name":"Gandaki","shortCode":"GA"},{"name":"Janakpur","shortCode":"JA"},{"name":"Karnali","shortCode":"KA"},{"name":"Kosi","shortCode":"KO"},{"name":"Lumbini","shortCode":"LU"},{"name":"Mahakali","shortCode":"MA"},{"name":"Mechi","shortCode":"ME"},{"name":"Narayani","shortCode":"NA"},{"name":"Rapti","shortCode":"RA"},{"name":"Sagarmatha","shortCode":"SA"},{"name":"Seti","shortCode":"SE"}]},{"countryName":"Netherlands","countryShortCode":"NLD","regions":[{"name":"Drenthe","shortCode":"DR"},{"name":"Flevoland","shortCode":"FL"},{"name":"Friesland","shortCode":"FR"},{"name":"Gelderland","shortCode":"GE"},{"name":"Groningen","shortCode":"GR"},{"name":"Limburg","shortCode":"LI"},{"name":"Noord-Brabant","shortCode":"NB"},{"name":"Noord-Holland","shortCode":"NH"},{"name":"Overijssel","shortCode":"OV"},{"name":"Utrecht","shortCode":"UT"},{"name":"Zeeland","shortCode":"ZE"},{"name":"Zuid-Holland","shortCode":"ZH"}]},{"countryName":"New Caledonia","countryShortCode":"NCL","regions":[{"name":"Iles Loyaute"},{"name":"Nord"},{"name":"Sud"}]},{"countryName":"New Zealand","countryShortCode":"NZL","regions":[{"name":"Auckland","shortCode":"AUK"},{"name":"Bay of Plenty","shortCode":"BOP"},{"name":"Canterbury","shortCode":"CAN"},{"name":"Gisborne","shortCode":"GIS"},{"name":"Hawke\'s Bay","shortCode":"HKB"},{"name":"Marlborough","shortCode":"MBH"},{"name":"Manawatu-Wanganui","shortCode":"MWT"},{"name":"Northland","shortCode":"NTL"},{"name":"Nelson","shortCode":"NSN"},{"name":"Otago","shortCode":"OTA"},{"name":"Southland","shortCode":"STL"},{"name":"Taranaki","shortCode":"TKI"},{"name":"Tasman","shortCode":"TAS"},{"name":"Waikato","shortCode":"WKO"},{"name":"Wellington","shortCode":"WGN"},{"name":"West Coast","shortCode":"WTC"},{"name":"Chatham Islands Territory","shortCode":"CIT"}]},{"countryName":"Nicaragua","countryShortCode":"NIC","regions":[{"name":"Boaco","shortCode":"BO"},{"name":"Carazo","shortCode":"CA"},{"name":"Chinandega","shortCode":"CI"},{"name":"Chontales","shortCode":"CO"},{"name":"Estel\u00ed","shortCode":"ES"},{"name":"Granada","shortCode":"GR"},{"name":"Jinotega","shortCode":"JI"},{"name":"Le\u00f3n","shortCode":"LE"},{"name":"Madriz","shortCode":"MD"},{"name":"Managua","shortCode":"MN"},{"name":"Masaya","shortCode":"MS"},{"name":"Matagalpa","shortCode":"MT"},{"name":"Nueva Segovia","shortCode":"NS"},{"name":"R\u00edo San Juan","shortCode":"SJ"},{"name":"Rivas","shortCode":"RI"},{"name":"Atl\u00e1ntico Norte","shortCode":"AN"},{"name":"Atl\u00e1ntico Sur","shortCode":"AS"}]},{"countryName":"Niger","countryShortCode":"NER","regions":[{"name":"Agadez","shortCode":"1"},{"name":"Diffa","shortCode":"2"},{"name":"Dosso","shortCode":"3"},{"name":"Maradi","shortCode":"4"},{"name":"Niamey","shortCode":"8"},{"name":"Tahoua","shortCode":"5"},{"name":"Tillab\u00e9ri","shortCode":"6"},{"name":"Zinder","shortCode":"7"}]},{"countryName":"Nigeria","countryShortCode":"NGA","regions":[{"name":"Abia","shortCode":"AB"},{"name":"Abuja Federal Capital Territory","shortCode":"FC"},{"name":"Adamawa","shortCode":"AD"},{"name":"Akwa Ibom","shortCode":"AK"},{"name":"Anambra","shortCode":"AN"},{"name":"Bauchi","shortCode":"BA"},{"name":"Bayelsa","shortCode":"BY"},{"name":"Benue","shortCode":"BE"},{"name":"Borno","shortCode":"BO"},{"name":"Cross River","shortCode":"CR"},{"name":"Delta","shortCode":"DE"},{"name":"Ebonyi","shortCode":"EB"},{"name":"Edo","shortCode":"ED"},{"name":"Ekiti","shortCode":"EK"},{"name":"Enugu","shortCode":"EN"},{"name":"Gombe","shortCode":"GO"},{"name":"Imo","shortCode":"IM"},{"name":"Jigawa","shortCode":"JI"},{"name":"Kaduna","shortCode":"KD"},{"name":"Kano","shortCode":"KN"},{"name":"Katsina","shortCode":"KT"},{"name":"Kebbi","shortCode":"KE"},{"name":"Kogi","shortCode":"KO"},{"name":"Kwara","shortCode":"KW"},{"name":"Lagos","shortCode":"LA"},{"name":"Nassarawa","shortCode":"NA"},{"name":"Niger","shortCode":"NI"},{"name":"Ogun","shortCode":"OG"},{"name":"Ondo","shortCode":"ON"},{"name":"Osun","shortCode":"OS"},{"name":"Oyo","shortCode":"OY"},{"name":"Plateau","shortCode":"PL"},{"name":"Rivers","shortCode":"RI"},{"name":"Sokoto","shortCode":"SO"},{"name":"Taraba","shortCode":"TA"},{"name":"Yobe","shortCode":"YO"},{"name":"Zamfara","shortCode":"ZA"}]},{"countryName":"Niue","countryShortCode":"NIU","regions":[{"name":"Niue"}]},{"countryName":"Norfolk Island","countryShortCode":"NFK","regions":[{"name":"Norfolk Island"}]},{"countryName":"Northern Mariana Islands","countryShortCode":"MNP","regions":[{"name":"Northern Islands"},{"name":"Rota"},{"name":"Saipan"},{"name":"Tinian"}]},{"countryName":"Norway","countryShortCode":"NOR","regions":[{"name":"Akershus","shortCode":"02"},{"name":"Aust-Agder","shortCode":"09"},{"name":"Buskerud","shortCode":"06"},{"name":"Finnmark","shortCode":"20"},{"name":"Hedmark","shortCode":"04"},{"name":"Hordaland","shortCode":"12"},{"name":"M\u00f8re og Romsdal","shortCode":"15"},{"name":"Nordland","shortCode":"18"},{"name":"Nord-Tr\u00f8ndelag","shortCode":"17"},{"name":"Oppland","shortCode":"05"},{"name":"Oslo","shortCode":"03"},{"name":"Rogaland","shortCode":"11"},{"name":"Sogn og Fjordane","shortCode":"14"},{"name":"S\u00f8r-Tr\u00f8ndelag","shortCode":"16"},{"name":"Telemark","shortCode":"08"},{"name":"Troms","shortCode":"19"},{"name":"Vest-Agder","shortCode":"10"},{"name":"Vestfold","shortCode":"07"},{"name":"\u00d8stfold","shortCode":"01"},{"name":"Jan Mayen","shortCode":"22"},{"name":"Svalbard","shortCode":"21"}]},{"countryName":"Oman","countryShortCode":"OMN","regions":[{"name":"Ad Dakhiliyah","shortCode":"DA"},{"name":"Al Buraymi","shortCode":"BU"},{"name":"Al Wusta","shortCode":"WU"},{"name":"Az Zahirah","shortCode":"ZA"},{"name":"Janub al Batinah","shortCode":"BS"},{"name":"Janub ash Sharqiyah","shortCode":"SS"},{"name":"Masqat","shortCode":"MA"},{"name":"Musandam","shortCode":"MU"},{"name":"Shamal al Batinah","shortCode":"BJ"},{"name":"Shamal ash Sharqiyah","shortCode":"SJ"},{"name":"Zufar","shortCode":"ZU"}]},{"countryName":"Pakistan","countryShortCode":"PAK","regions":[{"name":"\u0100z\u0101d Kashm\u012br","shortCode":"JK"},{"name":"Bal\u014dchist\u0101n","shortCode":"BA"},{"name":"Gilgit-Baltist\u0101n","shortCode":"GB"},{"name":"Isl\u0101m\u0101b\u0101d","shortCode":"IS"},{"name":"Kha\u012bbar Pakht\u016bnkhw\u0101s","shortCode":"KP"},{"name":"Punj\u0101b","shortCode":"PB"},{"name":"Sindh","shortCode":"SD"},{"name":"Federally Administered Tribal Areas","shortCode":"TA"}]},{"countryName":"Palau","countryShortCode":"PLW","regions":[{"name":"Aimeliik","shortCode":"002"},{"name":"Airai","shortCode":"004"},{"name":"Angaur","shortCode":"010"},{"name":"Hatobohei","shortCode":"050"},{"name":"Kayangel","shortCode":"100"},{"name":"Koror","shortCode":"150"},{"name":"Melekeok","shortCode":"212"},{"name":"Ngaraard","shortCode":"214"},{"name":"Ngarchelong","shortCode":"218"},{"name":"Ngardmau","shortCode":"222"},{"name":"Ngatpang","shortCode":"224"},{"name":"Ngchesar","shortCode":"226"},{"name":"Ngeremlengui","shortCode":"227"},{"name":"Ngiwal","shortCode":"228"},{"name":"Peleliu","shortCode":"350"},{"name":"Sonsoral","shortCode":"350"}]},{"countryName":"Palestine, State of","countryShortCode":"PSE","regions":[{"name":"Ak Khal\u012bl","shortCode":"HBN"},{"name":"Al Quds","shortCode":"JEM"},{"name":"Ar\u012b\u1e29\u0101 wal Aghw\u0101r","shortCode":"JRH"},{"name":"Bayt La\u1e29m","shortCode":"BTH"},{"name":"Dayr al Bala\u1e29","shortCode":"DEB"},{"name":"Ghazzah","shortCode":"GZA"},{"name":"Jan\u012bn","shortCode":"JEN"},{"name":"Kh\u0101n Y\u016bnis","shortCode":"KYS"},{"name":"N\u0101blus","shortCode":"NBS"},{"name":"Qalq\u012byah","shortCode":"QQA"},{"name":"Rafa\u1e29","shortCode":"RFH"},{"name":"R\u0101m All\u0101h wal B\u012brah","shortCode":"RBH"},{"name":"Salf\u012bt","shortCode":"SLT"},{"name":"Sham\u0101l Ghazzah","shortCode":"NGZ"},{"name":"\u0162\u016bb\u0101s","shortCode":"TBS"},{"name":"\u0162\u016blkarm","shortCode":"TKM"}]},{"countryName":"Panama","countryShortCode":"PAN","regions":[{"name":"Bocas del Toro","shortCode":"1"},{"name":"Chiriqu\u00ed","shortCode":"4"},{"name":"Cocl\u00e9","shortCode":"2"},{"name":"Col\u00f3n","shortCode":"3"},{"name":"Dari\u00e9n","shortCode":"5"},{"name":"Ember\u00e1","shortCode":"EM"},{"name":"Herrera","shortCode":"6"},{"name":"Kuna Yala","shortCode":"KY"},{"name":"Los Santos","shortCode":"7"},{"name":"Ng\u00e4be-Bugl\u00e9","shortCode":"NB"},{"name":"Panam\u00e1","shortCode":"8"},{"name":"Panam\u00e1 Oeste","shortCode":"10"},{"name":"Veraguas","shortCode":"9"}]},{"countryName":"Papua New Guinea","countryShortCode":"PNG","regions":[{"name":"Bougainville","shortCode":"NSB"},{"name":"Central","shortCode":"CPM"},{"name":"Chimbu","shortCode":"CPK"},{"name":"East New Britain","shortCode":"EBR"},{"name":"East Sepik","shortCode":"ESW"},{"name":"Eastern Highlands","shortCode":"EHG"},{"name":"Enga","shortCode":"EPW"},{"name":"Gulf","shortCode":"GPK"},{"name":"Hela","shortCode":"HLA"},{"name":"Jiwaka","shortCode":"JWK"},{"name":"Madang","shortCode":"MOM"},{"name":"Manus","shortCode":"MRL"},{"name":"Milne Bay","shortCode":"MBA"},{"name":"Morobe","shortCode":"MPL"},{"name":"Port Moresby","shortCode":"NCD"},{"name":"New Ireland","shortCode":"NIK"},{"name":"Northern","shortCode":"NPP"},{"name":"Southern Highlands","shortCode":"SHM"},{"name":"West New Britain","shortCode":"WBK"},{"name":"West Sepik","shortCode":"SAN"},{"name":"Western","shortCode":"WPD"},{"name":"Western Highlands","shortCode":"WHM"}]},{"countryName":"Paraguay","countryShortCode":"PRY","regions":[{"name":"Alto Paraguay","shortCode":"16"},{"name":"Alto Parana","shortCode":"10"},{"name":"Amambay","shortCode":"13"},{"name":"Asuncion","shortCode":"ASU"},{"name":"Caaguazu","shortCode":"5"},{"name":"Caazapa","shortCode":"6"},{"name":"Canindeyu","shortCode":"14"},{"name":"Central","shortCode":"11"},{"name":"Concepcion","shortCode":"1"},{"name":"Cordillera","shortCode":"3"},{"name":"Guaira","shortCode":"4"},{"name":"Itapua","shortCode":"7"},{"name":"Misiones","shortCode":"8"},{"name":"Neembucu","shortCode":"12"},{"name":"Paraguari","shortCode":"9"},{"name":"Presidente Hayes","shortCode":"15"},{"name":"San Pedro","shortCode":"2"}]},{"countryName":"Peru","countryShortCode":"PER","regions":[{"name":"Amazonas","shortCode":"AMA"},{"name":"Ancash","shortCode":"ANC"},{"name":"Apurimac","shortCode":"APU"},{"name":"Arequipa","shortCode":"ARE"},{"name":"Ayacucho","shortCode":"AYA"},{"name":"Cajamarca","shortCode":"CAJ"},{"name":"Callao","shortCode":"CAL"},{"name":"Cusco","shortCode":"CUS"},{"name":"Huancavelica","shortCode":"HUV"},{"name":"Huanuco","shortCode":"HUC"},{"name":"Ica","shortCode":"ICA"},{"name":"Junin","shortCode":"JUN"},{"name":"La Libertad","shortCode":"LAL"},{"name":"Lambayeque","shortCode":"LAM"},{"name":"Lima","shortCode":"LIM"},{"name":"Loreto","shortCode":"LOR"},{"name":"Madre de Dios","shortCode":"MDD"},{"name":"Moquegua","shortCode":"MOQ"},{"name":"Municipalidad Metropolitana de Lima","shortCode":"LMA"},{"name":"Pasco","shortCode":"PAS"},{"name":"Piura","shortCode":"PIU"},{"name":"Puno","shortCode":"PUN"},{"name":"San Martin","shortCode":"SAM"},{"name":"Tacna","shortCode":"TAC"},{"name":"Tumbes","shortCode":"TUM"},{"name":"Ucayali","shortCode":"UCA"}]},{"countryName":"Philippines","countryShortCode":"PHL","regions":[{"name":"Abra","shortCode":"ABR"},{"name":"Agusan del Norte","shortCode":"AGN"},{"name":"Agusan del Sur","shortCode":"AGS"},{"name":"Aklan","shortCode":"AKL"},{"name":"Albay","shortCode":"ALB"},{"name":"Antique","shortCode":"ANT"},{"name":"Apayao","shortCode":"APA"},{"name":"Aurora","shortCode":"AUR"},{"name":"Basilan","shortCode":"BAS"},{"name":"Bataan","shortCode":"BAN"},{"name":"Batanes","shortCode":"BTN"},{"name":"Batangas","shortCode":"BTG"},{"name":"Benguet","shortCode":"BEN"},{"name":"Biliran","shortCode":"BIL"},{"name":"Bohol","shortCode":"BOH"},{"name":"Bukidnon","shortCode":"BUK"},{"name":"Bulacan","shortCode":"BUL"},{"name":"Cagayan","shortCode":"CAG"},{"name":"Camarines Norte","shortCode":"CAN"},{"name":"Camarines Sur","shortCode":"CAS"},{"name":"Camiguin","shortCode":"CAM"},{"name":"Capiz","shortCode":"CAP"},{"name":"Catanduanes","shortCode":"CAT"},{"name":"Cavite","shortCode":"CAV"},{"name":"Cebu","shortCode":"CEB"},{"name":"Compostela","shortCode":"COM"},{"name":"Cotabato","shortCode":"NCO"},{"name":"Davao del Norte","shortCode":"DAV"},{"name":"Davao del Sur","shortCode":"DAS"},{"name":"Davao Occidental","shortCode":"DVO"},{"name":"Davao Oriental","shortCode":"DAO"},{"name":"Dinagat Islands","shortCode":"DIN"},{"name":"Eastern Samar","shortCode":"EAS"},{"name":"Guimaras","shortCode":"GUI"},{"name":"Ifugao","shortCode":"IFU"},{"name":"Ilocos Norte","shortCode":"ILN"},{"name":"Ilocos Sur","shortCode":"ILS"},{"name":"Iloilo","shortCode":"ILI"},{"name":"Isabela","shortCode":"ISA"},{"name":"Kalinga","shortCode":"KAL"},{"name":"La Union","shortCode":"LUN"},{"name":"Laguna","shortCode":"LAG"},{"name":"Lanao del Norte","shortCode":"LAN"},{"name":"Lanao del Sur","shortCode":"LAS"},{"name":"Leyte","shortCode":"LEY"},{"name":"Maguindanao","shortCode":"MAG"},{"name":"Masbate","shortCode":"MAS"},{"name":"Metro Manila","shortCode":"00"},{"name":"Mindoro Occidental","shortCode":"MDC"},{"name":"Mindoro Oriental","shortCode":"MDR"},{"name":"Misamis Occidental","shortCode":"MSC"},{"name":"Misamis Oriental","shortCode":"MSR"},{"name":"Mountain Province","shortCode":"MOU"},{"name":"Negros Occidental","shortCode":"NEC"},{"name":"Negros Oriental","shortCode":"NER"},{"name":"Northern Samar","shortCode":"NSA"},{"name":"Nueva Ecija","shortCode":"NUE"},{"name":"Nueva Vizcaya","shortCode":"NUV"},{"name":"Palawan","shortCode":"PLW"},{"name":"Pampanga","shortCode":"PAM"},{"name":"Pangasinan","shortCode":"PAN"},{"name":"Quezon","shortCode":"QUE"},{"name":"Quirino","shortCode":"QUI"},{"name":"Rizal","shortCode":"RIZ"},{"name":"Romblon","shortCode":"ROM"},{"name":"Samar","shortCode":"WSA"},{"name":"Sarangani","shortCode":"SAR"},{"name":"Siquijor","shortCode":"SIG"},{"name":"Sorsogon","shortCode":"SOR"},{"name":"Southern Leyte","shortCode":"SLE"},{"name":"Sultan Kudarat","shortCode":"AUK"},{"name":"Sulu","shortCode":"SLU"},{"name":"Surigao del Norte","shortCode":"SUN"},{"name":"Surigao del Sur","shortCode":"SUR"},{"name":"Tarlac","shortCode":"TAR"},{"name":"Tawi-Tawi","shortCode":"TAW"},{"name":"Zambales","shortCode":"ZMB"},{"name":"Zamboanga del Norte","shortCode":"ZAN"},{"name":"Zamboanga del Sur","shortCode":"ZAS"},{"name":"Zamboanga Sibugay","shortCode":"ZSI"}]},{"countryName":"Pitcairn","countryShortCode":"PCN","regions":[{"name":"Pitcairn Islands"}]},{"countryName":"Poland","countryShortCode":"POL","regions":[{"name":"Dolno\u015bl\u0105skie","shortCode":"DS"},{"name":"Kujawsko-pomorskie","shortCode":"KP"},{"name":"\u0141\u00f3dzkie","shortCode":"LD"},{"name":"Lubelskie","shortCode":"LU"},{"name":"Lubuskie","shortCode":"LB"},{"name":"Malopolskie","shortCode":"MA"},{"name":"Mazowieckie","shortCode":"MZ"},{"name":"Opolskie","shortCode":"OP"},{"name":"Podkarpackie","shortCode":"PK"},{"name":"Podlaskie","shortCode":"PD"},{"name":"Pomorskie","shortCode":"PM"},{"name":"\u015al\u0105skie","shortCode":"SL"},{"name":"\u015awi\u0119tokrzyskie","shortCode":"SK"},{"name":"Warmi\u0144sko-mazurskie","shortCode":"WN"},{"name":"Wielkopolskie","shortCode":"WP"},{"name":"Zachodniopomorskie","shortCode":"ZP"}]},{"countryName":"Portugal","countryShortCode":"PRT","regions":[{"name":"A\u00e7ores","shortCode":"20"},{"name":"Aveiro","shortCode":"01"},{"name":"Beja","shortCode":"02"},{"name":"Braga","shortCode":"03"},{"name":"Bragan\u00e7a","shortCode":"04"},{"name":"Castelo Branco","shortCode":"05"},{"name":"Coimbra","shortCode":"06"},{"name":"\u00c9vora","shortCode":"07"},{"name":"Faro","shortCode":"08"},{"name":"Guarda","shortCode":"09"},{"name":"Leiria","shortCode":"10"},{"name":"Lisboa","shortCode":"11"},{"name":"Madeira","shortCode":"30"},{"name":"Portalegre","shortCode":"12"},{"name":"Porto","shortCode":"13"},{"name":"Santar\u00e9m","shortCode":"14"},{"name":"Set\u00fabal","shortCode":"15"},{"name":"Viana do Castelo","shortCode":"16"},{"name":"Vila Real","shortCode":"17"},{"name":"Viseu","shortCode":"18"}]},{"countryName":"Puerto Rico","countryShortCode":"PRI","regions":[{"name":"Adjuntas"},{"name":"Aguada"},{"name":"Aguadilla"},{"name":"Aguas Buenas"},{"name":"Aibonito"},{"name":"Anasco"},{"name":"Arecibo"},{"name":"Arroyo"},{"name":"Barceloneta"},{"name":"Barranquitas"},{"name":"Bayamon"},{"name":"Cabo Rojo"},{"name":"Caguas"},{"name":"Camuy"},{"name":"Canovanas"},{"name":"Carolina"},{"name":"Cat"},{"name":"Ceiba"},{"name":"Ciales"},{"name":"Cidra"},{"name":"Coamo"},{"name":"Comerio"},{"name":"Corozal"},{"name":"Culebra"},{"name":"Dorado"},{"name":"Fajardo"},{"name":"Florida"},{"name":"Guanica"},{"name":"Guayama"},{"name":"Guayanilla"},{"name":"Guaynabo"},{"name":"Gurabo"},{"name":"Hatillo"},{"name":"Hormigueros"},{"name":"Humacao"},{"name":"Isabe"},{"name":"Juana Diaz"},{"name":"Juncos"},{"name":"Lajas"},{"name":"Lares"},{"name":"Las Marias"},{"name":"Las oiza"},{"name":"Luquillo"},{"name":"Manati"},{"name":"Maricao"},{"name":"Maunabo"},{"name":"Mayaguez"},{"name":"Moca"},{"name":"Morovis"},{"name":"Naguabo"},{"name":"Naranjito"},{"name":"Orocovis"},{"name":"Patillas"},{"name":"Penuelas"},{"name":"Ponce"},{"name":"Quebradillas"},{"name":"Rincon"},{"name":"Rio Grande"},{"name":"Sabana linas"},{"name":"San German"},{"name":"San Juan"},{"name":"San Lorenzo"},{"name":"San Sebastian"},{"name":"Santa Isabel"},{"name":"Toa Alta"},{"name":"Toa Baja"},{"name":"Trujillo Alto"},{"name":"Utuado"},{"name":"Vega Alta"},{"name":"Vega ues"},{"name":"Villalba"},{"name":"Yabucoa"},{"name":"Yauco"}]},{"countryName":"Qatar","countryShortCode":"QAT","regions":[{"name":"Ad Daw\u1e29ah","shortCode":"DA"},{"name":"Al Khawr wa adh Dhakh\u012brah","shortCode":"KH"},{"name":"Al Wakrah","shortCode":"WA"},{"name":"Ar Rayy\u0101n","shortCode":"RA"},{"name":"Ash Sham\u0101l","shortCode":"MS"},{"name":"Az\u0327 Za\u0327`\u0101yin","shortCode":"ZA"},{"name":"Umm \u015eal\u0101l","shortCode":"US"}]},{"countryName":"R\u00e9union","countryShortCode":"REU","regions":[{"name":"R\u00e9union"}]},{"countryName":"Romania","countryShortCode":"ROU","regions":[{"name":"Alba","shortCode":"AB"},{"name":"Arad","shortCode":"AR"},{"name":"Arges","shortCode":"AG"},{"name":"Bacau","shortCode":"BC"},{"name":"Bihor","shortCode":"BH"},{"name":"Bistrita-Nasaud","shortCode":"BN"},{"name":"Botosani","shortCode":"BT"},{"name":"Braila","shortCode":"BR"},{"name":"Brasov","shortCode":"BV"},{"name":"Bucuresti","shortCode":"B"},{"name":"Buzau","shortCode":"BZ"},{"name":"Calarasi","shortCode":"CL"},{"name":"Caras-Severin","shortCode":"CS"},{"name":"Cluj","shortCode":"CJ"},{"name":"Constanta","shortCode":"CT"},{"name":"Covasna","shortCode":"CV"},{"name":"Dambovita","shortCode":"DB"},{"name":"Dolj","shortCode":"DJ"},{"name":"Galati","shortCode":"GL"},{"name":"Giurgiu","shortCode":"GR"},{"name":"Gorj","shortCode":"GJ"},{"name":"Harghita","shortCode":"HR"},{"name":"Hunedoara","shortCode":"HD"},{"name":"Ialomita","shortCode":"IL"},{"name":"Iasi","shortCode":"IS"},{"name":"Maramures","shortCode":"MM"},{"name":"Mehedinti","shortCode":"MH"},{"name":"Mures","shortCode":"MS"},{"name":"Neamt","shortCode":"NT"},{"name":"Olt","shortCode":"OT"},{"name":"Prahova","shortCode":"PH"},{"name":"Salaj","shortCode":"SJ"},{"name":"Satu Mare","shortCode":"SM"},{"name":"Sibiu","shortCode":"SB"},{"name":"Suceava","shortCode":"SV"},{"name":"Teleorman","shortCode":"TR"},{"name":"Timis","shortCode":"TM"},{"name":"Tulcea","shortCode":"TL"},{"name":"Valcea","shortCode":"VL"},{"name":"Vaslui","shortCode":"VS"},{"name":"Vrancea","shortCode":"VN"}]},{"countryName":"Russian Federation","countryShortCode":"RUS","regions":[{"name":"Republic of Adygea","shortCode":"AD"},{"name":"Republic of Altai (Gorno-Altaysk)","shortCode":"AL"},{"name":"Altai Krai","shortCode":"ALT"},{"name":"Amur Oblast","shortCode":"AMU"},{"name":"Arkhangelsk Oblast","shortCode":"ARK"},{"name":"Astrakhan Oblast","shortCode":"AST"},{"name":"Republic of Bashkortostan","shortCode":"BA"},{"name":"Belgorod Oblast","shortCode":"BEL"},{"name":"Bryansk Oblast","shortCode":"BRY"},{"name":"Republic of Buryatia","shortCode":"BU"},{"name":"Chechen Republic","shortCode":"CE"},{"name":"Chelyabinsk Oblast","shortCode":"CHE"},{"name":"Chukotka Autonomous Okrug","shortCode":"CHU"},{"name":"Chuvash Republic","shortCode":"CU"},{"name":"Republic of Dagestan","shortCode":"DA"},{"name":"Republic of Ingushetia","shortCode":"IN"},{"name":"Irkutsk Oblast","shortCode":"IRK"},{"name":"Ivanovo Oblast","shortCode":"IVA"},{"name":"Jewish Autonomous Oblast","shortCode":"JEW"},{"name":"Kabardino-Balkar Republic","shortCode":"KB"},{"name":"Kaliningrad Oblast","shortCode":"KLN"},{"name":"Republic of Kalmykia","shortCode":"KL"},{"name":"Kaluga Oblast","shortCode":"KLU"},{"name":"Kamchatka Krai","shortCode":"KAM"},{"name":"Karachay-Cherkess Republic","shortCode":"KC"},{"name":"Republic of Karelia","shortCode":"KR"},{"name":"Khabarovsk Krai","shortCode":"KHA"},{"name":"Republic of Khakassia","shortCode":"KK"},{"name":"Khanty-Mansi Autonomous Okrug - Yugra","shortCode":"KHM"},{"name":"Kemerovo Oblast","shortCode":"KEM"},{"name":"Kirov Oblast","shortCode":"KIR"},{"name":"Komi Republic","shortCode":"KO"},{"name":"Kostroma Oblast","shortCode":"KOS"},{"name":"Krasnodar Krai","shortCode":"KDA"},{"name":"Krasnoyarsk Krai","shortCode":"KYA"},{"name":"Kurgan Oblast","shortCode":"KGN"},{"name":"Kursk Oblast","shortCode":"KRS"},{"name":"Leningrad Oblast","shortCode":"LEN"},{"name":"Lipetsk Oblast","shortCode":"LIP"},{"name":"Magadan Oblast","shortCode":"MAG"},{"name":"Mari El Republic","shortCode":"ME"},{"name":"Republic of Mordovia","shortCode":"MO"},{"name":"Moscow Oblast","shortCode":"MOS"},{"name":"Moscow","shortCode":"MOW"},{"name":"Murmansk Oblast","shortCode":"MU"},{"name":"Nenets Autonomous Okrug","shortCode":"NEN"},{"name":"Nizhny Novgorod Oblast","shortCode":"NIZ"},{"name":"Novgorod Oblast","shortCode":"NGR"},{"name":"Novosibirsk Oblast","shortCode":"NVS"},{"name":"Omsk Oblast","shortCode":"OMS"},{"name":"Orenburg Oblast","shortCode":"ORE"},{"name":"Oryol Oblast","shortCode":"ORL"},{"name":"Penza Oblast","shortCode":"PNZ"},{"name":"Perm Krai","shortCode":"PER"},{"name":"Primorsky Krai","shortCode":"PRI"},{"name":"Pskov Oblast","shortCode":"PSK"},{"name":"Rostov Oblast","shortCode":"ROS"},{"name":"Ryazan Oblast","shortCode":"RYA"},{"name":"Saint Petersburg","shortCode":"SPE"},{"name":"Sakha (Yakutia) Republic","shortCode":"SA"},{"name":"Sakhalin Oblast","shortCode":"SAK"},{"name":"Samara Oblast","shortCode":"SAM"},{"name":"Saratov Oblast","shortCode":"SAR"},{"name":"Republic of North Ossetia-Alania","shortCode":"NOA"},{"name":"Smolensk Oblast","shortCode":"SMO"},{"name":"Stavropol Krai","shortCode":"STA"},{"name":"Sverdlovsk Oblast","shortCode":"SVE"},{"name":"Tambov Oblast","shortCode":"TAM"},{"name":"Republic of Tatarstan","shortCode":"TA"},{"name":"Tomsk Oblast","shortCode":"TOM"},{"name":"Tuva Republic","shortCode":"TU"},{"name":"Tula Oblast","shortCode":"TUL"},{"name":"Tver Oblast","shortCode":"TVE"},{"name":"Tyumen Oblast","shortCode":"TYU"},{"name":"Udmurt Republic","shortCode":"UD"},{"name":"Ulyanovsk Oblast","shortCode":"ULY"},{"name":"Vladimir Oblast","shortCode":"VLA"},{"name":"Volgograd Oblast","shortCode":"VGG"},{"name":"Vologda Oblast","shortCode":"VLG"},{"name":"Voronezh Oblast","shortCode":"VOR"},{"name":"Yamalo-Nenets Autonomous Okrug","shortCode":"YAN"},{"name":"Yaroslavl Oblast","shortCode":"YAR"},{"name":"Zabaykalsky Krai","shortCode":"ZAB"}]},{"countryName":"Rwanda","countryShortCode":"RWA","regions":[{"name":"Kigali","shortCode":"01"},{"name":"Eastern","shortCode":"02"},{"name":"Northern","shortCode":"03"},{"name":"Western","shortCode":"04"},{"name":"Southern","shortCode":"05"}]},{"countryName":"Saint Barth\u00e9lemy","countryShortCode":"BLM","regions":[{"name":"Au Vent","shortCode":"02"},{"name":"Sous le Vent","shortCode":"01"}]},{"countryName":"Saint Helena, Ascension and Tristan da Cunha","countryShortCode":"SHN","regions":[{"name":"Ascension","shortCode":"AC"},{"name":"Saint Helena","shortCode":"HL"},{"name":"Tristan da Cunha","shortCode":"TA"}]},{"countryName":"Saint Kitts and Nevis","countryShortCode":"KNA","regions":[{"name":"Saint Kitts","shortCode":"K"},{"name":"Nevis","shortCode":"N"}]},{"countryName":"Saint Lucia","countryShortCode":"LCA","regions":[{"name":"Anse-la-Raye","shortCode":"01"},{"name":"Canaries","shortCode":"12"},{"name":"Castries","shortCode":"02"},{"name":"Choiseul","shortCode":"03"},{"name":"Dennery","shortCode":"05"},{"name":"Gros Islet","shortCode":"06"},{"name":"Laborie","shortCode":"07"},{"name":"Micoud","shortCode":"08"},{"name":"Soufriere","shortCode":"10"},{"name":"Vieux Fort","shortCode":"11"}]},{"countryName":"Saint Martin","countryShortCode":"MAF","regions":[{"name":"Saint Martin"}]},{"countryName":"Saint Pierre and Miquelon","countryShortCode":"SPM","regions":[{"name":"Miquelon"},{"name":"Saint Pierre"}]},{"countryName":"Saint Vincent and the Grenadines","countryShortCode":"VCT","regions":[{"name":"Charlotte","shortCode":"01"},{"name":"Grenadines","shortCode":"06"},{"name":"Saint Andrew","shortCode":"02"},{"name":"Saint David","shortCode":"03"},{"name":"Saint George","shortCode":"04"},{"name":"Saint Patrick","shortCode":"05"}]},{"countryName":"Samoa","countryShortCode":"WSM","regions":[{"name":"A\'ana","shortCode":"AA"},{"name":"Aiga-i-le-Tai","shortCode":"AL"},{"name":"Atua","shortCode":"AT"},{"name":"Fa\'asaleleaga","shortCode":"FA"},{"name":"Gaga\'emauga","shortCode":"GE"},{"name":"Gagaifomauga","shortCode":"GI"},{"name":"Palauli","shortCode":"PA"},{"name":"Satupa\'itea","shortCode":"SA"},{"name":"Tuamasaga","shortCode":"TU"},{"name":"Va\'a-o-Fonoti","shortCode":"VF"},{"name":"Vaisigano","shortCode":"VS"}]},{"countryName":"San Marino","countryShortCode":"SMR","regions":[{"name":"Acquaviva","shortCode":"01"},{"name":"Borgo Maggiore","shortCode":"06"},{"name":"Chiesanuova","shortCode":"02"},{"name":"Domagnano","shortCode":"03"},{"name":"Faetano","shortCode":"04"},{"name":"Fiorentino","shortCode":"05"},{"name":"Montegiardino","shortCode":"08"},{"name":"San Marino","shortCode":"07"},{"name":"Serravalle","shortCode":"09"}]},{"countryName":"Sao Tome and Principe","countryShortCode":"STP","regions":[{"name":"Principe","shortCode":"P"},{"name":"Sao Tome","shortCode":"S"}]},{"countryName":"Saudi Arabia","countryShortCode":"SAU","regions":[{"name":"\'Asir","shortCode":"14"},{"name":"Al Bahah","shortCode":"11"},{"name":"Al Hudud ash Shamaliyah","shortCode":"08"},{"name":"Al Jawf","shortCode":"12"},{"name":"Al Madinah al Munawwarah","shortCode":"03"},{"name":"Al Qasim","shortCode":"05"},{"name":"Ar Riyad","shortCode":"01"},{"name":"Ash Sharqiyah","shortCode":"04"},{"name":"Ha\'il","shortCode":"06"},{"name":"Jazan","shortCode":"09"},{"name":"Makkah al Mukarramah","shortCode":"02"},{"name":"Najran","shortCode":"10"},{"name":"Tabuk","shortCode":"07"}]},{"countryName":"Senegal","countryShortCode":"SEN","regions":[{"name":"Dakar","shortCode":"DK"},{"name":"Diourbel","shortCode":"DB"},{"name":"Fatick","shortCode":"FK"},{"name":"Kaffrine","shortCode":"KA"},{"name":"Kaolack","shortCode":"KL"},{"name":"Kedougou","shortCode":"KE"},{"name":"Kolda","shortCode":"KD"},{"name":"Louga","shortCode":"LG"},{"name":"Matam","shortCode":"MT"},{"name":"Saint-Louis","shortCode":"SL"},{"name":"Sedhiou","shortCode":"SE"},{"name":"Tambacounda","shortCode":"TC"},{"name":"Thies","shortCode":"TH"},{"name":"Ziguinchor","shortCode":"ZG"}]},{"countryName":"Serbia","countryShortCode":"SRB","regions":[{"name":"Beograd (Belgrade)","shortCode":"00"},{"name":"Borski","shortCode":"14"},{"name":"Brani\u010devski","shortCode":"11"},{"name":"Jablani\u010dki","shortCode":"23"},{"name":"Ju\u017enoba\u010dki","shortCode":"06"},{"name":"Ju\u017enobanatski","shortCode":"04"},{"name":"Kolubarski","shortCode":"09"},{"name":"Kosovski","shortCode":"25"},{"name":"Kosovsko-Mitrova\u010dki","shortCode":"28"},{"name":"Kosovsko-Pomoravski","shortCode":"29"},{"name":"Ma\u010dvanski","shortCode":"08"},{"name":"Moravi\u010dki","shortCode":"17"},{"name":"Ni\u0161avski","shortCode":"20"},{"name":"P\u010dinjski","shortCode":"24"},{"name":"Pe\u0107ki","shortCode":"26"},{"name":"Pirotski","shortCode":"22"},{"name":"Podunavski","shortCode":"10"},{"name":"Pomoravski","shortCode":"13"},{"name":"Prizrenski","shortCode":"27"},{"name":"Rasinski","shortCode":"19"},{"name":"Ra\u0161ki","shortCode":"18"},{"name":"Severnoba\u010dki","shortCode":"01"},{"name":"Severnobanatski","shortCode":"03"},{"name":"Srednjebanatski","shortCode":"02"},{"name":"Sremski","shortCode":"07"},{"name":"\u0160umadijski","shortCode":"12"},{"name":"Topli\u010dki","shortCode":"21"},{"name":"Zaje\u010darski","shortCode":"15"},{"name":"Zapadnoba\u010dki","shortCode":"05"},{"name":"Zlatiborski","shortCode":"16"}]},{"countryName":"Seychelles","countryShortCode":"SYC","regions":[{"name":"Anse aux Pins","shortCode":"01"},{"name":"Anse Boileau","shortCode":"02"},{"name":"Anse Etoile","shortCode":"03"},{"name":"Anse Royale","shortCode":"05"},{"name":"Anu Cap","shortCode":"04"},{"name":"Baie Lazare","shortCode":"06"},{"name":"Baie Sainte Anne","shortCode":"07"},{"name":"Beau Vallon","shortCode":"08"},{"name":"Bel Air","shortCode":"09"},{"name":"Bel Ombre","shortCode":"10"},{"name":"Cascade","shortCode":"11"},{"name":"Glacis","shortCode":"12"},{"name":"Grand\'Anse Mahe","shortCode":"13"},{"name":"Grand\'Anse Praslin","shortCode":"14"},{"name":"La Digue","shortCode":"15"},{"name":"La Riviere Anglaise","shortCode":"16"},{"name":"Les Mamelles","shortCode":"24"},{"name":"Mont Buxton","shortCode":"17"},{"name":"Mont Fleuri","shortCode":"18"},{"name":"Plaisance","shortCode":"19"},{"name":"Pointe La Rue","shortCode":"20"},{"name":"Port Glaud","shortCode":"21"},{"name":"Roche Caiman","shortCode":"25"},{"name":"Saint Louis","shortCode":"22"},{"name":"Takamaka","shortCode":"23"}]},{"countryName":"Sierra Leone","countryShortCode":"SLE","regions":[{"name":"Eastern","shortCode":"E"},{"name":"Northern","shortCode":"N"},{"name":"Southern","shortCode":"S"},{"name":"Western","shortCode":"W"}]},{"countryName":"Singapore","countryShortCode":"SGP","regions":[{"name":"Central Singapore","shortCode":"01"},{"name":"North East","shortCode":"02"},{"name":"North West","shortCode":"03"},{"name":"South East","shortCode":"04"},{"name":"South West","shortCode":"05"}]},{"countryName":"Sint Maarten (Dutch part)","countryShortCode":"SXM","regions":[{"name":"Sint Maarten"}]},{"countryName":"Slovakia","countryShortCode":"SVK","regions":[{"name":"Banskobystricky","shortCode":"BC"},{"name":"Bratislavsky","shortCode":"BL"},{"name":"Kosicky","shortCode":"KI"},{"name":"Nitriansky","shortCode":"NI"},{"name":"Presovsky","shortCode":"PV"},{"name":"Trenciansky","shortCode":"TC"},{"name":"Trnavsky","shortCode":"TA"},{"name":"Zilinsky","shortCode":"ZI"}]},{"countryName":"Slovenia","countryShortCode":"SVN","regions":[{"name":"Ajdovscina","shortCode":"001"},{"name":"Apace","shortCode":"195"},{"name":"Beltinci","shortCode":"002"},{"name":"Benedikt","shortCode":"148"},{"name":"Bistrica ob Sotli","shortCode":"149"},{"name":"Bled","shortCode":"003"},{"name":"Bloke","shortCode":"150"},{"name":"Bohinj","shortCode":"004"},{"name":"Borovnica","shortCode":"005"},{"name":"Bovec","shortCode":"006"},{"name":"Braslovce","shortCode":"151"},{"name":"Brda","shortCode":"007"},{"name":"Brezice","shortCode":"009"},{"name":"Brezovica","shortCode":"008"},{"name":"Cankova","shortCode":"152"},{"name":"Celje","shortCode":"011"},{"name":"Cerklje na Gorenjskem","shortCode":"012"},{"name":"Cerknica","shortCode":"013"},{"name":"Cerkno","shortCode":"014"},{"name":"Cerkvenjak","shortCode":"153"},{"name":"Cirkulane","shortCode":"196"},{"name":"Crensovci","shortCode":"015"},{"name":"Crna na Koroskem","shortCode":"016"},{"name":"Crnomelj","shortCode":"017"},{"name":"Destrnik","shortCode":"018"},{"name":"Divaca","shortCode":"019"},{"name":"Dobje","shortCode":"154"},{"name":"Dobrepolje","shortCode":"020"},{"name":"Dobrna","shortCode":"155"},{"name":"Dobrova-Polhov Gradec","shortCode":"021"},{"name":"Dobrovnik","shortCode":"156"},{"name":"Dol pri Ljubljani","shortCode":"022"},{"name":"Dolenjske Toplice","shortCode":"157"},{"name":"Domzale","shortCode":"023"},{"name":"Dornava","shortCode":"024"},{"name":"Dravograd","shortCode":"025"},{"name":"Duplek","shortCode":"026"},{"name":"Gorenja Vas-Poljane","shortCode":"027"},{"name":"Gorisnica","shortCode":"028"},{"name":"Gorje","shortCode":"207"},{"name":"Gornja Radgona","shortCode":"029"},{"name":"Gornji Grad","shortCode":"030"},{"name":"Gornji Petrovci","shortCode":"031"},{"name":"Grad","shortCode":"158"},{"name":"Grosuplje","shortCode":"032"},{"name":"Hajdina","shortCode":"159"},{"name":"Hoce-Slivnica","shortCode":"160"},{"name":"Hodos","shortCode":"161"},{"name":"Horjul","shortCode":"162"},{"name":"Hrastnik","shortCode":"034"},{"name":"Hrpelje-Kozina","shortCode":"035"},{"name":"Idrija","shortCode":"036"},{"name":"Ig","shortCode":"037"},{"name":"Ilirska Bistrica","shortCode":"038"},{"name":"Ivancna Gorica","shortCode":"039"},{"name":"Izola","shortCode":"040s"},{"name":"Jesenice","shortCode":"041"},{"name":"Jursinci","shortCode":"042"},{"name":"Kamnik","shortCode":"043"},{"name":"Kanal","shortCode":"044"},{"name":"Kidricevo","shortCode":"045"},{"name":"Kobarid","shortCode":"046"},{"name":"Kobilje","shortCode":"047"},{"name":"Kocevje","shortCode":"048"},{"name":"Komen","shortCode":"049"},{"name":"Komenda","shortCode":"164"},{"name":"Koper","shortCode":"050"},{"name":"Kodanjevica na Krki","shortCode":"197"},{"name":"Kostel","shortCode":"165"},{"name":"Kozje","shortCode":"051"},{"name":"Kranj","shortCode":"052"},{"name":"Kranjska Gora","shortCode":"053"},{"name":"Krizevci","shortCode":"166"},{"name":"Krsko","shortCode":"054"},{"name":"Kungota","shortCode":"055"},{"name":"Kuzma","shortCode":"056"},{"name":"Lasko","shortCode":"057"},{"name":"Lenart","shortCode":"058"},{"name":"Lendava","shortCode":"059"},{"name":"Litija","shortCode":"068"},{"name":"Ljubljana","shortCode":"061"},{"name":"Ljubno","shortCode":"062"},{"name":"Ljutomer","shortCode":"063"},{"name":"Log-Dragomer","shortCode":"208"},{"name":"Logatec","shortCode":"064"},{"name":"Loska Dolina","shortCode":"065"},{"name":"Loski Potok","shortCode":"066"},{"name":"Lovrenc na Pohorju","shortCode":"167"},{"name":"Lukovica","shortCode":"068"},{"name":"Luce","shortCode":"067"},{"name":"Majsperk","shortCode":"069"},{"name":"Makole","shortCode":"198"},{"name":"Maribor","shortCode":"070"},{"name":"Markovci","shortCode":"168"},{"name":"Medvode","shortCode":"071"},{"name":"Menges","shortCode":"072"},{"name":"Metlika","shortCode":"073"},{"name":"Mezica","shortCode":"074"},{"name":"Miklavz na Dravskem Polju","shortCode":"169"},{"name":"Miren-Kostanjevica","shortCode":"075"},{"name":"Mirna","shortCode":"212"},{"name":"Mirna Pec","shortCode":"170"},{"name":"Mislinja","shortCode":"076"},{"name":"Mokronog-Trebelno","shortCode":"199"},{"name":"Moravce","shortCode":"077"},{"name":"Moravske Toplice","shortCode":"078"},{"name":"Mozirje","shortCode":"079"},{"name":"Murska Sobota","shortCode":"080"},{"name":"Naklo","shortCode":"082"},{"name":"Nazarje","shortCode":"083"},{"name":"Nova Gorica","shortCode":"084"},{"name":"Novo Mesto","shortCode":"085"},{"name":"Odranci","shortCode":"086"},{"name":"Ormoz","shortCode":"087"},{"name":"Osilnica","shortCode":"088"},{"name":"Pesnica","shortCode":"089"},{"name":"Piran","shortCode":"090"},{"name":"Pivka","shortCode":"091"},{"name":"Podcetrtek","shortCode":"092"},{"name":"Podlehnik","shortCode":"172"},{"name":"Podvelka","shortCode":"093"},{"name":"Poljcane","shortCode":"200"},{"name":"Postojna","shortCode":"094"},{"name":"Prebold","shortCode":"174"},{"name":"Preddvor","shortCode":"095"},{"name":"Prevalje","shortCode":"175"},{"name":"Ptuj","shortCode":"096"},{"name":"Race-Fram","shortCode":"098"},{"name":"Radece","shortCode":"099"},{"name":"Radenci","shortCode":"100"},{"name":"Radlje ob Dravi","shortCode":"101"},{"name":"Radovljica","shortCode":"102"},{"name":"Ravne na Koroskem","shortCode":"103"},{"name":"Razkrizje","shortCode":"176"},{"name":"Recica ob Savinji","shortCode":"209"},{"name":"Rence-Vogrsko","shortCode":"201"},{"name":"Ribnica","shortCode":"104"},{"name":"Ribnica na Poboriu","shortCode":"177"},{"name":"Rogaska Slatina","shortCode":"106"},{"name":"Rogasovci","shortCode":"105"},{"name":"Rogatec","shortCode":"107"},{"name":"Ruse","shortCode":"108"},{"name":"Salovci","shortCode":"033"},{"name":"Selnica ob Dravi","shortCode":"178"},{"name":"Semic","shortCode":"109"},{"name":"Sempeter-Vrtojba","shortCode":"183"},{"name":"Sencur","shortCode":"117"},{"name":"Sentilj","shortCode":"118"},{"name":"Sentjernej","shortCode":"119"},{"name":"Sentjur","shortCode":"120"},{"name":"Sentrupert","shortCode":"211"},{"name":"Sevnica","shortCode":"110"},{"name":"Sezana","shortCode":"111"},{"name":"Skocjan","shortCode":"121"},{"name":"Skofja Loka","shortCode":"122"},{"name":"Skofljica","shortCode":"123"},{"name":"Slovenj Gradec","shortCode":"112"},{"name":"Slovenska Bistrica","shortCode":"113"},{"name":"Slovenske Konjice","shortCode":"114"},{"name":"Smarje pri elsah","shortCode":"124"},{"name":"Smarjeske Toplice","shortCode":"206"},{"name":"Smartno ob Paki","shortCode":"125"},{"name":"Smartno pri Litiji","shortCode":"194"},{"name":"Sodrazica","shortCode":"179"},{"name":"Solcava","shortCode":"180"},{"name":"Sostanj","shortCode":"126"},{"name":"Sredisce ob Dravi","shortCode":"202"},{"name":"Starse","shortCode":"115"},{"name":"Store","shortCode":"127"},{"name":"Straza","shortCode":"203"},{"name":"Sveta Ana","shortCode":"181"},{"name":"Sveta Trojica v Slovenskih Goricah","shortCode":"204"},{"name":"Sveta Andraz v Slovenskih Goricah","shortCode":"182"},{"name":"Sveti Jurij","shortCode":"116"},{"name":"Sveti Jurij v Slovenskih Goricah","shortCode":"210"},{"name":"Sveti Tomaz","shortCode":"205"},{"name":"Tabor","shortCode":"184"},{"name":"Tisina","shortCode":"128"},{"name":"Tolmin","shortCode":"128"},{"name":"Trbovlje","shortCode":"129"},{"name":"Trebnje","shortCode":"130"},{"name":"Trnovska Vas","shortCode":"185"},{"name":"Trzin","shortCode":"186"},{"name":"Trzic","shortCode":"131"},{"name":"Turnisce","shortCode":"132"},{"name":"Velenje","shortCode":"133"},{"name":"Velika Polana","shortCode":"187"},{"name":"Velike Lasce","shortCode":"134"},{"name":"Verzej","shortCode":"188"},{"name":"Videm","shortCode":"135"},{"name":"Vipava","shortCode":"136"},{"name":"Vitanje","shortCode":"137"},{"name":"Vodice","shortCode":"138"},{"name":"Vojnik","shortCode":"139"},{"name":"Vransko","shortCode":"189"},{"name":"Vrhnika","shortCode":"140"},{"name":"Vuzenica","shortCode":"141"},{"name":"Zagorje ob Savi","shortCode":"142"},{"name":"Zavrc","shortCode":"143"},{"name":"Zrece","shortCode":"144"},{"name":"Zalec","shortCode":"190"},{"name":"Zelezniki","shortCode":"146"},{"name":"Zetale","shortCode":"191"},{"name":"Ziri","shortCode":"147"},{"name":"Zirovnica","shortCode":"192"},{"name":"Zuzemberk","shortCode":"193"}]},{"countryName":"Solomon Islands","countryShortCode":"SLB","regions":[{"name":"Central","shortCode":"CE"},{"name":"Choiseul","shortCode":"CH"},{"name":"Guadalcanal","shortCode":"GU"},{"name":"Honiara","shortCode":"CT"},{"name":"Isabel","shortCode":"IS"},{"name":"Makira-Ulawa","shortCode":"MK"},{"name":"Malaita","shortCode":"ML"},{"name":"Rennell and Bellona","shortCode":"RB"},{"name":"Temotu","shortCode":"TE"},{"name":"Western","shortCode":"WE"}]},{"countryName":"Somalia","countryShortCode":"SOM","regions":[{"name":"Awdal","shortCode":"AW"},{"name":"Bakool","shortCode":"BK"},{"name":"Banaadir","shortCode":"BN"},{"name":"Bari","shortCode":"BR"},{"name":"Bay","shortCode":"BY"},{"name":"Galguduud","shortCode":"GA"},{"name":"Gedo","shortCode":"GE"},{"name":"Hiiraan","shortCode":"HI"},{"name":"Jubbada Dhexe","shortCode":"JD"},{"name":"Jubbada Hoose","shortCode":"JH"},{"name":"Mudug","shortCode":"MU"},{"name":"Nugaal","shortCode":"NU"},{"name":"Sanaag","shortCode":"SA"},{"name":"Shabeellaha Dhexe","shortCode":"SD"},{"name":"Shabeellaha Hoose","shortCode":"SH"},{"name":"Sool","shortCode":"SO"},{"name":"Togdheer","shortCode":"TO"},{"name":"Woqooyi Galbeed","shortCode":"WO"}]},{"countryName":"South Africa","countryShortCode":"ZAF","regions":[{"name":"Eastern Cape","shortCode":"EC"},{"name":"Free State","shortCode":"FS"},{"name":"Gauteng","shortCode":"GT"},{"name":"KwaZulu-Natal","shortCode":"NL"},{"name":"Limpopo","shortCode":"LP"},{"name":"Mpumalanga","shortCode":"MP"},{"name":"Northern Cape","shortCode":"NC"},{"name":"North West","shortCode":"NW"},{"name":"Western Cape","shortCode":"WC"}]},{"countryName":"South Georgia and South Sandwich Islands","countryShortCode":"SGS","regions":[{"name":"Bird Island"},{"name":"Bristol Island"},{"name":"Clerke Rocks"},{"name":"Montagu Island"},{"name":"Saunders Island"},{"name":"South Georgia"},{"name":"Southern Thule"},{"name":"Traversay Islands"}]},{"countryName":"South Sudan","countryShortCode":"SSD","regions":[{"name":"Central Equatoria","shortCode":"CE"},{"name":"Eastern Equatoria","shortCode":"EE"},{"name":"Jonglei","shortCode":"JG"},{"name":"Lakes","shortCode":"LK"},{"name":"Northern Bahr el Ghazal","shortCode":"BN"},{"name":"Unity","shortCode":"UY"},{"name":"Upper Nile","shortCode":"NU"},{"name":"Warrap","shortCode":"WR"},{"name":"Western Bahr el Ghazal","shortCode":"BW"},{"name":"Western Equatoria","shortCode":"EW"}]},{"countryName":"Spain","countryShortCode":"ESP","regions":[{"name":"Albacete","shortCode":"CM"},{"name":"Alicante","shortCode":"VC"},{"name":"Almer\u00eda","shortCode":"AN"},{"name":"Araba\/\u00c1lava","shortCode":"VI"},{"name":"Asturias","shortCode":"O"},{"name":"\u00c1vila","shortCode":"AV"},{"name":"Badajoz","shortCode":"BA"},{"name":"Barcelona","shortCode":"B"},{"name":"Bizkaia","shortCode":"BI"},{"name":"Burgos","shortCode":"BU"},{"name":"C\u00e1ceres","shortCode":"CC"},{"name":"C\u00e1diz","shortCode":"CA"},{"name":"Cantabria","shortCode":"S"},{"name":"Castell\u00f3n","shortCode":"CS"},{"name":"Cueta","shortCode":"CU"},{"name":"Ciudad Real","shortCode":"CR"},{"name":"C\u00f3rdoba","shortCode":"CO"},{"name":"A Coru\u00f1a","shortCode":"C"},{"name":"Cuenca","shortCode":"CU"},{"name":"Gipuzkoa","shortCode":"SS"},{"name":"Girona","shortCode":"GI"},{"name":"Granada","shortCode":"GR"},{"name":"Guadalajara","shortCode":"GU"},{"name":"Huelva","shortCode":"H"},{"name":"Huesca","shortCode":"HU"},{"name":"Illes Balears","shortCode":"PM"},{"name":"Ja\u00e9n","shortCode":"J"},{"name":"Le\u00f3n","shortCode":"LE"},{"name":"Lleida","shortCode":"L"},{"name":"Lugo","shortCode":"LU"},{"name":"Madrid","shortCode":"M"},{"name":"M\u00e1laga","shortCode":"MA"},{"name":"Melilla","shortCode":"ML"},{"name":"Murcia","shortCode":"MU"},{"name":"Navarre","shortCode":"NA"},{"name":"Ourense","shortCode":"OR"},{"name":"Palencia","shortCode":"P"},{"name":"Las Palmas","shortCode":"GC"},{"name":"Pontevedra","shortCode":"PO"},{"name":"La Rioja","shortCode":"LO"},{"name":"Salamanca","shortCode":"SA"},{"name":"Santa Cruz de Tenerife","shortCode":"TF"},{"name":"Segovia","shortCode":"SG"},{"name":"Sevilla","shortCode":"SE"},{"name":"Soria","shortCode":"SO"},{"name":"Tarragona","shortCode":"T"},{"name":"Teruel","shortCode":"TE"},{"name":"Toledo","shortCode":"TO"},{"name":"Valencia","shortCode":"V"},{"name":"Valladolid","shortCode":"VA"},{"name":"Zamora","shortCode":"ZA"},{"name":"Zaragoza","shortCode":"Z"}]},{"countryName":"Sri Lanka","countryShortCode":"LKA","regions":[{"name":"Basnahira","shortCode":"1"},{"name":"Dakunu","shortCode":"3"},{"name":"Madhyama","shortCode":"2"},{"name":"Naegenahira","shortCode":"5"},{"name":"Sabaragamuwa","shortCode":"9"},{"name":"Uturu","shortCode":"4"},{"name":"Uturumaeda","shortCode":"7"},{"name":"Vayamba","shortCode":"6"},{"name":"Uva","shortCode":"8"}]},{"countryName":"Sudan","countryShortCode":"SDN","regions":[{"name":"Al Bahr al Ahmar","shortCode":"RS"},{"name":"Al Jazirah","shortCode":"GZ"},{"name":"Al Khartum","shortCode":"KH"},{"name":"Al Qadarif","shortCode":"GD"},{"name":"An Nil al Abyad","shortCode":"NW"},{"name":"An Nil al Azraq","shortCode":"NB"},{"name":"Ash Shamaliyah","shortCode":"NO"},{"name":"Gharb Darfur","shortCode":"DW"},{"name":"Gharb Kurdufan","shortCode":"GK"},{"name":"Janub Darfur","shortCode":"DS"},{"name":"Janub Kurdufan","shortCode":"KS"},{"name":"Kassala","shortCode":"KA"},{"name":"Nahr an Nil","shortCode":"NR"},{"name":"Shamal Darfur","shortCode":"DN"},{"name":"Sharq Darfur","shortCode":"DE"},{"name":"Shiamal Kurdufan","shortCode":"KN"},{"name":"Sinnar","shortCode":"SI"},{"name":"Wasat Darfur Zalinjay","shortCode":"DC"}]},{"countryName":"Suriname","countryShortCode":"SUR","regions":[{"name":"Brokopondo","shortCode":"BR"},{"name":"Commewijne","shortCode":"CM"},{"name":"Coronie","shortCode":"CR"},{"name":"Marowijne","shortCode":"MA"},{"name":"Nickerie","shortCode":"NI"},{"name":"Para","shortCode":"PR"},{"name":"Paramaribo","shortCode":"PM"},{"name":"Saramacca","shortCode":"SA"},{"name":"Sipaliwini","shortCode":"SI"},{"name":"Wanica","shortCode":"WA"}]},{"countryName":"Swaziland","countryShortCode":"SWZ","regions":[{"name":"Hhohho","shortCode":"HH"},{"name":"Lubombo","shortCode":"LU"},{"name":"Manzini","shortCode":"MA"},{"name":"Shiselweni","shortCode":"SH"}]},{"countryName":"Sweden","countryShortCode":"SWE","regions":[{"name":"Blekinge","shortCode":"K"},{"name":"Dalarna","shortCode":"W"},{"name":"G\u00e4vleborg","shortCode":"X"},{"name":"Gotland","shortCode":"I"},{"name":"Halland","shortCode":"N"},{"name":"J\u00e4mtland","shortCode":"Z"},{"name":"J\u00f6nk\u00f6ping","shortCode":"F"},{"name":"Kalmar","shortCode":"H"},{"name":"Kronoberg","shortCode":"G"},{"name":"Norrbotten","shortCode":"BD"},{"name":"\u00d6rebro","shortCode":"T"},{"name":"\u00d6sterg\u00f6tland","shortCode":"E"},{"name":"Sk\u00e5ne","shortCode":"M"},{"name":"S\u00f6dermanland","shortCode":"D"},{"name":"Stockholm","shortCode":"AB"},{"name":"Uppsala","shortCode":"C"},{"name":"V\u00e4rmland","shortCode":"S"},{"name":"V\u00e4sterbotten","shortCode":"AC"},{"name":"V\u00e4sternorrland","shortCode":"Y"},{"name":"V\u00e4stmanland","shortCode":"U"},{"name":"V\u00e4stra G\u00f6taland","shortCode":"O"}]},{"countryName":"Switzerland","countryShortCode":"CHE","regions":[{"name":"Aargau","shortCode":"AG"},{"name":"Appenzell Ausserrhoden","shortCode":"AR"},{"name":"Appenzell Innerhoden","shortCode":"AI"},{"name":"Basel-Landschaft","shortCode":"BL"},{"name":"Basel-Stadt","shortCode":"BS"},{"name":"Bern","shortCode":"BE"},{"name":"Fribourg","shortCode":"FR"},{"name":"Gen\u00e8ve","shortCode":"GE"},{"name":"Glarus","shortCode":"GL"},{"name":"Graub\u00fcnden","shortCode":"GR"},{"name":"Jura","shortCode":"JU"},{"name":"Luzern","shortCode":"LU"},{"name":"Neuch\u00e2tel","shortCode":"NE"},{"name":"Nidwalden","shortCode":"NW"},{"name":"Obwalden","shortCode":"OW"},{"name":"Sankt Gallen","shortCode":"SG"},{"name":"Schaffhausen","shortCode":"SH"},{"name":"Schwyz","shortCode":"SZ"},{"name":"Solothurn","shortCode":"SO"},{"name":"Thurgau","shortCode":"TG"},{"name":"Ticino","shortCode":"TI"},{"name":"Uri","shortCode":"UR"},{"name":"Valais","shortCode":"VS"},{"name":"Vaud","shortCode":"VD"},{"name":"Zug","shortCode":"ZG"},{"name":"Z\u00fcrich","shortCode":"ZH"}]},{"countryName":"Syrian Arab Republic","countryShortCode":"SYR","regions":[{"name":"Al Hasakah","shortCode":"HA"},{"name":"Al Ladhiqiyah","shortCode":"LA"},{"name":"Al Qunaytirah","shortCode":"QU"},{"name":"Ar Raqqah","shortCode":"RA"},{"name":"As Suwayda\'","shortCode":"SU"},{"name":"Dar\'a","shortCode":"DR"},{"name":"Dayr az Zawr","shortCode":"DY"},{"name":"Dimashq","shortCode":"DI"},{"name":"Halab","shortCode":"HL"},{"name":"Hamah","shortCode":"HM"},{"name":"Hims","shortCode":"HI"},{"name":"Idlib","shortCode":"ID"},{"name":"Rif Dimashq","shortCode":"RD"},{"name":"Tartus","shortCode":"TA"}]},{"countryName":"Taiwan","countryShortCode":"TWN","regions":[{"name":"Chang-hua","shortCode":"CHA"},{"name":"Chia-i","shortCode":"CYQ"},{"name":"Hsin-chu","shortCode":"HSQ"},{"name":"Hua-lien","shortCode":"HUA"},{"name":"Kao-hsiung","shortCode":"KHH"},{"name":"Keelung","shortCode":"KEE"},{"name":"Kinmen","shortCode":"KIN"},{"name":"Lienchiang","shortCode":"LIE"},{"name":"Miao-li","shortCode":"MIA"},{"name":"Nan-t\'ou","shortCode":"NAN"},{"name":"P\'eng-hu","shortCode":"PEN"},{"name":"New Taipei","shortCode":"NWT"},{"name":"P\'ing-chung","shortCode":"PIF"},{"name":"T\'ai-chung","shortCode":"TXG"},{"name":"T\'ai-nan","shortCode":"TNN"},{"name":"T\'ai-pei","shortCode":"TPE"},{"name":"T\'ai-tung","shortCode":"TTT"},{"name":"T\'ao-yuan","shortCode":"TAO"},{"name":"Yi-lan","shortCode":"ILA"},{"name":"Yun-lin","shortCode":"YUN"}]},{"countryName":"Tajikistan","countryShortCode":"TJK","regions":[{"name":"Dushanbe","shortCode":"DU"},{"name":"K\u016dhistoni Badakhshon","shortCode":"GB"},{"name":"Khatlon","shortCode":"KT"},{"name":"Sughd","shortCode":"SU"}]},{"countryName":"Tanzania, United Republic of","countryShortCode":"TZA","regions":[{"name":"Arusha","shortCode":"01"},{"name":"Coast","shortCode":"19"},{"name":"Dar es Salaam","shortCode":"02"},{"name":"Dodoma","shortCode":"03"},{"name":"Iringa","shortCode":"04"},{"name":"Kagera","shortCode":"05"},{"name":"Kigoma","shortCode":"08"},{"name":"Kilimanjaro","shortCode":"09"},{"name":"Lindi","shortCode":"12"},{"name":"Manyara","shortCode":"26"},{"name":"Mara","shortCode":"13"},{"name":"Mbeya","shortCode":"14"},{"name":"Morogoro","shortCode":"16"},{"name":"Mtwara","shortCode":"17"},{"name":"Mwanza","shortCode":"18"},{"name":"Pemba North","shortCode":"06"},{"name":"Pemba South","shortCode":"10"},{"name":"Rukwa","shortCode":"20"},{"name":"Ruvuma","shortCode":"21"},{"name":"Shinyanga","shortCode":"22"},{"name":"Singida","shortCode":"23"},{"name":"Tabora","shortCode":"24"},{"name":"Tanga","shortCode":"25"},{"name":"Zanzibar North","shortCode":"07"},{"name":"Zanzibar Central\/South","shortCode":"11"},{"name":"Zanzibar Urban\/West","shortCode":"15"}]},{"countryName":"Thailand","countryShortCode":"THA","regions":[{"name":"Amnat Charoen","shortCode":"37"},{"name":"Ang Thong","shortCode":"15"},{"name":"Bueng Kan","shortCode":"38"},{"name":"Buri Ram","shortCode":"31"},{"name":"Chachoengsao","shortCode":"24"},{"name":"Chai Nat","shortCode":"18"},{"name":"Chaiyaphum","shortCode":"36"},{"name":"Chanthaburi","shortCode":"22"},{"name":"Chiang Mai","shortCode":"50"},{"name":"Chiang Rai","shortCode":"57"},{"name":"Chon Buri","shortCode":"20"},{"name":"Chumphon","shortCode":"86"},{"name":"Kalasin","shortCode":"46"},{"name":"Kamphaeng Phet","shortCode":"62"},{"name":"Kanchanaburi","shortCode":"71"},{"name":"Khon Kaen","shortCode":"40"},{"name":"Krabi","shortCode":"81"},{"name":"Krung Thep Mahanakhon (Bangkok)","shortCode":"10"},{"name":"Lampang","shortCode":"52"},{"name":"Lamphun","shortCode":"51"},{"name":"Loei","shortCode":"42"},{"name":"Lop Buri","shortCode":"16"},{"name":"Mae Hong Son","shortCode":"58"},{"name":"Maha Sarakham","shortCode":"44"},{"name":"Mukdahan","shortCode":"49"},{"name":"Nakhon Nayok","shortCode":"26"},{"name":"Nakhon Phathom","shortCode":"73"},{"name":"Nakhon Phanom","shortCode":"48"},{"name":"Nakhon Ratchasima","shortCode":"30"},{"name":"Nakhon Sawan","shortCode":"60"},{"name":"Nakhon Si Thammarat","shortCode":"80"},{"name":"Nan","shortCode":"55"},{"name":"Narathiwat","shortCode":"96"},{"name":"Nong Bua Lam Phu","shortCode":"39"},{"name":"Nong Khai","shortCode":"43"},{"name":"Nonthaburi","shortCode":"12"},{"name":"Pathum Thani","shortCode":"13"},{"name":"Pattani","shortCode":"94"},{"name":"Phangnga","shortCode":"82"},{"name":"Phatthalung","shortCode":"93"},{"name":"Phayao","shortCode":"56"},{"name":"Phetchabun","shortCode":"76"},{"name":"Phetchaburi","shortCode":"76"},{"name":"Phichit","shortCode":"66"},{"name":"Phitsanulok","shortCode":"65"},{"name":"Phra Nakhon Si Ayutthaya","shortCode":"14"},{"name":"Phrae","shortCode":"54"},{"name":"Phuket","shortCode":"83"},{"name":"Prachin Buri","shortCode":"25"},{"name":"Prachuap Khiri Khan","shortCode":"77"},{"name":"Ranong","shortCode":"85"},{"name":"Ratchaburi","shortCode":"70"},{"name":"Rayong","shortCode":"21"},{"name":"Roi Et","shortCode":"45"},{"name":"Sa Kaeo","shortCode":"27"},{"name":"Sakon Nakhon","shortCode":"47"},{"name":"Samut Prakan","shortCode":"11"},{"name":"Samut Sakhon","shortCode":"74"},{"name":"Samut Songkhram","shortCode":"75"},{"name":"Saraburi","shortCode":"19"},{"name":"Satun","shortCode":"91"},{"name":"Sing Buri","shortCode":"17"},{"name":"Si Sa ket","shortCode":"33"},{"name":"Songkhla","shortCode":"90"},{"name":"Sukhothai","shortCode":"64"},{"name":"Suphan Buri","shortCode":"72"},{"name":"Surat Thani","shortCode":"84"},{"name":"Surin","shortCode":"32"},{"name":"Tak","shortCode":"63"},{"name":"Trang","shortCode":"92"},{"name":"Trat","shortCode":"23"},{"name":"Ubon Ratchathani","shortCode":"34"},{"name":"Udon Thani","shortCode":"41"},{"name":"Uthai Thani","shortCode":"61"},{"name":"Uttaradit","shortCode":"53"},{"name":"Yala","shortCode":"95"},{"name":"Yasothon","shortCode":"35"}]},{"countryName":"Timor-Leste","countryShortCode":"TLS","regions":[{"name":"Aileu","shortCode":"AL"},{"name":"Ainaro","shortCode":"AN"},{"name":"Baucau","shortCode":"BA"},{"name":"Bobonaro","shortCode":"BO"},{"name":"Cova Lima","shortCode":"CO"},{"name":"Dili","shortCode":"DI"},{"name":"Ermera","shortCode":"ER"},{"name":"Lautem","shortCode":"LA"},{"name":"Liquica","shortCode":"LI"},{"name":"Manatuto","shortCode":"MT"},{"name":"Manufahi","shortCode":"MF"},{"name":"Oecussi","shortCode":"OE"},{"name":"Viqueque","shortCode":"VI"}]},{"countryName":"Togo","countryShortCode":"TGO","regions":[{"name":"Centre","shortCode":"C"},{"name":"Kara","shortCode":"K"},{"name":"Maritime","shortCode":"M"},{"name":"Plateaux","shortCode":"P"},{"name":"Savannes","shortCode":"S"}]},{"countryName":"Tokelau","countryShortCode":"TKL","regions":[{"name":"Atafu"},{"name":"Fakaofo"},{"name":"Nukunonu"}]},{"countryName":"Tonga","countryShortCode":"TON","regions":[{"name":"\'Eua","shortCode":"01"},{"name":"Ha\'apai","shortCode":"02"},{"name":"Niuas","shortCode":"03"},{"name":"Tongatapu","shortCode":"04"},{"name":"Vava\'u","shortCode":"05"}]},{"countryName":"Trinidad and Tobago","countryShortCode":"TTO","regions":[{"name":"Arima","shortCode":"ARI"},{"name":"Chaguanas","shortCode":"CHA"},{"name":"Couva-Tabaquite-Talparo","shortCode":"CTT"},{"name":"Diefo Martin","shortCode":"DMN"},{"name":"Mayaro-Rio Claro","shortCode":"MRC"},{"name":"Penal-Debe","shortCode":"PED"},{"name":"Point Fortin","shortCode":"PTF"},{"name":"Port-of-Spain","shortCode":"POS"},{"name":"Princes Town","shortCode":"PRT"},{"name":"San Fernando","shortCode":"SFO"},{"name":"San Juan-Laventille","shortCode":"SJL"},{"name":"Sangre Grande","shortCode":"SGE"},{"name":"Siparia","shortCode":"SIP"},{"name":"Tobago","shortCode":"TOB"},{"name":"Tunapuna-Piarco","shortCode":"TUP"}]},{"countryName":"Tunisia","countryShortCode":"TUN","regions":[{"name":"Ariana","shortCode":"12"},{"name":"Beja","shortCode":"31"},{"name":"Ben Arous","shortCode":"13"},{"name":"Bizerte","shortCode":"23"},{"name":"Gabes","shortCode":"81"},{"name":"Gafsa","shortCode":"71"},{"name":"Jendouba","shortCode":"32"},{"name":"Kairouan","shortCode":"41"},{"name":"Kasserine","shortCode":"42"},{"name":"Kebili","shortCode":"73"},{"name":"Kef","shortCode":"33"},{"name":"Mahdia","shortCode":"53"},{"name":"Medenine","shortCode":"82"},{"name":"Monastir","shortCode":"52"},{"name":"Nabeul","shortCode":"21"},{"name":"Sfax","shortCode":"61"},{"name":"Sidi Bouzid","shortCode":"43"},{"name":"Siliana","shortCode":"34"},{"name":"Sousse","shortCode":"51"},{"name":"Tataouine","shortCode":"83"},{"name":"Tozeur","shortCode":"72"},{"name":"Tunis","shortCode":"11"},{"name":"Zaghouan","shortCode":"22"}]},{"countryName":"Turkey","countryShortCode":"TUR","regions":[{"name":"Adana","shortCode":"01"},{"name":"Adiyaman","shortCode":"02"},{"name":"Afyonkarahisar","shortCode":"03"},{"name":"Agri","shortCode":"04"},{"name":"Aksaray","shortCode":"68"},{"name":"Amasya","shortCode":"05"},{"name":"Ankara","shortCode":"06"},{"name":"Antalya","shortCode":"07"},{"name":"Ardahan","shortCode":"75"},{"name":"Artvin","shortCode":"08"},{"name":"Aydin","shortCode":"09"},{"name":"Balikesir","shortCode":"10"},{"name":"Bartin","shortCode":"74"},{"name":"Batman","shortCode":"72"},{"name":"Bayburt","shortCode":"69"},{"name":"Bilecik","shortCode":"11"},{"name":"Bingol","shortCode":"12"},{"name":"Bitlis","shortCode":"13"},{"name":"Bolu","shortCode":"14"},{"name":"Burdur","shortCode":"15"},{"name":"Bursa","shortCode":"16"},{"name":"Canakkale","shortCode":"17"},{"name":"Cankiri","shortCode":"18"},{"name":"Corum","shortCode":"19"},{"name":"Denizli","shortCode":"20"},{"name":"Diyarbakir","shortCode":"21"},{"name":"Duzce","shortCode":"81"},{"name":"Edirne","shortCode":"22"},{"name":"Elazig","shortCode":"23"},{"name":"Erzincan","shortCode":"24"},{"name":"Erzurum","shortCode":"25"},{"name":"Eskisehir","shortCode":"26"},{"name":"Gaziantep","shortCode":"27"},{"name":"Giresun","shortCode":"28"},{"name":"Gumushane","shortCode":"29"},{"name":"Hakkari","shortCode":"30"},{"name":"Hatay","shortCode":"31"},{"name":"Igdir","shortCode":"76"},{"name":"Isparta","shortCode":"32"},{"name":"Istanbul","shortCode":"34"},{"name":"Izmir","shortCode":"35"},{"name":"Kahramanmaras","shortCode":"46"},{"name":"Karabuk","shortCode":"78"},{"name":"Karaman","shortCode":"70"},{"name":"Kars","shortCode":"36"},{"name":"Kastamonu","shortCode":"37"},{"name":"Kayseri","shortCode":"38"},{"name":"Kilis","shortCode":"79"},{"name":"Kirikkale","shortCode":"71"},{"name":"Kirklareli","shortCode":"39"},{"name":"Kirsehir","shortCode":"40"},{"name":"Kocaeli","shortCode":"41"},{"name":"Konya","shortCode":"42"},{"name":"Kutahya","shortCode":"43"},{"name":"Malatya","shortCode":"44"},{"name":"Manisa","shortCode":"45"},{"name":"Mardin","shortCode":"47"},{"name":"Mersin","shortCode":"33"},{"name":"Mugla","shortCode":"48"},{"name":"Mus","shortCode":"49"},{"name":"Nevsehir","shortCode":"50"},{"name":"Nigde","shortCode":"51"},{"name":"Ordu","shortCode":"52"},{"name":"Osmaniye","shortCode":"80"},{"name":"Rize","shortCode":"53"},{"name":"Sakarya","shortCode":"54"},{"name":"Samsun","shortCode":"55"},{"name":"Sanliurfa","shortCode":"63"},{"name":"Siirt","shortCode":"56"},{"name":"Sinop","shortCode":"57"},{"name":"Sirnak","shortCode":"73"},{"name":"Sivas","shortCode":"58"},{"name":"Tekirdag","shortCode":"59"},{"name":"Tokat","shortCode":"60"},{"name":"Trabzon","shortCode":"61"},{"name":"Tunceli","shortCode":"62"},{"name":"Usak","shortCode":"64"},{"name":"Van","shortCode":"65"},{"name":"Yalova","shortCode":"77"},{"name":"Yozgat","shortCode":"66"},{"name":"Zonguldak","shortCode":"67"}]},{"countryName":"Turkmenistan","countryShortCode":"TKM","regions":[{"name":"Ahal","shortCode":"A"},{"name":"Asgabat","shortCode":"S"},{"name":"Balkan","shortCode":"B"},{"name":"Dashoguz","shortCode":"D"},{"name":"Lebap","shortCode":"L"},{"name":"Mary","shortCode":"M"}]},{"countryName":"Turks and Caicos Islands","countryShortCode":"TCA","regions":[{"name":"Turks and Caicos Islands"}]},{"countryName":"Tuvalu","countryShortCode":"TUV","regions":[{"name":"Funafuti","shortCode":"FUN"},{"name":"Nanumanga","shortCode":"NMG"},{"name":"Nanumea","shortCode":"NMA"},{"name":"Niutao","shortCode":"NIT"},{"name":"Nui","shortCode":"NUI"},{"name":"Nukufetau","shortCode":"NKF"},{"name":"Nukulaelae","shortCode":"NKL"},{"name":"Vaitupu","shortCode":"VAU"}]},{"countryName":"Uganda","countryShortCode":"UGA","regions":[{"name":"Abim","shortCode":"317"},{"name":"Adjumani","shortCode":"301"},{"name":"Amolatar","shortCode":"314"},{"name":"Amuria","shortCode":"216"},{"name":"Amuru","shortCode":"319"},{"name":"Apac","shortCode":"302"},{"name":"Arua","shortCode":"303"},{"name":"Budaka","shortCode":"217"},{"name":"Bududa","shortCode":"223"},{"name":"Bugiri","shortCode":"201"},{"name":"Bukedea","shortCode":"224"},{"name":"Bukwa","shortCode":"218"},{"name":"Buliisa","shortCode":"419"},{"name":"Bundibugyo","shortCode":"401"},{"name":"Bushenyi","shortCode":"402"},{"name":"Busia","shortCode":"202"},{"name":"Butaleja","shortCode":"219"},{"name":"Dokolo","shortCode":"318"},{"name":"Gulu","shortCode":"304"},{"name":"Hoima","shortCode":"403"},{"name":"Ibanda","shortCode":"416"},{"name":"Iganga","shortCode":"203"},{"name":"Isingiro","shortCode":"417"},{"name":"Jinja","shortCode":"204"},{"name":"Kaabong","shortCode":"315"},{"name":"Kabale","shortCode":"404"},{"name":"Kabarole","shortCode":"405"},{"name":"Kaberamaido","shortCode":"213"},{"name":"Kalangala","shortCode":"101"},{"name":"Kaliro","shortCode":"220"},{"name":"Kampala","shortCode":"102"},{"name":"Kamuli","shortCode":"205"},{"name":"Kamwenge","shortCode":"413"},{"name":"Kanungu","shortCode":"414"},{"name":"Kapchorwa","shortCode":"206"},{"name":"Kasese","shortCode":"406"},{"name":"Katakwi","shortCode":"207"},{"name":"Kayunga","shortCode":"112"},{"name":"Kibaale","shortCode":"407"},{"name":"Kiboga","shortCode":"103"},{"name":"Kiruhura","shortCode":"418"},{"name":"Kisoro","shortCode":"408"},{"name":"Kitgum","shortCode":"305"},{"name":"Koboko","shortCode":"316"},{"name":"Kotido","shortCode":"306"},{"name":"Kumi","shortCode":"208"},{"name":"Kyenjojo","shortCode":"415"},{"name":"Lira","shortCode":"307"},{"name":"Luwero","shortCode":"104"},{"name":"Lyantonde","shortCode":"116"},{"name":"Manafwa","shortCode":"221"},{"name":"Maracha","shortCode":"320"},{"name":"Masaka","shortCode":"105"},{"name":"Masindi","shortCode":"409"},{"name":"Mayuge","shortCode":"214"},{"name":"Mbale","shortCode":"209"},{"name":"Mbarara","shortCode":"410"},{"name":"Mityana","shortCode":"114"},{"name":"Moroto","shortCode":"308"},{"name":"Moyo","shortCode":"309"},{"name":"Mpigi","shortCode":"106"},{"name":"Mubende","shortCode":"107"},{"name":"Mukono","shortCode":"108"},{"name":"Nakapiripirit","shortCode":"311"},{"name":"Nakaseke","shortCode":"115"},{"name":"Nakasongola","shortCode":"109"},{"name":"Namutumba","shortCode":"222"},{"name":"Nebbi","shortCode":"310"},{"name":"Ntungamo","shortCode":"411"},{"name":"Oyam","shortCode":"321"},{"name":"Pader","shortCode":"312"},{"name":"Pallisa","shortCode":"210"},{"name":"Rakai","shortCode":"110"},{"name":"Rukungiri","shortCode":"412"},{"name":"Sembabule","shortCode":"111"},{"name":"Sironko","shortCode":"215"},{"name":"Soroti","shortCode":"211"},{"name":"Tororo","shortCode":"212"},{"name":"Wakiso","shortCode":"113"},{"name":"Yumbe","shortCode":"313"}]},{"countryName":"Ukraine","countryShortCode":"UKR","regions":[{"name":"Cherkasy","shortCode":"71"},{"name":"Chernihiv","shortCode":"74"},{"name":"Chernivtsi","shortCode":"77"},{"name":"Dnipropetrovsk","shortCode":"12"},{"name":"Donetsk","shortCode":"14"},{"name":"Ivano-Frankivsk","shortCode":"26"},{"name":"Kharkiv","shortCode":"63"},{"name":"Kherson","shortCode":"65"},{"name":"Khmelnytskyi","shortCode":"68"},{"name":"Kiev","shortCode":"32"},{"name":"Kirovohrad","shortCode":"35"},{"name":"Luhansk","shortCode":"09"},{"name":"Lviv","shortCode":"46"},{"name":"Mykolaiv","shortCode":"48"},{"name":"Odessa","shortCode":"51"},{"name":"Poltava","shortCode":"53"},{"name":"Rivne","shortCode":"56"},{"name":"Sumy","shortCode":"59"},{"name":"Ternopil","shortCode":"61"},{"name":"Vinnytsia","shortCode":"05"},{"name":"Volyn","shortCode":"07"},{"name":"Zakarpattia","shortCode":"21"},{"name":"Zaporizhia","shortCode":"23"},{"name":"Zhytomyr","shortCode":"18"},{"name":"Avtonomna Respublika Krym","shortCode":"43"},{"name":"Ky\u00efv","shortCode":"30"},{"name":"Sevastopol","shortCode":"40"}]},{"countryName":"United Arab Emirates","countryShortCode":"ARE","regions":[{"name":"Abu Dhabi","shortCode":"AZ"},{"name":"Ajman","shortCode":"AJ"},{"name":"Dubai","shortCode":"DU"},{"name":"Fujairah","shortCode":"FU"},{"name":"Ras al Khaimah","shortCode":"RK"},{"name":"Sharjah","shortCode":"SH"},{"name":"Umm Al Quwain","shortCode":"UQ"}]},{"countryName":"United Kingdom","countryShortCode":"GBR","regions":[{"name":"Avon","shortCode":"AVN"},{"name":"Bedfordshire","shortCode":"BDF"},{"name":"Berkshire","shortCode":"BRK"},{"name":"Bristol, City of","shortCode":"COB"},{"name":"Buckinghamshire","shortCode":"BKM"},{"name":"Cambridgeshire","shortCode":"CAM"},{"name":"Cheshire","shortCode":"CHS"},{"name":"Cleveland","shortCode":"CLV"},{"name":"Cornwall","shortCode":"CON"},{"name":"Cumbria","shortCode":"CMA"},{"name":"Derbyshire","shortCode":"DBY"},{"name":"Devon","shortCode":"DEV"},{"name":"Dorset","shortCode":"DOR"},{"name":"Durham","shortCode":"DUR"},{"name":"East Sussex","shortCode":"SXE"},{"name":"Essex","shortCode":"ESS"},{"name":"Gloucestershire","shortCode":"GLS"},{"name":"Greater London","shortCode":"LND"},{"name":"Greater Manchester","shortCode":"GTM"},{"name":"Hampshire","shortCode":"HAM"},{"name":"Hereford and Worcester","shortCode":"HWR"},{"name":"Herefordshire","shortCode":"HEF"},{"name":"Hertfordshire","shortCode":"HRT"},{"name":"Isle of Wight","shortCode":"IOW"},{"name":"Kent","shortCode":"KEN"},{"name":"Lancashire","shortCode":"LAN"},{"name":"Leicestershire","shortCode":"LEI"},{"name":"Lincolnshire","shortCode":"LIN"},{"name":"London","shortCode":"LDN"},{"name":"Merseyside","shortCode":"MSY"},{"name":"Middlesex","shortCode":"MDX"},{"name":"Norfolk","shortCode":"NFK"},{"name":"Northamptonshire","shortCode":"NTH"},{"name":"Northumberland","shortCode":"NBL"},{"name":"North Humberside","shortCode":"NHM"},{"name":"North Yorkshire","shortCode":"NYK"},{"name":"Nottinghamshire","shortCode":"NTT"},{"name":"Oxfordshire","shortCode":"OXF"},{"name":"Rutland","shortCode":"RUT"},{"name":"Shropshire","shortCode":"SAL"},{"name":"Somerset","shortCode":"SOM"},{"name":"South Humberside","shortCode":"SHM"},{"name":"South Yorkshire","shortCode":"SYK"},{"name":"Staffordshire","shortCode":"STS"},{"name":"Suffolk","shortCode":"SFK"},{"name":"Surrey","shortCode":"SRY"},{"name":"Tyne and Wear","shortCode":"TWR"},{"name":"Warwickshire","shortCode":"WAR"},{"name":"West Midlands","shortCode":"WMD"},{"name":"West Sussex","shortCode":"SXW"},{"name":"West Yorkshire","shortCode":"WYK"},{"name":"Wiltshire","shortCode":"WIL"},{"name":"Worcestershire","shortCode":"WOR"},{"name":"Antrim","shortCode":"ANT"},{"name":"Armagh","shortCode":"ARM"},{"name":"Belfast, City of","shortCode":"BLF"},{"name":"Down","shortCode":"DOW"},{"name":"Fermanagh","shortCode":"FER"},{"name":"Londonderry","shortCode":"LDY"},{"name":"Derry, City of","shortCode":"DRY"},{"name":"Tyrone","shortCode":"TYR"},{"name":"Aberdeen, City of","shortCode":"AN"},{"name":"Aberdeenshire","shortCode":"ABD"},{"name":"Angus (Forfarshire)","shortCode":"ANS"},{"name":"Argyll","shortCode":"AGB"},{"name":"Ayrshire","shortCode":"ARG"},{"name":"Banffshire","shortCode":"BAN"},{"name":"Berwickshire","shortCode":"BEW"},{"name":"Bute","shortCode":"BUT"},{"name":"Caithness","shortCode":"CAI"},{"name":"Clackmannanshire","shortCode":"CLK"},{"name":"Cromartyshire","shortCode":"COC"},{"name":"Dumfriesshire","shortCode":"DFS"},{"name":"Dunbartonshire (Dumbarton)","shortCode":"DNB"},{"name":"Dundee, City of","shortCode":"DD"},{"name":"East Lothian (Haddingtonshire)","shortCode":"ELN"},{"name":"Edinburgh, City of","shortCode":"EB"},{"name":"Fife","shortCode":"FIF"},{"name":"Glasgow, City of","shortCode":"GLA"},{"name":"Inverness-shire","shortCode":"INV"},{"name":"Kincardineshire","shortCode":"KCD"},{"name":"Kinross-shire","shortCode":"KRS"},{"name":"Kirkcudbrightshire","shortCode":"KKD"},{"name":"Lanarkshire","shortCode":"LKS"},{"name":"Midlothian (County of Edinburgh)","shortCode":"MLN"},{"name":"Moray (Elginshire)","shortCode":"MOR"},{"name":"Nairnshire","shortCode":"NAI"},{"name":"Orkney","shortCode":"OKI"},{"name":"Peeblesshire","shortCode":"PEE"},{"name":"Perthshire","shortCode":"PER"},{"name":"Renfrewshire","shortCode":"RFW"},{"name":"Ross and Cromarty","shortCode":"ROC"},{"name":"Ross-shire","shortCode":"ROS"},{"name":"Roxburghshire","shortCode":"ROX"},{"name":"Selkirkshire","shortCode":"SEL"},{"name":"Shetland (Zetland)","shortCode":"SHI"},{"name":"Stirlingshire","shortCode":"STI"},{"name":"Sutherland","shortCode":"SUT"},{"name":"West Lothian (Linlithgowshire)","shortCode":"WLN"},{"name":"Wigtownshire","shortCode":"WIG"},{"name":"Clwyd","shortCode":"CWD"},{"name":"Dyfed","shortCode":"DFD"},{"name":"Gwent","shortCode":"GNT"},{"name":"Gwynedd","shortCode":"GWN"},{"name":"Mid Glamorgan","shortCode":"MGM"},{"name":"Powys","shortCode":"POW"},{"name":"South Glamorgan","shortCode":"SGM"},{"name":"West Glamorgan","shortCode":"WGM"}]},{"countryName":"United States","countryShortCode":"USA","regions":[{"name":"Alabama","shortCode":"AL"},{"name":"Alaska","shortCode":"AK"},{"name":"American Samoa","shortCode":"AS"},{"name":"Arizona","shortCode":"AZ"},{"name":"Arkansas","shortCode":"AR"},{"name":"California","shortCode":"CA"},{"name":"Colorado","shortCode":"CO"},{"name":"Connecticut","shortCode":"CT"},{"name":"Delaware","shortCode":"DE"},{"name":"District of Columbia","shortCode":"DC"},{"name":"Micronesia","shortCode":"FM"},{"name":"Florida","shortCode":"FL"},{"name":"Georgia","shortCode":"GA"},{"name":"Guam","shortCode":"GU"},{"name":"Hawaii","shortCode":"HI"},{"name":"Idaho","shortCode":"ID"},{"name":"Illinois","shortCode":"IL"},{"name":"Indiana","shortCode":"IN"},{"name":"Iowa","shortCode":"IA"},{"name":"Kansas","shortCode":"KS"},{"name":"Kentucky","shortCode":"KY"},{"name":"Louisiana","shortCode":"LA"},{"name":"Maine","shortCode":"ME"},{"name":"Marshall Islands","shortCode":"MH"},{"name":"Maryland","shortCode":"MD"},{"name":"Massachusetts","shortCode":"MA"},{"name":"Michigan","shortCode":"MI"},{"name":"Minnesota","shortCode":"MN"},{"name":"Mississippi","shortCode":"MS"},{"name":"Missouri","shortCode":"MO"},{"name":"Montana","shortCode":"MT"},{"name":"Nebraska","shortCode":"NE"},{"name":"Nevada","shortCode":"NV"},{"name":"New Hampshire","shortCode":"NH"},{"name":"New Jersey","shortCode":"NJ"},{"name":"New Mexico","shortCode":"NM"},{"name":"New York","shortCode":"NY"},{"name":"North Carolina","shortCode":"NC"},{"name":"North Dakota","shortCode":"ND"},{"name":"Northern Mariana Islands","shortCode":"MP"},{"name":"Ohio","shortCode":"OH"},{"name":"Oklahoma","shortCode":"OK"},{"name":"Oregon","shortCode":"OR"},{"name":"Palau","shortCode":"PW"},{"name":"Pennsylvania","shortCode":"PA"},{"name":"Puerto Rico","shortCode":"PR"},{"name":"Rhode Island","shortCode":"RI"},{"name":"South Carolina","shortCode":"SC"},{"name":"South Dakota","shortCode":"SD"},{"name":"Tennessee","shortCode":"TN"},{"name":"Texas","shortCode":"TX"},{"name":"Utah","shortCode":"UT"},{"name":"Vermont","shortCode":"VT"},{"name":"Virgin Islands","shortCode":"VI"},{"name":"Virginia","shortCode":"VA"},{"name":"Washington","shortCode":"WA"},{"name":"West Virginia","shortCode":"WV"},{"name":"Wisconsin","shortCode":"WI"},{"name":"Wyoming","shortCode":"WY"},{"name":"Armed Forces Americas","shortCode":"AA"},{"name":"Armed Forces Europe, Canada, Africa and Middle East","shortCode":"AE"},{"name":"Armed Forces Pacific","shortCode":"AP"}]},{"countryName":"United States Minor Outlying Islands","countryShortCode":"UMI","regions":[{"name":"Baker Island","shortCode":"81"},{"name":"Howland Island","shortCode":"84"},{"name":"Jarvis Island","shortCode":"86"},{"name":"Johnston Atoll","shortCode":"67"},{"name":"Kingman Reef","shortCode":"89"},{"name":"Midway Islands","shortCode":"71"},{"name":"Navassa Island","shortCode":"76"},{"name":"Palmyra Atoll","shortCode":"95"},{"name":"Wake Island","shortCode":"79"},{"name":"Bajo Nuevo Bank","shortCode":"BN"},{"name":"Serranilla Bank","shortCode":"SB"}]},{"countryName":"Uruguay","countryShortCode":"URY","regions":[{"name":"Artigas","shortCode":"AR"},{"name":"Canelones","shortCode":"CA"},{"name":"Cerro Largo","shortCode":"CL"},{"name":"Colonia","shortCode":"CO"},{"name":"Durazno","shortCode":"DU"},{"name":"Flores","shortCode":"FS"},{"name":"Florida","shortCode":"FD"},{"name":"Lavalleja","shortCode":"LA"},{"name":"Maldonado","shortCode":"MA"},{"name":"Montevideo","shortCode":"MO"},{"name":"Paysand\u00fa","shortCode":"PA"},{"name":"R\u00edo Negro","shortCode":"RN"},{"name":"Rivera","shortCode":"RV"},{"name":"Rocha","shortCode":"RO"},{"name":"Salto","shortCode":"SA"},{"name":"San Jos\u00e9","shortCode":"SJ"},{"name":"Soriano","shortCode":"SO"},{"name":"Tacuaremb\u00f3","shortCode":"TA"},{"name":"Treinta y Tres","shortCode":"TT"}]},{"countryName":"Uzbekistan","countryShortCode":"UZB","regions":[{"name":"Toshkent shahri","shortCode":"TK"},{"name":"Andijon","shortCode":"AN"},{"name":"Buxoro","shortCode":"BU"},{"name":"Farg\u2018ona","shortCode":"FA"},{"name":"Jizzax","shortCode":"JI"},{"name":"Namangan","shortCode":"NG"},{"name":"Navoiy","shortCode":"NW"},{"name":"Qashqadaryo (Qarshi)","shortCode":"QA"},{"name":"Samarqand","shortCode":"SA"},{"name":"Sirdaryo (Guliston)","shortCode":"SI"},{"name":"Surxondaryo (Termiz)","shortCode":"SU"},{"name":"Toshkent wiloyati","shortCode":"TO"},{"name":"Xorazm (Urganch)","shortCode":"XO"},{"name":"Qoraqalpog\u2018iston Respublikasi (Nukus)","shortCode":"QR"}]},{"countryName":"Vanuatu","countryShortCode":"VUT","regions":[{"name":"Malampa","shortCode":"MAP"},{"name":"P\u00e9nama","shortCode":"PAM"},{"name":"Sanma","shortCode":"SAM"},{"name":"Sh\u00e9fa","shortCode":"SEE"},{"name":"Taf\u00e9a","shortCode":"TAE"},{"name":"Torba","shortCode":"TOB"}]},{"countryName":"Venezuela, Bolivarian Republic of","countryShortCode":"VEN","regions":[{"name":"Dependencias Federales","shortCode":"W"},{"name":"Distrito Federal","shortCode":"A"},{"name":"Amazonas","shortCode":"Z"},{"name":"Anzo\u00e1tegui","shortCode":"B"},{"name":"Apure","shortCode":"C"},{"name":"Aragua","shortCode":"D"},{"name":"Barinas","shortCode":"E"},{"name":"Bol\u00edvar","shortCode":"F"},{"name":"Carabobo","shortCode":"G"},{"name":"Cojedes","shortCode":"H"},{"name":"Delta Amacuro","shortCode":"Y"},{"name":"Falc\u00f3n","shortCode":"I"},{"name":"Gu\u00e1rico","shortCode":"J"},{"name":"Lara","shortCode":"K"},{"name":"M\u00e9rida","shortCode":"L"},{"name":"Miranda","shortCode":"M"},{"name":"Monagas","shortCode":"N"},{"name":"Nueva Esparta","shortCode":"O"},{"name":"Portuguesa","shortCode":"P"},{"name":"Sucre","shortCode":"R"},{"name":"T\u00e1chira","shortCode":"S"},{"name":"Trujillo","shortCode":"T"},{"name":"Vargas","shortCode":"X"},{"name":"Yaracuy","shortCode":"U"},{"name":"Zulia","shortCode":"V"}]},{"countryName":"Vietnam","countryShortCode":"VNM","regions":[{"name":"\u0110\u1ed3ng Nai","shortCode":"39"},{"name":"\u0110\u1ed3ng Th\u00e1p","shortCode":"45"},{"name":"Gia Lai","shortCode":"30"},{"name":"H\u00e0 Giang","shortCode":"03"},{"name":"H\u00e0 Nam","shortCode":"63"},{"name":"H\u00e0 T\u00e2y","shortCode":"15"},{"name":"H\u00e0 T\u0129nh","shortCode":"23"},{"name":"H\u1ea3i D\u01b0\u01a1ng","shortCode":"61"},{"name":"H\u1eadu Giang","shortCode":"73"},{"name":"H\u00f2a B\u00ecnh","shortCode":"14"},{"name":"H\u01b0ng Y\u00ean","shortCode":"66"},{"name":"Kh\u00e1nh H\u00f2a","shortCode":"34"},{"name":"Ki\u00ean Giang","shortCode":"47"},{"name":"Kon Tum","shortCode":"28"},{"name":"Lai Ch\u00e2u","shortCode":"01"},{"name":"L\u00e2m \u0110\u1ed3ng","shortCode":"35"},{"name":"L\u1ea1ng S\u01a1n","shortCode":"09"},{"name":"L\u00e0o Cai","shortCode":"02"},{"name":"Long An","shortCode":"41"},{"name":"Nam \u0110\u1ecbnh","shortCode":"67"},{"name":"Ngh\u1ec7 An","shortCode":"22"},{"name":"Ninh B\u00ecnh","shortCode":"18"},{"name":"Ninh Thu\u1eadn","shortCode":"36"},{"name":"Ph\u00fa Th\u1ecd","shortCode":"68"},{"name":"Ph\u00fa Y\u00ean","shortCode":"32"},{"name":"Qu\u1ea3ng B\u00ecnh","shortCode":"24"},{"name":"Qu\u1ea3ng Nam","shortCode":"27"},{"name":"Qu\u1ea3ng Ng\u00e3i","shortCode":"29"},{"name":"Qu\u1ea3ng Ninh","shortCode":"13"},{"name":"Qu\u1ea3ng Tr\u1ecb","shortCode":"25"},{"name":"S\u00f3c Tr\u0103ng","shortCode":"52"},{"name":"S\u01a1n La","shortCode":"05"},{"name":"T\u00e2y Ninh","shortCode":"37"},{"name":"Th\u00e1i B\u00ecnh","shortCode":"20"},{"name":"Th\u00e1i Nguy\u00ean","shortCode":"69"},{"name":"Thanh H\u00f3a","shortCode":"21"},{"name":"Th\u1eeba Thi\u00ean\u2013Hu\u1ebf","shortCode":"26"},{"name":"Ti\u1ec1n Giang","shortCode":"46"},{"name":"Tr\u00e0 Vinh","shortCode":"51"},{"name":"Tuy\u00ean Quang","shortCode":"07"},{"name":"V\u0129nh Long","shortCode":"49"},{"name":"V\u0129nh Ph\u00fac","shortCode":"70"},{"name":"Y\u00ean B\u00e1i","shortCode":"06"},{"name":"C\u1ea7n Th\u01a1","shortCode":"CT"},{"name":"\u0110\u00e0 N\u1eb5ng","shortCode":"DN"},{"name":"H\u00e0 N\u1ed9i","shortCode":"HN"},{"name":"H\u1ea3i Ph\u00f2ng","shortCode":"HP"},{"name":"H\u1ed3 Ch\u00ed Minh (S\u00e0i G\u00f2n)","shortCode":"SG"}]},{"countryName":"Virgin Islands, British","countryShortCode":"VGB","regions":[{"name":"Anegada","shortCode":"ANG"},{"name":"Jost Van Dyke","shortCode":"JVD"},{"name":"Tortola","shortCode":"TTA"},{"name":"Virgin Gorda","shortCode":"VGD"}]},{"countryName":"Virgin Islands, U.S.","countryShortCode":"VIR","regions":[{"name":"St. Thomas","shortCode":"STH"},{"name":"St. John","shortCode":"SJO"},{"name":"St. Croix","shortCode":"SCR"}]},{"countryName":"Wallis and Futuna","countryShortCode":"WLF","regions":[{"name":"Alo","shortCode":"ALO"},{"name":"Sigave","shortCode":"SIG"},{"name":"Wallis","shortCode":"WAL"}]},{"countryName":"Western Sahara","countryShortCode":"ESH","regions":[{"name":"Es Smara","shortCode":"ESM"},{"name":"Boujdour","shortCode":"BOD"},{"name":"La\u00e2youne","shortCode":"LAA"},{"name":"Aousserd","shortCode":"AOU"},{"name":"Oued ed Dahab","shortCode":"OUD"}]},{"countryName":"Yemen","countryShortCode":"YEM","regions":[{"name":"Aby\u0101n","shortCode":"AB"},{"name":"\'Adan","shortCode":"AD"},{"name":"A\u1e11 \u1e10\u0101li\'","shortCode":"DA"},{"name":"Al Bay\u1e11\u0101\'","shortCode":"BA"},{"name":"Al \u1e28udaydah","shortCode":"HU"},{"name":"Al Jawf","shortCode":"JA"},{"name":"Al Mahrah","shortCode":"MR"},{"name":"Al Ma\u1e29w\u012bt","shortCode":"MW"},{"name":"\'Amr\u0101n","shortCode":"AM"},{"name":"Dham\u0101r","shortCode":"DH"},{"name":"\u1e28a\u1e11ramawt","shortCode":"HD"},{"name":"\u1e28ajjah","shortCode":"HJ"},{"name":"Ibb","shortCode":"IB"},{"name":"La\u1e29ij","shortCode":"LA"},{"name":"Ma\'rib","shortCode":"MA"},{"name":"Raymah","shortCode":"RA"},{"name":"\u015e\u0101\u2018dah","shortCode":"SD"},{"name":"\u015ean\u2018\u0101\'","shortCode":"SN"},{"name":"Shabwah","shortCode":"SH"},{"name":"T\u0101\u2018izz","shortCode":"TA"}]},{"countryName":"Zambia","countryShortCode":"ZMB","regions":[{"name":"Central","shortCode":"02"},{"name":"Copperbelt","shortCode":"08"},{"name":"Eastern","shortCode":"03"},{"name":"Luapula","shortCode":"04"},{"name":"Lusaka","shortCode":"09"},{"name":"Northern","shortCode":"05"},{"name":"North-Western","shortCode":"06"},{"name":"Southern","shortCode":"07"},{"name":"Western","shortCode":"01"}]},{"countryName":"Zimbabwe","countryShortCode":"ZWE","regions":[{"name":"Bulawayo","shortCode":"BU"},{"name":"Harare","shortCode":"HA"},{"name":"Manicaland","shortCode":"MA"},{"name":"Mashonaland Central","shortCode":"MC"},{"name":"Mashonaland East","shortCode":"ME"},{"name":"Mashonaland West","shortCode":"MW"},{"name":"Masvingo","shortCode":"MV"},{"name":"Matabeleland North","shortCode":"MN"},{"name":"Matabeleland South","shortCode":"MS"},{"name":"Midlands","shortCode":"MI"}]}]';
        }

        function __construct() {

            $this->_generate_ipn_url();
            $this->_generate_cancel_url();
            $this->_generate_failed_url();

            $this->on_creation();
            $this->init();

            add_action( 'tc_gateway_settings', array( &$this, 'gateway_admin_settings' ) );
            add_action( 'tc_handle_payment_return_' . $this->plugin_name, array( &$this, 'ipn' ) );

            add_action( 'template_redirect', array( &$this, '_checkout_confirmation_hook' ) );
            add_filter( 'tc_checkout_confirm_payment_' . $this->plugin_name, array( &$this, 'confirm_payment_form' ), 10, 2 );
            add_action( 'tc_payment_confirm_' . $this->plugin_name, array( &$this, 'process_payment' ), 10, 2 );
            add_filter( 'tc_order_notification_' . $this->plugin_name, array( &$this, 'order_confirmation_email' ), 10, 2 );
            add_action( 'tc_checkout_payment_pre_confirmation_' . $this->plugin_name, array( &$this, 'order_confirmation' ) );
            add_filter( 'tc_checkout_payment_confirmation_' . $this->plugin_name, array( &$this, 'order_confirmation_message' ), 10, 2 );

            add_action( 'template_redirect', array( &$this, 'force_ssl' ) );
            add_action( 'init', array( &$this, 'show_cart_errors' ) );
        }
    }
}

/**
 * Use this function to register your gateway plugin class
 *
 * @param string $class_name - the case sensitive name of your plugin class
 * @param string $plugin_name - the sanitized private name for your plugin
 * @param string $admin_name - pretty name of your gateway, for the admin side.
 * @param bool $global optional - whether the gateway supports global checkouts
 * @param bool $demo
 * @return bool
 */
function tc_register_gateway_plugin( $class_name, $plugin_name, $admin_name, $global = false, $demo = false ) {

    global $tc_gateway_plugins;

    if ( ! is_array( $tc_gateway_plugins ) ) {
        $tc_gateway_plugins = [];
    }

    if ( class_exists( $class_name ) ) {

        $tc_gateway_plugins[ $plugin_name ] = array( $class_name, $admin_name, $global, $demo );

        // Sort gateways by priority
        uksort( $tc_gateway_plugins, function ( $a, $b ) use($tc_gateway_plugins) {

            $settings = get_option( 'tc_settings' );

            $a_priority = ( isset( $settings[ 'gateways' ][ $a ][ 'gateway_show_priority' ] ) && is_numeric( $settings[ 'gateways' ][ $a ][ 'gateway_show_priority' ] ) )
                ? $settings[ 'gateways' ][ $a ][ 'gateway_show_priority' ]
                : 0;

            $b_priority = ( isset( $settings[ 'gateways' ][ $b ][ 'gateway_show_priority' ] ) && is_numeric( $settings[ 'gateways' ][ $b ][ 'gateway_show_priority' ] ) )
                ? $settings[ 'gateways' ][ $b ][ 'gateway_show_priority' ]
                : 0;

            return $a_priority - $b_priority;
        });

    } else {
        return false;
    }
}
