<?php

// wrapped from https://github.com/magaras/greek_stemmer
class SearchWP_GreekStemmer {
	/**
	 *  Greek Stemmer.
	 * @author Magarisiotis Konstantinos <magarisiotis.kostas@hotmail.com>
	 *
	 *  Original Stemmer by "Spyros Saroukos".
	 *
	 *
	 *  This library is free software; you can redistribute it and/or
	 *  modify it under the terms of the GNU Lesser General Public
	 *  License as published by the Free Software Foundation; either
	 *  version 2.1 of the License, or (at your option) any later version.
	 *
	 *  This library is distributed in the hope that it will be useful,
	 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
	 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
	 *  Lesser General Public License for more details.
	 *
	 *  You should have received a copy of the GNU Lesser General Public
	 *  License along with this library; if not, write to the Free Software
	 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
	 */
	/**
	 *  The call function of the stemmer.
	 *  It receives the text and returnes the stemmed words.
	 *
	 * @param string $word :   The text we want to stem
	 * @param bool $simple_stemmer :   If TRUE then it returns all the stemmed words.
	 *                                  If FALSE, then it returns only the words that have meaning for
	 *                                  the content of the whole text.
	 *
	 * @return array               :   The stemmed words in an array
	 */

	function start( $word, $simple_stemmer ) {
		// The array that holds the stemmed words
		$stem_con = array();
		// Pre - manipulation
		// Remove the EndLines and dots (.) from the ending of the words.
		$w = trim( $word );         // Remove empty spaces
		$w = explode( "\n", $w );    // Explode the string based on new line characters
		$w = implode( ' ', $w );     // Reunite the string (it does not contain any new lines)
		$w = explode( '.', $w );     // Explode the string based on dots.
		$w = implode( ' ', $w );     // Reunite the string
		$w = explode( ' ', $w );     // Explode it into an array
		// Stem each one word
		for ( $i = 0; $i < sizeof( $w ); $i ++ ) {

			// Pass each word into the stemmer
			$stemmed = $this->stem_word( $w[ $i ], $simple_stemmer );
			// Push the stemmed word into the array, only if a value is returned
			if ( $stemmed === null ) {
				continue;
			} else {
				array_push( $stem_con, $stemmed );
			}
		}

		if ( false == $simple_stemmer ) {
			$content_stem_con = [ ];
			foreach ( $stem_con as $word ) {
				if ( array_key_exists( $word, $content_stem_con ) ) {
					$value = $content_stem_con[ $word ];
					$value ++;
					$content_stem_con[ $word ] = $value;
				} else {
					$content_stem_con[ $word ] = 1;
				}
			}

			return $content_stem_con;
		}

		// Return the stemmed words
		return $stem_con;
	}

	/**
	 *  The stemmer
	 *
	 * @param string $w :   The word to be stemmed
	 *
	 * @param $simple_stemmer
	 *
	 * @return string
	 */
	function stem_word( $w, $simple_stemmer ) {
		// Change the encoding to "ISO-8859-7" and keep track of the change.
		$encoding_changed = false;
		if ( mb_check_encoding( $w, 'UTF-8' ) ) {
			// Flag - the encoding has changed
			$encoding_changed = true;
			$w = mb_convert_encoding( $w, 'ISO-8859-7', 'UTF-8' );
		}

		// Remove all special characters and numbers
		$w = str_replace( '<br />', '', $w );
		$w = str_replace( '.', '', $w );
		$w = str_replace( ',', '', $w );
		$w = str_replace( ';', '', $w );
		$w = str_replace( '/', '', $w );
		$w = str_replace( '?', '', $w );
		$w = str_replace( ':', '', $w );
		$w = str_replace( "'", '', $w );
		$w = str_replace( '"', '', $w );
		$w = str_replace( '\\', '', $w );
		$w = str_replace( '[', '', $w );
		$w = str_replace( '{', '', $w );
		$w = str_replace( ']', '', $w );
		$w = str_replace( '}', '', $w );
		$w = str_replace( '|', '', $w );
		$w = str_replace( '~', '', $w );
		$w = str_replace( '`', '', $w );
		$w = str_replace( '!', '', $w );
		$w = str_replace( '@', '', $w );
		$w = str_replace( '#', '', $w );
		$w = str_replace( '$', '', $w );
		$w = str_replace( '%', '', $w );
		$w = str_replace( '%', '', $w );
		$w = str_replace( '^', '', $w );
		$w = str_replace( '&', '', $w );
		$w = str_replace( '*', '', $w );
		$w = str_replace( '(', '', $w );
		$w = str_replace( ')', '', $w );
		$w = str_replace( '-', '', $w );
		$w = str_replace( '_', '', $w );
		$w = str_replace( '+', '', $w );
		$w = str_replace( '=', '', $w );
		$w = str_replace( '0', '', $w );
		$w = str_replace( '1', '', $w );
		$w = str_replace( '2', '', $w );
		$w = str_replace( '3', '', $w );
		$w = str_replace( '4', '', $w );
		$w = str_replace( '5', '', $w );
		$w = str_replace( '6', '', $w );
		$w = str_replace( '7', '', $w );
		$w = str_replace( '8', '', $w );
		$w = str_replace( '9', '', $w );
		$w = str_replace( '«', '', $w );
		$w = str_replace( '»', '', $w );

		// Convert to uppercase
		$w = mb_strtoupper( $w, 'ISO-8859-7' );

		// Replace some values - remove the tonnation
		$unacceptedLetters = [
			'á' => 'Á',
			'â' => 'Â',
			'ã' => 'Ã',
			'ä' => 'Ä',
			'å' => 'Å',
			'æ' => 'Æ',
			'ç' => 'Ç',
			'è' => 'È',
			'é' => 'É',
			'ê' => 'Ê',
			'ë' => 'Ë',
			'ì' => 'Ì',
			'í' => 'Í',
			'î' => 'Î',
			'ï' => 'Ï',
			'ð' => 'Ð',
			'ñ' => 'Ñ',
			'ó' => 'Ó',
			'ô' => 'Ô',
			'õ' => 'Õ',
			'ö' => 'Ö',
			'÷' => '×',
			'ø' => 'Ø',
			'ù' => 'Ù',
			'Ü' => 'Á',
			'Ý' => 'Å',
			'Þ' => 'Ç',
			'ß' => 'É',
			'ü' => 'Ï',
			'ý' => 'Õ',
			'þ' => 'Ù',
			'ú' => 'É',
			'û' => 'Õ',
			'¶' => 'Á',
			'¸' => 'Å',
			'¹' => 'Ç',
			'º' => 'É',
			'¼' => 'Ï',
			'¾' => 'Õ',
			'¿' => 'Ù',
			'ò' => 'Ó'
		];

		// Replace any occurrance of the above letters
		for ( $i = 0; $i < strlen( $w ); $i ++ ) {
			if ( array_key_exists( $w[ $i ], $unacceptedLetters ) ) {
				$w[ $i ] = $unacceptedLetters[ $w[ $i ] ];
			}
		}

		// If we choose to return only the valuable words, then we remove the Stop Words.
		if ( false == $simple_stemmer ) {
			// ********** The following rules could be modified.
			// Stop Words Removal
			$stop_w = array(
				'ÅÊÏ',
				'ÁÂÁ',
				'ÁÃÁ',
				'ÁÃÇ',
				'ÁÃÙ',
				'ÁÄÇ',
				'ÁÄÙ',
				'ÁÅ',
				'ÁÅÉ',
				'ÁÈÙ',
				'ÁÉ',
				'ÁÉÊ',
				'ÁÊÇ',
				'ÁÊÏÌÁ',
				'ÁÊÏÌÇ',
				'ÁÊÑÉÂÙÓ',
				'ÁËÁ',
				'ÁËÇÈÅÉÁ',
				'ÁËÇÈÉÍÁ',
				'ÁËËÁ×ÏÕ',
				'ÁËËÉÙÓ',
				'ÁËËÉÙÔÉÊÁ',
				'ÁËËÏÉÙÓ',
				'ÁËËÏÉÙÔÉÊÁ',
				'ÁËËÏÔÅ',
				'ÁËÔ',
				'ÁËÙ',
				'ÁÌÁ',
				'ÁÌÅ',
				'ÁÌÅÓÁ',
				'ÁÌÅÓÙÓ',
				'ÁÌÙ',
				'ÁÍ',
				'ÁÍÁ',
				'ÁÍÁÌÅÓÁ',
				'ÁÍÁÌÅÔÁÎÕ',
				'ÁÍÅÕ',
				'ÁÍÔÉ',
				'ÁÍÔÉÐÅÑÁ',
				'ÁÍÔÉÓ',
				'ÁÍÙ',
				'ÁÍÙÔÅÑÙ',
				'ÁÎÁÖÍÁ',
				'ÁÐ',
				'ÁÐÅÍÁÍÔÉ',
				'ÁÐÏ',
				'ÁÐÏØÅ',
				'ÁÐÙ',
				'ÁÑÁ',
				'ÁÑÁÃÅ',
				'ÁÑÅ',
				'ÁÑÊ',
				'ÁÑÊÅÔÁ',
				'ÁÑË',
				'ÁÑÌ',
				'ÁÑÔ',
				'ÁÑÕ',
				'ÁÑÙ',
				'ÁÓ',
				'ÁÓÁ',
				'ÁÓÏ',
				'ÁÔÁ',
				'ÁÔÅ',
				'ÁÔÇ',
				'ÁÔÉ',
				'ÁÔÌ',
				'ÁÔÏ',
				'ÁÕÑÉÏ',
				'ÁÖÇ',
				'ÁÖÏÔÏÕ',
				'ÁÖÏÕ',
				'Á×',
				'Á×Å',
				'Á×Ï',
				'ÁØÁ',
				'ÁØÅ',
				'ÁØÇ',
				'ÁØÕ',
				'ÁÙÅ',
				'ÁÙÏ',
				'ÂÁÍ',
				'ÂÁÔ',
				'ÂÁ×',
				'ÂÅÁ',
				'ÂÅÂÁÉÏÔÁÔÁ',
				'ÂÇÎ',
				'ÂÉÁ',
				'ÂÉÅ',
				'ÂÉÇ',
				'ÂÉÏ',
				'ÂÏÇ',
				'ÂÏÙ',
				'ÂÑÅ',
				'ÃÁ',
				'ÃÁÂ',
				'ÃÁÑ',
				'ÃÅÍ',
				'ÃÅÓ',
				'',
				'ÃÇ',
				'ÃÇÍ',
				'ÃÉ',
				'ÃÉÁ',
				'ÃÉÅ',
				'ÃÉÍ',
				'ÃÉÏ',
				'ÃÊÉ',
				'ÃÉÁÔÉ',
				'ÃÊÕ',
				'ÃÏÇ',
				'ÃÏÏ',
				'ÃÑÇÃÏÑÁ',
				'ÃÑÉ',
				'ÃÑÕ',
				'ÃÕÇ',
				'ÃÕÑÙ',
				'ÄÁ',
				'ÄÅ',
				'ÄÅÇ',
				'ÄÅÉ',
				'ÄÅÍ',
				'ÄÅÓ',
				'ÄÇ',
				'ÄÇÈÅÍ',
				'ÄÇËÁÄÇ',
				'ÄÇÙ',
				'ÄÉ',
				'ÄÉÁ',
				'ÄÉÁÑÊÙÓ',
				'ÄÉÏËÏÕ',
				'ÄÉÓ',
				'ÄÉ×ÙÓ',
				'ÄÏË',
				'ÄÏÍ',
				'ÄÑÁ',
				'ÄÑÕ',
				'ÄÑ×',
				'ÄÕÅ',
				'ÄÕÏ',
				'ÄÙ',
				'ÅÁÌ',
				'ÅÁÍ',
				'ÅÁÑ',
				'ÅÈÇ',
				'ÅÉ',
				'ÅÉÄÅÌÇ',
				'ÅÉÈÅ',
				'ÅÉÌÁÉ',
				'ÅÉÌÁÓÔÅ',
				'ÅÉÍÁÉ',
				'ÅÉÓ',
				'ÅÉÓÁÉ',
				'ÅÉÓÁÓÔÅ',
				'ÅÉÓÔÅ',
				'ÅÉÔÅ',
				'ÅÉ×Á',
				'ÅÉ×ÁÌÅ',
				'ÅÉ×ÁÍ',
				'ÅÉ×ÁÔÅ',
				'ÅÉ×Å',
				'ÅÉ×ÅÓ',
				'ÅÊ',
				'ÅÊÅÉ',
				'ÅËÁ',
				'ÅËÉ',
				'ÅÌÐ',
				'ÅÍ',
				'ÅÍÔÅËÙÓ',
				'ÅÍÔÏÓ',
				'ÅÍÔÙÌÅÔÁÎÕ',
				'ÅÍÙ',
				'ÅÎ',
				'ÅÎÁÖÍÁ',
				'ÅÎÉ',
				'ÅÎÉÓÏÕ',
				'ÅÎÙ',
				'ÅÏÊ',
				'ÅÐÁÍÙ',
				'ÅÐÅÉÄÇ',
				'ÅÐÅÉÔÁ',
				'ÅÐÇ',
				'ÅÐÉ',
				'ÅÐÉÓÇÓ',
				'ÅÐÏÌÅÍÙÓ',
				'ÅÑÁ',
				'ÅÓ',
				'ÅÓÁÓ',
				'ÅÓÅ',
				'ÅÓÅÉÓ',
				'ÅÓÅÍÁ',
				'ÅÓÇ',
				'ÅÓÔÙ',
				'ÅÓÕ',
				'ÅÓÙ',
				'ÅÔÉ',
				'ÅÔÓÉ',
				'ÅÕ',
				'ÅÕÁ',
				'ÅÕÃÅ',
				'ÅÕÈÕÓ',
				'ÅÕÔÕ×ÙÓ',
				'ÅÖÅ',
				'ÅÖÅÎÇÓ',
				'ÅÖÔ',
				'Å×Å',
				'Å×ÅÉ',
				'Å×ÅÉÓ',
				'Å×ÅÔÅ',
				'Å×ÈÅÓ',
				'Å×ÏÌÅ',
				'Å×ÏÕÌÅ',
				'Å×ÏÕÍ',
				'Å×ÔÅÓ',
				'Å×Ù',
				'ÅÙÓ',
				'ÆÅÁ',
				'ÆÅÇ',
				'ÆÅÉ',
				'ÆÅÍ',
				'ÆÇÍ',
				'ÆÙ',
				'Ç',
				'ÇÄÇ',
				'ÇÄÕ',
				'ÇÈÇ',
				'ÇËÏ',
				'ÇÌÉ',
				'ÇÐÁ',
				'ÇÓÁÓÔÅ',
				'ÇÓÏÕÍ',
				'ÇÔÁ',
				'ÇÔÁÍ',
				'ÇÔÁÍÅ',
				'ÇÔÏÉ',
				'ÇÔÔÏÍ',
				'ÇÙ',
				'ÈÁ',
				'ÈÕÅ',
				'ÈÙÑ',
				'É',
				'ÉÁ',
				'ÉÂÏ',
				'ÉÄÇ',
				'ÉÄÉÙÓ',
				'ÉÅ',
				'ÉÉ',
				'ÉÉÉ',
				'ÉÊÁ',
				'ÉËÏ',
				'ÉÌÁ',
				'ÉÍÁ',
				'ÉÍÙ',
				'ÉÎÅ',
				'ÉÎÏ',
				'ÉÏ',
				'ÉÏÉ',
				'ÉÓÁ',
				'ÉÓÁÌÅ',
				'ÉÓÅ',
				'ÉÓÇ',
				'ÉÓÉÁ',
				'ÉÓÏ',
				'ÉÓÙÓ',
				'ÉÙÂ',
				'ÉÙÍ',
				'ÉÙÓ',
				'ÉÁÍ',
				'ÊÁÈ',
				'ÊÁÈÅ',
				'ÊÁÈÅÔÉ',
				'ÊÁÈÏËÏÕ',
				'ÊÁÈÙÓ',
				'ÊÁÉ',
				'ÊÁÍ',
				'ÊÁÐÏÔÅ',
				'ÊÁÐÏÕ',
				'ÊÁÐÙÓ',
				'ÊÁÔ',
				'ÊÁÔÁ',
				'ÊÁÔÉ',
				'ÊÁÔÉÔÉ',
				'ÊÁÔÏÐÉÍ',
				'ÊÁÔÙ',
				'ÊÁÙ',
				'ÊÂÏ',
				'ÊÅÁ',
				'ÊÅÉ',
				'ÊÅÍ',
				'ÊÉ',
				'ÊÉÌ',
				'ÊÉÏËÁÓ',
				'ÊÉÔ',
				'ÊÉ×',
				'ÊÊÅ',
				'ÊËÉÓÅ',
				'ÊËÐ',
				'ÊÏÊ',
				'ÊÏÍÔÁ',
				'ÊÏ×',
				'ÊÔË',
				'ÊÕÑ',
				'ÊÕÑÉÙÓ',
				'ÊÙ',
				'ÊÙÍ',
				'ËÁ',
				'ËÅÁ',
				'ËÅÍ',
				'ËÅÏ',
				'ËÉÁ',
				'ËÉÃÁÊÉ',
				'ËÉÃÏÕËÁÊÉ',
				'ËÉÃÏ',
				'ËÉÃÙÔÅÑÏ',
				'ËÉÏ',
				'ËÉÑ',
				'ËÏÃÙ',
				'ËÏÉÐÁ',
				'ËÏÉÐÏÍ',
				'ËÏÓ',
				'ËÓ',
				'ËÕÙ',
				'ÌÁ',
				'ÌÁÆÉ',
				'ÌÁÊÁÑÉ',
				'ÌÁËÉÓÔÁ',
				'ÌÁËËÏÍ',
				'ÌÁÍ',
				'ÌÁÎ',
				'ÌÁÓ',
				'ÌÁÔ',
				'ÌÅ',
				'ÌÅÈÁÕÑÉÏ',
				'ÌÅÉ',
				'ÌÅÉÏÍ',
				'ÌÅË',
				'ÌÅËÅÉ',
				'ÌÅËËÅÔÁÉ',
				'ÌÅÌÉÁÓ',
				'ÌÅÍ',
				'ÌÅÓ',
				'ÌÅÓÁ',
				'ÌÅÔ',
				'ÌÅÔÁ',
				'ÌÅÔÁÎÕ',
				'ÌÅ×ÑÉ',
				'ÌÇ',
				'ÌÇÄÅ',
				'ÌÇÍ',
				'ÌÇÐÙÓ',
				'ÌÇÔÅ',
				'ÌÉ',
				'ÌÉÎ',
				'ÌÉÓ',
				'ÌÌÅ',
				'ÌÍÁ',
				'ÌÏÂ',
				'ÌÏËÉÓ',
				'ÌÏËÏÍÏÔÉ',
				'ÌÏÍÁ×Á',
				'ÌÏÍÏÌÉÁÓ',
				'ÌÉÁ',
				'ÌÏÕ',
				'ÌÐÁ',
				'ÌÐÏÑÅÉ',
				'ÌÐÏÑÏÕÍ',
				'ÌÐÑÁÂÏ',
				'ÌÐÑÏÓ',
				'ÌÐÙ',
				'ÌÕ',
				'ÌÕÁ',
				'ÌÕÍ',
				'ÍÁ',
				'ÍÁÅ',
				'ÍÁÉ',
				'ÍÁÏ',
				'ÍÄ',
				'ÍÅÀ',
				'ÍÅÁ',
				'ÍÅÅ',
				'ÍÅÏ',
				'ÍÉ',
				'ÍÉÁ',
				'ÍÉÊ',
				'ÍÉË',
				'ÍÉÍ',
				'ÍÉÏ',
				'ÍÔÁ',
				'ÍÔÅ',
				'ÍÔÉ',
				'ÍÔÏ',
				'ÍÕÍ',
				'ÍÙÅ',
				'ÍÙÑÉÓ',
				'ÎÁÍÁ',
				'ÎÁÖÍÉÊÁ',
				'ÎÅÙ',
				'ÎÉ',
				'Ï',
				'ÏÁ',
				'ÏÁÐ',
				'ÏÄÏ',
				'ÏÅ',
				'ÏÆÏ',
				'ÏÇÅ',
				'ÏÉ',
				'ÏÉÁ',
				'ÏÉÇ',
				'ÏÊÁ',
				'ÏËÏÃÕÑÁ',
				'ÏËÏÍÅÍ',
				'ÏËÏÔÅËÁ',
				'ÏËÙÓÄÉÏËÏÕ',
				'ÏÌÙÓ',
				'¼ÌÙÓ',
				'¼ÐÙÓ',
				'ÏÍ',
				'ÏÍÅ',
				'ÏÍÏ',
				'ÏÐÁ',
				'ÏÐÅ',
				'ÏÐÇ',
				'ÏÐÏ',
				'ÏÐÏÉÁÄÇÐÏÔÅ',
				'ÏÐÏÉÁÍÄÇÐÏÔÅ',
				'ÏÐÏÉÁÓÄÇÐÏÔÅ',
				'ÏÐÏÉÄÇÐÏÔÅ',
				'ÏÐÏÉÅÓÄÇÐÏÔÅ',
				'ÏÐÏÉÏÄÇÐÏÔÅ',
				'ÏÐÏÉÏÍÄÇÐÏÔÅ',
				'ÏÐÏÉÏÓÄÇÐÏÔÅ',
				'ÏÐÏÉÏÕÄÇÐÏÔÅ',
				'ÏÐÏÉÏÕÓÄÇÐÏÔÅ',
				'ÏÐÏÉÙÍÄÇÐÏÔÅ',
				'ÏÐÏÔÅÄÇÐÏÔÅ',
				'ÏÐÏÕ',
				'ÏÐÏÕÄÇÐÏÔÅ',
				'ÏÐÙÓ',
				'ÏÑÁ',
				'ÏÑÅ',
				'ÏÑÇ',
				'ÏÑÏ',
				'ÏÑÖ',
				'ÏÑÙ',
				'ÏÓÁ',
				'ÏÓÁÄÇÐÏÔÅ',
				'ÏÓÅ',
				'ÏÓÅÓÄÇÐÏÔÅ',
				'ÏÓÇÄÇÐÏÔÅ',
				'ÏÓÇÍÄÇÐÏÔÅ',
				'ÏÓÇÓÄÇÐÏÔÅ',
				'ÏÓÏÄÇÐÏÔÅ',
				'ÏÓÏÉÄÇÐÏÔÅ',
				'ÏÓÏÍÄÇÐÏÔÅ',
				'ÏÓÏÓÄÇÐÏÔÅ',
				'ÏÓÏÕÄÇÐÏÔÅ',
				'ÏÓÏÕÓÄÇÐÏÔÅ',
				'ÏÓÙÍÄÇÐÏÔÅ',
				'ÏÔÁÍ',
				'ÏÔÅ',
				'ÏÔÉ',
				'ÏÔÉÄÇÐÏÔÅ',
				'ÏÕ',
				'ÏÕÄÅ',
				'ÏÕÊ',
				'ÏÕÓ',
				'ÏÕÔÅ',
				'ÏÕÖ',
				'Ï×É',
				'ÏØÁ',
				'ÏØÅ',
				'ÏØÇ',
				'ÏØÉ',
				'ÏØÏ',
				'ÐÁ',
				'ÐÁËÉ',
				'ÐÁÍ',
				'ÐÁÍÔÏÔÅ',
				'ÐÁÍÔÏÕ',
				'ÐÁÍÔÙÓ',
				'ÐÁÐ',
				'ÐÁÑ',
				'ÐÁÑÁ',
				'ÐÅÉ',
				'ÐÅÑ',
				'ÐÅÑÁ',
				'ÐÅÑÉ',
				'ÐÅÑÉÐÏÕ',
				'ÐÅÑÓÉ',
				'ÐÅÑÕÓÉ',
				'ÐÅÓ',
				'ÐÉ',
				'ÐÉÁ',
				'ÐÉÈÁÍÏÍ',
				'ÐÉÊ',
				'ÐÉÏ',
				'ÐÉÓÙ',
				'ÐÉÔ',
				'ÐÉÙ',
				'ÐËÁÉ',
				'ÐËÅÏÍ',
				'ÐËÇÍ',
				'ÐËÙ',
				'ÐÌ',
				'ÐÏÁ',
				'ÐÏÅ',
				'ÐÏË',
				'ÐÏËÕ',
				'ÐÏÐ',
				'ÐÏÔÅ',
				'ÐÏÕ',
				'ÐÏÕÈÅ',
				'ÐÏÕÈÅÍÁ',
				'ÐÑÅÐÅÉ',
				'ÐÑÉ',
				'ÐÑÉÍ',
				'ÐÑÏ',
				'ÐÑÏÊÅÉÌÅÍÏÕ',
				'ÐÑÏÊÅÉÔÁÉ',
				'ÐÑÏÐÅÑÓÉ',
				'ÐÑÏÓ',
				'ÐÑÏÔÏÕ',
				'ÐÑÏ×ÈÅÓ',
				'ÐÑÏ×ÔÅÓ',
				'ÐÑÙÔÕÔÅÑÁ',
				'ÐÕÁ',
				'ÐÕÎ',
				'ÐÕÏ',
				'ÐÕÑ',
				'Ð×',
				'ÐÙ',
				'ÐÙË',
				'ÐÙÓ',
				'ÑÁ',
				'ÑÁÉ',
				'ÑÁÐ',
				'ÑÁÓ',
				'ÑÅ',
				'ÑÅÁ',
				'ÑÅÅ',
				'ÑÅÉ',
				'ÑÇÓ',
				'ÑÈÙ',
				'ÑÉÏ',
				'ÑÏ',
				'ÑÏÀ',
				'ÑÏÅ',
				'ÑÏÆ',
				'ÑÏÇ',
				'ÑÏÈ',
				'ÑÏÉ',
				'ÑÏÊ',
				'ÑÏË',
				'ÑÏÍ',
				'ÑÏÓ',
				'ÑÏÕ',
				'ÓÁÉ',
				'ÓÁÍ',
				'ÓÁÏ',
				'ÓÁÓ',
				'ÓÅ',
				'ÓÅÉÓ',
				'ÓÅÊ',
				'ÓÅÎ',
				'ÓÅÑ',
				'ÓÅÔ',
				'ÓÅÖ',
				'ÓÇÌÅÑÁ',
				'ÓÉ',
				'ÓÉÁ',
				'ÓÉÃÁ',
				'ÓÉÊ',
				'ÓÉ×',
				'ÓÊÉ',
				'ÓÏÉ',
				'ÓÏÊ',
				'ÓÏË',
				'ÓÏÍ',
				'ÓÏÓ',
				'ÓÏÕ',
				'ÓÑÉ',
				'ÓÔÁ',
				'ÓÔÇ',
				'ÓÔÇÍ',
				'ÓÔÇÓ',
				'ÓÔÉÓ',
				'ÓÔÏ',
				'ÓÔÏÍ',
				'ÓÔÏÕ',
				'ÓÔÏÕÓ',
				'ÓÔÙÍ',
				'ÓÕ',
				'ÓÕÃ×ÑÏÍÙÓ',
				'ÓÕÍ',
				'ÓÕÍÁÌÁ',
				'ÓÕÍÅÐÙÓ',
				'ÓÕÍÇÈÙÓ',
				'Ó×ÅÄÏÍ',
				'ÓÙÓÔÁ',
				'ÔÁ',
				'ÔÁÄÅ',
				'ÔÁÊ',
				'ÔÁÍ',
				'ÔÁÏ',
				'ÔÁÕ',
				'ÔÁ×Á',
				'ÔÁ×ÁÔÅ',
				'ÔÅ',
				'ÔÅÉ',
				'ÔÅË',
				'ÔÅËÉÊÁ',
				'ÔÅËÉÊÙÓ',
				'ÔÅÓ',
				'ÔÅÔ',
				'ÔÆÏ',
				'ÔÇ',
				'ÔÇË',
				'ÔÇÍ',
				'ÔÇÓ',
				'ÔÉ',
				'ÔÉÊ',
				'ÔÉÌ',
				'ÔÉÐÏÔÁ',
				'ÔÉÐÏÔÅ',
				'ÔÉÓ',
				'ÔÍÔ',
				'ÔÏ',
				'ÔÏÉ',
				'ÔÏÊ',
				'ÔÏÌ',
				'ÔÏÍ',
				'ÔÏÐ',
				'ÔÏÓ',
				'ÔÏÓ?Í',
				'ÔÏÓÁ',
				'ÔÏÓÅÓ',
				'ÔÏÓÇ',
				'ÔÏÓÇÍ',
				'ÔÏÓÇÓ',
				'ÔÏÓÏ',
				'ÔÏÓÏÉ',
				'ÔÏÓÏÍ',
				'ÔÏÓÏÓ',
				'ÔÏÓÏÕ',
				'ÔÏÓÏÕÓ',
				'ÔÏÔÅ',
				'ÔÏÕ',
				'ÔÏÕËÁ×ÉÓÔÏ',
				'ÔÏÕËÁ×ÉÓÔÏÍ',
				'ÔÏÕÓ',
				'ÔÓ',
				'ÔÓÁ',
				'ÔÓÅ',
				'ÔÕ×ÏÍ',
				'ÔÙ',
				'ÔÙÍ',
				'ÔÙÑÁ',
				'ÕÁÓ',
				'ÕÂÁ',
				'ÕÂÏ',
				'ÕÉÅ',
				'ÕÉÏ',
				'ÕËÁ',
				'ÕËÇ',
				'ÕÍÉ',
				'ÕÐ',
				'ÕÐÅÑ',
				'ÕÐÏ',
				'ÕÐÏØÇ',
				'ÕÐÏØÉÍ',
				'ÕÓÔÅÑÁ',
				'ÕÖÇ',
				'ÕØÇ',
				'ÖÁ',
				'ÖÁÀ',
				'ÖÁÅ',
				'ÖÁÍ',
				'ÖÁÎ',
				'ÖÁÓ',
				'ÖÁÙ',
				'ÖÅÆ',
				'ÖÅÉ',
				'ÖÅÔÏÓ',
				'ÖÅÕ',
				'ÖÉ',
				'ÖÉË',
				'ÖÉÓ',
				'ÖÏÎ',
				'ÖÐÁ',
				'ÖÑÉ',
				'×Á',
				'×ÁÇ',
				'×ÁË',
				'×ÁÍ',
				'×ÁÖ',
				'×Å',
				'×ÅÉ',
				'×ÈÅÓ',
				'×É',
				'×ÉÁ',
				'×ÉË',
				'×ÉÏ',
				'×ËÌ',
				'×Ì',
				'×ÏÇ',
				'×ÏË',
				'×ÑÙ',
				'×ÔÅÓ',
				'×ÙÑÉÓ',
				'×ÙÑÉÓÔÁ',
				'ØÅÓ',
				'ØÇËÁ',
				'ØÉ',
				'ØÉÔ',
				'Ù',
				'ÙÁ',
				'ÙÁÓ',
				'ÙÄÅ',
				'ÙÅÓ',
				'ÙÈÙ',
				'ÙÌÁ',
				'ÙÌÅ',
				'ÙÍ',
				'ÙÏ',
				'ÙÏÍ',
				'ÙÏÕ',
				'ÙÓ',
				'ÙÓÁÍ',
				'ÙÓÇ',
				'ÙÓÏÔÏÕ',
				'ÙÓÐÏÕ',
				'ÙÓÔÅ',
				'ÙÓÔÏÓÏ',
				'ÙÔÁ',
				'Ù×',
				'ÙÙÍ',
				'ÁÄÉÁÊÏÐÁ',
				'ÁÉ',
				'ÁÊÏÌÁ',
				'ÁÊÏÌÇ',
				'ÁÊÑÉÂÙÓ',
				'ÁËÇÈÅÉÁ',
				'ÁËÇÈÉÍÁ',
				'ÁËËÁ',
				'ÁËËÁ×ÏÕ',
				'ÁËËÅÓ',
				'ÁËËÇ',
				'ÁËËÇÍ',
				'ÁËËÇÓ',
				'ÁËËÉÙÓ',
				'ÁËËÉÙÔÉÊÁ',
				'ÁËËÏ',
				'ÁËËÏÉ',
				'ÁËËÏÉÙÓ',
				'ÁËËÏÉÙÔÉÊÁ',
				'ÁËËÏÍ',
				'ÁËËÏÓ',
				'ÁËËÏÔÅ',
				'ÁËËÏÕ',
				'ÁËËÏÕÓ',
				'ÁËËÙÍ',
				'ÁÌÁ',
				'ÁÌÅÓÁ',
				'ÁÌÅÓÙÓ',
				'ÁÍ',
				'ÁÍÁ',
				'ÁÍÁÌÅÓÁ',
				'ÁÍÁÌÅÔÁÎÕ',
				'ÁÍÅÕ',
				'ÁÍÔÉ',
				'ÁÍÔÉÐÅÑÁ',
				'ÁÍÔÉÓ',
				'ÁÍÙ',
				'ÁÍÙÔÅÑÙ',
				'ÁÎÁÖÍÁ',
				'ÁÐ',
				'ÁÐÅÍÁÍÔÉ',
				'ÁÐÏ',
				'ÁÐÏØÅ',
				'ÁÑÁ',
				'ÁÑÁÃÅ',
				'ÁÑÃÁ',
				'ÁÑÃÏÔÅÑÏ',
				'ÁÑÉÓÔÅÑÁ',
				'ÁÑÊÅÔÁ',
				'ÁÑ×ÉÊÁ',
				'ÁÓ',
				'ÁÕÑÉÏ',
				'ÁÕÔÁ',
				'ÁÕÔÅÓ',
				'ÁÕÔÇ',
				'ÁÕÔÇÍ',
				'ÁÕÔÇÓ',
				'ÁÕÔÏ',
				'ÁÕÔÏÉ',
				'ÁÕÔÏÍ',
				'ÁÕÔÏÓ',
				'ÁÕÔÏÕ',
				'ÁÕÔÏÕÓ',
				'ÁÕÔÙÍ',
				'ÁÖÏÔÏÕ',
				'ÁÖÏÕ',
				'ÂÅÂÁÉÁ',
				'ÂÅÂÁÉÏÔÁÔÁ',
				'ÃÉ',
				'ÃÉÁ',
				'ÃÑÇÃÏÑÁ',
				'ÃÕÑÙ',
				'ÄÁ',
				'ÄÅ',
				'ÄÅÉÍÁ',
				'ÄÅÍ',
				'ÄÅÎÉÁ',
				'ÄÇÈÅÍ',
				'ÄÇËÁÄÇ',
				'ÄÉ',
				'ÄÉÁ',
				'ÄÉÁÑÊÙÓ',
				'ÄÉÊÁ',
				'ÄÉÊÏ',
				'ÄÉÊÏÉ',
				'ÄÉÊÏÓ',
				'ÄÉÊÏÕ',
				'ÄÉÊÏÕÓ',
				'ÄÉÏËÏÕ',
				'ÄÉÐËÁ',
				'ÄÉ×ÙÓ',
				'ÅÁÍ',
				'ÅÁÕÔÏ',
				'ÅÁÕÔÏÍ',
				'ÅÁÕÔÏÕ',
				'ÅÁÕÔÏÕÓ',
				'ÅÁÕÔÙÍ',
				'ÅÃÊÁÉÑÁ',
				'ÅÃÊÁÉÑÙÓ',
				'ÅÃÙ',
				'ÅÄÙ',
				'ÅÉÄÅÌÇ',
				'ÅÉÈÅ',
				'ÅÉÌÁÉ',
				'ÅÉÌÁÓÔÅ',
				'ÅÉÍÁÉ',
				'ÅÉÓ',
				'ÅÉÓÁÉ',
				'ÅÉÓÁÓÔÅ',
				'ÅÉÓÔÅ',
				'ÅÉÔÅ',
				'ÅÉ×Á',
				'ÅÉ×ÁÌÅ',
				'ÅÉ×ÁÍ',
				'ÅÉ×ÁÔÅ',
				'ÅÉ×Å',
				'ÅÉ×ÅÓ',
				'ÅÊÁÓÔÁ',
				'ÅÊÁÓÔÅÓ',
				'ÅÊÁÓÔÇ',
				'ÅÊÁÓÔÇÍ',
				'ÅÊÁÓÔÇÓ',
				'ÅÊÁÓÔÏ',
				'ÅÊÁÓÔÏÉ',
				'ÅÊÁÓÔÏÍ',
				'ÅÊÁÓÔÏÓ',
				'ÅÊÁÓÔÏÕ',
				'ÅÊÁÓÔÏÕÓ',
				'ÅÊÁÓÔÙÍ',
				'ÅÊÅÉ',
				'ÅÊÅÉÍÁ',
				'ÅÊÅÉÍÅÓ',
				'ÅÊÅÉÍÇ',
				'ÅÊÅÉÍÇÍ',
				'ÅÊÅÉÍÇÓ',
				'ÅÊÅÉÍÏ',
				'ÅÊÅÉÍÏÉ',
				'ÅÊÅÉÍÏÍ',
				'ÅÊÅÉÍÏÓ',
				'ÅÊÅÉÍÏÕ',
				'ÅÊÅÉÍÏÕÓ',
				'ÅÊÅÉÍÙÍ',
				'ÅÊÔÏÓ',
				'ÅÌÁÓ',
				'ÅÌÅÉÓ',
				'ÅÌÅÍÁ',
				'ÅÌÐÑÏÓ',
				'ÅÍ',
				'ÅÍÁ',
				'ÅÍÁÍ',
				'ÅÍÁÓ',
				'ÅÍÏÓ',
				'ÅÍÔÅËÙÓ',
				'ÅÍÔÏÓ',
				'ÅÍÔÙÌÅÔÁÎÕ',
				'ÅÍÙ',
				'ÅÎ',
				'ÅÎÁÖÍÁ',
				'ÅÎÇÓ',
				'ÅÎÉÓÏÕ',
				'ÅÎÙ',
				'ÅÐÁÍÙ',
				'ÅÐÅÉÄÇ',
				'ÅÐÅÉÔÁ',
				'ÅÐÉ',
				'ÅÐÉÓÇÓ',
				'ÅÐÏÌÅÍÙÓ',
				'ÅÓÁÓ',
				'ÅÓÅÉÓ',
				'ÅÓÅÍÁ',
				'ÅÓÔÙ',
				'ÅÓÕ',
				'ÅÔÅÑÁ',
				'ÅÔÅÑÁÉ',
				'ÅÔÅÑÁÓ',
				'ÅÔÅÑÅÓ',
				'ÅÔÅÑÇ',
				'ÅÔÅÑÇÓ',
				'ÅÔÅÑÏ',
				'ÅÔÅÑÏÉ',
				'ÅÔÅÑÏÍ',
				'ÅÔÅÑÏÓ',
				'ÅÔÅÑÏÕ',
				'ÅÔÅÑÏÕÓ',
				'ÅÔÅÑÙÍ',
				'ÅÔÏÕÔÁ',
				'ÅÔÏÕÔÅÓ',
				'ÅÔÏÕÔÇ',
				'ÅÔÏÕÔÇÍ',
				'ÅÔÏÕÔÇÓ',
				'ÅÔÏÕÔÏ',
				'ÅÔÏÕÔÏÉ',
				'ÅÔÏÕÔÏÍ',
				'ÅÔÏÕÔÏÓ',
				'ÅÔÏÕÔÏÕ',
				'ÅÔÏÕÔÏÕÓ',
				'ÅÔÏÕÔÙÍ',
				'ÅÔÓÉ',
				'ÅÕÃÅ',
				'ÅÕÈÕÓ',
				'ÅÕÔÕ×ÙÓ',
				'ÅÖÅÎÇÓ',
				'Å×ÅÉ',
				'Å×ÅÉÓ',
				'Å×ÅÔÅ',
				'Å×ÈÅÓ',
				'Å×ÏÌÅ',
				'Å×ÏÕÌÅ',
				'Å×ÏÕÍ',
				'Å×ÔÅÓ',
				'Å×Ù',
				'ÅÙÓ',
				'Ç',
				'ÇÄÇ',
				'ÇÌÁÓÔÁÍ',
				'ÇÌÁÓÔÅ',
				'ÇÌÏÕÍ',
				'ÇÓÁÓÔÁÍ',
				'ÇÓÁÓÔÅ',
				'ÇÓÏÕÍ',
				'ÇÔÁÍ',
				'ÇÔÁÍÅ',
				'ÇÔÏÉ',
				'ÇÔÔÏÍ',
				'ÈÁ',
				'É',
				'ÉÄÉÁ',
				'ÉÄÉÁÍ',
				'ÉÄÉÁÓ',
				'ÉÄÉÅÓ',
				'ÉÄÉÏ',
				'ÉÄÉÏÉ',
				'ÉÄÉÏÍ',
				'ÉÄÉÏÓ',
				'ÉÄÉÏÕ',
				'ÉÄÉÏÕÓ',
				'ÉÄÉÙÍ',
				'ÉÄÉÙÓ',
				'ÉÉ',
				'ÉÉÉ',
				'ÉÓÁÌÅ',
				'ÉÓÉÁ',
				'ÉÓÙÓ',
				'ÊÁÈÅ',
				'ÊÁÈÅÌÉÁ',
				'ÊÁÈÅÌÉÁÓ',
				'ÊÁÈÅÍÁ',
				'ÊÁÈÅÍÁÓ',
				'ÊÁÈÅÍÏÓ',
				'ÊÁÈÅÔÉ',
				'ÊÁÈÏËÏÕ',
				'ÊÁÈÙÓ',
				'ÊÁÉ',
				'ÊÁÊÁ',
				'ÊÁÊÙÓ',
				'ÊÁËÁ',
				'ÊÁËÙÓ',
				'ÊÁÌÉÁ',
				'ÊÁÌÉÁÍ',
				'ÊÁÌÉÁÓ',
				'ÊÁÌÐÏÓÁ',
				'ÊÁÌÐÏÓÅÓ',
				'ÊÁÌÐÏÓÇ',
				'ÊÁÌÐÏÓÇÍ',
				'ÊÁÌÐÏÓÇÓ',
				'ÊÁÌÐÏÓÏ',
				'ÊÁÌÐÏÓÏÉ',
				'ÊÁÌÐÏÓÏÍ',
				'ÊÁÌÐÏÓÏÓ',
				'ÊÁÌÐÏÓÏÕ',
				'ÊÁÌÐÏÓÏÕÓ',
				'ÊÁÌÐÏÓÙÍ',
				'ÊÁÍÅÉÓ',
				'ÊÁÍÅÍ',
				'ÊÁÍÅÍÁ',
				'ÊÁÍÅÍÁÍ',
				'ÊÁÍÅÍÁÓ',
				'ÊÁÍÅÍÏÓ',
				'ÊÁÐÏÉÁ',
				'ÊÁÐÏÉÁÍ',
				'ÊÁÐÏÉÁÓ',
				'ÊÁÐÏÉÅÓ',
				'ÊÁÐÏÉÏ',
				'ÊÁÐÏÉÏÉ',
				'ÊÁÐÏÉÏÍ',
				'ÊÁÐÏÉÏÓ',
				'ÊÁÐÏÉÏÕ',
				'ÊÁÐÏÉÏÕÓ',
				'ÊÁÐÏÉÙÍ',
				'ÊÁÐÏÔÅ',
				'ÊÁÐÏÕ',
				'ÊÁÐÙÓ',
				'ÊÁÔ',
				'ÊÁÔÁ',
				'ÊÁÔÉ',
				'ÊÁÔÉÔÉ',
				'ÊÁÔÏÐÉÍ',
				'ÊÁÔÙ',
				'ÊÉÏËÁÓ',
				'ÊËÐ',
				'ÊÏÍÔÁ',
				'ÊÔË',
				'ÊÕÑÉÙÓ',
				'ËÉÃÁÊÉ',
				'ËÉÃÏ',
				'ËÉÃÙÔÅÑÏ',
				'ËÏÃÙ',
				'ËÏÉÐÁ',
				'ËÏÉÐÏÍ',
				'ÌÁ',
				'ÌÁÆÉ',
				'ÌÁÊÁÑÉ',
				'ÌÁÊÑÕÁ',
				'ÌÁËÉÓÔÁ',
				'ÌÁËËÏÍ',
				'ÌÁÓ',
				'ÌÅ',
				'ÌÅÈÁÕÑÉÏ',
				'ÌÅÉÏÍ',
				'ÌÅËÅÉ',
				'ÌÅËËÅÔÁÉ',
				'ÌÅÌÉÁÓ',
				'ÌÅÍ',
				'ÌÅÑÉÊÁ',
				'ÌÅÑÉÊÅÓ',
				'ÌÅÑÉÊÏÉ',
				'ÌÅÑÉÊÏÕÓ',
				'ÌÅÑÉÊÙÍ',
				'ÌÅÓÁ',
				'ÌÅÔ',
				'ÌÅÔÁ',
				'ÌÅÔÁÎÕ',
				'ÌÅ×ÑÉ',
				'ÌÇ',
				'ÌÇÄÅ',
				'ÌÇÍ',
				'ÌÇÐÙÓ',
				'ÌÇÔÅ',
				'ÌÉÁ',
				'ÌÉÁÍ',
				'ÌÉÁÓ',
				'ÌÏËÉÓ',
				'ÌÏËÏÍÏÔÉ',
				'ÌÏÍÁ×Á',
				'ÌÏÍÅÓ',
				'ÌÏÍÇ',
				'ÌÏÍÇÍ',
				'ÌÏÍÇÓ',
				'ÌÏÍÏ',
				'ÌÏÍÏÉ',
				'ÌÏÍÏÌÉÁÓ',
				'ÌÏÍÏÓ',
				'ÌÏÍÏÕ',
				'ÌÏÍÏÕÓ',
				'ÌÏÍÙÍ',
				'ÌÏÕ',
				'ÌÐÏÑÅÉ',
				'ÌÐÏÑÏÕÍ',
				'ÌÐÑÁÂÏ',
				'ÌÐÑÏÓ',
				'ÍÁ',
				'ÍÁÉ',
				'ÍÙÑÉÓ',
				'ÎÁÍÁ',
				'ÎÁÖÍÉÊÁ',
				'Ï',
				'ÏÉ',
				'ÏËÁ',
				'ÏËÅÓ',
				'ÏËÇ',
				'ÏËÇÍ',
				'ÏËÇÓ',
				'ÏËÏ',
				'ÏËÏÃÕÑÁ',
				'ÏËÏÉ',
				'ÏËÏÍ',
				'ÏËÏÍÅÍ',
				'ÏËÏÓ',
				'ÏËÏÔÅËÁ',
				'ÏËÏÕ',
				'ÏËÏÕÓ',
				'ÏËÙÍ',
				'ÏËÙÓ',
				'ÏËÙÓÄÉÏËÏÕ',
				'ÏÌÙÓ',
				'ÏÐÏÉÁ',
				'ÏÐÏÉÁÄÇÐÏÔÅ',
				'ÏÐÏÉÁÍ',
				'ÏÐÏÉÁÍÄÇÐÏÔÅ',
				'ÏÐÏÉÁÓ',
				'ÏÐÏÉÁÓÄÇÐÏÔÅ',
				'ÏÐÏÉÄÇÐÏÔÅ',
				'ÏÐÏÉÅÓ',
				'ÏÐÏÉÅÓÄÇÐÏÔÅ',
				'ÏÐÏÉÏ',
				'ÏÐÏÉÏÄÇÐÏÔÅ',
				'ÏÐÏÉÏÉ',
				'ÏÐÏÉÏÍ',
				'ÏÐÏÉÏÍÄÇÐÏÔÅ',
				'ÏÐÏÉÏÓ',
				'ÏÐÏÉÏÓÄÇÐÏÔÅ',
				'ÏÐÏÉÏÕ',
				'ÏÐÏÉÏÕÄÇÐÏÔÅ',
				'ÏÐÏÉÏÕÓ',
				'ÏÐÏÉÏÕÓÄÇÐÏÔÅ',
				'ÏÐÏÉÙÍ',
				'ÏÐÏÉÙÍÄÇÐÏÔÅ',
				'ÏÐÏÔÅ',
				'ÏÐÏÔÅÄÇÐÏÔÅ',
				'ÏÐÏÕ',
				'ÏÐÏÕÄÇÐÏÔÅ',
				'ÏÐÙÓ',
				'ÏÑÉÓÌÅÍÁ',
				'ÏÑÉÓÌÅÍÅÓ',
				'ÏÑÉÓÌÅÍÙÍ',
				'ÏÑÉÓÌÅÍÙÓ',
				'ÏÓÁ',
				'ÏÓÁÄÇÐÏÔÅ',
				'ÏÓÅÓ',
				'ÏÓÅÓÄÇÐÏÔÅ',
				'ÏÓÇ',
				'ÏÓÇÄÇÐÏÔÅ',
				'ÏÓÇÍ',
				'ÏÓÇÍÄÇÐÏÔÅ',
				'ÏÓÇÓ',
				'ÏÓÇÓÄÇÐÏÔÅ',
				'ÏÓÏ',
				'ÏÓÏÄÇÐÏÔÅ',
				'ÏÓÏÉ',
				'ÏÓÏÉÄÇÐÏÔÅ',
				'ÏÓÏÍ',
				'ÏÓÏÍÄÇÐÏÔÅ',
				'ÏÓÏÓ',
				'ÏÓÏÓÄÇÐÏÔÅ',
				'ÏÓÏÕ',
				'ÏÓÏÕÄÇÐÏÔÅ',
				'ÏÓÏÕÓ',
				'ÏÓÏÕÓÄÇÐÏÔÅ',
				'ÏÓÙÍ',
				'ÏÓÙÍÄÇÐÏÔÅ',
				'ÏÔÁÍ',
				'ÏÔÉ',
				'ÏÔÉÄÇÐÏÔÅ',
				'ÏÔÏÕ',
				'ÏÕ',
				'ÏÕÄÅ',
				'ÏÕÔÅ',
				'Ï×É',
				'ÐÁËÉ',
				'ÐÁÍÔÏÔÅ',
				'ÐÁÍÔÏÕ',
				'ÐÁÍÔÙÓ',
				'ÐÁÑÁ',
				'ÐÅÑÁ',
				'ÐÅÑÉ',
				'ÐÅÑÉÐÏÕ',
				'ÐÅÑÉÓÓÏÔÅÑÏ',
				'ÐÅÑÓÉ',
				'ÐÅÑÕÓÉ',
				'ÐÉÁ',
				'ÐÉÈÁÍÏÍ',
				'ÐÉÏ',
				'ÐÉÓÙ',
				'ÐËÁÉ',
				'ÐËÅÏÍ',
				'ÐËÇÍ',
				'ÐÏÉÁ',
				'ÐÏÉÁÍ',
				'ÐÏÉÁÓ',
				'ÐÏÉÅÓ',
				'ÐÏÉÏ',
				'ÐÏÉÏÉ',
				'ÐÏÉÏÍ',
				'ÐÏÉÏÓ',
				'ÐÏÉÏÕ',
				'ÐÏÉÏÕÓ',
				'ÐÏÉÙÍ',
				'ÐÏËÕ',
				'ÐÏÓÅÓ',
				'ÐÏÓÇ',
				'ÐÏÓÇÍ',
				'ÐÏÓÇÓ',
				'ÐÏÓÏÉ',
				'ÐÏÓÏÓ',
				'ÐÏÓÏÕÓ',
				'ÐÏÔÅ',
				'ÐÏÕ',
				'ÐÏÕÈÅ',
				'ÐÏÕÈÅÍÁ',
				'ÐÑÅÐÅÉ',
				'ÐÑÉÍ',
				'ÐÑÏ',
				'ÐÑÏÊÅÉÌÅÍÏÕ',
				'ÐÑÏÊÅÉÔÁÉ',
				'ÐÑÏÐÅÑÓÉ',
				'ÐÑÏÓ',
				'ÐÑÏÔÏÕ',
				'ÐÑÏ×ÈÅÓ',
				'ÐÑÏ×ÔÅÓ',
				'ÐÑÙÔÕÔÅÑÁ',
				'ÐÙÓ',
				'ÓÁÍ',
				'ÓÁÓ',
				'ÓÅ',
				'ÓÅÉÓ',
				'ÓÇÌÅÑÁ',
				'ÓÉÃÁ',
				'ÓÏÕ',
				'ÓÔÁ',
				'ÓÔÇ',
				'ÓÔÇÍ',
				'ÓÔÇÓ',
				'ÓÔÉÓ',
				'ÓÔÏ',
				'ÓÔÏÍ',
				'ÓÔÏÕ',
				'ÓÔÏÕÓ',
				'ÓÔÙÍ',
				'ÓÕÃ×ÑÏÍÙÓ',
				'ÓÕÍ',
				'ÓÕÍÁÌÁ',
				'ÓÕÍÅÐÙÓ',
				'ÓÕÍÇÈÙÓ',
				'ÓÕ×ÍÁ',
				'ÓÕ×ÍÁÓ',
				'ÓÕ×ÍÅÓ',
				'ÓÕ×ÍÇ',
				'ÓÕ×ÍÇÍ',
				'ÓÕ×ÍÇÓ',
				'ÓÕ×ÍÏ',
				'ÓÕ×ÍÏÉ',
				'ÓÕ×ÍÏÍ',
				'ÓÕ×ÍÏÓ',
				'ÓÕ×ÍÏÕ',
				'ÓÕ×ÍÏÕ',
				'ÓÕ×ÍÏÕÓ',
				'ÓÕ×ÍÙÍ',
				'ÓÕ×ÍÙÓ',
				'Ó×ÅÄÏÍ',
				'ÓÙÓÔÁ',
				'ÔÁ',
				'ÔÁÄÅ',
				'ÔÁÕÔÁ',
				'ÔÁÕÔÅÓ',
				'ÔÁÕÔÇ',
				'ÔÁÕÔÇÍ',
				'ÔÁÕÔÇÓ',
				'ÔÁÕÔÏ',
				'ÔÁÕÔÏÍ',
				'ÔÁÕÔÏÓ',
				'ÔÁÕÔÏÕ',
				'ÔÁÕÔÙÍ',
				'ÔÁ×Á',
				'ÔÁ×ÁÔÅ',
				'ÔÅËÉÊÁ',
				'ÔÅËÉÊÙÓ',
				'ÔÅÓ',
				'ÔÅÔÏÉÁ',
				'ÔÅÔÏÉÁÍ',
				'ÔÅÔÏÉÁÓ',
				'ÔÅÔÏÉÅÓ',
				'ÔÅÔÏÉÏ',
				'ÔÅÔÏÉÏÉ',
				'ÔÅÔÏÉÏÍ',
				'ÔÅÔÏÉÏÓ',
				'ÔÅÔÏÉÏÕ',
				'ÔÅÔÏÉÏÕÓ',
				'ÔÅÔÏÉÙÍ',
				'ÔÇ',
				'ÔÇÍ',
				'ÔÇÓ',
				'ÔÉ',
				'ÔÉÐÏÔÁ',
				'ÔÉÐÏÔÅ',
				'ÔÉÓ',
				'ÔÏ',
				'ÔÏÉ',
				'ÔÏÍ',
				'ÔÏÓ',
				'ÔÏÓÁ',
				'ÔÏÓÅÓ',
				'ÔÏÓÇ',
				'ÔÏÓÇÍ',
				'ÔÏÓÇÓ',
				'ÔÏÓÏ',
				'ÔÏÓÏÉ',
				'ÔÏÓÏÍ',
				'ÔÏÓÏÓ',
				'ÔÏÓÏÕ',
				'ÔÏÓÏÕÓ',
				'ÔÏÓÙÍ',
				'ÔÏÔÅ',
				'ÔÏÕ',
				'ÔÏÕËÁ×ÉÓÔÏ',
				'ÔÏÕËÁ×ÉÓÔÏÍ',
				'ÔÏÕÓ',
				'ÔÏÕÔÁ',
				'ÔÏÕÔÅÓ',
				'ÔÏÕÔÇ',
				'ÔÏÕÔÇÍ',
				'ÔÏÕÔÇÓ',
				'ÔÏÕÔÏ',
				'ÔÏÕÔÏÉ',
				'ÔÏÕÔÏÉÓ',
				'ÔÏÕÔÏÍ',
				'ÔÏÕÔÏÓ',
				'ÔÏÕÔÏÕ',
				'ÔÏÕÔÏÕÓ',
				'ÔÏÕÔÙÍ',
				'ÔÕ×ÏÍ',
				'ÔÙÍ',
				'ÔÙÑÁ',
				'ÕÐ',
				'ÕÐÅÑ',
				'ÕÐÏ',
				'ÕÐÏØÇ',
				'ÕÐÏØÉÍ',
				'ÕÓÔÅÑÁ',
				'ÖÅÔÏÓ',
				'×ÁÌÇËÁ',
				'×ÈÅÓ',
				'×ÔÅÓ',
				'×ÙÑÉÓ',
				'×ÙÑÉÓÔÁ',
				'ØÇËÁ',
				'Ù',
				'ÙÑÁÉÁ',
				'ÙÓ',
				'ÙÓÁÍ',
				'ÙÓÏÔÏÕ',
				'ÙÓÐÏÕ',
				'ÙÓÔÅ',
				'ÙÓÔÏÓÏ',
				'Ù×',
				'ÅÍÏÓ',
				'ÄÕÏ',
				'ÔÑÉÁ',
				'ÔÑÉÙÍ',
				'ÔÅÓÓÅÑÁ',
				'ÔÅÓÓÁÑÙÍ',
				'ÐÅÍÔÅ',
				'ÅÎÉ',
				'ÅÖÔÁ',
				'ÏÊÔÙ',
				'Ï×ÔÙ',
				'ÅÍÍÉÁ',
			);
			if ( in_array( $w, $stop_w ) ) {
				return null;
			}
			// Verbs Removal
			$re = '/(.+?)(ÉÆÁ|ÉÓÁ|ÉÓÙ|ÉÓÏÕÍ|ÉÆÅÓ|ÉÆÅ|ÉÆÁÌÅ|ÉÆÁÔÅ|ÉÆÁÍ|ÉÆÁÍÅ|ÉÆÙ|ÉÆÅÉÓ|ÉÆÅÉ|ÉÆÏÕÌÅ|ÉÆÅÔÅ|ÉÆÏÕÍ|ÉÆÏÕÍÅ|ÅÉ|ÉÓÅ|ÇÓÅ|ÏÍÔÁÉ|ÏÕÍ|ÏÍÔÁÓ)$/';
			if ( preg_match( $re, $w ) ) {
				return null;
			}
		}
		// ------------------------- Stemming Rules : Original Stemmer

		// step1list is used in Step 1. 41 stems
		$step1list               = array();
		$step1list['ÖÁÃÉÁ']      = 'ÖÁ';
		$step1list['ÖÁÃÉÏÕ']     = 'ÖÁ';
		$step1list['ÖÁÃÉÙÍ']     = 'ÖÁ';
		$step1list['ÓÊÁÃÉÁ']     = 'ÓÊÁ';
		$step1list['ÓÊÁÃÉÏÕ']    = 'ÓÊÁ';
		$step1list['ÓÊÁÃÉÙÍ']    = 'ÓÊÁ';
		$step1list['ÏËÏÃÉÏÕ']    = 'ÏËÏ';
		$step1list['ÏËÏÃÉÁ']     = 'ÏËÏ';
		$step1list['ÏËÏÃÉÙÍ']    = 'ÏËÏ';
		$step1list['ÓÏÃÉÏÕ']     = 'ÓÏ';
		$step1list['ÓÏÃÉÁ']      = 'ÓÏ';
		$step1list['ÓÏÃÉÙÍ']     = 'ÓÏ';
		$step1list['ÔÁÔÏÃÉÁ']    = 'ÔÁÔÏ';
		$step1list['ÔÁÔÏÃÉÏÕ']   = 'ÔÁÔÏ';
		$step1list['ÔÁÔÏÃÉÙÍ']   = 'ÔÁÔÏ';
		$step1list['ÊÑÅÁÓ']      = 'ÊÑÅ';
		$step1list['ÊÑÅÁÔÏÓ']    = 'ÊÑÅ';
		$step1list['ÊÑÅÁÔÁ']     = 'ÊÑÅ';
		$step1list['ÊÑÅÁÔÙÍ']    = 'ÊÑÅ';
		$step1list['ÐÅÑÁÓ']      = 'ÐÅÑ';
		$step1list['ÐÅÑÁÔÏÓ']    = 'ÐÅÑ';
		$step1list['ÐÅÑÁÔÇ']     = 'ÐÅÑ'; //Added by Spyros . also at $re in step1
		$step1list['ÐÅÑÁÔÁ']     = 'ÐÅÑ';
		$step1list['ÐÅÑÁÔÙÍ']    = 'ÐÅÑ';
		$step1list['ÔÅÑÁÓ']      = 'ÔÅÑ';
		$step1list['ÔÅÑÁÔÏÓ']    = 'ÔÅÑ';
		$step1list['ÔÅÑÁÔÁ']     = 'ÔÅÑ';
		$step1list['ÔÅÑÁÔÙÍ']    = 'ÔÅÑ';
		$step1list['ÖÙÓ']        = 'ÖÙ';
		$step1list['ÖÙÔÏÓ']      = 'ÖÙ';
		$step1list['ÖÙÔÁ']       = 'ÖÙ';
		$step1list['ÖÙÔÙÍ']      = 'ÖÙ';
		$step1list['ÊÁÈÅÓÔÙÓ']   = 'ÊÁÈÅÓÔ';
		$step1list['ÊÁÈÅÓÔÙÔÏÓ'] = 'ÊÁÈÅÓÔ';
		$step1list['ÊÁÈÅÓÔÙÔÁ']  = 'ÊÁÈÅÓÔ';
		$step1list['ÊÁÈÅÓÔÙÔÙÍ'] = 'ÊÁÈÅÓÔ';
		$step1list['ÃÅÃÏÍÏÓ']    = 'ÃÅÃÏÍ';
		$step1list['ÃÅÃÏÍÏÔÏÓ']  = 'ÃÅÃÏÍ';
		$step1list['ÃÅÃÏÍÏÔÁ']   = 'ÃÅÃÏÍ';
		$step1list['ÃÅÃÏÍÏÔÙÍ']  = 'ÃÅÃÏÍ';
		$v                       = '(Á|Å|Ç|É|Ï|Õ|Ù)'; // vowel
		$v2                      = '(Á|Å|Ç|É|Ï|Ù)';      //vowel without Y
		$test1                   = true;              // Variable needed later
		/*
		 * preg_match definition
		 *
		 * preg_match( $pattern, $input_string, $matches);
		 * $pattern     :   the pattern to search for as a string
		 * $input_string:   the input string
		 * $matches     :   if $matches is provided, then it is filled with the results of search
		 */

		$match = [ ];

		//Step S1. 14 stems
		$re       = '/^(.+?)(ÉÆÁ|ÉÆÅÓ|ÉÆÅ|ÉÆÁÌÅ|ÉÆÁÔÅ|ÉÆÁÍ|ÉÆÁÍÅ|ÉÆÙ|ÉÆÅÉÓ|ÉÆÅÉ|ÉÆÏÕÌÅ|ÉÆÅÔÅ|ÉÆÏÕÍ|ÉÆÏÕÍÅ)$/';
		$exceptS1 = '/^(ÁÍÁÌÐÁ|ÅÌÐÁ|ÅÐÁ|ÎÁÍÁÐÁ|ÐÁ|ÐÅÑÉÐÁ|ÁÈÑÏ|ÓÕÍÁÈÑÏ|ÄÁÍÅ)$/';
		$exceptS2 = '/^(ÌÁÑÊ|ÊÏÑÍ|ÁÌÐÁÑ|ÁÑÑ|ÂÁÈÕÑÉ|ÂÁÑÊ|Â|ÂÏËÂÏÑ|ÃÊÑ|ÃËÕÊÏÑ|ÃËÕÊÕÑ|ÉÌÐ|Ë|ËÏÕ|ÌÁÑ|Ì|ÐÑ|ÌÐÑ|ÐÏËÕÑ|Ð|Ñ|ÐÉÐÅÑÏÑ)$/';
		if ( preg_match( $re, $w, $match ) ) {
			$stem   = $match[1];
			$suffix = $match[2];
			$w      = $stem . $step1list[ $suffix ];
			$test1  = false;
			if ( preg_match( $exceptS1, $w ) ) {
				$w = $w . 'I';
			}
			if ( preg_match( $exceptS2, $w ) ) {
				$w = $w . 'IÆ';
			}

			return $this->return_stem( $w, $encoding_changed );
		}
		//Step S2. 7 stems
		$re       = '/^(.+?)(ÙÈÇÊÁ|ÙÈÇÊÅÓ|ÙÈÇÊÅ|ÙÈÇÊÁÌÅ|ÙÈÇÊÁÔÅ|ÙÈÇÊÁÍ|ÙÈÇÊÁÍÅ)$/';
		$exceptS1 = '/^(ÁË|ÂÉ|ÅÍ|ÕØ|ËÉ|ÆÙ|Ó|×)$/';
		if ( preg_match( $re, $w, $match ) ) {
			$stem   = $match[1];
			$suffix = $match[2];
			$w      = $stem . $step1list[ $suffix ];
			$test1  = false;
			if ( preg_match( $exceptS1, $w ) ) {
				$w = $w . 'ÙÍ';
			}

			return $this->return_stem( $w, $encoding_changed );
		}
		//Step S3. 7 stems
		$re       = '/^(.+?)(ÉÓÁ|ÉÓÅÓ|ÉÓÅ|ÉÓÁÌÅ|ÉÓÁÔÅ|ÉÓÁÍ|ÉÓÁÍÅ)$/';
		$exceptS1 = '/^(ÁÍÁÌÐÁ|ÁÈÑÏ|ÅÌÐÁ|ÅÓÅ|ÅÓÙÊËÅ|ÅÐÁ|ÎÁÍÁÐÁ|ÅÐÅ|ÐÅÑÉÐÁ|ÁÈÑÏ|ÓÕÍÁÈÑÏ|ÄÁÍÅ|ÊËÅ|×ÁÑÔÏÐÁ|ÅÎÁÑ×Á|ÌÅÔÅÐÅ|ÁÐÏÊËÅ|ÁÐÅÊËÅ|ÅÊËÅ|ÐÅ|ÐÅÑÉÐÁ)$/';
		$exceptS2 = '/^(ÁÍ|ÁÖ|ÃÅ|ÃÉÃÁÍÔÏÁÖ|ÃÊÅ|ÄÇÌÏÊÑÁÔ|ÊÏÌ|ÃÊ|Ì|Ð|ÐÏÕÊÁÌ|ÏËÏ|ËÁÑ)$/';
		if ( $w == 'ÉÓÁ' ) {
			$w = 'ÉÓ';

			return $w;
		}
		if ( preg_match( $re, $w, $match ) ) {
			$stem   = $match[1];
			$suffix = $match[2];
			$w      = $stem . $step1list[ $suffix ];
			$test1  = false;
			if ( preg_match( $exceptS1, $w ) ) {
				$w = $w . 'É';
			}

			return $this->return_stem( $w, $encoding_changed );
		}
		//Step S4. 7 stems
		$re       = '/^(.+?)(ÉÓÙ|ÉÓÅÉÓ|ÉÓÅÉ|ÉÓÏÕÌÅ|ÉÓÅÔÅ|ÉÓÏÕÍ|ÉÓÏÕÍÅ)$/';
		$exceptS1 = '/^(ÁÍÁÌÐÁ|ÅÌÐÁ|ÅÓÅ|ÅÓÙÊËÅ|ÅÐÁ|ÎÁÍÁÐÁ|ÅÐÅ|ÐÅÑÉÐÁ|ÁÈÑÏ|ÓÕÍÁÈÑÏ|ÄÁÍÅ|ÊËÅ|×ÁÑÔÏÐÁ|ÅÎÁÑ×Á|ÌÅÔÅÐÅ|ÁÐÏÊËÅ|ÁÐÅÊËÅ|ÅÊËÅ|ÐÅ|ÐÅÑÉÐÁ)$/';
		if ( preg_match( $re, $w, $match ) ) {
			$stem   = $match[1];
			$suffix = $match[2];
			$w      = $stem . $step1list[ $suffix ];
			$test1  = false;
			if ( preg_match( $exceptS1, $w ) ) {
				$w = $w . 'É';
			}

			return $this->return_stem( $w, $encoding_changed );
		}


		//Step S5. 11 stems
		$re       = '/^(.+?)(ÉÓÔÏÓ|ÉÓÔÏÕ|ÉÓÔÏ|ÉÓÔÅ|ÉÓÔÏÉ|ÉÓÔÙÍ|ÉÓÔÏÕÓ|ÉÓÔÇ|ÉÓÔÇÓ|ÉÓÔÁ|ÉÓÔÅÓ)$/';
		$exceptS1 = '/^(Ì|Ð|ÁÐ|ÁÑ|ÇÄ|ÊÔ|ÓÊ|Ó×|ÕØ|ÖÁ|×Ñ|×Ô|ÁÊÔ|ÁÏÑ|ÁÓ×|ÁÔÁ|Á×Í|Á×Ô|ÃÅÌ|ÃÕÑ|ÅÌÐ|ÅÕÐ|Å×È|ÇÖÁ|¹ÖÁ|ÊÁÈ|ÊÁÊ|ÊÕË|ËÕÃ|ÌÁÊ|ÌÅÃ|ÔÁ×|ÖÉË|×ÙÑ)$/';
		$exceptS2 = '/^(ÄÁÍÅ|ÓÕÍÁÈÑÏ|ÊËÅ|ÓÅ|ÅÓÙÊËÅ|ÁÓÅ|ÐËÅ)$/';
		if ( preg_match( $re, $w, $match ) ) {
			$stem   = $match[1];
			$suffix = $match[2];
			$w      = $stem . $step1list[ $suffix ];
			$test1  = false;
			if ( preg_match( $exceptS1, $w ) ) {
				$w = $w . 'ÉÓÔ';
			}
			if ( preg_match( $exceptS2, $w ) ) {
				$w = $w . 'É';
			}

			return $this->return_stem( $w, $encoding_changed );
		}


		//Step S6. 6 stems
		$re       = '/^(.+?)(ÉÓÌÏ|ÉÓÌÏÉ|ÉÓÌÏÓ|ÉÓÌÏÕ|ÉÓÌÏÕÓ|ÉÓÌÙÍ)$/';
		$exceptS1 = '/^(ÁÃÍÙÓÔÉÊ|ÁÔÏÌÉÊ|ÃÍÙÓÔÉÊ|ÅÈÍÉÊ|ÅÊËÅÊÔÉÊ|ÓÊÅÐÔÉÊ|ÔÏÐÉÊ)$/';
		$exceptS2 = '/^(ÓÅ|ÌÅÔÁÓÅ|ÌÉÊÑÏÓÅ|ÅÃÊËÅ|ÁÐÏÊËÅ)$/';
		$exceptS3 = '/^(ÄÁÍÅ|ÁÍÔÉÄÁÍÅ)$/';
		$exceptS4 = '/^(ÁËÅÎÁÍÄÑÉÍ|ÂÕÆÁÍÔÉÍ|ÈÅÁÔÑÉÍ)$/';
		if ( preg_match( $re, $w, $match ) ) {
			$stem   = $match[1];
			$suffix = $match[2];
			$w      = $stem;
			$test1  = false;
			if ( preg_match( $exceptS1, $w ) ) {
				$w = str_replace( 'ÉÊ', '', $w );
			}
			if ( preg_match( $exceptS2, $w ) ) {
				$w = $w . 'ÉÓÌ';
			}
			if ( preg_match( $exceptS3, $w ) ) {
				$w = $w . 'É';
			}
			if ( preg_match( $exceptS4, $w ) ) {
				$w = str_replace( 'ÉÍ', '', $w );
			}

			return $this->return_stem( $w, $encoding_changed );
		}


		//Step S7. 4 stems
		$re       = '/^(.+?)(ÁÑÁÊÉ|ÁÑÁÊÉÁ|ÏÕÄÁÊÉ|ÏÕÄÁÊÉÁ)$/';
		$exceptS1 = '/^(Ó|×)$/';
		if ( preg_match( $re, $w, $match ) ) {
			$stem   = $match[1];
			$suffix = $match[2];
			$w      = $stem;
			$test1  = false;
			if ( preg_match( $exceptS1, $w ) ) {
				$w = $w . 'AÑÁÊ';
			}

			return $this->return_stem( $w, $encoding_changed );
		}


		//Step S8. 8 stems
		$re       = '/^(.+?)(ÁÊÉ|ÁÊÉÁ|ÉÔÓÁ|ÉÔÓÁÓ|ÉÔÓÅÓ|ÉÔÓÙÍ|ÁÑÁÊÉ|ÁÑÁÊÉÁ)$/';
		$exceptS1 = '/^(ÁÍÈÑ|ÂÁÌÂ|ÂÑ|ÊÁÉÌ|ÊÏÍ|ÊÏÑ|ËÁÂÑ|ËÏÕË|ÌÅÑ|ÌÏÕÓÔ|ÍÁÃÊÁÓ|ÐË|Ñ|ÑÕ|Ó|ÓÊ|ÓÏÊ|ÓÐÁÍ|ÔÆ|ÖÁÑÌ|×|ÊÁÐÁÊ|ÁËÉÓÖ|ÁÌÂÑ|ÁÍÈÑ|Ê|ÖÕË|ÊÁÔÑÁÐ|ÊËÉÌ|ÌÁË|ÓËÏÂ|Ö|ÓÖ|ÔÓÅ×ÏÓËÏÂ)$/';
		$exceptS2 = '/^(Â|ÂÁË|ÃÉÁÍ|ÃË|Æ|ÇÃÏÕÌÅÍ|ÊÁÑÄ|ÊÏÍ|ÌÁÊÑÕÍ|ÍÕÖ|ÐÁÔÅÑ|Ð|ÓÊ|ÔÏÓ|ÔÑÉÐÏË)$/';
		$exceptS3 = '/(ÊÏÑ)$/'; // for words like ÐËÏÕÓÉÏÊÏÑÉÔÓÁ, ÐÁËÉÏÊÏÑÉÔÓÁ etc
		if ( preg_match( $re, $w, $match ) ) {
			$stem   = $match[1];
			$suffix = $match[2];
			$w      = $stem;
			$test1  = false;
			if ( preg_match( $exceptS1, $w ) ) {
				$w = $w . 'ÁÊ';
			}
			if ( preg_match( $exceptS2, $w ) ) {
				$w = $w . 'ÉÔÓ';
			}
			if ( preg_match( $exceptS3, $w ) ) {
				$w = $w . 'ÉÔÓ';
			}

			return $this->return_stem( $w, $encoding_changed );
		}


		//Step S9. 3 stems
		$re       = '/^(.+?)(ÉÄÉÏ|ÉÄÉÁ|ÉÄÉÙÍ)$/';
		$exceptS1 = '/^(ÁÉÖÍ|ÉÑ|ÏËÏ|ØÁË)$/';
		$exceptS2 = '/(Å|ÐÁÉ×Í)$/';
		if ( preg_match( $re, $w, $match ) ) {
			$stem   = $match[1];
			$suffix = $match[2];
			$w      = $stem;
			$test1  = false;
			if ( preg_match( $exceptS1, $w ) ) {
				$w = $w . 'ÉÄ';
			}
			if ( preg_match( $exceptS2, $w ) ) {
				$w = $w . 'ÉÄ';
			}

			return $this->return_stem( $w, $encoding_changed );
		}


		//Step S10. 4 stems
		$re       = '/^(.+?)(ÉÓÊÏÓ|ÉÓÊÏÕ|ÉÓÊÏ|ÉÓÊÅ)$/';
		$exceptS1 = '/^(Ä|ÉÂ|ÌÇÍ|Ñ|ÖÑÁÃÊ|ËÕÊ|ÏÂÅË)$/';
		if ( preg_match( $re, $w, $match ) ) {
			$stem   = $match[1];
			$suffix = $match[2];
			$w      = $stem;
			$test1  = false;
			if ( preg_match( $exceptS1, $w ) ) {
				$w = $w . 'ÉÓÊ';
			}

			return $this->return_stem( $w, $encoding_changed );
		}
		//Step1
		$re = '/(.*)(ÖÁÃÉÁ|ÖÁÃÉÏÕ|ÖÁÃÉÙÍ|ÓÊÁÃÉÁ|ÓÊÁÃÉÏÕ|ÓÊÁÃÉÙÍ|ÏËÏÃÉÏÕ|ÏËÏÃÉÁ|ÏËÏÃÉÙÍ|ÓÏÃÉÏÕ|ÓÏÃÉÁ|ÓÏÃÉÙÍ|ÔÁÔÏÃÉÁ|ÔÁÔÏÃÉÏÕ|ÔÁÔÏÃÉÙÍ|ÊÑÅÁÓ|ÊÑÅÁÔÏÓ|ÊÑÅÁÔÁ|ÊÑÅÁÔÙÍ|ÐÅÑÁÓ|ÐÅÑÁÔÏÓ|ÐÅÑÁÔÇ|ÐÅÑÁÔÁ|ÐÅÑÁÔÙÍ|ÔÅÑÁÓ|ÔÅÑÁÔÏÓ|ÔÅÑÁÔÁ|ÔÅÑÁÔÙÍ|ÖÙÓ|ÖÙÔÏÓ|ÖÙÔÁ|ÖÙÔÙÍ|ÊÁÈÅÓÔÙÓ|ÊÁÈÅÓÔÙÔÏÓ|ÊÁÈÅÓÔÙÔÁ|ÊÁÈÅÓÔÙÔÙÍ|ÃÅÃÏÍÏÓ|ÃÅÃÏÍÏÔÏÓ|ÃÅÃÏÍÏÔÁ|ÃÅÃÏÍÏÔÙÍ)$/';
		if ( preg_match( $re, $w, $match ) ) {
			//debug($w,1);
			$stem   = $match[1];
			$suffix = $match[2];
			$w      = $stem . $step1list[ $suffix ];
			$test1  = false;
		}

		// Step 2a. 2 stems
		$re = '/^(.+?)(ÁÄÅÓ|ÁÄÙÍ)$/';
		if ( preg_match( $re, $w, $match ) ) {
			$stem = $match[1];
			$w    = $stem;
			$re   = '/(ÏÊ|ÌÁÌ|ÌÁÍ|ÌÐÁÌÐ|ÐÁÔÅÑ|ÃÉÁÃÉ|ÍÔÁÍÔ|ÊÕÑ|ÈÅÉ|ÐÅÈÅÑ)$/';
			if ( ! preg_match( $re, $w ) ) {
				$w = $w . 'ÁÄ';
			}
		}

		//Step 2b. 2 stems
		$re = '/^(.+?)(ÅÄÅÓ|ÅÄÙÍ)$/';
		if ( preg_match( $re, $w ) ) {
			preg_match( $re, $w, $match );
			$stem   = $match[1];
			$w      = $stem;
			$exept2 = '/(ÏÐ|ÉÐ|ÅÌÐ|ÕÐ|ÃÇÐ|ÄÁÐ|ÊÑÁÓÐ|ÌÉË)$/';
			if ( preg_match( $exept2, $w ) ) {
				$w = $w . 'ÅÄ';
			}
		}

		//Step 2c
		$re = '/^(.+?)(ÏÕÄÅÓ|ÏÕÄÙÍ)$/';
		if ( preg_match( $re, $w ) ) {
			preg_match( $re, $w, $match );
			$stem   = $match[1];
			$w      = $stem;
			$exept3 = '/(ÁÑÊ|ÊÁËÉÁÊ|ÐÅÔÁË|ËÉ×|ÐËÅÎ|ÓÊ|Ó|ÖË|ÖÑ|ÂÅË|ËÏÕË|×Í|ÓÐ|ÔÑÁÃ|ÖÅ)$/';
			if ( preg_match( $exept3, $w ) ) {
				$w = $w . 'ÏÕÄ';
			}
		}

		//Step 2d
		$re = '/^(.+?)(ÅÙÓ|ÅÙÍ)$/';
		if ( preg_match( $re, $w ) ) {
			preg_match( $re, $w, $match );
			$stem   = $match[1];
			$w      = $stem;
			$test1  = false;
			$exept4 = '/^(È|Ä|ÅË|ÃÁË|Í|Ð|ÉÄ|ÐÁÑ)$/';
			if ( preg_match( $exept4, $w ) ) {
				$w = $w . 'Å';
			}
		}

		$fp = [ ];
		//Step 3
		$re = '/^(.+?)(ÉÁ|ÉÏÕ|ÉÙÍ)$/';
		if ( preg_match( $re, $w, $fp ) ) {
			$stem  = $fp[1];
			$w     = $stem;
			$re    = '/' . $v . '$/';
			$test1 = false;
			if ( preg_match( $re, $w ) ) {
				$w = $stem . 'É';
			}
		}

		//Step 4
		$re = '/^(.+?)(ÉÊÁ|ÉÊÏ|ÉÊÏÕ|ÉÊÙÍ)$/';
		if ( preg_match( $re, $w ) ) {
			preg_match( $re, $w, $match );
			$stem   = $match[1];
			$w      = $stem;
			$test1  = false;
			$re     = '/' . $v . '$/';
			$exept5 = '/^(ÁË|ÁÄ|ÅÍÄ|ÁÌÁÍ|ÁÌÌÏ×ÁË|ÇÈ|ÁÍÇÈ|ÁÍÔÉÄ|ÖÕÓ|ÂÑÙÌ|ÃÅÑ|ÅÎÙÄ|ÊÁËÐ|ÊÁËËÉÍ|ÊÁÔÁÄ|ÌÏÕË|ÌÐÁÍ|ÌÐÁÃÉÁÔ|ÌÐÏË|ÌÐÏÓ|ÍÉÔ|ÎÉÊ|ÓÕÍÏÌÇË|ÐÅÔÓ|ÐÉÔÓ|ÐÉÊÁÍÔ|ÐËÉÁÔÓ|ÐÏÓÔÅËÍ|ÐÑÙÔÏÄ|ÓÅÑÔ|ÓÕÍÁÄ|ÔÓÁÌ|ÕÐÏÄ|ÖÉËÏÍ|ÖÕËÏÄ|×ÁÓ)$/';
			if ( preg_match( $re, $w ) || preg_match( $exept5, $w ) ) {
				$w = $w . 'ÉÊ';
			}
		}

		//step 5a
		$re  = '/^(.+?)(ÁÌÅ)$/';
		$re2 = '/^(.+?)(ÁÃÁÌÅ|ÇÓÁÌÅ|ÏÕÓÁÌÅ|ÇÊÁÌÅ|ÇÈÇÊÁÌÅ)$/';
		if ( $w == 'ÁÃÁÌÅ' ) {
			$w = 'ÁÃÁÌ';
		}
		if ( preg_match( $re2, $w ) ) {
			preg_match( $re2, $w, $match );
			$stem  = $match[1];
			$w     = $stem;
			$test1 = false;
		}
		if ( preg_match( $re, $w ) ) {
			preg_match( $re, $w, $match );
			$stem   = $match[1];
			$w      = $stem;
			$test1  = false;
			$exept6 = '/^(ÁÍÁÐ|ÁÐÏÈ|ÁÐÏÊ|ÁÐÏÓÔ|ÂÏÕÂ|ÎÅÈ|ÏÕË|ÐÅÈ|ÐÉÊÑ|ÐÏÔ|ÓÉ×|×)$/';
			if ( preg_match( $exept6, $w ) ) {
				$w = $w . 'ÁÌ';
			}
		}

		//Step 5b
		$re2 = '/^(.+?)(ÁÍÅ)$/';
		$re3 = '/^(.+?)(ÁÃÁÍÅ|ÇÓÁÍÅ|ÏÕÓÁÍÅ|ÉÏÍÔÁÍÅ|ÉÏÔÁÍÅ|ÉÏÕÍÔÁÍÅ|ÏÍÔÁÍÅ|ÏÔÁÍÅ|ÏÕÍÔÁÍÅ|ÇÊÁÍÅ|ÇÈÇÊÁÍÅ)$/';
		if ( preg_match( $re3, $w ) ) {
			preg_match( $re3, $w, $match );
			$stem  = $match[1];
			$w     = $stem;
			$test1 = false;
			$re3   = '/^(ÔÑ|ÔÓ)$/';
			if ( preg_match( $re3, $w ) ) {
				$w = $w . 'ÁÃÁÍ';
			}
		}
		if ( preg_match( $re2, $w ) ) {
			preg_match( $re2, $w, $match );
			$stem   = $match[1];
			$w      = $stem;
			$test1  = false;
			$re2    = '/' . $v2 . '$/';
			$exept7 = '/^(ÂÅÔÅÑ|ÂÏÕËÊ|ÂÑÁ×Ì|Ã|ÄÑÁÄÏÕÌ|È|ÊÁËÐÏÕÆ|ÊÁÓÔÅË|ÊÏÑÌÏÑ|ËÁÏÐË|ÌÙÁÌÅÈ|Ì|ÌÏÕÓÏÕËÌ|Í|ÏÕË|Ð|ÐÅËÅÊ|ÐË|ÐÏËÉÓ|ÐÏÑÔÏË|ÓÁÑÁÊÁÔÓ|ÓÏÕËÔ|ÔÓÁÑËÁÔ|ÏÑÖ|ÔÓÉÃÃ|ÔÓÏÐ|ÖÙÔÏÓÔÅÖ|×|ØÕ×ÏÐË|ÁÃ|ÏÑÖ|ÃÁË|ÃÅÑ|ÄÅÊ|ÄÉÐË|ÁÌÅÑÉÊÁÍ|ÏÕÑ|ÐÉÈ|ÐÏÕÑÉÔ|Ó|ÆÙÍÔ|ÉÊ|ÊÁÓÔ|ÊÏÐ|ËÉ×|ËÏÕÈÇÑ|ÌÁÉÍÔ|ÌÅË|ÓÉÃ|ÓÐ|ÓÔÅÃ|ÔÑÁÃ|ÔÓÁÃ|Ö|ÅÑ|ÁÄÁÐ|ÁÈÉÃÃ|ÁÌÇ×|ÁÍÉÊ|ÁÍÏÑÃ|ÁÐÇÃ|ÁÐÉÈ|ÁÔÓÉÃÃ|ÂÁÓ|ÂÁÓÊ|ÂÁÈÕÃÁË|ÂÉÏÌÇ×|ÂÑÁ×ÕÊ|ÄÉÁÔ|ÄÉÁÖ|ÅÍÏÑÃ|ÈÕÓ|ÊÁÐÍÏÂÉÏÌÇ×|ÊÁÔÁÃÁË|ÊËÉÂ|ÊÏÉËÁÑÖ|ËÉÂ|ÌÅÃËÏÂÉÏÌÇ×|ÌÉÊÑÏÂÉÏÌÇ×|ÍÔÁÂ|ÎÇÑÏÊËÉÂ|ÏËÉÃÏÄÁÌ|ÏËÏÃÁË|ÐÅÍÔÁÑÖ|ÐÅÑÇÖ|ÐÅÑÉÔÑ|ÐËÁÔ|ÐÏËÕÄÁÐ|ÐÏËÕÌÇ×|ÓÔÅÖ|ÔÁÂ|ÔÅÔ|ÕÐÅÑÇÖ|ÕÐÏÊÏÐ|×ÁÌÇËÏÄÁÐ|ØÇËÏÔÁÂ)$/';
			if ( preg_match( $re2, $w ) || preg_match( $exept7, $w ) ) {
				$w = $w . 'ÁÍ';
			}
		}

		//Step 5c
		$re3 = '/^(.+?)(ÅÔÅ)$/';
		$re4 = '/^(.+?)(ÇÓÅÔÅ)$/';
		if ( preg_match( $re4, $w ) ) {
			preg_match( $re4, $w, $match );
			$stem  = $match[1];
			$w     = $stem;
			$test1 = false;
		}
		if ( preg_match( $re3, $w ) ) {
			preg_match( $re3, $w, $match );
			$stem   = $match[1];
			$w      = $stem;
			$test1  = false;
			$re3    = '/' . $v2 . '$/';
			$exept8 = '/(ÏÄ|ÁÉÑ|ÖÏÑ|ÔÁÈ|ÄÉÁÈ|Ó×|ÅÍÄ|ÅÕÑ|ÔÉÈ|ÕÐÅÑÈ|ÑÁÈ|ÅÍÈ|ÑÏÈ|ÓÈ|ÐÕÑ|ÁÉÍ|ÓÕÍÄ|ÓÕÍ|ÓÕÍÈ|×ÙÑ|ÐÏÍ|ÂÑ|ÊÁÈ|ÅÕÈ|ÅÊÈ|ÍÅÔ|ÑÏÍ|ÁÑÊ|ÂÁÑ|ÂÏË|ÙÖÅË)$/';
			$exept9 = '/^(ÁÂÁÑ|ÂÅÍ|ÅÍÁÑ|ÁÂÑ|ÁÄ|ÁÈ|ÁÍ|ÁÐË|ÂÁÑÏÍ|ÍÔÑ|ÓÊ|ÊÏÐ|ÌÐÏÑ|ÍÉÖ|ÐÁÃ|ÐÁÑÁÊÁË|ÓÅÑÐ|ÓÊÅË|ÓÕÑÖ|ÔÏÊ|Õ|Ä|ÅÌ|ÈÁÑÑ|È)$/';
			if ( preg_match( $re3, $w ) || preg_match( $exept8, $w ) || preg_match( $exept9, $w ) ) {
				$w = $w . 'ÅÔ';
			}
		}

		//Step 5d
		$re = '/^(.+?)(ÏÍÔÁÓ|ÙÍÔÁÓ)$/';
		if ( preg_match( $re, $w ) ) {
			preg_match( $re, $w, $match );
			$stem    = $match[1];
			$w       = $stem;
			$test1   = false;
			$exept10 = '/^(ÁÑ×)$/';
			$exept11 = '/(ÊÑÅ)$/';
			if ( preg_match( $exept10, $w ) ) {
				$w = $w . 'ÏÍÔ';
			}
			if ( preg_match( $exept11, $w ) ) {
				$w = $w . 'ÙÍÔ';
			}
		}

		//Step 5e
		$re = '/^(.+?)(ÏÌÁÓÔÅ|ÉÏÌÁÓÔÅ)$/';
		if ( preg_match( $re, $w ) ) {
			preg_match( $re, $w, $match );
			$stem    = $match[1];
			$w       = $stem;
			$test1   = false;
			$exept11 = '/^(ÏÍ)$/';
			if ( preg_match( $exept11, $w ) ) {
				$w = $w . 'ÏÌÁÓÔ';
			}
		}

		//Step 5f
		$re  = '/^(.+?)(ÅÓÔÅ)$/';
		$re2 = '/^(.+?)(ÉÅÓÔÅ)$/';
		if ( preg_match( $re2, $w ) ) {
			preg_match( $re2, $w, $match );
			$stem  = $match[1];
			$w     = $stem;
			$test1 = false;
			$re2   = '/^(Ð|ÁÐ|ÓÕÌÐ|ÁÓÕÌÐ|ÁÊÁÔÁÐ|ÁÌÅÔÁÌÖ)$/';
			if ( preg_match( $re2, $w ) ) {
				$w = $w . 'ÉÅÓÔ';
			}
		}

		if ( preg_match( $re, $w ) ) {
			preg_match( $re, $w, $match );
			$stem    = $match[1];
			$w       = $stem;
			$test1   = false;
			$exept12 = '/^(ÁË|ÁÑ|ÅÊÔÅË|Æ|Ì|Î|ÐÁÑÁÊÁË|ÁÑ|ÐÑÏ|ÍÉÓ)$/';
			if ( preg_match( $exept12, $w ) ) {
				$w = $w . 'ÅÓÔ';
			}
		}

		//Step 5g
		$re  = '/^(.+?)(ÇÊÁ|ÇÊÅÓ|ÇÊÅ)$/';
		$re2 = '/^(.+?)(ÇÈÇÊÁ|ÇÈÇÊÅÓ|ÇÈÇÊÅ)$/';
		if ( preg_match( $re2, $w ) ) {
			preg_match( $re2, $w, $match );
			$stem  = $match[1];
			$w     = $stem;
			$test1 = false;
		}

		if ( preg_match( $re, $w ) ) {
			preg_match( $re, $w, $match );
			$stem    = $match[1];
			$w       = $stem;
			$test1   = false;
			$exept13 = '/(ÓÊÙË|ÓÊÏÕË|ÍÁÑÈ|ÓÖ|ÏÈ|ÐÉÈ)$/';
			$exept14 = '/^(ÄÉÁÈ|È|ÐÁÑÁÊÁÔÁÈ|ÐÑÏÓÈ|ÓÕÍÈ|)$/';
			if ( preg_match( $exept13, $w ) || preg_match( $exept14, $w ) ) {
				$w = $w . 'ÇÊ';
			}
		}

		//Step 5h
		$re = '/^(.+?)(ÏÕÓÁ|ÏÕÓÅÓ|ÏÕÓÅ)$/';
		if ( preg_match( $re, $w ) ) {
			preg_match( $re, $w, $match );
			$stem    = $match[1];
			$w       = $stem;
			$test1   = false;
			$exept15 = '/^(ÖÁÑÌÁÊ|×ÁÄ|ÁÃÊ|ÁÍÁÑÑ|ÂÑÏÌ|ÅÊËÉÐ|ËÁÌÐÉÄ|ËÅ×|Ì|ÐÁÔ|Ñ|Ë|ÌÅÄ|ÌÅÓÁÆ|ÕÐÏÔÅÉÍ|ÁÌ|ÁÉÈ|ÁÍÇÊ|ÄÅÓÐÏÆ|ÅÍÄÉÁÖÅÑ|ÄÅ|ÄÅÕÔÅÑÅÕ|ÊÁÈÁÑÅÕ|ÐËÅ|ÔÓÁ)$/';
			$exept16 = '/(ÐÏÄÁÑ|ÂËÅÐ|ÐÁÍÔÁ×|ÖÑÕÄ|ÌÁÍÔÉË|ÌÁËË|ÊÕÌÁÔ|ËÁ×|ËÇÃ|ÖÁÃ|ÏÌ|ÐÑÙÔ)$/';
			if ( preg_match( $exept15, $w ) || preg_match( $exept16, $w ) ) {
				$w = $w . 'ÏÕÓ';
			}
		}

		//Step 5i
		$re = '/^(.+?)(ÁÃÁ|ÁÃÅÓ|ÁÃÅ)$/';
		if ( preg_match( $re, $w ) ) {
			preg_match( $re, $w, $match );
			$stem    = $match[1];
			$w       = $stem;
			$test1   = false;
			$exept17 = '/^(ØÏÖ|ÍÁÕËÏ×)$/';
			$exept20 = '/(ÊÏËË)$/';
			$exept18 = '/^(ÁÂÁÓÔ|ÐÏËÕÖ|ÁÄÇÖ|ÐÁÌÖ|Ñ|ÁÓÐ|ÁÖ|ÁÌÁË|ÁÌÁËËÉ|ÁÍÕÓÔ|ÁÐÅÑ|ÁÓÐÁÑ|Á×ÁÑ|ÄÅÑÂÅÍ|ÄÑÏÓÏÐ|ÎÅÖ|ÍÅÏÐ|ÍÏÌÏÔ|ÏËÏÐ|ÏÌÏÔ|ÐÑÏÓÔ|ÐÑÏÓÙÐÏÐ|ÓÕÌÐ|ÓÕÍÔ|Ô|ÕÐÏÔ|×ÁÑ|ÁÅÉÐ|ÁÉÌÏÓÔ|ÁÍÕÐ|ÁÐÏÔ|ÁÑÔÉÐ|ÄÉÁÔ|ÅÍ|ÅÐÉÔ|ÊÑÏÊÁËÏÐ|ÓÉÄÇÑÏÐ|Ë|ÍÁÕ|ÏÕËÁÌ|ÏÕÑ|Ð|ÔÑ|Ì)$/';
			$exept19 = '/(ÏÖ|ÐÅË|×ÏÑÔ|ËË|ÓÖ|ÑÐ|ÖÑ|ÐÑ|ËÏ×|ÓÌÇÍ)$/';
			if ( ( preg_match( $exept18, $w ) || preg_match( $exept19, $w ) ) && ! ( preg_match( $exept17, $w ) || preg_match( $exept20, $w ) ) ) {
				$w = $w . 'ÁÃ';
			}
		}

		//Step 5j
		$re = '/^(.+?)(ÇÓÅ|ÇÓÏÕ|ÇÓÁ)$/';
		if ( preg_match( $re, $w ) ) {
			preg_match( $re, $w, $match );
			$stem    = $match[1];
			$w       = $stem;
			$test1   = false;
			$exept21 = '/^(Í|×ÅÑÓÏÍ|ÄÙÄÅÊÁÍ|ÅÑÇÌÏÍ|ÌÅÃÁËÏÍ|ÅÐÔÁÍ)$/';
			if ( preg_match( $exept21, $w ) ) {
				$w = $w . 'ÇÓ';
			}
		}

		//Step 5k
		$re = '/^(.+?)(ÇÓÔÅ)$/';
		if ( preg_match( $re, $w ) ) {
			preg_match( $re, $w, $match );
			$stem    = $match[1];
			$w       = $stem;
			$test1   = false;
			$exept22 = '/^(ÁÓÂ|ÓÂ|Á×Ñ|×Ñ|ÁÐË|ÁÅÉÌÍ|ÄÕÓ×Ñ|ÅÕ×Ñ|ÊÏÉÍÏ×Ñ|ÐÁËÉÌØ)$/';
			if ( preg_match( $exept22, $w ) ) {
				$w = $w . 'ÇÓÔ';
			}
		}

		//Step 5l
		$re = '/^(.+?)(ÏÕÍÅ|ÇÓÏÕÍÅ|ÇÈÏÕÍÅ)$/';
		if ( preg_match( $re, $w ) ) {
			preg_match( $re, $w, $match );
			$stem    = $match[1];
			$w       = $stem;
			$test1   = false;
			$exept23 = '/^(Í|Ñ|ÓÐÉ|ÓÔÑÁÂÏÌÏÕÔÓ|ÊÁÊÏÌÏÕÔÓ|ÅÎÙÍ)$/';
			if ( preg_match( $exept23, $w ) ) {
				$w = $w . 'ÏÕÍ';
			}
		}

		//Step 5l
		$re = '/^(.+?)(ÏÕÌÅ|ÇÓÏÕÌÅ|ÇÈÏÕÌÅ)$/';
		if ( preg_match( $re, $w ) ) {
			preg_match( $re, $w, $match );
			$stem    = $match[1];
			$w       = $stem;
			$test1   = false;
			$exept24 = '/^(ÐÁÑÁÓÏÕÓ|Ö|×|ÙÑÉÏÐË|ÁÆ|ÁËËÏÓÏÕÓ|ÁÓÏÕÓ)$/';
			if ( preg_match( $exept24, $w ) ) {
				$w = $w . 'ÏÕÌ';
			}
		}

		// Step 6
		$re  = '/^(.+?)(ÌÁÔÁ|ÌÁÔÙÍ|ÌÁÔÏÓ)$/';
		$re2 = '/^(.+?)(Á|ÁÃÁÔÅ|ÁÃÁÍ|ÁÅÉ|ÁÌÁÉ|ÁÍ|ÁÓ|ÁÓÁÉ|ÁÔÁÉ|ÁÙ|Å|ÅÉ|ÅÉÓ|ÅÉÔÅ|ÅÓÁÉ|ÅÓ|ÅÔÁÉ|É|ÉÅÌÁÉ|ÉÅÌÁÓÔÅ|ÉÅÔÁÉ|ÉÅÓÁÉ|ÉÅÓÁÓÔÅ|ÉÏÌÁÓÔÁÍ|ÉÏÌÏÕÍ|ÉÏÌÏÕÍÁ|ÉÏÍÔÁÍ|ÉÏÍÔÏÕÓÁÍ|ÉÏÓÁÓÔÁÍ|ÉÏÓÁÓÔÅ|ÉÏÓÏÕÍ|ÉÏÓÏÕÍÁ|ÉÏÔÁÍ|ÉÏÕÌÁ|ÉÏÕÌÁÓÔÅ|ÉÏÕÍÔÁÉ|ÉÏÕÍÔÁÍ|Ç|ÇÄÅÓ|ÇÄÙÍ|ÇÈÅÉ|ÇÈÅÉÓ|ÇÈÅÉÔÅ|ÇÈÇÊÁÔÅ|ÇÈÇÊÁÍ|ÇÈÏÕÍ|ÇÈÙ|ÇÊÁÔÅ|ÇÊÁÍ|ÇÓ|ÇÓÁÍ|ÇÓÁÔÅ|ÇÓÅÉ|ÇÓÅÓ|ÇÓÏÕÍ|ÇÓÙ|Ï|ÏÉ|ÏÌÁÉ|ÏÌÁÓÔÁÍ|ÏÌÏÕÍ|ÏÌÏÕÍÁ|ÏÍÔÁÉ|ÏÍÔÁÍ|ÏÍÔÏÕÓÁÍ|ÏÓ|ÏÓÁÓÔÁÍ|ÏÓÁÓÔÅ|ÏÓÏÕÍ|ÏÓÏÕÍÁ|ÏÔÁÍ|ÏÕ|ÏÕÌÁÉ|ÏÕÌÁÓÔÅ|ÏÕÍ|ÏÕÍÔÁÉ|ÏÕÍÔÁÍ|ÏÕÓ|ÏÕÓÁÍ|ÏÕÓÁÔÅ|Õ|ÕÓ|Ù|ÙÍ)$/';
		if ( preg_match( $re, $w, $match ) ) {
			//debug($w,6);
			$stem = $match[1];
			$w    = $stem . 'ÌÁ';

		}

		if ( preg_match( $re2, $w ) && $test1 ) {
			//debug($w,'6-re2');
			preg_match( $re2, $w, $match );
			$stem = $match[1];
			$w    = $stem;

		}

		// Step 7 (ÐÁÑÁÈÅÔÉÊÁ)
		$re = '/^(.+?)(ÅÓÔÅÑ|ÅÓÔÁÔ|ÏÔÅÑ|ÏÔÁÔ|ÕÔÅÑ|ÕÔÁÔ|ÙÔÅÑ|ÙÔÁÔ)$/';
		if ( preg_match( $re, $w ) ) {
			preg_match( $re, $w, $match );
			$stem = $match[1];
			$w    = $stem;
		}

		return $this->return_stem( $w, $encoding_changed );
	}

	/**
	 *  Reverts the encoding if needed and returns the stemmed
	 *
	 * @param string $w :   The stemmed word
	 * @param bool $encoding_changed :   The encoding has changed
	 *
	 * @return string                    :   The stemmed word
	 */
	function return_stem( $w, $encoding_changed ) {
		// SearchWP always wants all lowercase
		$w = mb_strtolower( $w, 'ISO-8859-7' );
		if ( true == $encoding_changed ) {
			$w = mb_convert_encoding( $w, 'UTF-8', 'ISO-8859-7' );
		}

		return $w;
	}
}
