<?php
/**
 * MonthlyBreakdown.php
 *
 * @package   edd-commissions
 * @copyright Copyright (c) 2021, Sandhills Development, LLC
 * @license   GPL2+
 * @since     3.5
 */

namespace EDD_Commissions\Reports;

class MonthlyBreakdown {

	/**
	 * @var int ID of the user to generate the report for.
	 */
	public $user_id;

	/**
	 * @var int Year to generate the report for.
	 */
	public $year;

	/**
	 * @var \EDDC_DB
	 */
	protected $db;

	/**
	 * @var \wpdb
	 */
	protected $wpdb;

	/**
	 * @var array
	 */
	public $report_data;

	/**
	 * MonthlyBreakdown constructor.
	 *
	 * @param int      $user_id User to generate the report for.
	 * @param int|null $year    Year to generate the report for.
	 */
	public function __construct( $user_id, $year = null ) {
		$this->user_id = $user_id;
		$this->db      = edd_commissions()->commissions_db;
		$this->year    = $year ? : date( 'Y' );

		global $wpdb;
		$this->wpdb = $wpdb;
	}

	/**
	 * Retrieves the distinct years available for reports.
	 *
	 * @since 3.5
	 * @return int[]
	 */
	public function get_distinct_years() {
		$years = $this->wpdb->get_col( $this->wpdb->prepare(
			"SELECT DISTINCT YEAR(date_created) FROM {$this->db->table_name} WHERE user_id = %d ORDER BY date_created DESC",
			$this->user_id
		) );

		return array_map( 'intval', $years );
	}

	/**
	 * Generates an array of data, grouped by month. Example:
	 *
	 * array(
	 *        // 1 is for "January".
	 *        1 => array(
	 *            'total_count' => 0,
	 *            'paid_count' => 0,
	 *            'revoked_count' => 0,
	 *            'total_earnings' => 0.00,
	 *            'paid_earnings' => 0.00,
	 *            'revoked_earnings' => 0.00,
	 *        ),
	 *        2 => array( ... )
	 * )
	 *
	 * @since 3.5
	 * @return self
	 */
	public function generate() {
		$results = array_fill( 1, 12, array(
			'total_count'      => 0,
			'paid_count'       => 0,
			'revoked_count'    => 0,
			'total_earnings'   => 0.00,
			'paid_earnings'    => 0.00,
			'revoked_earnings' => 0.00, // Handles both refunded & revoked
		) );

		$this->report_data = $results;

		$this->query_sales();
		$this->query_earnings();

		return $this;
	}

	/**
	 * Fills in record data in the report data array.
	 *
	 * @param array  $records
	 * @param string $destination_key
	 *
	 * @since 3.5
	 */
	protected function fill_record_data( $records, $destination_key ) {
		if ( empty( $records ) ) {
			return;
		}

		foreach ( $records as $record ) {
			if ( ! isset( $record->month ) || ! isset( $record->value ) ) {
				continue;
			}

			$this->report_data[ (int) $record->month ][ $destination_key ] = $record->value;
		}
	}

	/**
	 * Retrieves a list of destination keys with their associated commission status(es).
	 *
	 * @since 3.5
	 *
	 * @param string $type Either `count` or `earnings`.
	 *
	 * @return array
	 */
	private function get_destination_key_statuses( $type = 'count' ) {
		if ( ! in_array( $type, array( 'count', 'earnings' ) ) ) {
			throw new \InvalidArgumentException( sprintf( 'Invalid type %s', $type ) );
		}

		/*
		 * Key is the "destination key" (key in the report data);
		 * value is the commission status.
		 */
		return array(
			'total_' . $type   => '', // All statuses
			'paid_' . $type    => 'paid',
			'revoked_' . $type => array( 'refunded', 'revoked' ),
		);
	}

	/**
	 * Queries for sales.
	 *
	 * @since 3.5
	 */
	private function query_sales() {
		$statuses = $this->get_destination_key_statuses();

		foreach ( $statuses as $destination_key => $status ) {
			$this->fill_record_data( $this->query_sales_by_status( $status ), $destination_key );
		}
	}

	/**
	 * Queries for sales by status or an array of statuses.
	 *
	 * @since 3.5
	 *
	 * @param string|string[] $status
	 *
	 * @return object[]
	 */
	private function query_sales_by_status( $status ) {
		$status_sql = '';
		$date_col   = 'date_created';

		if ( 'paid' === $status ) {
			// Note: I don't think we need a status clause here, because the `date_paid` column handles it.
			$date_col = 'date_paid';
		} else {
			$status_sql = $this->get_status_sql( $status );
		}

		return $this->wpdb->get_results( $this->wpdb->prepare(
			"SELECT MONTH({$date_col}) AS month, COUNT(id) AS value FROM {$this->db->table_name}
			WHERE user_id = %d
			AND YEAR({$date_col}) = %d
			{$status_sql}
			GROUP BY MONTH({$date_col})",
			$this->user_id,
			$this->year
		) );
	}

	/**
	 * Queries earnings by status
	 *
	 * Right now we do a separate query for each status. We could probably replace that with
	 * one query that groups by status, but I suspect multiple queries may perform faster than
	 * adding a second group by.
	 *
	 * @since 3.5
	 */
	private function query_earnings() {
		$statuses = $this->get_destination_key_statuses( 'earnings' );

		foreach ( $statuses as $destination_key => $status ) {
			$this->fill_record_data( $this->query_earnings_by_status( $status ), $destination_key );
		}
	}

	/**
	 * Queries for earnings by status, and groups them by month.
	 *
	 * @param string|array $status Single status or array of statuses.
	 *
	 * @return object[]
	 */
	private function query_earnings_by_status( $status ) {
		$status_sql = '';
		$date_col   = 'date_created';

		if ( 'paid' === $status ) {
			// Note: I don't think we need a status clause here, because the `date_paid` column handles it.
			$date_col = 'date_paid';
		} else {
			$status_sql = $this->get_status_sql( $status );
		}

		return $this->wpdb->get_results( $this->wpdb->prepare(
			"SELECT MONTH({$date_col}) AS month, SUM(amount) as value FROM {$this->db->table_name}
			WHERE user_id = %d
			{$status_sql}
			AND YEAR({$date_col}) = %d
			GROUP BY MONTH({$date_col})",
			$this->user_id,
			$this->year
		) );
	}

	/**
	 * Builds the SQL WHERE clause for status or an array of statuses.
	 *
	 * @param string|string[] $status
	 *
	 * @return string
	 */
	private function get_status_sql( $status ) {
		if ( empty( $status ) ) {
			return '';
		}

		if ( is_array( $status ) ) {
			$placeholder_string = implode( ', ', array_fill( 0, count( $status ), '%s' ) );

			return $this->wpdb->prepare( "AND STATUS IN({$placeholder_string})", $status );
		} else {
			return $this->wpdb->prepare( "AND status = %s", $status );
		}
	}

	/**
	 * Retrieves the total across all months for a specific key.
	 *
	 * @since 3.5
	 *
	 * @param string $key Key to check in: `paid_earnings``, etc.
	 *
	 * @return float|int
	 * @throws \Exception
	 */
	public function get_total( $key ) {
		if ( ! isset( $this->report_data ) ) {
			throw new \Exception( 'Total cannot be retrieved until report has been generated.', 400 );
		}

		$values = wp_list_pluck( $this->report_data, $key );

		return array_sum( $values );
	}

}
