<?php

namespace DHGI\App\Providers;

use DHGI\WpMVC\Contracts\Provider;
use HelpGent\App\DTO\AttachmentDTO;
use HelpGent\App\DTO\ContactInfoDTO;
use HelpGent\App\Models\Form;
use HelpGent\App\Models\Response;
use HelpGent\App\Models\ResponseAttachment;
use HelpGent\App\Repositories\FormRepository;
use HelpGent\App\Utils\User;
use HelpGent\WpMVC\Database\Query\Builder;
use HelpGent\WpMVC\Database\Query\JoinClause;
use HelpGent\WpMVC\View\View;
use WP_REST_Request;

class MenuServiceProvider implements Provider
{
    public function boot() {
        add_action( 'directorist_after_dashboard_navigation', [ $this, 'nav_items' ] );
        add_action( 'directorist_after_dashboard_contents', [ $this, 'nav_contents' ] );
        add_filter( 'helpgent_form_query', [$this, 'form_query'], 10, 2 );
        add_filter( 'helpgent_chat_bubble_form_query', [$this, 'chat_bubble_form_query'] );
        add_filter( 'helpgent_lead_forms_query', [$this, 'lead_forms_query'] );
        add_filter( 'helpgent_single_form_query', [$this, 'single_form_query'] );
        add_filter( 'helpgent_is_user_allowed_response_message', [$this, 'is_user_allowed_response_message'], 10, 2 );
        add_filter( 'helpgent_allowed_form_preview', [$this, 'allowed_form_preview'], 10, 2 );
        add_filter( 'helpgent_can_user_access_attachment', [$this, 'can_user_access_attachment'], 10, 3 );
        add_filter( 'helpgent_admin_form_tab_total_query', [$this, 'admin_form_tab_total_query'] );
        add_filter( 'helpgent_get_forms', [$this, 'get_forms'] );
        add_filter( 'helpgent_contact_info_dto', [$this, 'contact_info_dto'] );
    }

    public function get_forms( array $form_data ) {
        $directorist_forms = Form::query( 'form' )->left_join(
            "helpgent_form_meta as dir_form_meta", function( JoinClause $join ) {
                $join->on_column( "dir_form_meta.form_id", "=", "form.id" )->on( "dir_form_meta.meta_key", "directorist" );
            }
        )->where_not_is_null( "dir_form_meta.meta_key" )->distinct()->count( 'form.id' );

        $form_data['directorist_total_forms'] = $directorist_forms;

        return $form_data;
    }

    public function admin_form_tab_total_query( Builder $query ) {
        return $query->left_join(
            "helpgent_form_meta as dir_form_meta", function( JoinClause $join ) {
                $join->on_column( "dir_form_meta.form_id", "=", "form.id" )->on( "dir_form_meta.meta_key", "directorist" );
            }
        )->where_is_null( "dir_form_meta.meta_key" );
    }

    public function contact_info_dto( ContactInfoDTO $dto ) {
        /**
         * @var FormRepository $form_repository
         */
        $form_repository = helpgent_singleton( FormRepository::class );
        $is_directorist  = $form_repository->get_meta_value( $dto->form_id, 'directorist' );

        if ( $is_directorist ) {
            $form_data                   = $dto->form_data;
            $form_data['is_directorist'] = $is_directorist;
            $dto->set_form_data( $form_data );
        }

        return $dto;
    }

    public function can_user_access_attachment( int $can_access, AttachmentDTO $attachment, ?User $user ) {
        if ( $can_access || ! is_user_logged_in() ) {
            return $can_access;
        }

        $response_table      = Response::get_table_name();
        $response_attachment = ResponseAttachment::get_table_name();

        $form = Form::query( 'form' )->select( 1 )->where( 'form.created_by', $user->get_id() )
        ->join( $response_table, "form.id", "{$response_table}.form_id" )
        ->join( $response_attachment, "{$response_table}.id", '=', "{$response_attachment}.response_id" )
        ->where( "{$response_attachment}.attachment_id", $attachment->get_id() )
        ->first();

        if ( empty( $form ) ) {
            return $can_access;
        }

        return true;
    }

    public function allowed_form_preview( bool $is_allowed, int $form_id ) {
        if ( $is_allowed || ! is_user_logged_in() ) {
            return $is_allowed;
        }

        $form = Form::query()->select( 1 )->where( 'id', $form_id )->where( 'created_by', wp_get_current_user()->ID )->first();

        return ! empty( $form );
    }

    public function is_user_allowed_response_message( bool $is_user_allowed, int $response_id ) {
        if ( $is_user_allowed || ! is_user_logged_in() ) {
            return $is_user_allowed;
        }

        $form_table_name = Form::get_table_name();
        $response        = Response::query( 'response' )->select( 1 )->where( 'response.id', $response_id )
        ->left_join( $form_table_name, "{$form_table_name}.id", 'response.form_id' )
        ->where( "{$form_table_name}.created_by", wp_get_current_user()->ID )->first();

        if ( ! $response ) {
            return $is_user_allowed;
        }

        return true;
    }

    public function single_form_query( Builder $query ) {
        add_filter( 'helpgent_single_form_query_join_meta', '__return_true' );

        $columns   = $query->columns;
        $columns[] = "COALESCE(MAX(CASE WHEN form_meta.meta_key = 'directorist' THEN 1 ELSE 0 END), 0) as is_directorist";
        return $query->select( $columns );
    }

    public function lead_forms_query( Builder $query ) {
        /**
         * @var WP_REST_Request $wp_rest_request
         */
        $wp_rest_request = helpgent_pro_singleton( WP_REST_Request::class );

        $route = ( fn() => $this->route )->call( $wp_rest_request );

        if ( "/helpgent-pro/admin/contact/forms" !== $route ) {
            return $query;
        }

        add_filter( "helpgent_lead_forms_query_join_meta", "__return_true" );

        $query->left_join( 'users', 'form.created_by', '=', 'users.ID' )
        ->select(
            "form.id",
            "CASE
                WHEN COALESCE(MAX(CASE WHEN form_meta.meta_key = 'directorist' THEN 1 ELSE 0 END), 0) = 'directorist' AND users.user_nicename IS NOT NULL
                THEN CONCAT(form.title, ' (', users.user_nicename, ')')
                ELSE form.title
            END as title"
        );
        return $query;
    }

    public function chat_bubble_form_query( Builder $query ) {
        add_filter( 'helpgent_chat_bubble_form_query_join_meta', '__return_true' );

        $current_page_id = helpgent_get_current_page_id();

        if ( 0 !== $current_page_id ) {
            $listing = get_post( $current_page_id );

            if ( $listing instanceof \WP_Post && 'at_biz_dir' === $listing->post_type ) {
                $query->where_raw( "(form_meta.meta_key is null or (form_meta.meta_key = 'directorist' and form.created_by = '{$listing->post_author}'))" );
            } else {
                $query->where_raw( "(form_meta.meta_key is null or form_meta.meta_key != 'directorist')" );
            }
        } else {
            $query->where_raw( "(form_meta.meta_key is null or form_meta.meta_key != 'directorist')" );
        }

        $columns   = $query->columns;
        $columns[] = "COALESCE(MAX(CASE WHEN form_meta.meta_key = 'directorist' THEN 1 ELSE 0 END), 0) as is_directorist";
        return $query->select( $columns );
    }

    /**
     * Admin dashboard form query
     *
     * @param Builder $query
     * @param string $type
     * @return Builder
     */
    public function form_query( Builder $query, string $type ) {
        /**
         * @var WP_REST_Request $wp_rest_request
         */
        $wp_rest_request = helpgent_singleton( WP_REST_Request::class );

        $route = ( fn() => $this->route )->call( $wp_rest_request );

        if ( '/helpgent/admin/form' !== $route ) {
            return $query;
        }

        if ( 'directorist' === $type ) {
            add_filter( "helpgent_form_query_join_meta", "__return_true" );
            add_filter( 'helpgent_form_count_query', [$this, 'form_count_query'], 10, 2 );
            return $query->where( "form_meta.meta_key", "directorist" );
        } else {
            add_filter( 'helpgent_form_count_query', [$this, 'dir_form_query'], 10, 2 );
            return $this->dir_form_query( $query );
        }
    }

    public function form_count_query( Builder $query ) {
        return $query->where( "form_meta.meta_key", "directorist" );
    }

    public function dir_form_query( Builder $query ) {
        return $query->left_join(
            "helpgent_form_meta as dir_form_meta", function( JoinClause $join ) {
                $join->on_column( "form.id", "dir_form_meta.form_id" )->on( "dir_form_meta.meta_key", "directorist" );
            }
        )->where_is_null( "dir_form_meta.meta_key" );
    }

    public function nav_items() {
        ?>
        <li class="directorist-tab__nav__item atbdp_tab_nav--has-child atbdp_all_dhgi_nav">
            <a href="#" target="hg-forms" class="atbdp_all_dhgi_nav-link directorist-tab__nav__link atbd-dash-nav-dropdown">
                <span class="directorist_menuItem-text">
                    <span class="directorist_menuItem-icon">
                        <?php directorist_icon( 'las la-clipboard-list' ); ?>
                    </span>
                <?php esc_html_e( "Messages & Forms", "directorist-helpgent-integration" ); ?>
            </span>
            <?php directorist_icon( 'fas fa-angle-down' ); ?>
            </a>
            <ul class="helpgent-dashboard-nav atbd-dashboard-nav">
                <li>
                    <a
                        href="#/hg-forms" class="directorist-tab__nav__link hg_form_nav__link" target="hg-forms"
                    >
                        <span class="directorist_menuItem-text">
                            <?php directorist_icon( 'las la-file' ); esc_html_e( "Forms", "directorist-helpgent-integration" ); ?>
                        </span>
                    </a>

                    <a style="display: none;" data-parent-nav='.hg_form_nav__link' href="#/hg-forms/\d+/edit" class="directorist-tab__nav__link" target="hg-forms"></a>
                    <a style="display: none;" data-parent-nav='.hg_form_nav__link' href="#/hg-forms/\d+/responses" class="directorist-tab__nav__link" target="hg-forms"></a>
                    <a style="display: none;" data-parent-nav='.hg_form_nav__link' href="#/hg-forms/\d+/summary" class="directorist-tab__nav__link" target="hg-forms"></a>
                </li>
                <li>
                    <a href="#/hg-tags" class="directorist-tab__nav__link" target="hg-forms">
                        <span class="directorist_menuItem-text">
                            <?php directorist_icon( 'las la-tags' );  esc_html_e( "Tags", "directorist-helpgent-integration" ); ?>
                        </span>
                    </a>
                </li>
                <?php if ( function_exists( "helpgent_pro" ) && dhgi_get_settings( false )['display_the_lead'] ) : ?>
                    <li>
                        <a href="#/hg-leads" class="directorist-tab__nav__link" target="hg-forms">
                            <span class="directorist_menuItem-text">
                                <?php directorist_icon( 'las la-user' );  esc_html_e( "Leads", "directorist-helpgent-integration" ); ?>
                            </span>
                        </a>
                    </li>
                <?php endif; ?>
            </ul>
        </li>
        <?php
    }

    public function nav_contents() {
        wp_enqueue_media();

        if ( is_file( helpgent_dir( 'enqueues/enqueue-common.php' ) ) ) {
            include_once helpgent_dir( 'enqueues/enqueue-common.php' );
        }

        wp_enqueue_script( 'helpgent-pro/elements' );
        wp_enqueue_script( 'helpgent/notification' );

        wp_enqueue_script( "dhgi-listing-owner" );
        wp_enqueue_style( "dhgi-listing-owner" );

        wp_enqueue_style( 'wordpress-dashicon', includes_url( 'css/dashicons.css' ), [], get_bloginfo( 'version' ) );

        ob_start();
        wp_editor( '', 'email_template_editor' );
        $email_template_editor = ob_get_clean();

        View::render( 'font-family' );
        $email_sent = get_user_meta( wp_get_current_user()->ID, EmailNotificationServiceProvider::EMAIL_COUNT_METAKEY, true );
        $total_sent = 0;

        if ( ! empty( $email_sent ) ) {
            $today = helpgent_now()->format( 'Y-m-d' );
            if ( $today === $email_sent['date'] ) {
                $total_sent = $email_sent['total'];
            }
        }

        ?>
        <div class="directorist-tab__pane helpgent-dashboard-tab" id="hg-forms" data-total_email_sent="<?php helpgent_render( $total_sent );?>">
            <?php dhgi_render( '<div class="helpgent-root" class="wpwax-vm-app-container-full" data-email_template_editor="' . base64_encode( $email_template_editor ) . '"></div>' ) ?>
        </div>
        <?php
    }
}