<?php

namespace DHGI\App\Providers;

use DHGI\App\Models\User;
use DHGI\WpMVC\Contracts\Provider;
use HelpGent\WpMVC\Database\Query\Builder;
use stdClass;
use WP_REST_Request;

class EmailNotificationServiceProvider implements Provider {

    const EMAIL_COUNT_METAKEY = 'dhgi_email_sent';

    public function boot() {
        add_action( 'helpgent_after_sending_email', [$this, 'after_sending_email'], 10, 2 );
        add_filter( 'helpgent_response_store_from_query', [$this, 'response_store_form_query'] );
        add_filter( 'helpgent_response_incomplete_from_query', [$this, 'incomplete_response_form_query'], 10, 2 );
        add_filter( 'helpgent_send_admin_email_notification', [$this, 'send_email_notification'], 10, 2 );
        add_filter( 'helpgent_send_user_email_notification', [$this, 'send_email_notification'], 10, 2 );
        add_filter( 'helpgent_email_notification_admin_data', [$this, 'email_notification_admin_data'], 10, 2 );
        add_filter( 'helpgent_admin_email_dashboard_url', [$this, 'admin_email_dashboard_url'], 10, 2 );
        add_filter( 'helpgent_message_store_from_query', [$this, 'message_store_from_query'] );
        add_filter( 'helpgent_response_form_query', [$this, 'response_form_query'] );
    }

    /**
     * Count email after email sent
     *
     * @param boolean $sent
     * @param stdClass $form
     * @return void
     */
    public function after_sending_email( bool $sent, ?stdClass $form = null ) {
        if ( ! $form || '0' === $form->is_directorist ) {
            return;
        }

        $email_sent = get_user_meta( $form->created_by, self::EMAIL_COUNT_METAKEY, true );
        $today      = $this->get_today();

        if ( empty( $email_sent ) ) {
            add_user_meta(
                $form->created_by, self::EMAIL_COUNT_METAKEY, [
                    'date'  => $today,
                    'total' => 1
                ]
            );
        } else {
            if ( $email_sent['date'] === $today ) {
                $total = $email_sent['total'] + 1;
            } else {
                $total = 1;
            }
            update_user_meta(
                $form->created_by, self::EMAIL_COUNT_METAKEY, [
                    'date'  => $today,
                    'total' => $total
                ]
            );
        }
    }

    public function response_store_form_query( Builder $query ) {
        add_filter( 'helpgent_response_store_from_query_join_meta', '__return_true' );
        return $this->full_response_form_query( $query );
    }

    public function incomplete_response_form_query( Builder $query, WP_REST_Request $wp_rest_request ) {
        if ( ! $wp_rest_request->has_param( 'submit' ) ) {
            return $query;
        }

        add_filter( 'helpgent_response_incomplete_from_query_join_meta', '__return_true' );

        return $this->full_response_form_query( $query );
    }

    public function send_email_notification( bool $send, stdClass $form ) {
        if ( ! $send ) {
            return $send;
        }

        if ( '0' === $form->is_directorist ) {
            return $send;
        }

        $allowed_email = intval( dhgi_get_settings( false )['allowed_email'] );

        if ( 0 === $allowed_email ) {
            return $send;
        }

        $email_sent = get_user_meta( $form->created_by, self::EMAIL_COUNT_METAKEY, true );

        if ( empty( $email_sent ) ) {
            return $send;
        }

        $today = $this->get_today();

        if ( $email_sent['date'] !== $today ) {
            return $send;
        }

        if ( $allowed_email > $email_sent['total'] ) {
            return $send;
        }

        return false;
    }

    public function email_notification_admin_data( ?array $admin, stdClass $form ) {
        if ( '0' === $form->is_directorist ) {
            return $admin;
        }

        $user = User::query( 'user' )->where( 'user.ID', $form->created_by )->first();

        if ( ! $user ) {
            return $admin;
        }

        return [ 
            'id'    => $user->ID, 
            'name'  => $user->display_name, 
            'email' => $user->user_email 
        ];
    }

    public function admin_email_dashboard_url( string $dashboard_url, stdClass $form ) {
        if ( '0' === $form->is_directorist ) {
            return $dashboard_url;
        }

        return \ATBDP_Permalink::get_dashboard_page_link() . "#/hg-forms/{$form->id}/responses";
    }

    public function message_store_from_query( Builder $query ) {
        add_filter( 'helpgent_message_store_from_query_join_meta', '__return_true' );
        return $this->full_response_form_query( $query );
    }

    public function response_form_query( Builder $query ) {
        add_filter( 'helpgent_response_form_query_join_meta', '__return_true' );
        return $this->full_response_form_query( $query );
    }

    public function full_response_form_query( Builder $query ) {
        $columns   = $query->columns;
        $columns[] = "COALESCE(MAX(CASE WHEN form_meta.meta_key = 'directorist' THEN 1 ELSE 0 END), 0) as is_directorist";
        return $query->select( $columns );
    }

    protected function get_today() {
        return helpgent_now()->format( 'Y-m-d' );
    }
}