<?php

namespace DHGI\App\Http\Controllers;

use DHGI\App\Http\Controllers\Controller;
use HelpGent\App\Http\Controllers\Admin\TagController as HelpGentTagController;
use HelpGent\App\Repositories\TagRepository;
use HelpGent\WpMVC\RequestValidator\Validator;
use HelpGent\WpMVC\Database\Query\Builder;
use DHGI\WpMVC\Routing\Response;
use WP_REST_Request;

class TagController extends Controller {
    public HelpGentTagController $controller;

    public TagRepository $repository;

    public function __construct( HelpGentTagController $controller, TagRepository $repository ) {
        $this->controller = $controller;
        $this->repository = $repository;
    }

    public function index( Validator $validator, WP_REST_Request $wp_rest_request ) {
        add_filter( "helpgent_tags_query", [ $this, "tags_query"] );
        add_filter( "helpgent_tag_query_join_form_meta", "__return_true" );
        return $this->controller->index( $validator, $wp_rest_request );
    }

    public function tags_query( Builder $query ) {
        return $query->where( "tag.created_by", wp_get_current_user()->ID )->where( "form_meta.meta_key", "directorist" );
    }

    public function store( Validator $validator, WP_REST_Request $wp_rest_request ) {
        $validator->validate(
            [
                'form_id' => 'required|numeric',
            ]
        );

        if ( $validator->is_fail() ) {
            return Response::send(
                [
                    'messages' => $validator->errors
                ], 422
            );
        }

        $form = $this->repository->form_repository->get_by_id( $wp_rest_request->get_param( 'form_id' ) );

        if ( ! $form ) {
            return Response::send(
                [
                    'message' => esc_html__( 'Form not found.', 'directorist-helpgent-integration' )
                ], 404
            );
        }

        if ( wp_get_current_user()->ID !== intval( $form->created_by ) ) {
            return Response::send(
                [
                    'message' => esc_html__( 'Sorry, you are not authorized to access this form data.', 'directorist-helpgent-integration' )
                ], 500
            );
        }

        return $this->controller->store( $validator, $wp_rest_request );
    }

    public function update( Validator $validator, WP_REST_Request $wp_rest_request ) {
        $error = $this->is_user_can_access( $validator, $wp_rest_request );

        if ( is_array( $error ) ) {
            return $error;
        }

        return $this->controller->update( $validator, $wp_rest_request );
    }

    public function delete( Validator $validator, WP_REST_Request $wp_rest_request ) {
        $error = $this->is_user_can_access( $validator, $wp_rest_request );

        if ( is_array( $error ) ) {
            return $error;
        }

        return $this->controller->delete( $validator, $wp_rest_request );
    }

    public function is_user_can_access( Validator $validator, WP_REST_Request $wp_rest_request ) {
        $validator->validate(
            [
                'id' => 'required|numeric',
            ]
        );

        if ( $validator->is_fail() ) {
            return Response::send(
                [
                    'messages' => $validator->errors
                ], 422
            );
        }

        $tag = $this->repository->get_by_id( $wp_rest_request->get_param( 'id' ) );

        if ( ! $tag ) {
            return Response::send(
                [
                    'message' => esc_html__( 'Tag not found.', 'directorist-helpgent-integration' )
                ], 404
            );
        }

        if ( wp_get_current_user()->ID !== intval( $tag->created_by ) ) {
            return Response::send(
                [
                    'message' => esc_html__( 'Sorry, you are not authorized to access this tag data.', 'directorist-helpgent-integration' )
                ], 500
            );
        }
    }
}