<?php

namespace DHGI\App\Http\Controllers;

use DHGI\App\Http\Controllers\Controller;
use HelpGent\App\DTO\FormDTO;
use HelpGent\App\Models\Form;
use HelpGent\App\Repositories\FormRepository;
use HelpGent\WpMVC\Database\Query\JoinClause;
use HelpGent\WpMVC\RequestValidator\Validator;
use HelpGent\WpMVC\Database\Query\Builder;
use HelpGent\App\Http\Controllers\Admin\FormController as HelpGentFormController;
use DHGI\WpMVC\Routing\Response;
use WP_REST_Request;

class FormController extends Controller {
    public HelpGentFormController $controller;

    public FormRepository $form_repository;

    public function __construct( HelpGentFormController $controller, FormRepository $form_repository ) {
        $this->controller      = $controller;
        $this->form_repository = $form_repository;
    }

    public function index( Validator $validator, WP_REST_Request $wp_rest_request ) {
        add_filter( "helpgent_form_query_join_meta", "__return_true" );
        add_filter( "helpgent_form_query", [$this, "form_query"] );
        add_filter( "helpgent_form_count_query", [$this, "form_query"] );

        return $this->controller->index( $validator, $wp_rest_request );
    }

    public function store( Validator $validator, WP_REST_Request $wp_rest_request ) {
        $user_id = helpgent_get_current_user()->get_id();

        $total_forms = Form::query( 'form' )->where( 'form.created_by', $user_id )->left_join(
            "helpgent_form_meta as form_meta", function( JoinClause $join ) {
                $join->on_column( "form_meta.form_id", "=", "form.id" )->on( "form_meta.meta_key", "directorist" );
            }
        )->where( "form_meta.meta_key", "directorist" )->count();

        $form_limit = dhgi_get_settings()['maximum_form_creation_limit'];

        if ( $form_limit !== 0 && $form_limit <= $total_forms ) {
            return Response::send(
                [
                    'message' => sprintf( esc_html__( "Sorry, you can't can create more than %d forms.", 'directorist-helpgent-integration' ), $form_limit )
                ], 500
            );
        }

        add_action( "helpgent_after_create_form", [$this, "after_create_form"] );

        return $this->controller->store( $validator, $wp_rest_request );
    }

    public function update( Validator $validator, WP_REST_Request $wp_rest_request ) {
        $error = $this->is_user_can_edit( $validator, $wp_rest_request );

        if ( is_array( $error ) ) {
            return $error;
        }

        /**
         * Remove email_template_settings from Form update data, because email_template_settings only form admin
         */
        $params = $wp_rest_request->get_params();
        unset( $params['email_template_settings'] );
        $wp_rest_request->set_body_params( $params );

        return $this->controller->update( $validator, $wp_rest_request );
    }

    public function auto_save( Validator $validator, WP_REST_Request $wp_rest_request ) {
        $error = $this->is_user_can_edit( $validator, $wp_rest_request );

        if ( is_array( $error ) ) {
            return $error;
        }

        return $this->controller->auto_save( $validator, $wp_rest_request );
    }

    public function show( Validator $validator, WP_REST_Request $wp_rest_request ) {
        $error = $this->is_user_can_edit( $validator, $wp_rest_request );

        if ( is_array( $error ) ) {
            return $error;
        }

        return $this->controller->show( $validator, $wp_rest_request );
    }

    public function delete_bulk_form( Validator $validator, WP_REST_Request $wp_rest_request ) {
        $error = $this->is_user_can_edit_bulk_items( $validator, $wp_rest_request );

        if ( is_array( $error ) ) {
            return $error;
        }

        return $this->controller->delete_bulk_form( $validator, $wp_rest_request );
    }

    public function update_bulk_status( Validator $validator, WP_REST_Request $wp_rest_request ) {
        $error = $this->is_user_can_edit_bulk_items( $validator, $wp_rest_request );

        if ( is_array( $error ) ) {
            return $error;
        }

        return $this->controller->update_bulk_status( $validator, $wp_rest_request );
    }

    public function delete( Validator $validator, WP_REST_Request $wp_rest_request ) {
        $error = $this->is_user_can_edit( $validator, $wp_rest_request );

        if ( is_array( $error ) ) {
            return $error;
        }

        return $this->controller->delete( $validator, $wp_rest_request );
    }

    public function update_status( Validator $validator, WP_REST_Request $wp_rest_request ) {
        $error = $this->is_user_can_edit( $validator, $wp_rest_request );

        if ( is_array( $error ) ) {
            return $error;
        }

        return $this->controller->update_status( $validator, $wp_rest_request );
    }

    public function update_title( Validator $validator, WP_REST_Request $wp_rest_request ) {
        $error = $this->is_user_can_edit( $validator, $wp_rest_request );

        if ( is_array( $error ) ) {
            return $error;
        }

        return $this->controller->update_title( $validator, $wp_rest_request );
    }

    public function tags( Validator $validator, WP_REST_Request $wp_rest_request ) {
        $error = $this->is_user_can_edit( $validator, $wp_rest_request );

        if ( is_array( $error ) ) {
            return $error;
        }

        return $this->controller->tags( $validator, $wp_rest_request );
    }

    public function insert_media( Validator $validator, WP_REST_Request $wp_rest_request ) {
        return $this->controller->insert_media( $validator, $wp_rest_request );
    }

    public function delete_element_option( Validator $validator, WP_REST_Request $wp_rest_request ) {
        $error = $this->is_user_can_edit( $validator, $wp_rest_request );

        if ( is_array( $error ) ) {
            return $error;
        }

        return $this->controller->delete_element_option( $validator, $wp_rest_request );
    }

    public function delete_element( WP_REST_Request $wp_rest_request, Validator $validator ) {
        $error = $this->is_user_can_edit( $validator, $wp_rest_request );

        if ( is_array( $error ) ) {
            return $error;
        }

        return $this->controller->delete_element( $validator, $wp_rest_request );
    }

    public function select( Validator $validator, WP_REST_Request $wp_rest_request ) {
        add_filter( 'helpgent_form_select_query_join_meta', '__return_true' );
        add_filter( 'helpgent_form_select_list_query', [ $this,'form_query'] );

        return $this->controller->select( $validator, $wp_rest_request );
    }

    protected function is_user_can_edit_bulk_items( Validator $validator, WP_REST_Request $wp_rest_request ) {
        $validator->validate(
            [
                'ids' => 'required|array'
            ]
        );

        if ( $validator->is_fail() ) {
            return Response::send(
                [
                    'messages' => $validator->errors
                ], 422
            );
        }

        $form_ids = $wp_rest_request->get_param( 'ids' );

        if ( empty( $form_ids ) || ! helpgent_is_one_level_array( $form_ids ) ) {
            return Response::send(
                [
                    'message' => esc_html__( 'Sorry, Something was wrong.', 'directorist-helpgent-integration' )
                ]
            );
        }

        $total_forms = Form::query()->where_in( 'id', $form_ids )->where( 'created_by', wp_get_current_user()->ID )->count();

        if ( count( $form_ids ) !== $total_forms ) {   
            return Response::send(
                [
                    'message' => esc_html__( 'Sorry, you are not allowed to edit these forms.', 'directorist-helpgent-integration' )
                ]
            );
        }
    }

    public function is_user_can_edit( Validator $validator, WP_REST_Request $wp_rest_request ) {
        $validator->validate(
            [
                'id' => 'required|numeric',
            ]
        );

        if ( $validator->is_fail() ) {
            return Response::send(
                [
                    'messages' => $validator->errors
                ], 422
            );
        }

        $form = $this->form_repository->get_by_id( $wp_rest_request->get_param( 'id' ) );

        if ( ! $form ) {
            return Response::send(
                [
                    'message' => esc_html__( 'Form not found.', 'directorist-helpgent-integration' )
                ], 404
            );
        }

        if ( wp_get_current_user()->ID !== intval( $form->created_by ) ) {
            return Response::send(
                [
                    'message' => esc_html__( 'Sorry, you are not authorized to access this form data.', 'directorist-helpgent-integration' )
                ], 500
            );
        }
    }

    public function form_query( Builder $query ) {
        return $query->where( "form.created_by", wp_get_current_user()->ID )->where( "form_meta.meta_key", "directorist" );
    }

    public function after_create_form( FormDTO $form_dto ) {
        $this->form_repository->add_meta( $form_dto->get_id(), "directorist", 1 );
    }
}