<?php

defined( 'ABSPATH' ) || exit;

use DHGI\WpMVC\App;
use DHGI\DI\Container;

function dhgi():App {
    return App::$instance;
}

function dhgi_config( string $config_key ) {
    return dhgi()::$config->get( $config_key );
}

function dhgi_app_config( string $config_key ) {
    return dhgi_config( "app.{$config_key}" );
}

function dhgi_version() {
    return dhgi_app_config( 'version' );
}

function dhgi_container():Container {
    return dhgi()::$container;
}

function dhgi_singleton( string $class ) {
    return dhgi_container()->get( $class );
}

function dhgi_url( string $url = '' ) {
    return dhgi()->get_url( $url );
}

function dhgi_dir( string $dir = '' ) {
    return dhgi()->get_dir( $dir );
}

function dhgi_render( string $content ) {
    //phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
    echo $content;
}

function dhgi_get_default_settings( $key = '' ) {
    static $settings = [
        'max_video_duration'          => 2,
        'max_voice_duration'          => 2,
        'max_upload_size'             => 50,
        'max_upload_size_unit'        => 'mb',
        'video_resolution'            => 720,
        'form_bubble_position'        => 'right',
        'form_bubble_offset_x'        => 30,
        'form_bubble_offset_y'        => 30,
        'form_bubble_size'            => 'medium',
        'allowed_email'               => 0,
        'maximum_form_creation_limit' => 0,
        'display_the_lead'            => true,
    ];

    if ( ! empty( $key ) ) {
        return array_key_exists( $key, $settings ) ? $settings[ $key ] : null;
    }

    return $settings;
}

function dhgi_get_settings( bool $return_formatted_settings = true ): array {
    $settings = dhgi_get_default_settings();
    $options  = ( array ) get_option( 'atbdp_option', [] );

    foreach ( $settings as $key => $value ) {
        $prefixed_key = 'hgd_' . $key;

        if ( array_key_exists( $prefixed_key, $options ) ) {
            $settings[ $key ] = $options[ $prefixed_key ];
        }
    }

    if ( ! $return_formatted_settings ) {
        return $settings;
    }

    $formatted_settings = [
        'allowed_email'               => $settings['allowed_email'],
        'maximum_form_creation_limit' => $settings['maximum_form_creation_limit'],
        'display_the_lead'            => $settings['display_the_lead'],
        
        'media'                       => [
            'max_video_duration' => [ 'value' => $settings['max_video_duration'] ],
            'max_voice_duration' => [ 'value' => $settings['max_voice_duration'] ],
            'max_upload_size'    => [ 
                'value' => $settings['max_upload_size'],
                'unit'  => $settings['max_upload_size_unit'],
            ],
            'video_resolution'   => [ 'value' => $settings['video_resolution'] ],
        ],

        'bubble'                      => [
            'position' => $settings['form_bubble_position'],
            'offset_x' => $settings['form_bubble_offset_x'],
            'offset_y' => $settings['form_bubble_offset_y'],
            'size'     => $settings['form_bubble_size']
        ],
    ];

    return $formatted_settings; 
}

function dhgi_is_pro_active(): bool {
    $is_pro_active = is_plugin_active( 'helpgent-pro/helpgent-pro.php' );
    return $is_pro_active;
}

function dhgi_get_resolution_options( int $max = 0 ): array {
    $resolutions = [
        360  => [ 'width' => 640, 'height' => 360 ],
        480  => [ 'width' => 640, 'height' => 480 ],
        720  => [ 'width' => 1280, 'height' => 720 ],
        1080 => [ 'width' => 1920, 'height' => 1080 ],
        1440 => [ 'width' => 2560, 'height' => 1440 ],
        2160 => [ 'width' => 3840, 'height' => 2160, 'alias' => '4K' ],
        7680 => [ 'width' => 7680, 'height' => 4320, 'alias' => '8K' ],
    ];

    if ( $max ) {
        foreach ( $resolutions as $key => $value ) {
            if ( $key > $max ) {
                unset( $resolutions[ $key ] );
            }
        }
    }

    return array_map(
        function( $key ) use( $resolutions ) {
            $resolution = ( object ) $resolutions[ $key ];
            $alias      = ! empty( $resolution->alias ) ? " ({$resolution->alias})" : '';

            return [
                'value' => $key,
                'label' => "{$resolution->height}p{$alias} : {$resolution->width}x{$resolution->height}",
            ];
        }, array_keys( $resolutions ) 
    );
}