<?php
/**
 * @package  Directorist - Compare Listing
 */

if (!class_exists('AjaxHandler')) {
    /**
     * All ajax handler class
     */
    class AjaxHandler
    {
        /**
         * Class constructor
         * Call all hooks which we wanna triger
         */
        public function register()
        {
            // Get Selected Listings
            add_action('wp_ajax_get_the_selected_listings_sidebar', array($this, 'get_the_selected_listings_sidebar'));
            add_action('wp_ajax_nopriv_get_the_selected_listings_sidebar', array($this, 'get_the_selected_listings_sidebar'));
            
            // Listing add & remove handeller hook
            add_action('wp_ajax_selected_listing', array($this, 'add_or_remove_selected_listing'));
            add_action('wp_ajax_nopriv_selected_listing', array($this, 'add_or_remove_selected_listing'));

            // Display select window
            add_action('wp_ajax_display_select_window', array($this, 'display_select_window'));
            add_action('wp_ajax_nopriv_display_select_window', array($this, 'display_select_window'));

            // Listings remover hook
            add_action('wp_ajax_remove_listing_from_compare', array($this, 'remove_listing_from_compare'));
            add_action('wp_ajax_nopriv_remove_listing_from_compare', array($this, 'remove_listing_from_compare'));

            // license and auto update handler
            add_action('wp_ajax_atbdp_compare_listings_license_activation', array($this, 'atbdp_compare_listings_license_activation'));

            // license deactivation
            add_action('wp_ajax_atbdp_compare_listings_license_deactivation', array($this, 'atbdp_compare_listings_license_deactivation'));

            // Hook remover
            add_action('plugins_loaded', array($this, 'hook_remover'));
        }

        /**
         * Get the selected listings sidebar
         * 
         * @return array Response
         */
        public function get_the_selected_listings_sidebar() {
            $selected_listings_id = HelperFunctions::get_selected_ids();

            if ( empty( $selected_listings_id ) ) {
                wp_send_json([ 
                    'selected_listings' => $selected_listings_id,
                    'content'           => '',
                ]);
            }

            $args = array(
                'post_type'   => 'at_biz_dir',
                'post_status' => 'publish',
                'post__in'    => $selected_listings_id
            );

            $listings_query = new WP_Query( $args );

            if ( ! $listings_query->have_posts() ) {
                wp_send_json([ 
                    'selected_listings' => $selected_listings_id,
                    'content'           => '',
                ]);
            }

            $total_selected       = $listings_query->found_posts;
            $compare_button_title = __('Compare', 'directorist-compare-listing');
            $window_title         = get_directorist_option('atdlc_select_window_title', '' );
            $window_title         = ! empty( $window_title ) ? $window_title : __('Compare Listings', 'directorist-compare-listing');
            // Make compare page URL
            $compare_page_id        = get_directorist_option("compare_listings_page");
            $compare_page_url       = get_the_permalink($compare_page_id);

            $template_file = ATDLC_PLUGIN_TEMPLATES_DIR . 'selected-listings-sidebar.php';

            if ( ! file_exists( $template_file ) ) {
                wp_send_json([ 
                    'selected_listings' => $selected_listings_id,
                    'content'           => '',
                ]);
            }

            ob_start();
            include_once( $template_file );

            $content = ob_get_clean();

            wp_send_json([
                'content'              => $content,
                'selected_listings_id' => $selected_listings_id,
            ]);
        }


        public function atbdp_compare_listings_license_deactivation()
        {
            $license                            = !empty($_POST['compare_listings_license']) ? trim($_POST['compare_listings_license']) : '';
            $options                            = get_option('atbdp_option');
            $options['compare_listings_license'] = $license;

            update_option('atbdp_option', $options);
            update_option('directorist_compare_listings_license', $license);

            $data = $this->license_controller($license, 'deactivate_license');

            wp_send_json($data);
            die();
        }



        public function atbdp_compare_listings_license_activation()
        {
            $license                            = !empty($_POST['compare_listings_license']) ? trim($_POST['compare_listings_license']) : '';
            $options                            = get_option('atbdp_option');
            $options['compare_listings_license'] = $license;
            update_option('atbdp_option', $options);
            update_option('directorist_compare_listings_license', $license);
            $data = $this->license_controller($license, 'activate_license');
            wp_send_json($data);
            die();
        }



        private function license_controller($license, $action)
        {
            $data = array();
            if (!empty($license)) {
                // data to send in our API request
                $api_params = array(
                    'edd_action' => $action,
                    'license'    => $license,
                    'item_id'    => ATBDP_ATDLC_POST_ID, // The ID of the item in EDD
                    'url'        => home_url(),
                );

                // Call the custom API.
                $response = wp_remote_post(ATBDP_AUTHOR_URL, array('timeout' => 15, 'sslverify' => false, 'body' => $api_params));
                // make sure the response came back okay
                if (is_wp_error($response) || 200 !== wp_remote_retrieve_response_code($response)) {

                    $data['msg']    = (is_wp_error($response) && !empty($response->get_error_message())) ? $response->get_error_message() : __('An error occurred, please try again.', 'directorist-compare-listing');
                    $data['status'] = false;
                } else {
                    $license_data = json_decode(wp_remote_retrieve_body($response));
                    if (!$license_data) {
                        $data['status'] = false;
                        $data['msg']    = __('Response not found!', 'directorist-compare-listing');
                        wp_send_json($data);
                        die();
                    }

                    update_option('directorist_compare_listings_license_status', $license_data->license);
                    if (false === $license_data->success) {
                        switch ($license_data->error) {
                            case 'expired':
                                $data['msg'] = sprintf(
                                    __('Your license key expired on %s.', 'directorist-compare-listing'),
                                    date_i18n(get_option('date_format'), strtotime($license_data->expires, current_time('timestamp')))
                                );
                                $data['status'] = false;
                                break;

                            case 'revoked':
                                $data['status'] = false;
                                $data['msg']    = __('Your license key has been disabled.', 'directorist-compare-listing');
                                break;

                            case 'missing':

                                $data['msg']    = __('Invalid license.', 'directorist-compare-listing');
                                $data['status'] = false;
                                break;

                            case 'invalid':
                            case 'site_inactive':

                                $data['msg']    = __('Your license is not active for this URL.', 'directorist-compare-listing');
                                $data['status'] = false;
                                break;

                            case 'item_name_mismatch':

                                $data['msg']    = sprintf(__('This appears to be an invalid license key for %s.', 'directorist-compare-listing'), 'Directorist - Listings with Map');
                                $data['status'] = false;
                                break;

                            case 'no_activations_left':

                                $data['msg']    = __('Your license key has reached its activation limit.', 'directorist-compare-listing');
                                $data['status'] = false;
                                break;

                            default:
                                $data['msg']    = __('An error occurred, please try again.', 'directorist-compare-listing');
                                $data['status'] = false;
                                break;
                        }
                    } else {
                        $data['status'] = true;
                        if ('activate_license' === $action) {
                            $data['msg']    = __('License activated successfully!', 'directorist-compare-listing');
                        } else {
                            $data['msg']    = __('License deactivated successfully!', 'directorist-compare-listing');
                        }
                    }
                }
            } else {
                $data['status'] = false;
                $data['msg']    = __('License not found!', 'directorist-compare-listing');
            }
            return $data;
        }



        /**
         * Get & set to cookies the selected listing ID
         */
        public function add_or_remove_selected_listing()
        {
            // Get listing select limit
            $select_limit           = get_directorist_option('atd_lg_select_limit', 5);
            $select_limit           = !empty($select_limit) ? (int)$select_limit : '';

            // Count, current selected listings
            $selected_listings      = HelperFunctions::get_selected_ids();
            $selected_count         = count($selected_listings);

            // Get selected listings ID
            $selected_listing_id    = isset($_POST['listing_id']) ? (int)$_POST['listing_id'] : '';

            // Autheticate, is current selected ID was selected
            if (!in_array($selected_listing_id, $selected_listings)) {

                // Check, is select limit over
                if ($select_limit > $selected_count) {

                    // Set to cookies the selected listings ID
                    $cookies_name = ATDLC_COOKIES_NAME;

                    if (!isset($_COOKIE[$cookies_name])) {
                        $cookies_value = $selected_listing_id;
                        setcookie($cookies_name, $cookies_value, 0, COOKIEPATH, ATDLC_COOKIES_DOMAIN);
                        if (SITECOOKIEPATH != COOKIEPATH) {
                            setcookie($cookies_name, $cookies_value, 0, SITECOOKIEPATH, ATDLC_COOKIES_DOMAIN);
                        }
                    } else {
                        $cookies_value = $_COOKIE[$cookies_name] . "," . $selected_listing_id;
                        setcookie($cookies_name, $cookies_value, 0, COOKIEPATH, ATDLC_COOKIES_DOMAIN);
                        if (SITECOOKIEPATH != COOKIEPATH) {
                            setcookie($cookies_name, $cookies_value, 0, SITECOOKIEPATH, ATDLC_COOKIES_DOMAIN);
                        }
                    }

                    $compare_btn_title      = __('Compare', 'directorist-compare-listing');
                    $default_sw_title       = __('Compare Listings', 'directorist-compare-listing');
                    $select_window_title    = get_directorist_option('atdlc_select_window_title');
                    $select_window_title    = !empty($select_window_title) ? $select_window_title : $default_sw_title;

                    // Get listing title
                    $listing_title          = get_the_title($selected_listing_id);

                    // Get listing image
                    $default_img    = get_directorist_option('default_preview_image', ATBDP_PUBLIC_ASSETS . 'images/
                    grid.jpg');
                    $prv_img        = get_post_meta($selected_listing_id, '_listing_prv_img', true);
                    $prv_img_src    = atbdp_get_image_source($prv_img);
                    $img_src        = !empty($prv_img_src) ? $prv_img_src : $default_img;

                    // Make compare page URL
                    $compare_page_id        = get_directorist_option("compare_listings_page");
                    $compare_page_url       = get_the_permalink($compare_page_id);

                    // Count, current selected listings
                    $selected_listings_id   = HelperFunctions::get_selected_ids();
                    $total_selected         = count($selected_listings_id) + 1;

                    if (1 === (int)$total_selected) {
                        // If, there has no listing selected, then render
                        $info['frame'] = require_once(ATDLC_PLUGIN_TEMPLATES_DIR . "ajax-select-window.php");
                    } else {
                        // If, there has minimum 1 listing selected, then render
                        $info['selected_listing'] = "
                            <div id='directorist-compare-listing-single$selected_listing_id' class='directorist-compare-listing-single'>
                                <div class='directorist-compare-listing-single__img'>
                                    <img src='$img_src' alt=''>
                                </div>
                                <div class='directorist-compare-listing-single__content'>
                                    <p class='directorist-compare-listing-single__title'>$listing_title</p>
                                    <div class='directorist-compare-listing-single__action'>
                                        <div id='directorist-compare-listing-single__action__btn' remove_btn_id='$selected_listing_id' class='directorist-compare-listing-single__action__btn'>
                                        " . directorist_icon( 'las la-trash', false ) . "
                                        </div>
                                    </div>
                                </div>
                            </div>
                        ";
                    }

                    $info['listing_id']                 = $selected_listing_id;
                    $info['current_items']              = $total_selected;
                    $info['page_url']                   = $compare_page_url;
                    $info['tooltip_text']               = __('Remove from compare', 'directorist-compare-listing');
                    $info['is_listing_add_or_remove']   = 'add';

                    echo json_encode($info);
                    die();
                } else {
                    // If select limit is over then rander this block
                    $limit_alert_text  = __("You can not add more then $select_limit products to the compare.", 'directorist-compare-listing');

                    // Select limit aleart window
                    $info['limit_alert_message'] = "
                    <div class='atdlc_limit-aleart-message' id='atdlc_limit-aleart-message'>
                        <p>$limit_alert_text</p>
                    </div>
                    ";

                    echo json_encode($info);
                    die();
                }
            } else {
                // Remove selected listing ID
                if (in_array($selected_listing_id, $selected_listings)) {
                    $key = array_search($selected_listing_id, $selected_listings);
                    unset($selected_listings[$key]);
                }

                // Make selected listings array to a string
                $update_cookies     = implode(',', $selected_listings);

                // Update cookies by update selected listings array
                $cookies_value      = $update_cookies;
                $cookies_name       = ATDLC_COOKIES_NAME;
                setcookie($cookies_name, $cookies_value, 0, COOKIEPATH, ATDLC_COOKIES_DOMAIN);
                if (SITECOOKIEPATH != COOKIEPATH) {
                    setcookie($cookies_name, $cookies_value, 0, SITECOOKIEPATH, ATDLC_COOKIES_DOMAIN);
                }

                // Current selected listings ID count
                $current_listings   = HelperFunctions::get_selected_ids();
                $total_selected     = count($current_listings) - 1;

                // AJAX requist result
                $info['listing_id']                 = $selected_listing_id;
                $info['current_items']              = $total_selected;
                $info['tooltip_text']               = __('Add to compare', 'directorist-compare-listing');
                $info['is_listing_add_or_remove']   = 'remove';

                echo json_encode($info);
                die();
            }
        }



        /**
         * Remove listing from select window & compare table
         */
        public function remove_listing_from_compare()
        {
            // Get removed listings ID
            $removed_listing_id = isset($_POST['listing_id']) ? (int)$_POST['listing_id'] : '';

            // Get current selected listings
            $selected_listings  = HelperFunctions::get_selected_ids();

            // Remove selected listing ID
            if (in_array($removed_listing_id, $selected_listings)) {
                $key = array_search($removed_listing_id, $selected_listings);
                unset($selected_listings[$key]);
            }

            // Make selected listings array to a string
            $update_cookies     = implode(',', $selected_listings);

            // Update cookies by update selected listings array
            $cookies_value      = $update_cookies;
            $cookies_name       = ATDLC_COOKIES_NAME;
            setcookie($cookies_name, $cookies_value, 0, COOKIEPATH, ATDLC_COOKIES_DOMAIN);
            if (SITECOOKIEPATH != COOKIEPATH) {
                setcookie($cookies_name, $cookies_value, 0, SITECOOKIEPATH, ATDLC_COOKIES_DOMAIN);
            }

            // Current selected listings ID count
            $current_listings   = HelperFunctions::get_selected_ids();
            $total_selected     = count($current_listings) - 1;

            // AJAX requist result
            $info['current_items']          = $total_selected;
            $info['tooltip_text']           = __('Add to compare', 'directorist-compare-listing');
            $info['empty_listings_message'] = __('No result found.', 'directorist-compare-listing');

            echo json_encode($info);
            die();
        }


        /**
         * Display all selected listings & select window
         */
        public function display_select_window()
        {
            $compare_btn_title      = __('Compare', 'directorist-compare-listing');
            $default_sw_title       = __('Compare Listings', 'directorist-compare-listing');

            $select_window_title    = get_directorist_option('atdlc_select_window_title');
            $select_window_title    = !empty($select_window_title) ? $select_window_title : $default_sw_title;

            // Make compare page URL
            $compare_page_id        = get_directorist_option("compare_listings_page");
            $compare_page_url       = get_the_permalink($compare_page_id);

            // Count, current selected listings
            $selected_listings_id   = HelperFunctions::get_selected_ids();
            $total_selected         = count($selected_listings_id);

            if (0 !== (int)$total_selected) {
                require_once(ATDLC_PLUGIN_TEMPLATES_DIR . "select-window.php");
            }

            die();
        }



        /**
         * Remove hooks when Compare Listings is disable
         */
        public function hook_remover()
        {
            // Is listing compare extension enable
            $is_cl_enable   = get_directorist_option('atd_lg_compare_is_enable', 1);

            // Authenticate, listing compare is enable or not
            if (1 !== (int)$is_cl_enable) {
                remove_action('wp_ajax_display_select_window', array($this, 'display_select_window'));
                remove_action('wp_ajax_nopriv_display_select_window', array($this, 'display_select_window'));
            }
        }
    } // End class
}
