<?php

/**
 * Plugin Name: Directorist - Post Your Need
 * Plugin URI: https://directorist.com/product/directorist-post-your-need/
 * Description: This is an extension for Directorist Plugin. You can accept users need post by this extension.
 * Version: 1.2.6
 * Author: wpWax
 * Author URI: https://wpwax.com
 * License: GPLv2 or later
 * Text Domain: directorist-post-your-need
 * Domain Path: /languages
 */

// prevent direct access to the file
defined('ABSPATH') || die('No direct script access allowed!');

use \Directorist\Multi_Directory_Manager;
 
if (!class_exists('Post_Your_Need')) {
    final class Post_Your_Need
    {

        /** Singleton *************************************************************/

        /**
         * @var Post_Your_Need The one true Post_Your_Need
         * @since 1.0
         */
        private static $instance;

        /**
         * @since 1.1.6
         */
        public $controller;

        /**
         * @since 1.1.6
         */
        public $settings;

        /**
         * Main Post_Your_Need Instance.
         *
         * Insures that only one instance of Post_Your_Need exists in memory at any one
         * time. Also prevents needing to define globals all over the place.
         *
         * @return object|Post_Your_Need The one true Post_Your_Need
         * @uses Post_Your_Need::setup_constants() Setup the constants needed.
         * @uses Post_Your_Need::includes() Include the required files.
         * @uses Post_Your_Need::load_textdomain() load the language files.
         * @see  Post_Your_Need()
         * @since 1.0
         * @static
         * @static_var array $instance
         */
        public static function instance()
        {
            if (!isset(self::$instance) && !(self::$instance instanceof Post_Your_Need)) {
                self::$instance = new Post_Your_Need;
                self::$instance->setup_constants();
                self::$instance->includes();
                self::$instance->settings = new PYN_Settings;
                add_action('plugins_loaded', array(self::$instance, 'load_textdomain'));
                self::$instance->controller = new PYN_Controller;
                add_action( 'init', [ self::$instance, 'import_default_directory_type' ], 30 );
                register_deactivation_hook( __FILE__, array( self::$instance, 'deactivate_plugin' ) );
            }
            return self::$instance;
        }

        private function __construct()
        {
            /*making it private prevents constructing the object*/
        }

        /**
         * Throw error on object clone.
         *
         * The whole idea of the singleton design pattern is that there is a single
         * object therefore, we don't want the object to be cloned.
         *
         * @return void
         * @since 1.0
         * @access protected
         */
        public function __clone()
        {
            // Cloning instances of the class is forbidden.
            _doing_it_wrong(__FUNCTION__, __('Cheatin&#8217; huh?', 'directorist-post-your-need'), '1.0');
        }

        /**
         * Disable unserializing of the class.
         *
         * @return void
         * @since 1.0
         * @access protected
         */
        public function __wakeup()
        {
            // Unserializing instances of the class is forbidden.
            _doing_it_wrong(__FUNCTION__, __('Cheatin&#8217; huh?', 'directorist-post-your-need'), '1.0');
        }


        public function deactivate_plugin() {
            delete_option( 'need_listings_updated_v1' );
        }


        // import_default_directory_type
        public function import_default_directory_type() {

            if( get_option( 'need_listings_updated_v1' ) ){
                return;
            }
        
            $need_listings = get_term_by( 'slug', 'need-listings', ATBDP_DIRECTORY_TYPE );

            if ( $need_listings ) { 
                $directory_type = $need_listings->term_id;
            }else{

                $file = trailingslashit( dirname( PYN_FILE ) )  . 'assets/sample-data/directory.json';
                if ( ! file_exists( $file ) ) { return; }
                
                $file_contents = file_get_contents( $file );

                if ( ! class_exists( '\Directorist\Multi_Directory_Manager' ) ) { return; }

                $multi_directory_manager = new Directorist\Multi_Directory_Manager();

                $multi_directory_manager->prepare_settings();

                $add_directory = $multi_directory_manager->add_directory([
                    'directory_name' => 'Need Listings',
                    'fields_value'   => $file_contents,
                    'is_json'        => true
                ]);

                $directory_type = $add_directory['term_id'];
            }
            
            if ( $directory_type ) {
    
                // Add directory type to all listings
                $listings = new \WP_Query([
                    'post_type' => ATBDP_POST_TYPE,
                    'status'    => 'publish',
                    'per_page'  => -1,
                    'fields'    => 'ids',
                    'meta_query' => array(
                        array(
                            'key'       => '_need_post',
                            'value'     => 'yes',
                            'compare'   => '='
                        )
                    ),
                ]);
    
                if ( $listings->have_posts() ) {
                    foreach( $listings->posts as $listing_id ){

                        $needed_on = get_post_meta( $listing_id, '_pyn_deadline', true );
                        $needed_on__ = get_post_meta( $listing_id, '_date', true );
                        if( $needed_on && ! $needed_on__ ){
                           update_post_meta( $listing_id, '_date', $needed_on );
                        }
                        $hourly = get_post_meta( $listing_id, '_pyn_is_hourly', true );
                        $hourly__ = get_post_meta( $listing_id, '_hourly', true );
                        if( $hourly && ! $hourly__ ){
                           update_post_meta( $listing_id, '_hourly', $hourly );
                        }
                        $budget = get_post_meta( $listing_id, '_price', true );
                        $budget__ = get_post_meta( $listing_id, '_text', true );
                        if( $budget && ! $budget__ ){
                           update_post_meta( $listing_id, '_text', $budget );
                        }
                        $offer = get_post_meta( $listing_id, '_service_offer', true );
                        $offer__ = get_post_meta( $listing_id, '_requirments', true );
                        if( $offer && ! $offer__ ){
                           update_post_meta( $listing_id, '_text', $offer );
                        }

                        wp_set_object_terms( $listing_id, $directory_type, 'atbdp_listing_types' );
                        update_post_meta( $listing_id, '_directory_type', $directory_type );

                    }
                }
            }

            update_option( 'need_listings_updated_v1', true );
        }

        /**
         * It register the text domain to the WordPress
         */
        public function load_textdomain()
        {
            load_plugin_textdomain('directorist-post-your-need', false, PYN_LANG_DIR);
        }

        /**
         * It Includes and requires necessary files.
         *
         * @access private
         * @return void
         * @since 1.0
         */
        private function includes()
        {
            require_once PYN_INC_DIR . 'class-settings.php';
            require_once PYN_INC_DIR . 'class-controller.php';
            require_once PYN_INC_DIR . 'helper-functions.php';
        }

        /**
         * Setup plugin constants.
         *
         * @access private
         * @return void
         * @since 1.0
         */
        private function setup_constants()
        {
            if ( ! defined( 'PYN_FILE' ) ) { define( 'PYN_FILE', __FILE__ ); }

            require_once plugin_dir_path(__FILE__) . '/inc/config-helper.php'; // loads constant from a file so that it can be available on all files.
            require_once plugin_dir_path(__FILE__) . '/config.php'; // loads constant from a file so that it can be available on all files.
        }
    }

    if ( ! function_exists( 'directorist_is_plugin_active' ) ) {
        function directorist_is_plugin_active( $plugin ) {
            return in_array( $plugin, (array) get_option( 'active_plugins', array() ), true ) || directorist_is_plugin_active_for_network( $plugin );
        }
    }
    
    if ( ! function_exists( 'directorist_is_plugin_active_for_network' ) ) {
        function directorist_is_plugin_active_for_network( $plugin ) {
            if ( ! is_multisite() ) {
                return false;
            }
                    
            $plugins = get_site_option( 'active_sitewide_plugins' );
            if ( isset( $plugins[ $plugin ] ) ) {
                    return true;
            }
    
            return false;
        }
    }

    /**
     * The main function for that returns Post_Your_Need
     *
     * The main function responsible for returning the one true Post_Your_Need
     * Instance to functions everywhere.
     *
     * Use this function like you would a global variable, except without needing
     * to declare the global.
     *
     *
     * @return object|Post_Your_Need The one true Post_Your_Need Instance.
     * @since 1.0
     */
    function Post_Your_Need()
    {
        return Post_Your_Need::instance();
    }

    // Instantiate Directorist Stripe gateway only if our directorist plugin is active
    if ( directorist_is_plugin_active( 'directorist/directorist-base.php' ) ) {
        Post_Your_Need(); // get the plugin running
    }
}
