<?php

namespace DirectoristDigitalMarketplace\Module\WooCommerce;

use DirectoristDigitalMarketplace\Module\WooCommerce\Model\Product as Product_Data;
use DirectoristDigitalMarketplace\Helper\Listing as Listing;
use DirectoristDigitalMarketplace\Global_Helpers as Global_Helpers;
use WP_Query;
use function DirectoristDigitalMarketplace\Global_Helpers\ddm_get_first_array_value;

class WC_Helper {

	/**
	 * Constuctor
	 * 
	 */
	function __construct() {		
	}

	/**
	 * Adds listing to cart.
	 *
	 * @param  object $listing Listing object.
	 * @param array  $args Form arguments.
	 * @param array  $meta Cart meta.
	 * @return bool
	 */
	public static function add_to_cart( $listing_id, $args = [], $meta = [] ) {

		// Get product.
		$product_q = new WP_Query([
			'post_type' => 'product',
			'post_parent' => $listing_id,
			'posts_per_page' => 1,
			'fields' => 'ids'
		]);
		$product = ddm_get_first_array_value( $product_q->posts );
		$listing = Listing::get_listing( $listing_id );
		
		if ( ! $product || get_post_status( $product ) !== 'publish' ) {
			return false;
		}

		// Filter cart.
		$cart = apply_filters(
			'directorist_digital_product_add_to_cart',
			[
				'args' => $args,
				'meta' => $meta,
			],
			$listing
		);

		$args = $cart['args'];
		$meta = $cart['meta'];
		
		// Get quantity.
		// $quantity = ! empty( $cart['args']['quantity'] ) ? $cart['args']['quantity'] : '';
		$quantity = max( 1, absint( ddm_get_first_array_value( $args, 'quantity' ) ) );

		// Get meta.
		$meta = array_merge(
			[
				'quantity'     => 0,
				'fees'     	   => [],
				'price_tier'   => null,
				'price_change' => 0,
			],
			$meta
		);

		$price = Listing::get_price( $listing_id );
		if ( $args['tiers'] ) {
			$tier	= Global_Helpers\ddm_get_first_array_value( Listing::get_price_tiers( $listing_id ) );
			$price	= $args['tiers'] ? floatval( $args['tiers'] ) : $tier;
		}
		$meta['price'] = $price;

		if ( $args['extras'] ) {

			// Get extras.
			$extras = [];

			if ( $meta['price_extras'] ) {
				$extras = $meta['price_extras'];

				$meta['price_extras'] = [];
			} elseif ( Listing::get_price_extras( $listing_id ) ) {
				$extra_ids = ddm_get_first_array_value( $args, '_extras', [] );

				if ( $extra_ids ) {
					$extras = array_intersect_key( Listing::get_price_extras( $listing_id ), array_flip( array_map( 'absint', (array) $extra_ids ) ) );
				}
			}

			if ( $extras ) {

				// Set meta.
				foreach ( $extras as $name => $price ) {
					$extra_price = floatval( $price );

					if ( $meta['quantity'] > 1 ) {
						$extra_price *= $meta['quantity'];
					}

					$meta['fees'][] = [
						'name'   => $name,
						'amount' => $extra_price,
					];

					$meta['price_change'] += $extra_price;

				}

			}
		}
		// return $meta;

		// Load cart.
		if ( is_null( WC()->cart ) ) {
			wc_load_cart();
		}
		
		// Empty cart.
		if( ! WC()->cart->is_empty() ) {
			WC()->cart->empty_cart();
		}

		// unset($meta['ddm_fees']);
		return [
			'meta' => $meta,
			'args' => $args,
			'quantity' => $quantity,
			'product' => $product,
			'cart' => WC()->cart->add_to_cart( $product, $quantity, 0, [], $meta ),
		];

		return WC()->cart->add_to_cart( $product, $quantity, 0, [], $meta );
	}


	/**
	 * Formats price.
	 *
	 * @param float $price Price.
	 * @return string
	 */
	public static function format_price( $price ) {
		return wp_strip_all_tags( wc_price( $price ) );
	}

	public static function commission_per_order( $order ) {
		if( ! is_object( $order ) ) {
			$order = wc_get_order( $order );
		}
		$total = $order->get_total();

		if( empty( $total ) ) {
			return;
		}
		return ( (int)$total * (int)self::commission_rate() ) / 100;
	}

	public static function commission_rate() {
		return round( floatval( get_directorist_option( 'vendor_commission_rate' ) ), 2 );
	}
		
	/**
	 * Formats price.
	 *
	 * @param float $price Price.
	 * @return string
	 */
	public static function update_vendor_balance( $vendor, $new_balance = '', $context = '+' ) {
		if ( is_object( $vendor ) ) {
			$vendor = $vendor->ID;
		}
		$available_payout = get_user_meta( $vendor, '_ddm_available_payout', true );
		if( '+' === $context ) {
			$available_payout += (int)$new_balance;
		}elseif( '-' === $context ) {
			$available_payout -= (int)$new_balance;
		}elseif( '#' === $context ) {
			$available_payout = 0;
		}

		update_user_meta( $vendor, '_ddm_available_payout', $available_payout );
		return $available_payout;
	}

	/**
	 * Update vendor earnings.
	 *
	 * @param string $vendor user id.
	 * @param array $orders Price.
	 * @return string $available_payout
	 */
	public static function update_vendor_balance_by_orders( $vendor, $orders ) {
		if ( empty( $orders ) ) {
			update_user_meta( $vendor, '_ddm_available_payout', 0 );
		}
		$available_payout = 0;
		foreach( $orders as $order ) {
			$order = ! is_object( $order ) ? wc_get_order( $order ) : $order;
			$available_payout += (int)$order->get_total() - (int)WC_Helper::commission_per_order( $order );
		}

		update_user_meta( $vendor, '_ddm_available_payout', $available_payout );
		return $available_payout;
	}

	public static function tier_file_rul( $price_tier, $listing_id ) {
		$tiers = Listing::get_price_tiers( $listing_id );
		foreach( $tiers as $tier ){
			if( $tier['price'] == $price_tier ) {
				return $tier['file'];
			}
		}
	}


}
