<?php
namespace DirectoristDigitalMarketplace\Module\WooCommerce;
use function DirectoristDigitalMarketplace\Global_Helpers\ddm_get_first_array_value;

use ATBDP_Permalink;
use DirectoristDigitalMarketplace\Module\WooCommerce\WC_Helper;
use DirectoristDigitalMarketplace\Global_Helpers as Helper;

class Order {
	/**
	 * Constuctor
	 * 
	 */
	function __construct() {
		add_action( 'woocommerce_new_order', [ $this, 'create_order' ], 10, 2 );
		add_filter( 'woocommerce_payment_complete_order_status', [ $this, 'set_order_status' ], 100, 3 );
		add_action( 'woocommerce_order_status_changed', [ $this, 'update_order_status' ], 10, 4 );
		
		// manage user downloads
		// add_filter( 'woocommerce_customer_available_downloads', [ $this, 'update_download_file' ], 10, 2 );

		// Manage refunds.
		// add_action( 'woocommerce_order_refunded', [ $this, 'create_refund' ], 10, 2 );
		// add_action( 'woocommerce_refund_deleted', [ $this, 'delete_refund' ], 10, 2 );
		
		// add_action( 'woocommerce_after_order_notes', [ $this, 'add_custom_checkout_hidden_field' ] );
	}

	public function add_custom_checkout_hidden_field( $checkout ) {

		$tier = ! empty( $_GET['ddm_tier'] ) ? directorist_clean( wp_unslash( $_GET['ddm_tier'] ) ) : '';
		if( $tier ) {
			echo '<div id="user_link_hidden_checkout_field">
			<input type="hidden" class="input-hidden" name="ddm_price_tier" id="ddm_price_tier" value="' . $tier . '">
			</div>';
		}
	}

	public function update_download_file( $downloads, $customer_id ) {
		
		if( ! empty( $downloads ) ) {
			foreach( $downloads as $key => $download ) {
				// Get order.
				$order = wc_get_order( $download['order_id'] );
				if ( $order->get_meta( 'ddm_vendor' ) ) {
					$tier = $order->get_meta( 'ddm_price_tier' );
					if( $tier ) {
						// Get item.
						$item = ddm_get_first_array_value( $order->get_items() );

						if ( ! $item || ! $item->get_product_id() ) {
							return;
						}
						
						// Get listing id
						$listing = get_post_parent( $item->get_product_id() );
						$listing_id = is_object( $listing ) ? $listing->ID : '';
						
						$file_url = WC_Helper::tier_file_rul( $tier, $listing_id );

						$downloads[$key]['download_url'] = $file_url;
					}
				}
			}
		}
		return $downloads;
	}

	/**
	 * Creates refund.
	 *
	 * @param int $order_id Order ID.
	 * @param int $refund_id Refund ID.
	 */
	public function create_refund( $order_id, $refund_id ) {

		// Get order.
		$order = wc_get_order( $order_id );

		if ( ! $order->get_meta( 'ddm_vendor' ) ) {
			return;
		}

		// Get user ID.
		$user_id = absint( get_post_field( 'post_author', $order_id ) );

		// Update refund.
		wp_update_post(
			[
				'ID'          => $refund_id,
				'post_author' => $user_id,
			]
		);

		update_post_meta( $refund_id, 'ddm_commission_rate', $order->get_meta( 'ddm_commission_rate' ) );

		// Update balance.
		$balance = WC_Helper::commission_per_order( $order );
		WC_Helper::update_vendor_balance( $user_id, $balance, '-' );
	}

	/**
	 * Deletes refund.
	 *
	 * @param int $refund_id Refund ID.
	 * @param int $order_id Order ID.
	 */
	public function delete_refund( $refund_id, $order_id ) {

		// Get order.
		$order = wc_get_order( $order_id );

		if ( ! $order->get_meta( 'hp_vendor' ) ) {
			return;
		}
		// Get user ID.
		$user_id = absint( get_post_field( 'post_author', $order_id ) );
		// Update balance.
		$balance = WC_Helper::commission_per_order( $order );
		WC_Helper::update_vendor_balance( $user_id, $balance );
	}

	/**
	 * Sets order status.
	 *
	 * @param string   $status Order status.
	 * @param int      $order_id Order ID.
	 * @param WC_Order $order Order object.
	 */
	public function set_order_status( $status, $order_id, $order ) {
		if ( $order->get_meta( 'ddm_vendor' ) && ! $order->has_downloadable_item() ) {
			$status = 'processing';
		}

		return $status;
	}

	/**
	 * Updates order status.
	 *
	 * @param int      $order_id Order ID.
	 * @param string   $old_status Old status.
	 * @param string   $new_status New status.
	 * @param WC_Order $order Order object.
	 */
	public function update_order_status( $order_id, $old_status, $new_status, $order ) {

		// Check vendor.
		if ( ! $order->get_meta( 'ddm_vendor' ) ) {
			return;
		}

		$vendor = get_post_field( 'post_author', $order->get_id() );

		if( ( $new_status === 'refunded' ) && ( 'completed' === $old_status ) ) {
			// Update balance.
			$balance = WC_Helper::commission_per_order( $order );
			WC_Helper::update_vendor_balance( $vendor, $balance, '-' );
		}

		if( $new_status === 'completed' ) {
			// Update balance.
			$balance = WC_Helper::commission_per_order( $order );
			WC_Helper::update_vendor_balance( $vendor, $balance );
			// add commission 
			$user = get_userdata( $vendor );
			$args = [
				'post_title' => 'New commssion from ' . $user->display_name,
				'meta_input' => [
					'_amount' 	=> $order->get_total(),
					'_earning' 	=> $balance,
					'_order' 	=> $order->get_id(),
				],
			];
			Helper\add_commission( $vendor, $args );
			// send notice to vendor
			$this->send_email_confirmation_to_vendor( $vendor, $order );
		}

	}

	public function send_email_confirmation_to_vendor( $vendor, $order ) {
		if( ! $vendor ) {
			return;
		}
		if( ! $order ) {
			return;
		}

		if( get_directorist_option( 'disable_email_notification' ) ) {
			return;
		}

		if ( is_object( $vendor ) ) {
			$vendor = $vendor->ID;
		}

		$item = ddm_get_first_array_value( $order->get_items() );

		if ( ! $item || ! $item->get_product_id() ) {
			return;
		}
		
		// Get listing id
		$listing_id = get_post_parent( $item->get_product_id() );
		
		if ( is_object( $listing_id ) ) {
			$listing_id = $listing_id->ID;
		}
		
		$user                   = get_userdata( $vendor );
		$site_name              = get_bloginfo( 'name' );
		$site_url               = get_bloginfo( 'url' );
		$current_user_email     = $user->user_email;
		$listing_title          = get_the_title( $listing_id );
		$listing_url            = get_permalink( $listing_id );
		$date_format            = get_option( 'date_format' );
		$time_format            = get_option( 'time_format' );
		$current_time           = current_time( 'timestamp' );
		$contact_email_subject  = get_directorist_option('email_sub_approved_order');
		$contact_email_body     = get_directorist_option('email_tmpl_approved_order');
		$placeholders = array(
			'==NAME=='            => $user->display_name,
			'==USERNAME=='        => $user->user_login,
			'==SITE_NAME=='       => $site_name,
			'==SITE_LINK=='       => sprintf( '<a href="%s">%s</a>', $site_url, $site_name ),
			'==SITE_URL=='        => sprintf( '<a href="%s">%s</a>', $site_url, $site_url ),
			'==LISTING_TITLE=='   => $listing_title,
			'==DASHBOARD_LINK=='  => ATBDP_Permalink::get_dashboard_page_link(),
			'==LISTING_LINK=='    => sprintf( '<a href="%s">%s</a>', $listing_url, $listing_title ),
			'==LISTING_URL=='     => sprintf( '<a href="%s">%s</a>', $listing_url, $listing_url ),
			'==SENDER_NAME=='     => $site_name,
			'==TODAY=='           => date_i18n( $date_format, $current_time ),
			'==NOW=='             => date_i18n( $date_format . ' ' . $time_format, $current_time )
		);

		$to      = $current_user_email;
		$subject = strtr( $contact_email_subject, $placeholders );
		$message = strtr( $contact_email_body, $placeholders );
		$message = nl2br( $message );

		$headers  = "From: {$site_name} <{$current_user_email}>\r\n";
		
		ATBDP()->email->send_mail( $to, $subject, $message, $headers );

	}

	/**
	 * Creates order.
	 *
	 * @param int      $order_id Order ID.
	 * @param WC_Order $order Order object.
	 */
	public function create_order( $order_id, $order ) {

		// Check order.
		if ( ! $order || $order->get_meta( 'ddm_vendor' ) ) {
			return;
		}

		// Get item.
		$item = ddm_get_first_array_value( $order->get_items() );

		if ( ! $item || ! $item->get_product_id() ) {
			return;
		}
		
		// Get listing id
		$listing_id = get_post_parent( $item->get_product_id() );
		

		if ( ! $listing_id ) {
			return;
		}

		// Get vendor.
		$vendor = get_post_field( 'post_author', $listing_id );

		if ( ! $vendor ) {
			return;
		}

		// Update order.
		wp_update_post(
			[
				'ID'          => $order->get_id(),
				'post_author' => $vendor,
			]
		);
		$tier = ! empty( $_POST['ddm_price_tier'] ) ? directorist_clean( wp_unslash( $_POST['ddm_price_tier'] ) ) : '';
		if( $tier ) {
			update_post_meta( $order->get_id(), 'ddm_price_tier', $tier );
		}
		update_post_meta( $order->get_id(), 'ddm_vendor', $vendor );
		update_post_meta( $order->get_id(), 'ddn_commission_rate', WC_Helper::commission_rate() );

		// $balance = WC_Helper::commission_per_order( $order );
		// WC_Helper::update_vendor_balance( $vendor, $balance, '#' );

	}

}